/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Ant", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */
package org.apache.commons.cactus.ant;

import java.net.*;
import java.io.*;

import org.apache.tools.ant.*;
import org.apache.tools.ant.taskdefs.*;

/**
 * A helper class for an Ant Task that does the following :
 * <ul>
 *   <li>create a java thread,</li>
 *   <li>start another Ant target in that thread. This target must be a
 *       blocking target that starts a web server/servlet engine,</li>
 *   <li>wait for that server to be started. This is done by continuously
 *       trying to call a URL.</li>
 * </ul>.
 *
 * @version @version@
 */
public class StartServerHelper implements Runnable
{
    /**
     * The URL that is continuously pinged to verify if the server is running.
     */
    private URL m_TestURL;

    /**
     * The Ant target name that will start the web server/servlet engine.
     */
    private String m_StartTarget;

    /**
     * The tasks that wraps around this helper class
     */
    private Task m_Task;

    /**
     *
     */
    public StartServerHelper(Task theTask)
    {
        m_Task = theTask;
    }

    /**
     * Executes the task.
     */
    public void execute() throws BuildException
    {
        // Verify that a test URL has been specified
        if (m_TestURL == null) {
            throw new BuildException("A testURL attribute must be specified");
        }

        // Verify that a start target has been specified
        if (m_StartTarget == null) {
            throw new BuildException("A startTarget Ant target name must be specified");
        }

        // Try connecting in case the server is already running. If so, does
        // nothing
        try {
            HttpURLConnection connection = (HttpURLConnection)m_TestURL.openConnection();
            connection.connect();
            readFully(connection);
            connection.disconnect();

            // Server is already running. Make this task a no-op.
            return;
        } catch (IOException e) {
            // An error occurred. It just measn the server is not running. Do
            // nothing        
        }

        // Call the target that starts the server, in another thread. The called
        // target must be blocking.
        Thread thread = new Thread(this);

        thread.start();

        // Wait a few ms more (just to make sur the servlet engine is
        // ready to accept connections)
        try {
            Thread.sleep(1000);
        } catch (InterruptedException e) {
            throw new BuildException("Interruption during sleep", e);
        }

        // Continuously try calling the test URL until it succeeds
        while (true) {

            try {
                HttpURLConnection connection = (HttpURLConnection)m_TestURL.openConnection();
                connection.connect();
                readFully(connection);
                connection.disconnect();
            } catch (IOException e) {
                try {
                    Thread.sleep(500);
                } catch (InterruptedException ee) {
                    throw new BuildException("Interruption during sleep", ee);
                }
                continue;
            }

            break;
        }

        // Wait a few ms more (just to be sure !)
        try {
            Thread.sleep(500);
        } catch (InterruptedException e) {
            throw new BuildException("Interruption during sleep", e);
        }

        // We're done ... Ant will continue processing other tasks
    }

    static void readFully(HttpURLConnection connection) throws IOException
    {
        // finish reading it to prevent (harmless) server-side exceptions
        BufferedInputStream is = new BufferedInputStream(connection.getInputStream());
        byte[] buffer = new byte[256];
        while((is.read(buffer)) > 0) {}
        is.close();
    }

    /**
     * The thread that calls the Ant target that starts the web server/servlet
     * engine. Must be a blocking target.
     */
    public void run()
    {
        // Call the Ant target using the "antcall" task.
        CallTarget callee;
        callee = (CallTarget)(m_Task.getProject().createTask("antcall"));
        callee.setOwningTarget(m_Task.getOwningTarget());
        callee.setTaskName(m_Task.getTaskName());
        callee.setLocation(m_Task.getLocation());

        callee.init();

        callee.setTarget(m_StartTarget);
        callee.execute();

        // Should never reach this point as the target is blocking, unless the
        // server is stopped.
    }

    /**
     * @param theTestURL the test URL to ping
     */
    public void setTestURL(String theTestURL)
    {
        try {
            m_TestURL = new URL(theTestURL);
        } catch (MalformedURLException e) {
            throw new BuildException("Bad URL [" + theTestURL + "]", e);
        }
    }

    /**
     * @param theStartTarget the Ant target to call
     */
    public void setStartTarget(String theStartTarget)
    {
        m_StartTarget = theStartTarget;
    }

}