/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Ant", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */
package org.apache.commons.cactus.client;

import java.io.*;
import java.net.HttpURLConnection;

import junit.framework.*;

/**
 * Unit tests of the <code>AutoReadHttpURLConnection</code> class.
 *
 * @version @version@
 */
public class TestAutoReadHttpURLConnection extends TestCase
{
    static final byte[] TEST_CONTENT = new String("Test content").getBytes();

    // used by tests to read the test content input stream
    byte[] buffer = new byte[TEST_CONTENT.length];

    MockHttpURLConnection mockConnection;
    AutoReadHttpURLConnection autoReader;

    /**
     * Defines the testcase name for JUnit.
     *
     * @param theName the testcase's name.
     */
    public TestAutoReadHttpURLConnection(String theName)
    {
        super(theName);
    }

    /**
     * Start the tests.
     *
     * @param theArgs the arguments. Not used
     */
    public static void main(String[] theArgs)
    {
        junit.ui.TestRunner.main(new String[]
             {TestAutoReadHttpURLConnection.class.getName()});
    }

    /**
     * Run prior to each test to create an autoreader of
     * a mock http url connection that returns test data
     */
    public void setUp() {
        mockConnection = new MockHttpURLConnection();
        autoReader = new AutoReadHttpURLConnection(mockConnection);
    }

    /**
     * Tests that the wrapped connection's input stream is automatically
     * read on first call to getInputStream
     */
    public void testReadFully() throws Exception {
        InputStream is = autoReader.getInputStream();
        assert("Expected mock InputStream to be fully read",
                 mockConnection.isInputStreamRead());

        is.read(buffer);
        assert("Expected actual InputStream to be fully read", -1 == is.read());
        assertEquals("InputStream content",
                new String(TEST_CONTENT),
                new String(buffer));
    }

    /**
     * Tests where the test data size is not exactly divisible by the chunk size
     */
    public void testChunkSizeRemainder() throws Exception {
        autoReader.chunkSize = TEST_CONTENT.length - 1;
        testReadFully();
    }

    /**
     * Tests where the test data size is exactly divisible by the chunk size
     */
    public void testChunkSizeDivisible() throws Exception {
        autoReader.chunkSize = 1;
        testReadFully();
    }
    

    // MockHttpURLConnection to test when input stream is read
    static class MockHttpURLConnection extends HttpURLConnection
    {
        ByteArrayInputStream is = new ByteArrayInputStream(TEST_CONTENT);
        
        MockHttpURLConnection()
        {
            super(null);
        }

        public InputStream getInputStream()
        {
            return is;
        }

        boolean isInputStreamRead()
        {
            return -1 == is.read();
        }

        // Dummy implementation of abstract methods
        public void connect()
        {
        }

        public void disconnect()
        {
        }
        
        public boolean usingProxy()
        {
            return false;
        }
        
    }
}
