/*
 * Copyright 2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License")
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.commons.configuration;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import junit.framework.TestCase;
import junitx.framework.ListAssert;

/**
 * Abstract TestCase for implementations of {@link AbstractConfiguration}.
 *
 * @author Emmanuel Bourg
 * @version $Revision: 155408 $, $Date: 2005-02-26 13:56:39 +0100 (Sa, 26 Feb 2005) $
 */
public abstract class TestAbstractConfiguration extends TestCase
{
    /**
     * Return an abstract configuration with 2 key/value pairs:<br>
     * <pre>
     * key1 = value1
     * key2 = value2
     * list = value1, value2
     * </pre>
     */
    protected abstract AbstractConfiguration getConfiguration();

    /**
     * Return an empty configuration.
     */
    protected abstract AbstractConfiguration getEmptyConfiguration();

    public void testGetProperty()
    {
        Configuration config = getConfiguration();
        assertEquals("key1", "value1", config.getProperty("key1"));
        assertEquals("key2", "value2", config.getProperty("key2"));
        assertNull("key3", config.getProperty("key3"));
    }

    public void testList()
    {
        Configuration config = getConfiguration();

        List list = config.getList("list");
        assertNotNull("list not found", config.getProperty("list"));
        assertEquals("list size", 2, list.size());
        assertTrue("'value1' is not in the list", list.contains("value1"));
        assertTrue("'value2' is not in the list", list.contains("value2"));
    }

    public void testAddPropertyDirect()
    {
        AbstractConfiguration config = getConfiguration();
        config.addPropertyDirect("key3", "value3");
        assertEquals("key3", "value3", config.getProperty("key3"));
    }

    public void testIsEmpty()
    {
        Configuration config = getConfiguration();
        assertFalse("the configuration is empty", config.isEmpty());
        assertTrue("the configuration is not empty", getEmptyConfiguration().isEmpty());
    }

    public void testContainsKey()
    {
        Configuration config = getConfiguration();
        assertTrue("key1 not found", config.containsKey("key1"));
        assertFalse("key3 found", config.containsKey("key3"));
    }

    public void testClearProperty()
    {
        Configuration config = getConfiguration();
        config.clearProperty("key2");
        assertFalse("key2 not cleared", config.containsKey("key2"));
    }

    public void testGetKeys()
    {
        Configuration config = getConfiguration();
        Iterator keys = config.getKeys();

        List expectedKeys = new ArrayList();
        expectedKeys.add("key1");
        expectedKeys.add("key2");
        expectedKeys.add("list");

        assertNotNull("null iterator", keys);
        assertTrue("empty iterator", keys.hasNext());

        List actualKeys = new ArrayList();
        while (keys.hasNext())
        {
            actualKeys.add(keys.next());
        }

        ListAssert.assertEquals("keys", expectedKeys, actualKeys);
    }

}
