/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.dbcp2;

import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.lang.management.ThreadMXBean;
import java.sql.Connection;
import java.sql.SQLException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Properties;
import java.util.Set;
import java.util.stream.Stream;
import javax.management.AttributeNotFoundException;
import javax.management.MBeanAttributeInfo;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.sql.DataSource;
import org.apache.commons.dbcp2.BasicDataSource;
import org.apache.commons.dbcp2.BasicDataSourceFactory;
import org.apache.commons.dbcp2.DelegatingConnection;
import org.apache.commons.dbcp2.PoolableConnectionFactory;
import org.apache.commons.dbcp2.StackMessageLog;
import org.apache.commons.dbcp2.TestConnectionPool;
import org.apache.commons.dbcp2.TesterClassLoader;
import org.apache.commons.dbcp2.TesterConnRequestCountDriver;
import org.apache.commons.dbcp2.TesterConnection;
import org.apache.commons.logging.LogFactory;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.Test;

public class TestBasicDataSource
extends TestConnectionPool {
    private static final String CATALOG = "test catalog";
    protected BasicDataSource ds;

    @BeforeAll
    public static void setUpClass() {
        LogFactory.getFactory().setAttribute("org.apache.commons.logging.Log", (Object)"org.apache.commons.dbcp2.StackMessageLog");
    }

    protected BasicDataSource createDataSource() throws Exception {
        return new BasicDataSource();
    }

    @Override
    protected Connection getConnection() throws Exception {
        return this.ds.getConnection();
    }

    @BeforeEach
    public void setUp() throws Exception {
        this.ds = this.createDataSource();
        this.ds.setDriverClassName("org.apache.commons.dbcp2.TesterDriver");
        this.ds.setUrl("jdbc:apache:commons:testdriver");
        this.ds.setMaxTotal(this.getMaxTotal());
        this.ds.setMaxWait(this.getMaxWaitDuration());
        this.ds.setDefaultAutoCommit(Boolean.TRUE);
        this.ds.setDefaultReadOnly(Boolean.FALSE);
        this.ds.setDefaultTransactionIsolation(2);
        this.ds.setDefaultCatalog(CATALOG);
        this.ds.setUsername("userName");
        this.ds.setPassword("password");
        this.ds.setValidationQuery("SELECT DUMMY FROM DUAL");
        this.ds.setConnectionInitSqls(Arrays.asList("SELECT 1", "SELECT 2"));
        this.ds.setDriverClassLoader((ClassLoader)new TesterClassLoader());
        this.ds.setJmxName("org.apache.commons.dbcp2:name=test");
    }

    @Override
    @AfterEach
    public void tearDown() throws Exception {
        super.tearDown();
        this.ds.close();
        this.ds = null;
    }

    @Test
    public void testAccessToUnderlyingConnectionAllowed() throws Exception {
        this.ds.setAccessToUnderlyingConnectionAllowed(true);
        Assertions.assertTrue((boolean)this.ds.isAccessToUnderlyingConnectionAllowed());
        try (Connection conn = this.getConnection();){
            Connection dconn = ((DelegatingConnection)conn).getDelegate();
            Assertions.assertNotNull((Object)dconn);
            dconn = ((DelegatingConnection)conn).getInnermostDelegate();
            Assertions.assertNotNull((Object)dconn);
            Assertions.assertTrue((boolean)(dconn instanceof TesterConnection));
        }
    }

    @Test
    public void testClose() throws Exception {
        this.ds.setAccessToUnderlyingConnectionAllowed(true);
        Connection activeConnection = this.getConnection();
        Connection rawActiveConnection = ((DelegatingConnection)activeConnection).getInnermostDelegate();
        Assertions.assertFalse((boolean)activeConnection.isClosed());
        Assertions.assertFalse((boolean)rawActiveConnection.isClosed());
        Connection idleConnection = this.getConnection();
        Connection rawIdleConnection = ((DelegatingConnection)idleConnection).getInnermostDelegate();
        Assertions.assertFalse((boolean)idleConnection.isClosed());
        Assertions.assertFalse((boolean)rawIdleConnection.isClosed());
        idleConnection.close();
        Assertions.assertTrue((boolean)idleConnection.isClosed());
        Assertions.assertFalse((boolean)rawIdleConnection.isClosed());
        this.ds.close();
        Assertions.assertTrue((boolean)rawIdleConnection.isClosed());
        Assertions.assertFalse((boolean)activeConnection.isClosed());
        Assertions.assertFalse((boolean)rawActiveConnection.isClosed());
        activeConnection.close();
        Assertions.assertTrue((boolean)activeConnection.isClosed());
        Assertions.assertTrue((boolean)rawActiveConnection.isClosed());
        Assertions.assertThrows(SQLException.class, () -> this.getConnection());
        this.ds.close();
    }

    @Test
    public void testConcurrentInitBorrow() throws Exception {
        this.ds.setDriverClassName("org.apache.commons.dbcp2.TesterConnectionDelayDriver");
        this.ds.setUrl("jdbc:apache:commons:testerConnectionDelayDriver:50");
        this.ds.setInitialSize(8);
        TestConnectionPool.TestThread testThread = new TestConnectionPool.TestThread(this, 1, 0);
        Thread t = new Thread(testThread);
        t.start();
        Thread.sleep(100L);
        try (Connection conn = this.ds.getConnection();){
            Assertions.assertTrue((this.ds.getConnectionPool().getNumIdle() > 5 ? 1 : 0) != 0);
            t.join();
            Assertions.assertFalse((boolean)testThread.failed());
        }
        this.ds.close();
    }

    @Test
    public void testConcurrentInvalidateBorrow() throws Exception {
        this.ds.setDriverClassName("org.apache.commons.dbcp2.TesterConnRequestCountDriver");
        this.ds.setUrl("jdbc:apache:commons:testerConnRequestCountDriver");
        this.ds.setTestOnBorrow(true);
        this.ds.setValidationQuery("SELECT DUMMY FROM DUAL");
        this.ds.setMaxTotal(8);
        this.ds.setLifo(true);
        this.ds.setMaxWait(Duration.ofMillis(-1L));
        TestConnectionPool.TestThread testThread1 = new TestConnectionPool.TestThread(this, 1000, 0);
        Thread t1 = new Thread(testThread1);
        t1.start();
        TestConnectionPool.TestThread testThread2 = new TestConnectionPool.TestThread(this, 1000, 0);
        Thread t2 = new Thread(testThread1);
        t2.start();
        for (int i = 0; i < 1000; ++i) {
            Connection conn = this.ds.getConnection();
            this.ds.invalidateConnection(conn);
        }
        t1.join();
        t2.join();
        Assertions.assertFalse((boolean)testThread1.failed());
        Assertions.assertFalse((boolean)testThread2.failed());
        this.ds.close();
    }

    @Test
    public void testCreateConnectionFactoryWithConnectionFactoryClassName() throws Exception {
        Properties properties = new Properties();
        properties = new Properties();
        properties.put("initialSize", "1");
        properties.put("driverClassName", "org.apache.commons.dbcp2.TesterDriver");
        properties.put("url", "jdbc:apache:commons:testdriver");
        properties.put("username", "foo");
        properties.put("password", "bar");
        properties.put("connectionFactoryClassName", "org.apache.commons.dbcp2.TesterConnectionFactory");
        try (BasicDataSource ds = BasicDataSourceFactory.createDataSource((Properties)properties);
             Connection conn = ds.getConnection();){
            Assertions.assertNotNull((Object)conn);
        }
    }

    @Test
    public void testCreateConnectionFactoryWithoutConnectionFactoryClassName() throws Exception {
        Properties properties = new Properties();
        properties.put("initialSize", "1");
        properties.put("driverClassName", "org.apache.commons.dbcp2.TesterDriver");
        properties.put("url", "jdbc:apache:commons:testdriver");
        properties.put("username", "foo");
        properties.put("password", "bar");
        try (BasicDataSource ds = BasicDataSourceFactory.createDataSource((Properties)properties);
             Connection conn = ds.getConnection();){
            Assertions.assertNotNull((Object)conn);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testCreateDataSourceCleanupEvictor() throws Exception {
        this.ds.close();
        this.ds = null;
        this.ds = this.createDataSource();
        this.ds.setDriverClassName("org.apache.commons.dbcp2.TesterConnRequestCountDriver");
        this.ds.setUrl("jdbc:apache:commons:testerConnRequestCountDriver");
        this.ds.setValidationQuery("SELECT DUMMY FROM DUAL");
        this.ds.setUsername("userName");
        this.ds.setPassword("wrong");
        this.ds.setDurationBetweenEvictionRuns(Duration.ofMillis(100L));
        this.ds.setMinIdle(2);
        Class<TesterConnRequestCountDriver> clazz = TesterConnRequestCountDriver.class;
        synchronized (TesterConnRequestCountDriver.class) {
            TesterConnRequestCountDriver.initConnRequestCount();
            for (int i = 0; i < 10; ++i) {
                try {
                    DataSource dataSource = this.ds.createDataSource();
                    continue;
                }
                catch (SQLException sQLException) {
                    // empty catch block
                }
            }
            Thread.sleep(1000L);
            Assertions.assertEquals((int)10, (int)TesterConnRequestCountDriver.getConnectionRequestCount());
            // ** MonitorExit[var1_1] (shouldn't be in output)
            Assertions.assertNull((Object)this.ds.getConnectionPool());
            return;
        }
    }

    @Test
    public void testCreateDataSourceCleanupThreads() throws Exception {
        this.ds.close();
        this.ds = null;
        this.ds = this.createDataSource();
        this.ds.setDriverClassName("org.apache.commons.dbcp2.TesterDriver");
        this.ds.setUrl("jdbc:apache:commons:testdriver");
        this.ds.setMaxTotal(this.getMaxTotal());
        this.ds.setMaxWait(this.getMaxWaitDuration());
        this.ds.setDefaultAutoCommit(Boolean.TRUE);
        this.ds.setDefaultReadOnly(Boolean.FALSE);
        this.ds.setDefaultTransactionIsolation(2);
        this.ds.setDefaultCatalog(CATALOG);
        this.ds.setUsername("userName");
        this.ds.setDurationBetweenEvictionRuns(Duration.ofMillis(100L));
        this.ds.setPassword("wrong");
        this.ds.setValidationQuery("SELECT DUMMY FROM DUAL");
        int threadCount = Thread.activeCount();
        for (int i = 0; i < 10; ++i) {
            try {
                Connection c = this.ds.getConnection();
                if (c == null) continue;
                c.close();
                continue;
            }
            catch (SQLException sQLException) {
                // empty catch block
            }
        }
        Assertions.assertTrue((Thread.activeCount() <= threadCount + 1 ? 1 : 0) != 0);
    }

    @Test
    public void testDefaultCatalog() throws Exception {
        Connection[] c = new Connection[this.getMaxTotal()];
        for (int i = 0; i < c.length; ++i) {
            c[i] = this.getConnection();
            Assertions.assertNotNull((Object)c[i]);
            Assertions.assertEquals((Object)CATALOG, (Object)c[i].getCatalog());
        }
        for (Connection element : c) {
            element.setCatalog("error");
            element.close();
        }
        for (int i = 0; i < c.length; ++i) {
            c[i] = this.getConnection();
            Assertions.assertNotNull((Object)c[i]);
            Assertions.assertEquals((Object)CATALOG, (Object)c[i].getCatalog());
        }
        for (Connection element : c) {
            element.close();
        }
    }

    @Test
    public void testDeprecatedAccessors() throws SQLException {
        try (BasicDataSource bds = new BasicDataSource();){
            int i = 0;
            bds.setDefaultQueryTimeout(Integer.valueOf(++i));
            Assertions.assertEquals((int)i, (Integer)bds.getDefaultQueryTimeout());
            Assertions.assertEquals((Object)Duration.ofSeconds(i), (Object)bds.getDefaultQueryTimeoutDuration());
            bds.setMaxConnLifetimeMillis((long)(++i));
            Assertions.assertEquals((long)i, (long)bds.getMaxConnLifetimeMillis());
            Assertions.assertEquals((Object)Duration.ofMillis(i), (Object)bds.getMaxConnDuration());
            bds.setMaxWaitMillis((long)(++i));
            Assertions.assertEquals((long)i, (long)bds.getMaxWaitMillis());
            Assertions.assertEquals((Object)Duration.ofMillis(i), (Object)bds.getMaxWaitDuration());
            bds.setMinEvictableIdleTimeMillis((long)(++i));
            Assertions.assertEquals((long)i, (long)bds.getMinEvictableIdleTimeMillis());
            Assertions.assertEquals((Object)Duration.ofMillis(i), (Object)bds.getMinEvictableIdleDuration());
            bds.setRemoveAbandonedTimeout(++i);
            Assertions.assertEquals((int)i, (int)bds.getRemoveAbandonedTimeout());
            Assertions.assertEquals((Object)Duration.ofSeconds(i), (Object)bds.getRemoveAbandonedTimeoutDuration());
            bds.setSoftMinEvictableIdleTimeMillis((long)(++i));
            Assertions.assertEquals((long)i, (long)bds.getSoftMinEvictableIdleTimeMillis());
            Assertions.assertEquals((Object)Duration.ofMillis(i), (Object)bds.getSoftMinEvictableIdleDuration());
            bds.setTimeBetweenEvictionRunsMillis((long)(++i));
            Assertions.assertEquals((long)i, (long)bds.getTimeBetweenEvictionRunsMillis());
            Assertions.assertEquals((Object)Duration.ofMillis(i), (Object)bds.getDurationBetweenEvictionRuns());
            ++i;
            bds.setValidationQueryTimeout(1);
            Assertions.assertEquals((int)1, (int)bds.getValidationQueryTimeout());
            Assertions.assertEquals((Object)Duration.ofSeconds(1L), (Object)bds.getValidationQueryTimeoutDuration());
        }
    }

    @Test
    public void testDisconnectSqlCodes() throws Exception {
        ArrayList<String> disconnectionSqlCodes = new ArrayList<String>();
        disconnectionSqlCodes.add("XXX");
        this.ds.setDisconnectionSqlCodes(disconnectionSqlCodes);
        this.ds.setFastFailValidation(true);
        try (Connection conn = this.ds.getConnection();){
            PoolableConnectionFactory pcf = (PoolableConnectionFactory)this.ds.getConnectionPool().getFactory();
            Assertions.assertTrue((boolean)pcf.isFastFailValidation());
            Assertions.assertTrue((boolean)pcf.getDisconnectionSqlCodes().contains("XXX"));
            Assertions.assertEquals((int)1, (int)pcf.getDisconnectionSqlCodes().size());
        }
    }

    @Test
    public void testDriverClassLoader() throws Exception {
        try (Connection conn = this.getConnection();){
            ClassLoader cl = this.ds.getDriverClassLoader();
            Assertions.assertNotNull((Object)cl);
            Assertions.assertTrue((boolean)(cl instanceof TesterClassLoader));
            Assertions.assertTrue((boolean)((TesterClassLoader)cl).didLoad(this.ds.getDriverClassName()));
        }
    }

    @Test
    public void testEmptyInitConnectionSql() throws Exception {
        this.ds.setConnectionInitSqls(Arrays.asList("", "   "));
        Assertions.assertNotNull((Object)this.ds.getConnectionInitSqls());
        Assertions.assertEquals((int)0, (int)this.ds.getConnectionInitSqls().size());
        this.ds.setConnectionInitSqls(null);
        Assertions.assertNotNull((Object)this.ds.getConnectionInitSqls());
        Assertions.assertEquals((int)0, (int)this.ds.getConnectionInitSqls().size());
    }

    @Test
    public void testEmptyValidationQuery() throws Exception {
        Assertions.assertNotNull((Object)this.ds.getValidationQuery());
        this.ds.setValidationQuery("");
        Assertions.assertNull((Object)this.ds.getValidationQuery());
        this.ds.setValidationQuery("   ");
        Assertions.assertNull((Object)this.ds.getValidationQuery());
    }

    @Test
    @Disabled
    public void testEvict() throws Exception {
        long delay = 1000L;
        this.ds.setInitialSize(10);
        this.ds.setMaxIdle(10);
        this.ds.setMaxTotal(10);
        this.ds.setMinIdle(5);
        this.ds.setNumTestsPerEvictionRun(3);
        this.ds.setMinEvictableIdle(Duration.ofMillis(100L));
        this.ds.setDurationBetweenEvictionRuns(Duration.ofMillis(1000L));
        this.ds.setPoolPreparedStatements(true);
        Connection conn = this.ds.getConnection();
        if (conn != null) {
            conn.close();
        }
        ThreadMXBean threadBean = ManagementFactory.getThreadMXBean();
        while (Stream.of(threadBean.getThreadInfo(threadBean.getAllThreadIds())).anyMatch(t -> t.getThreadName().equals("commons-pool-evictor-thread")) && this.ds.getNumIdle() > this.ds.getMinIdle()) {
            Thread.sleep(1000L);
        }
        if (this.ds.getNumIdle() > this.ds.getMinIdle()) {
            Assertions.fail((String)("EvictionTimer thread was destroyed with numIdle=" + this.ds.getNumIdle() + "(expected: less or equal than " + this.ds.getMinIdle() + ")"));
        }
    }

    @Test
    public void testInitialSize() throws Exception {
        this.ds.setMaxTotal(20);
        this.ds.setMaxIdle(20);
        this.ds.setInitialSize(10);
        try (Connection conn = this.getConnection();){
            Assertions.assertNotNull((Object)conn);
        }
        Assertions.assertEquals((int)0, (int)this.ds.getNumActive());
        Assertions.assertEquals((int)10, (int)this.ds.getNumIdle());
    }

    @Test
    public void testInstanceNotFoundExceptionLogSuppressed() throws Exception {
        ObjectName objectName;
        MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
        Connection c = this.ds.getConnection();
        if (c != null) {
            c.close();
        }
        if (mbs.isRegistered(objectName = new ObjectName(this.ds.getJmxName()))) {
            mbs.unregisterMBean(objectName);
        }
        StackMessageLog.clear();
        this.ds.close();
        MatcherAssert.assertThat((Object)StackMessageLog.popMessage(), (Matcher)CoreMatchers.not((Matcher)CoreMatchers.containsString((String)"InstanceNotFoundException")));
        Assertions.assertNull((Object)this.ds.getRegisteredJmxName());
    }

    @Test
    public void testInvalidateConnection() throws Exception {
        this.ds.setMaxTotal(2);
        try (Connection conn1 = this.ds.getConnection();
             Connection conn2 = this.ds.getConnection();){
            this.ds.invalidateConnection(conn1);
            Assertions.assertTrue((boolean)conn1.isClosed());
            Assertions.assertEquals((int)1, (int)this.ds.getNumActive());
            Assertions.assertEquals((int)0, (int)this.ds.getNumIdle());
            try (Connection conn3 = this.ds.getConnection();){
                conn2.close();
            }
        }
    }

    @Test
    public void testInvalidConnectionInitSql() {
        this.ds.setConnectionInitSqls(Arrays.asList("SELECT 1", "invalid"));
        SQLException e = (SQLException)Assertions.assertThrows(SQLException.class, () -> ((BasicDataSource)this.ds).getConnection());
        Assertions.assertTrue((boolean)e.toString().contains("invalid"));
    }

    @Test
    public void testInvalidValidationQuery() {
        this.ds.setValidationQuery("invalid");
        SQLException e = (SQLException)Assertions.assertThrows(SQLException.class, () -> ((BasicDataSource)this.ds).getConnection());
        Assertions.assertTrue((boolean)e.toString().contains("invalid"));
    }

    @Test
    public void testIsClosedFailure() throws SQLException {
        this.ds.setAccessToUnderlyingConnectionAllowed(true);
        Connection conn = this.ds.getConnection();
        Assertions.assertNotNull((Object)conn);
        Assertions.assertEquals((int)1, (int)this.ds.getNumActive());
        TesterConnection tconn = (TesterConnection)((DelegatingConnection)conn).getInnermostDelegate();
        tconn.setFailure(new IOException("network error"));
        Assertions.assertThrows(SQLException.class, () -> conn.close());
        Assertions.assertEquals((int)0, (int)this.ds.getNumActive());
    }

    @Test
    public void testIsWrapperFor() throws Exception {
        Assertions.assertTrue((boolean)this.ds.isWrapperFor(BasicDataSource.class));
        Assertions.assertTrue((boolean)this.ds.isWrapperFor(AutoCloseable.class));
        Assertions.assertFalse((boolean)this.ds.isWrapperFor(String.class));
        Assertions.assertFalse((boolean)this.ds.isWrapperFor(null));
    }

    @Test
    public void testJmxDisabled() throws Exception {
        MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
        ObjectName commons = new ObjectName("org.apache.commons.*:*");
        Set<ObjectName> results = mbs.queryNames(commons, null);
        for (ObjectName result : results) {
            mbs.unregisterMBean(result);
        }
        this.ds.setJmxName(null);
        this.ds.setPoolPreparedStatements(true);
        try (Connection conn = this.ds.getConnection();){
            Assertions.assertEquals((int)0, (int)mbs.queryNames(commons, null).size());
        }
    }

    @Test
    public void testConnectionMBeansDisabled() throws Exception {
        MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
        ObjectName commons = new ObjectName("org.apache.commons.*:*");
        Set<ObjectName> results = mbs.queryNames(commons, null);
        for (ObjectName result : results) {
            mbs.unregisterMBean(result);
        }
        this.ds.setRegisterConnectionMBean(false);
        try (Connection conn = this.ds.getConnection();){
            ObjectName connections = new ObjectName("org.apache.commons.*:connection=*,*");
            Assertions.assertEquals((int)0, (int)mbs.queryNames(connections, null).size());
        }
    }

    @Test
    public void testJmxDoesNotExposePassword() throws Exception {
        ObjectName objectName;
        MBeanAttributeInfo[] attributes;
        MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
        Connection c = this.ds.getConnection();
        if (c != null) {
            c.close();
        }
        Assertions.assertTrue(((attributes = mbs.getMBeanInfo(objectName = new ObjectName(this.ds.getJmxName())).getAttributes()) != null && attributes.length > 0 ? 1 : 0) != 0);
        Arrays.asList(attributes).forEach(attrInfo -> Assertions.assertFalse((boolean)"password".equalsIgnoreCase(attrInfo.getName())));
        Assertions.assertThrows(AttributeNotFoundException.class, () -> mbs.getAttribute(objectName, "Password"));
    }

    @Test
    public void testManualConnectionEvict() throws Exception {
        this.ds.setMinIdle(0);
        this.ds.setMaxIdle(4);
        this.ds.setMinEvictableIdle(Duration.ofMillis(10L));
        this.ds.setNumTestsPerEvictionRun(2);
        try (Connection ds2 = this.ds.createDataSource().getConnection();
             Connection ds3 = this.ds.createDataSource().getConnection();){
            Assertions.assertEquals((int)0, (int)this.ds.getNumIdle());
        }
        Thread.sleep(100L);
        Assertions.assertEquals((int)2, (int)this.ds.getNumIdle());
        this.ds.evict();
        Assertions.assertEquals((int)0, (int)this.ds.getNumIdle());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testMaxConnLifetimeExceeded() throws Exception {
        try {
            StackMessageLog.lock();
            this.ds.setMaxConn(Duration.ofMillis(100L));
            try (Connection conn = this.ds.getConnection();){
                Assertions.assertEquals((int)1, (int)this.ds.getNumActive());
                Thread.sleep(500L);
            }
            Assertions.assertEquals((int)0, (int)this.ds.getNumIdle());
            String message = StackMessageLog.popMessage();
            Assertions.assertNotNull((Object)message);
            Assertions.assertTrue((message.indexOf("exceeds the maximum permitted value") > 0 ? 1 : 0) != 0);
        }
        finally {
            StackMessageLog.clear();
            StackMessageLog.unLock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testMaxConnLifetimeExceededMutedLog() throws Exception {
        try {
            StackMessageLog.lock();
            StackMessageLog.clear();
            this.ds.setMaxConn(Duration.ofMillis(100L));
            this.ds.setLogExpiredConnections(false);
            try (Connection conn = this.ds.getConnection();){
                Assertions.assertEquals((int)1, (int)this.ds.getNumActive());
                Thread.sleep(500L);
            }
            Assertions.assertEquals((int)0, (int)this.ds.getNumIdle());
            Assertions.assertTrue((boolean)StackMessageLog.isEmpty(), (String)StackMessageLog.getAll().toString());
        }
        finally {
            StackMessageLog.clear();
            StackMessageLog.unLock();
        }
    }

    @Test
    public void testMaxTotalZero() throws Exception {
        this.ds.setMaxTotal(0);
        Assertions.assertThrows(SQLException.class, () -> ((BasicDataSource)this.ds).getConnection());
    }

    @Test
    public void testMutateAbandonedConfig() throws Exception {
        Properties properties = new Properties();
        properties.put("initialSize", "1");
        properties.put("driverClassName", "org.apache.commons.dbcp2.TesterDriver");
        properties.put("url", "jdbc:apache:commons:testdriver");
        properties.put("username", "foo");
        properties.put("password", "bar");
        try (BasicDataSource ds = BasicDataSourceFactory.createDataSource((Properties)properties);){
            boolean original = ds.getConnectionPool().getLogAbandoned();
            ds.setLogAbandoned(!original);
            Assertions.assertNotEquals((Object)original, (Object)ds.getConnectionPool().getLogAbandoned());
        }
    }

    @Test
    public void testNoAccessToUnderlyingConnectionAllowed() throws Exception {
        Assertions.assertFalse((boolean)this.ds.isAccessToUnderlyingConnectionAllowed());
        try (Connection conn = this.getConnection();){
            Connection dconn = ((DelegatingConnection)conn).getDelegate();
            Assertions.assertNull((Object)dconn);
            dconn = ((DelegatingConnection)conn).getInnermostDelegate();
            Assertions.assertNull((Object)dconn);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testPoolCloseCheckedException() throws Exception {
        TesterConnection tc;
        this.ds.setAccessToUnderlyingConnectionAllowed(true);
        try (Connection conn = this.ds.getConnection();){
            tc = (TesterConnection)((DelegatingConnection)conn).getInnermostDelegate();
        }
        tc.setFailure(new SQLException("bang"));
        try {
            StackMessageLog.lock();
            StackMessageLog.clear();
            this.ds.close();
            String message = StackMessageLog.popMessage();
            Assertions.assertNotNull((Object)message);
            Assertions.assertTrue((message.indexOf("bang") > 0 ? 1 : 0) != 0);
        }
        catch (SQLException ex) {
            Assertions.assertTrue((ex.getMessage().indexOf("Cannot close") > 0 ? 1 : 0) != 0);
            Assertions.assertTrue((ex.getCause().getMessage().indexOf("bang") > 0 ? 1 : 0) != 0);
        }
        finally {
            StackMessageLog.unLock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testPoolCloseRTE() throws Exception {
        TesterConnection tc;
        this.ds.setAccessToUnderlyingConnectionAllowed(true);
        try (Connection conn = this.ds.getConnection();){
            tc = (TesterConnection)((DelegatingConnection)conn).getInnermostDelegate();
        }
        tc.setFailure(new IllegalStateException("boom"));
        try {
            StackMessageLog.lock();
            StackMessageLog.clear();
            this.ds.close();
            String message = StackMessageLog.popMessage();
            Assertions.assertNotNull((Object)message);
            Assertions.assertTrue((message.indexOf("boom") > 0 ? 1 : 0) != 0);
        }
        catch (IllegalStateException ex) {
            Assertions.assertTrue((ex.getMessage().indexOf("boom") > 0 ? 1 : 0) != 0);
        }
        finally {
            StackMessageLog.unLock();
        }
    }

    @Override
    @Test
    public void testPooling() throws Exception {
        this.ds.setAccessToUnderlyingConnectionAllowed(true);
        super.testPooling();
    }

    @Test
    public void testPropertyTestOnReturn() throws Exception {
        this.ds.setValidationQuery("select 1 from dual");
        this.ds.setTestOnBorrow(false);
        this.ds.setTestWhileIdle(false);
        this.ds.setTestOnReturn(true);
        try (Connection conn = this.ds.getConnection();){
            Assertions.assertNotNull((Object)conn);
            Assertions.assertFalse((boolean)this.ds.getConnectionPool().getTestOnBorrow());
            Assertions.assertFalse((boolean)this.ds.getConnectionPool().getTestWhileIdle());
            Assertions.assertTrue((boolean)this.ds.getConnectionPool().getTestOnReturn());
        }
    }

    @Test
    public void testRestart() throws Exception {
        this.ds.setMaxTotal(2);
        this.ds.setDurationBetweenEvictionRuns(Duration.ofMillis(100L));
        this.ds.setNumTestsPerEvictionRun(2);
        this.ds.setMinEvictableIdle(Duration.ofMinutes(1L));
        this.ds.setInitialSize(2);
        this.ds.setDefaultCatalog("foo");
        try (Connection conn1 = this.ds.getConnection();){
            Thread.sleep(200L);
            this.ds.setDefaultCatalog("bar");
            this.ds.setInitialSize(1);
            this.ds.restart();
            Assertions.assertEquals((Object)"bar", (Object)this.ds.getDefaultCatalog());
            Assertions.assertEquals((int)1, (int)this.ds.getInitialSize());
            this.ds.getLogWriter();
            Assertions.assertEquals((int)0, (int)this.ds.getNumActive());
            Assertions.assertEquals((int)1, (int)this.ds.getNumIdle());
        }
        Assertions.assertEquals((int)1, (int)this.ds.getNumIdle());
        this.ds.close();
    }

    @Test
    public void testRollbackReadOnly() throws Exception {
        this.ds.setDefaultReadOnly(Boolean.TRUE);
        this.ds.setDefaultAutoCommit(Boolean.FALSE);
        try (Connection conn = this.ds.getConnection();){
            Assertions.assertNotNull((Object)conn);
        }
    }

    @Test
    public void testSetAutoCommitTrueOnClose() throws Exception {
        Connection dconn;
        this.ds.setAccessToUnderlyingConnectionAllowed(true);
        this.ds.setDefaultAutoCommit(Boolean.FALSE);
        try (Connection conn = this.getConnection();){
            Assertions.assertNotNull((Object)conn);
            Assertions.assertFalse((boolean)conn.getAutoCommit());
            dconn = ((DelegatingConnection)conn).getInnermostDelegate();
            Assertions.assertNotNull((Object)dconn);
            Assertions.assertFalse((boolean)dconn.getAutoCommit());
        }
        Assertions.assertTrue((boolean)dconn.getAutoCommit());
    }

    @Test
    public void testSetProperties() throws Exception {
        this.ds.setConnectionProperties("name1=value1;name2=value2;name3=value3");
        Assertions.assertEquals((int)3, (int)this.ds.getConnectionProperties().size());
        Assertions.assertEquals((Object)"value1", (Object)this.ds.getConnectionProperties().getProperty("name1"));
        Assertions.assertEquals((Object)"value2", (Object)this.ds.getConnectionProperties().getProperty("name2"));
        Assertions.assertEquals((Object)"value3", (Object)this.ds.getConnectionProperties().getProperty("name3"));
        this.ds.setConnectionProperties("name1=value1;name2=value2");
        Assertions.assertEquals((int)2, (int)this.ds.getConnectionProperties().size());
        Assertions.assertEquals((Object)"value1", (Object)this.ds.getConnectionProperties().getProperty("name1"));
        Assertions.assertEquals((Object)"value2", (Object)this.ds.getConnectionProperties().getProperty("name2"));
        Assertions.assertFalse((boolean)this.ds.getConnectionProperties().containsKey("name3"));
        this.ds.setConnectionProperties("name1=value1;name2");
        Assertions.assertEquals((int)2, (int)this.ds.getConnectionProperties().size());
        Assertions.assertEquals((Object)"value1", (Object)this.ds.getConnectionProperties().getProperty("name1"));
        Assertions.assertEquals((Object)"", (Object)this.ds.getConnectionProperties().getProperty("name2"));
        this.ds.setConnectionProperties("name1=value1;name2=");
        Assertions.assertEquals((int)2, (int)this.ds.getConnectionProperties().size());
        Assertions.assertEquals((Object)"value1", (Object)this.ds.getConnectionProperties().getProperty("name1"));
        Assertions.assertEquals((Object)"", (Object)this.ds.getConnectionProperties().getProperty("name2"));
        this.ds.setConnectionProperties("name1=value1");
        Assertions.assertEquals((int)1, (int)this.ds.getConnectionProperties().size());
        Assertions.assertEquals((Object)"value1", (Object)this.ds.getConnectionProperties().getProperty("name1"));
        this.ds.setConnectionProperties("name1=value1;");
        Assertions.assertEquals((int)1, (int)this.ds.getConnectionProperties().size());
        Assertions.assertEquals((Object)"value1", (Object)this.ds.getConnectionProperties().getProperty("name1"));
        this.ds.setConnectionProperties("name1");
        Assertions.assertEquals((int)1, (int)this.ds.getConnectionProperties().size());
        Assertions.assertEquals((Object)"", (Object)this.ds.getConnectionProperties().getProperty("name1"));
        Assertions.assertThrows(NullPointerException.class, () -> this.ds.setConnectionProperties(null));
    }

    @Test
    public void testSetValidationTestProperties() {
        Assertions.assertTrue((boolean)this.ds.getTestOnBorrow());
        Assertions.assertFalse((boolean)this.ds.getTestOnReturn());
        Assertions.assertFalse((boolean)this.ds.getTestWhileIdle());
        this.ds.setTestOnBorrow(true);
        this.ds.setTestOnReturn(true);
        this.ds.setTestWhileIdle(true);
        Assertions.assertTrue((boolean)this.ds.getTestOnBorrow());
        Assertions.assertTrue((boolean)this.ds.getTestOnReturn());
        Assertions.assertTrue((boolean)this.ds.getTestWhileIdle());
        this.ds.setTestOnBorrow(false);
        this.ds.setTestOnReturn(false);
        this.ds.setTestWhileIdle(false);
        Assertions.assertFalse((boolean)this.ds.getTestOnBorrow());
        Assertions.assertFalse((boolean)this.ds.getTestOnReturn());
        Assertions.assertFalse((boolean)this.ds.getTestWhileIdle());
    }

    @Test
    public void testStart() throws Exception {
        this.ds.setAccessToUnderlyingConnectionAllowed(true);
        this.ds.setMaxTotal(2);
        DelegatingConnection conn1 = (DelegatingConnection)this.ds.getConnection();
        DelegatingConnection conn2 = (DelegatingConnection)this.ds.getConnection();
        Connection inner1 = conn1.getInnermostDelegate();
        Connection inner2 = conn2.getInnermostDelegate();
        Assertions.assertFalse((boolean)inner2.isClosed());
        conn2.close();
        Assertions.assertFalse((boolean)inner2.isClosed());
        this.ds.close();
        Assertions.assertFalse((boolean)conn1.isClosed());
        Assertions.assertTrue((boolean)inner2.isClosed());
        Assertions.assertEquals((int)0, (int)this.ds.getNumIdle());
        this.ds.start();
        Connection conn3 = this.ds.getConnection();
        Connection conn4 = this.ds.getConnection();
        conn3.close();
        conn4.close();
        conn1.close();
        Assertions.assertTrue((boolean)inner1.isClosed());
    }

    @Test
    public void testStartInitializes() throws Exception {
        this.ds.setInitialSize(2);
        Assertions.assertEquals((int)0, (int)this.ds.getNumIdle());
        Assertions.assertNull((Object)this.ds.getRegisteredJmxName());
        this.ds.start();
        Assertions.assertEquals((int)2, (int)this.ds.getNumIdle());
        Assertions.assertNotNull((Object)this.ds.getRegisteredJmxName());
    }

    @Test
    public void testTransactionIsolationBehavior() throws Exception {
        try (Connection conn = this.getConnection();){
            Assertions.assertNotNull((Object)conn);
            Assertions.assertEquals((int)2, (int)conn.getTransactionIsolation());
            conn.setTransactionIsolation(1);
        }
        Connection conn2 = this.getConnection();
        Assertions.assertEquals((int)2, (int)conn2.getTransactionIsolation());
        Connection conn3 = this.getConnection();
        Assertions.assertEquals((int)2, (int)conn3.getTransactionIsolation());
        conn2.close();
        conn3.close();
    }

    @Test
    public void testUnwrap() throws Exception {
        Assertions.assertSame((Object)this.ds.unwrap(BasicDataSource.class), (Object)this.ds);
        Assertions.assertSame((Object)this.ds.unwrap(AutoCloseable.class), (Object)this.ds);
        Assertions.assertThrows(SQLException.class, () -> this.ds.unwrap(String.class));
        Assertions.assertThrows(SQLException.class, () -> this.ds.unwrap(null));
    }

    @Test
    public void testValidationQueryTimeoutNegative() throws Exception {
        this.ds.setTestOnBorrow(true);
        this.ds.setTestOnReturn(true);
        this.ds.setValidationQueryTimeout(Duration.ofSeconds(-1L));
        Connection con = this.ds.getConnection();
        if (con != null) {
            con.close();
        }
    }

    @Test
    public void testValidationQueryTimeoutSucceed() throws Exception {
        this.ds.setTestOnBorrow(true);
        this.ds.setTestOnReturn(true);
        this.ds.setValidationQueryTimeout(Duration.ofMillis(100L));
        Connection con = this.ds.getConnection();
        if (con != null) {
            con.close();
        }
    }

    @Test
    public void testValidationQueryTimeoutZero() throws Exception {
        this.ds.setTestOnBorrow(true);
        this.ds.setTestOnReturn(true);
        this.ds.setValidationQueryTimeout(Duration.ZERO);
        Connection con = this.ds.getConnection();
        if (con != null) {
            con.close();
        }
    }

    @Test
    public void testValidationQueryTimoutFail() {
        this.ds.setTestOnBorrow(true);
        this.ds.setValidationQueryTimeout(Duration.ofSeconds(3L));
        SQLException e = (SQLException)Assertions.assertThrows(SQLException.class, () -> ((BasicDataSource)this.ds).getConnection());
        Assertions.assertTrue((boolean)e.toString().contains("timeout"));
    }
}

