/*
 * Copyright 2002-2005 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.commons.io.filefilter;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

/**
 * A {@link java.io.FileFilter} providing conditional AND logic across a list of
 * file filters. This filter returns <code>true</code> if all filters in the
 * list return <code>true</code>. Otherwise, it returns <code>false</code>.
 * Checking of the file filter list stops when the first filter returns
 * <code>false</code>.
 *
 * @since Commons IO 1.0
 * @version $Revision: 232900 $ $Date: 2005-08-15 23:22:30 +0100 (Mon, 15 Aug 2005) $
 *
 * @author Steven Caswell
 */
public class AndFileFilter
        extends AbstractFileFilter
        implements IOFileFilter, ConditionalFileFilter {

    /** The list of file filters. */
    private List fileFilters;

    /**
     * Constructs a new instance of <code>AndFileFilter</code>.
     *
     * @since Commons IO 1.1
     */
    public AndFileFilter() {
        this.fileFilters = new ArrayList();
    }

    /**
     * Constructs a new instance of <code>AndFileFilter</code>
     * with the specified list of filters.
     *
     * @param fileFilters  a List of IOFileFilter instances, copied, null ignored
     * @since Commons IO 1.1
     */
    public AndFileFilter(final List fileFilters) {
        if (fileFilters == null) {
            this.fileFilters = new ArrayList();
        } else {
            this.fileFilters = new ArrayList(fileFilters);
        }
    }

    /**
     * Constructs a new file filter that ANDs the result of two other filters.
     *
     * @param filter1  the first filter, must not be null
     * @param filter2  the second filter, must not be null
     * @throws IllegalArgumentException if either filter is null
     */
    public AndFileFilter(IOFileFilter filter1, IOFileFilter filter2) {
        if (filter1 == null || filter2 == null) {
            throw new IllegalArgumentException("The filters must not be null");
        }
        this.fileFilters = new ArrayList();
        addFileFilter(filter1);
        addFileFilter(filter2);
    }

    /**
     * {@inheritDoc}
     */
    public void addFileFilter(final IOFileFilter ioFileFilter) {
        this.fileFilters.add(ioFileFilter);
    }

    /**
     * {@inheritDoc}
     */
    public List getFileFilters() {
        return Collections.unmodifiableList(this.fileFilters);
    }

    /**
     * {@inheritDoc}
     */
    public boolean removeFileFilter(final IOFileFilter ioFileFilter) {
        return this.fileFilters.remove(ioFileFilter);
    }

    /**
     * {@inheritDoc}
     */
    public void setFileFilters(final List fileFilters) {
        this.fileFilters = new ArrayList(fileFilters);
    }

    /**
     * {@inheritDoc}
     */
    public boolean accept(final File file) {
        if (this.fileFilters.size() == 0) {
            return false;
        }
        for (Iterator iter = this.fileFilters.iterator(); iter.hasNext();) {
            IOFileFilter fileFilter = (IOFileFilter) iter.next();
            if (!fileFilter.accept(file)) {
                return false;
            }
        }
        return true;
    }

    /**
     * {@inheritDoc}
     */
    public boolean accept(final File file, final String name) {
        if (this.fileFilters.size() == 0) {
            return false;
        }
        for (Iterator iter = this.fileFilters.iterator(); iter.hasNext();) {
            IOFileFilter fileFilter = (IOFileFilter) iter.next();
            if (!fileFilter.accept(file, name)) {
                return false;
            }
        }
        return true;
    }

}
