/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.commons.lang3.mutable;

import junit.framework.TestCase;

/**
 * JUnit tests.
 * 
 * @since 2.2
 * @see MutableBoolean
 * @author Apache Software Foundation
 * @version $Id: MutableBooleanTest.java 892118 2009-12-18 03:39:13Z sebb $
 */
public class MutableBooleanTest extends TestCase {

    public MutableBooleanTest(String testName) {
        super(testName);
    }

    public void testCompareTo() {
        final MutableBoolean mutBool = new MutableBoolean(false);

        assertEquals(0, mutBool.compareTo(new MutableBoolean(false)));
        assertEquals(-1, mutBool.compareTo(new MutableBoolean(true)));
        mutBool.setValue(true);
        assertEquals(+1, mutBool.compareTo(new MutableBoolean(false)));
        assertEquals(0, mutBool.compareTo(new MutableBoolean(true)));

        try {
            mutBool.compareTo(null);
            fail();
        } catch (NullPointerException ex) {
        }
    }

    // ----------------------------------------------------------------
    public void testConstructors() {
        assertEquals(false, new MutableBoolean().booleanValue());

        assertEquals(true, new MutableBoolean(true).booleanValue());
        assertEquals(false, new MutableBoolean(false).booleanValue());

        assertEquals(true, new MutableBoolean(Boolean.TRUE).booleanValue());
        assertEquals(false, new MutableBoolean(Boolean.FALSE).booleanValue());

        try {
            new MutableBoolean(null);
            fail();
        } catch (NullPointerException ex) {
        }
    }

    public void testEquals() {
        final MutableBoolean mutBoolA = new MutableBoolean(false);
        final MutableBoolean mutBoolB = new MutableBoolean(false);
        final MutableBoolean mutBoolC = new MutableBoolean(true);

        assertEquals(true, mutBoolA.equals(mutBoolA));
        assertEquals(true, mutBoolA.equals(mutBoolB));
        assertEquals(true, mutBoolB.equals(mutBoolA));
        assertEquals(true, mutBoolB.equals(mutBoolB));
        assertEquals(false, mutBoolA.equals(mutBoolC));
        assertEquals(false, mutBoolB.equals(mutBoolC));
        assertEquals(true, mutBoolC.equals(mutBoolC));
        assertEquals(false, mutBoolA.equals(null));
        assertEquals(false, mutBoolA.equals(Boolean.FALSE));
        assertEquals(false, mutBoolA.equals("false"));
    }

    public void testGetSet() {
        assertEquals(false, new MutableBoolean().booleanValue());
        assertEquals(Boolean.FALSE, new MutableBoolean().getValue());
        
        final MutableBoolean mutBool = new MutableBoolean(false);
        assertEquals(Boolean.FALSE, mutBool.toBoolean());
        assertEquals(false, mutBool.booleanValue());
        assertEquals(true, mutBool.isFalse());
        assertEquals(false, mutBool.isTrue());

        mutBool.setValue(Boolean.TRUE);
        assertEquals(Boolean.TRUE, mutBool.toBoolean());
        assertEquals(true, mutBool.booleanValue());
        assertEquals(false, mutBool.isFalse());
        assertEquals(true, mutBool.isTrue());

        mutBool.setValue(false);
        assertEquals(false, mutBool.booleanValue());

        mutBool.setValue(true);
        assertEquals(true, mutBool.booleanValue());

        try {
            mutBool.setValue(null);
            fail();
        } catch (NullPointerException ex) {
        }
    }

    public void testHashCode() {
        final MutableBoolean mutBoolA = new MutableBoolean(false);
        final MutableBoolean mutBoolB = new MutableBoolean(false);
        final MutableBoolean mutBoolC = new MutableBoolean(true);

        assertEquals(true, mutBoolA.hashCode() == mutBoolA.hashCode());
        assertEquals(true, mutBoolA.hashCode() == mutBoolB.hashCode());
        assertEquals(false, mutBoolA.hashCode() == mutBoolC.hashCode());
        assertEquals(true, mutBoolA.hashCode() == Boolean.FALSE.hashCode());
        assertEquals(true, mutBoolC.hashCode() == Boolean.TRUE.hashCode());
    }

    public void testToString() {
        assertEquals(Boolean.FALSE.toString(), new MutableBoolean(false).toString());
        assertEquals(Boolean.TRUE.toString(), new MutableBoolean(true).toString());
    }

}
