/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.commons.lang3.mutable;

import junit.framework.TestCase;

/**
 * JUnit tests.
 * 
 * @version $Id: MutableShortTest.java 892118 2009-12-18 03:39:13Z sebb $
 * @see MutableShort
 */
public class MutableShortTest extends TestCase {

    public MutableShortTest(String testName) {
        super(testName);
    }

    // ----------------------------------------------------------------
    public void testConstructors() {
        assertEquals((short) 0, new MutableShort().shortValue());
        
        assertEquals((short) 1, new MutableShort((short) 1).shortValue());
        
        assertEquals((short) 2, new MutableShort(new Short((short) 2)).shortValue());
        assertEquals((short) 3, new MutableShort(new MutableShort((short) 3)).shortValue());

        assertEquals((short) 2, new MutableShort("2").shortValue());

        try {
            new MutableShort((Number)null);
            fail();
        } catch (NullPointerException ex) {}
    }

    public void testGetSet() {
        final MutableShort mutNum = new MutableShort((short) 0);
        assertEquals((short) 0, new MutableShort().shortValue());
        assertEquals(new Short((short) 0), new MutableShort().getValue());
        
        mutNum.setValue((short) 1);
        assertEquals((short) 1, mutNum.shortValue());
        assertEquals(new Short((short) 1), mutNum.getValue());
        
        mutNum.setValue(new Short((short) 2));
        assertEquals((short) 2, mutNum.shortValue());
        assertEquals(new Short((short) 2), mutNum.getValue());
        
        mutNum.setValue(new MutableShort((short) 3));
        assertEquals((short) 3, mutNum.shortValue());
        assertEquals(new Short((short) 3), mutNum.getValue());
        try {
            mutNum.setValue(null);
            fail();
        } catch (NullPointerException ex) {}
    }

    public void testEquals() {
        final MutableShort mutNumA = new MutableShort((short) 0);
        final MutableShort mutNumB = new MutableShort((short) 0);
        final MutableShort mutNumC = new MutableShort((short) 1);

        assertEquals(true, mutNumA.equals(mutNumA));
        assertEquals(true, mutNumA.equals(mutNumB));
        assertEquals(true, mutNumB.equals(mutNumA));
        assertEquals(true, mutNumB.equals(mutNumB));
        assertEquals(false, mutNumA.equals(mutNumC));
        assertEquals(false, mutNumB.equals(mutNumC));
        assertEquals(true, mutNumC.equals(mutNumC));
        assertEquals(false, mutNumA.equals(null));
        assertEquals(false, mutNumA.equals(new Short((short) 0)));
        assertEquals(false, mutNumA.equals("0"));
    }

    public void testHashCode() {
        final MutableShort mutNumA = new MutableShort((short) 0);
        final MutableShort mutNumB = new MutableShort((short) 0);
        final MutableShort mutNumC = new MutableShort((short) 1);

        assertEquals(true, mutNumA.hashCode() == mutNumA.hashCode());
        assertEquals(true, mutNumA.hashCode() == mutNumB.hashCode());
        assertEquals(false, mutNumA.hashCode() == mutNumC.hashCode());
        assertEquals(true, mutNumA.hashCode() == new Short((short) 0).hashCode());
    }

    public void testCompareTo() {
        final MutableShort mutNum = new MutableShort((short) 0);

        assertEquals((short) 0, mutNum.compareTo(new MutableShort((short) 0)));
        assertEquals((short) +1, mutNum.compareTo(new MutableShort((short) -1)));
        assertEquals((short) -1, mutNum.compareTo(new MutableShort((short) 1)));
        try {
            mutNum.compareTo(null);
            fail();
        } catch (NullPointerException ex) {}
    }

    public void testPrimitiveValues() {
        MutableShort mutNum = new MutableShort( (short) 1 );
        
        assertEquals( 1.0F, mutNum.floatValue(), 0 );
        assertEquals( 1.0, mutNum.doubleValue(), 0 );
        assertEquals( (byte) 1, mutNum.byteValue() );
        assertEquals( (short) 1, mutNum.shortValue() );
        assertEquals( 1, mutNum.intValue() );
        assertEquals( 1L, mutNum.longValue() );
    }

    public void testToShort() {
        assertEquals(new Short((short) 0), new MutableShort((short) 0).toShort());
        assertEquals(new Short((short) 123), new MutableShort((short) 123).toShort());
    }

    public void testIncrement() {
        MutableShort mutNum = new MutableShort((short) 1);
        mutNum.increment();
        
        assertEquals(2, mutNum.intValue());
        assertEquals(2L, mutNum.longValue());
    }

    public void testDecrement() {
        MutableShort mutNum = new MutableShort((short) 1);
        mutNum.decrement();
        
        assertEquals(0, mutNum.intValue());
        assertEquals(0L, mutNum.longValue());
    }

    public void testAddValuePrimitive() {
        MutableShort mutNum = new MutableShort((short) 1);
        mutNum.add((short) 1);
        
        assertEquals((short) 2, mutNum.shortValue());
    }

    public void testAddValueObject() {
        MutableShort mutNum = new MutableShort((short) 1);
        mutNum.add(new Short((short) 1));
        
        assertEquals((short) 2, mutNum.shortValue());
    }

    public void testSubtractValuePrimitive() {
        MutableShort mutNum = new MutableShort((short) 1);
        mutNum.subtract((short) 1);
        
        assertEquals((short) 0, mutNum.shortValue());
    }

    public void testSubtractValueObject() {
        MutableShort mutNum = new MutableShort((short) 1);
        mutNum.subtract(new Short((short) 1));
        
        assertEquals((short) 0, mutNum.shortValue());
    }

    public void testToString() {
        assertEquals("0", new MutableShort((short) 0).toString());
        assertEquals("10", new MutableShort((short) 10).toString());
        assertEquals("-123", new MutableShort((short) -123).toString());
    }

}
