/*
 * $Header: /home/cvs/jakarta-commons/modeler/src/java/org/apache/commons/modeler/BaseModelMBean.java,v 1.1.1.1 2002/04/30 20:58:51 craigmcc Exp $
 * $Revision: 1.1.1.1 $
 * $Date: 2002/04/30 20:58:51 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Commons", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */


package org.apache.commons.modeler;


import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Iterator;
import javax.management.Attribute;
import javax.management.AttributeChangeNotification;
import javax.management.AttributeList;
import javax.management.AttributeNotFoundException;
import javax.management.Descriptor;
import javax.management.DescriptorAccess;
import javax.management.ListenerNotFoundException;
import javax.management.MBeanException;
import javax.management.MBeanInfo;
import javax.management.MBeanNotificationInfo;
import javax.management.MBeanParameterInfo;
import javax.management.Notification;
import javax.management.NotificationFilter;
import javax.management.NotificationListener;
import javax.management.InstanceNotFoundException;
import javax.management.ReflectionException;
import javax.management.RuntimeErrorException;
import javax.management.RuntimeOperationsException;
import javax.management.ServiceNotFoundException;
import javax.management.modelmbean.DescriptorSupport;
import javax.management.modelmbean.InvalidTargetObjectTypeException;
import javax.management.modelmbean.ModelMBean;
import javax.management.modelmbean.ModelMBeanAttributeInfo;
import javax.management.modelmbean.ModelMBeanConstructorInfo;
import javax.management.modelmbean.ModelMBeanInfo;
import javax.management.modelmbean.ModelMBeanInfoSupport;
import javax.management.modelmbean.ModelMBeanNotificationBroadcaster;
import javax.management.modelmbean.ModelMBeanNotificationInfo;
import javax.management.modelmbean.ModelMBeanOperationInfo;


/**
 * <p>Basic implementation of the <code>ModelMBean</code> interface, which
 * supports the minimal requirements of the interface contract as follows:</p>
 * <ul>
 * <li>Only managed resources of type <code>objectReference</code> are
 *     supportd.</li>
 * <li>Caching of attribute values and operation results is not supported.
 *     All calls to <code>invoke()</code> are immediately executed.</li>
 * <li>Logging (under control of descriptors) is not supported.</li>
 * <li>Persistence of MBean attributes and operations is not supported.</li>
 * <li>All classes referenced as attribute types, operation parameters, or
 *     operation return values must be one of the following:
 *     <ul>
 *     <li>One of the Java primitive types (boolean, byte, char, double,
 *         float, integer, long, short).  Corresponding value will be wrapped
 *         in the appropriate wrapper class automatically.</li>
 *     <li>Operations that return no value should declare a return type of
 *         <code>void</code>.</li>
 *     </ul>
 * </ul>
 *
 * @author Craig R. McClanahan
 * @version $Revision: 1.1.1.1 $ $Date: 2002/04/30 20:58:51 $
 */

public class BaseModelMBean implements ModelMBean {


    // ----------------------------------------------------------- Constructors


    /**
     * Construct a <code>ModelMBean</code> with default
     * <code>ModelMBeanInfo</code> information.
     *
     * @exception MBeanException if the initializer of an object
     *  throws an exception
     * @exception RuntimeOperationsException if an IllegalArgumentException
     *  occurs
     */
    public BaseModelMBean() throws MBeanException, RuntimeOperationsException {

        super();
        setModelMBeanInfo(createDefaultModelMBeanInfo());

    }


    /**
     * Construct a <code>ModelMBean</code> associated with the specified
     * <code>ModelMBeanInfo</code> information.
     *
     * @param info ModelMBeanInfo for this MBean
     *
     * @exception MBeanException if the initializer of an object
     *  throws an exception
     * @exception RuntimeOperationsException if an IllegalArgumentException
     *  occurs
     */
    public BaseModelMBean(ModelMBeanInfo info)
        throws MBeanException, RuntimeOperationsException {

        super();
        setModelMBeanInfo(info);

    }


    // ----------------------------------------------------- Instance Variables


    /**
     * Notification broadcaster for attribute changes.
     */
    protected BaseNotificationBroadcaster attributeBroadcaster = null;


    /**
     * Notification broadcaster for general notifications.
     */
    protected BaseNotificationBroadcaster generalBroadcaster = null;


    /**
     * The <code>ModelMBeanInfo</code> object that controls our activity.
     */
    protected ModelMBeanInfo info = null;


    /**
     * The managed resource this MBean is associated with (if any).
     */
    protected Object resource = null;


    // --------------------------------------------------- DynamicMBean Methods


    /**
     * Obtain and return the value of a specific attribute of this MBean.
     *
     * @param name Name of the requested attribute
     *
     * @exception AttributeNotFoundException if this attribute is not
     *  supported by this MBean
     * @exception MBeanException if the initializer of an object
     *  throws an exception
     * @exception ReflectionException if a Java reflection exception
     *  occurs when invoking the getter
     */
    public Object getAttribute(String name)
        throws AttributeNotFoundException, MBeanException,
        ReflectionException {

        // Validate the input parameters
        if (name == null)
            throw new RuntimeOperationsException
                (new IllegalArgumentException("Attribute name is null"),
                 "Attribute name is null");

        // Look up the actual operation to be used
        ModelMBeanAttributeInfo attrInfo = info.getAttribute(name);
        if (attrInfo == null)
            throw new AttributeNotFoundException
                ("Cannot find attribute " + name);
        Descriptor attrDesc = attrInfo.getDescriptor();
        if (attrDesc == null)
            throw new AttributeNotFoundException
                ("Cannot find attribute " + name + " descriptor");
        String getMethod = (String) attrDesc.getFieldValue("getMethod");
        if (getMethod == null)
            throw new AttributeNotFoundException
                ("Cannot find attribute " + name + " get method name");

        // Invoke the specified get method and return the results
        return (invoke(getMethod,
                       new Object[] { },
                       new String[] { }));

    }


    /**
     * Obtain and return the values of several attributes of this MBean.
     *
     * @param names Names of the requested attributes
     */
    public AttributeList getAttributes(String names[]) {

        // Validate the input parameters
        if (names == null)
            throw new RuntimeOperationsException
                (new IllegalArgumentException("Attribute names list is null"),
                 "Attribute names list is null");

        // Prepare our response, eating all exceptions
        AttributeList response = new AttributeList();
        for (int i = 0; i < names.length; i++) {
            try {
                response.add(getAttribute(names[i]));
            } catch (Exception e) {
                ; // Not having a particular attribute in the response
                ; // is the indication of a getter problem
            }
        }
        return (response);

    }


    /**
     * Return the <code>MBeanInfo</code> object for this MBean.
     */
    public MBeanInfo getMBeanInfo() {

        return ((MBeanInfo) info.clone());

    }


    /**
     * Invoke a particular method on this MBean, and return any returned
     * value.
     *
     * <p><strong>IMPLEMENTATION NOTE</strong> - This implementation will
     * attempt to invoke this method on the MBean itself, or (if not
     * available) on the managed resource object associated with this
     * MBean.</p>
     *
     * @param name Name of the operation to be invoked
     * @param params Array containing the method parameters of this operation
     * @param signature Array containing the class names representing
     *  the signature of this operation
     *
     * @exception MBeanException if the initializer of an object
     *  throws an exception
     * @exception ReflectioNException if a Java reflection exception
     *  occurs when invoking a method
     */
    public Object invoke(String name, Object params[], String signature[])
        throws MBeanException, ReflectionException {

        // Validate the input parameters
        if (name == null)
            throw new RuntimeOperationsException
                (new IllegalArgumentException("Method name is null"),
                 "Method name is null");
        if (params == null)
            params = new Object[0];
        if (signature == null)
            signature = new String[0];
        if (params.length != signature.length)
            throw new RuntimeOperationsException
                (new IllegalArgumentException("Inconsistent arguments and signature"),
                 "Inconsistent arguments and signature");

        // Acquire the ModelMBeanOperationInfo information for
        // the requested operation
        ModelMBeanOperationInfo opInfo = info.getOperation(name);
        if (opInfo == null)
            throw new MBeanException
                (new ServiceNotFoundException("Cannot find operation " + name),
                 "Cannot find operation " + name);

        // Prepare the signature required by Java reflection APIs
        // FIXME - should we use the signature from opInfo?
        Class types[] = new Class[signature.length];
        for (int i = 0; i < signature.length; i++) {
            if (signature[i].equals(Boolean.TYPE.getName()))
                types[i] = Boolean.TYPE;
            else if (signature[i].equals(Byte.TYPE.getName()))
                types[i] = Byte.TYPE;
            else if (signature[i].equals(Character.TYPE.getName()))
                types[i] = Character.TYPE;
            else if (signature[i].equals(Double.TYPE.getName()))
                types[i] = Double.TYPE;
            else if (signature[i].equals(Float.TYPE.getName()))
                types[i] = Float.TYPE;
            else if (signature[i].equals(Integer.TYPE.getName()))
                types[i] = Integer.TYPE;
            else if (signature[i].equals(Long.TYPE.getName()))
                types[i] = Long.TYPE;
            else if (signature[i].equals(Short.TYPE.getName()))
                types[i] = Short.TYPE;
            else {
                try {
                    types[i] = Class.forName(signature[i]);
                } catch (ClassNotFoundException e) {
                    throw new ReflectionException
                        (e, "Cannot find Class for " + signature[i]);
                }
            }
        }

        // Locate the method to be invoked, either in this MBean itself
        // or in the corresponding managed resource
        // FIXME - Accessible methods in superinterfaces?
        Method method = null;
        Object object = null;
        Exception exception = null;
        try {
            object = this;
            method = object.getClass().getMethod(name, types);
        } catch (NoSuchMethodException e) {
            exception = e;;
        }
        try {
            if ((method == null) && (resource != null)) {
                object = resource;
                method = object.getClass().getMethod(name, types);
            }
        } catch (NoSuchMethodException e) {
            exception = e;
        }
        if (method == null) {
            throw new ReflectionException(exception,
                                          "Cannot find method " + name +
                                          " with this signature");
        }

        // Invoke the selected method on the appropriate object
        Object result = null;
        try {
            result = method.invoke(object, params);
        } catch (InvocationTargetException e) {
            Throwable t = e.getTargetException();
            if (t == null)
                t = e;
            if (t instanceof RuntimeException)
                throw new RuntimeOperationsException
                    ((RuntimeException) t, "Exception invoking method " + name);
            else if (t instanceof Error)
                throw new RuntimeErrorException
                    ((Error) t, "Error invoking method " + name);
            else
                throw new MBeanException
                    (e, "Exception invoking method " + name);
        } catch (Exception e) {
            throw new MBeanException
                (e, "Exception invoking method " + name);
        }

        // Return the results of this method invocation
        // FIXME - should we validate the return type?
        return (result);

    }


    /**
     * Set the value of a specific attribute of this MBean.
     *
     * @param attribute The identification of the attribute to be set
     *  and the new value
     *
     * @exception AttributeNotFoundException if this attribute is not
     *  supported by this MBean
     * @exception MBeanException if the initializer of an object
     *  throws an exception
     * @exception ReflectionException if a Java reflection exception
     *  occurs when invoking the getter
     */
    public void setAttribute(Attribute attribute)
        throws AttributeNotFoundException, MBeanException,
        ReflectionException {

        // Validate the input parameters
        if (attribute == null)
            throw new RuntimeOperationsException
                (new IllegalArgumentException("Attribute is null"),
                 "Attribute is null");
        String name = attribute.getName();
        Object value = attribute.getValue();
        if (name == null)
            throw new RuntimeOperationsException
                (new IllegalArgumentException("Attribute name is null"),
                 "Attribute name is null");

        // Look up the actual operation to be used
        ModelMBeanAttributeInfo attrInfo = info.getAttribute(name);
        if (attrInfo == null)
            throw new AttributeNotFoundException
                ("Cannot find attribute " + name);
        Descriptor attrDesc = attrInfo.getDescriptor();
        if (attrDesc == null)
            throw new AttributeNotFoundException
                ("Cannot find attribute " + name + " descriptor");
        String setMethod = (String) attrDesc.getFieldValue("setMethod");
        if (setMethod == null)
            throw new AttributeNotFoundException
                ("Cannot find attribute " + name + " set method name");

        // Invoke the specified set method and ignore any results
        invoke(setMethod,
               new Object[] { value },
               new String[] { attrInfo.getType() });

    }


    /**
     * Set the values of several attributes of this MBean.
     *
     * @param attributes THe names and values to be set
     *
     * @return The list of attributes that were set and their new values
     */
    public AttributeList setAttributes(AttributeList attributes) {

        // Validate the input parameters
        if (attributes == null)
            throw new RuntimeOperationsException
                (new IllegalArgumentException("Attributes list is null"),
                 "Attributes list is null");

        // Prepare and return our response, eating all exceptions
        AttributeList response = new AttributeList();
        String names[] = new String[attributes.size()];
        int n = 0;
        Iterator items = attributes.iterator();
        while (items.hasNext()) {
            Attribute item = (Attribute) items.next();
            names[n++] = item.getName();
            try {
                setAttribute(item);
            } catch (Exception e) {
                ; // Ignore all exceptions
            }
        }

        return (getAttributes(names));

    }


    // ----------------------------------------------------- ModelMBean Methods


    /**
     * Get the instance handle of the object against which we execute
     * all methods in this ModelMBean management interface.
     *
     * @exception InstanceNotFoundException if the managed resource object
     *  cannot be found
     * @exception MBeanException if the initializer of the object throws
     *  an exception
     * @exception RuntimeOperationsException if the managed resource or the
     *  resource type is <code>null</code> or invalid
     */
    public Object getManagedResource()
        throws InstanceNotFoundException, InvalidTargetObjectTypeException,
        MBeanException, RuntimeOperationsException {

        if (resource == null)
            throw new RuntimeOperationsException
                (new IllegalArgumentException("Managed resource is null"),
                 "Managed resource is null");

        return resource;

    }


    /**
     * Set the instance handle of the object against which we will execute
     * all methods in this ModelMBean management interface.
     *
     * @param resource The resource object to be managed
     * @param type The type of reference for the managed resource
     *  ("ObjectReference", "Handle", "IOR", "EJBHandle", or
     *  "RMIReference")
     *
     * @exception InstanceNotFoundException if the managed resource object
     *  cannot be found
     * @exception InvalidTargetObjectTypeException if this ModelMBean is
     *  asked to handle a reference type it cannot deal with
     * @exception MBeanException if the initializer of the object throws
     *  an exception
     * @exception RuntimeOperationsException if the managed resource or the
     *  resource type is <code>null</code> or invalid
     */
    public void setManagedResource(Object resource, String type)
        throws InstanceNotFoundException, InvalidTargetObjectTypeException,
        MBeanException, RuntimeOperationsException {

        if (resource == null)
            throw new RuntimeOperationsException
                (new IllegalArgumentException("Managed resource is null"),
                 "Managed resource is null");

        if (!"objectreference".equalsIgnoreCase(type))
            throw new InvalidTargetObjectTypeException(type);

        this.resource = resource;

    }


    /**
     * Initialize the <code>ModelMBeanInfo</code> associated with this
     * <code>ModelMBean</code>.  After the information and associated
     * descriptors have been customized, the <code>ModelMBean</code> should
     * be registered with the associated <code>MBeanServer</code>.
     *
     * @param info The ModelMBeanInfo object to be used by this ModelMBean
     *
     * @exception MBeanException If an exception occurs recording this
     *  ModelMBeanInfo information
     * @exception RuntimeOperations if the specified parameter is
     *  <code>null</code> or invalid
     */
    public void setModelMBeanInfo(ModelMBeanInfo info)
        throws MBeanException, RuntimeOperationsException {

        if (info == null)
            throw new RuntimeOperationsException
                (new IllegalArgumentException("ModelMBeanInfo is null"),
                 "ModelMBeanInfo is null");

        if (!isModelMBeanInfoValid(info))
            throw new RuntimeOperationsException
                (new IllegalArgumentException("ModelMBeanInfo is invalid"),
                 "ModelMBeanInfo is invalid");

        this.info = (ModelMBeanInfo) info.clone();

    }


    // ------------------------------ ModelMBeanNotificationBroadcaster Methods


    /**
     * Add an attribute change notification event listener to this MBean.
     *
     * @param listener Listener that will receive event notifications
     * @param name Name of the attribute of interest, or <code>null</code>
     *  to indicate interest in all attributes
     * @param handback Handback object to be sent along with event
     *  notifications
     *
     * @exception IllegalArgumentException if the listener parameter is null
     */
    public void addAttributeChangeNotificationListener
        (NotificationListener listener, String name, Object handback)
        throws IllegalArgumentException {

        if (listener == null)
            throw new IllegalArgumentException("Listener is null");
        if (attributeBroadcaster == null)
            attributeBroadcaster = new BaseNotificationBroadcaster();

        BaseAttributeFilter filter = new BaseAttributeFilter(name);
        attributeBroadcaster.addNotificationListener
            (listener, filter, handback);

    }


    /**
     * Remove an attribute change notification event listener from
     * this MBean.
     *
     * @param listener The listener to be removed
     * @param name The attribute name for which no more events are required
     *
     *
     * @exception ListenerNotFoundException if this listener is not
     *  registered in the MBean
     */
    public void removeAttributeChangeNotificationListener
        (NotificationListener listener, String name)
        throws ListenerNotFoundException {

        if (listener == null)
            throw new IllegalArgumentException("Listener is null");
        if (attributeBroadcaster == null)
            attributeBroadcaster = new BaseNotificationBroadcaster();

        // FIXME - currently this removes *all* notifications for this listener
        attributeBroadcaster.removeNotificationListener(listener);

    }


    /**
     * Remove an attribute change notification event listener from
     * this MBean.
     *
     * @param listener The listener to be removed
     * @param name The attribute name for which no more events are required
     * @param handback Handback object to be sent along with event
     *  notifications
     *
     *
     * @exception ListenerNotFoundException if this listener is not
     *  registered in the MBean
     */
    public void removeAttributeChangeNotificationListener
        (NotificationListener listener, String attributeName, Object handback)
        throws ListenerNotFoundException {

        removeAttributeChangeNotificationListener(listener, attributeName);

    }


    /**
     * Send an <code>AttributeChangeNotification</code> to all registered
     * listeners.
     *
     * @param notification The <code>AttributeChangeNotification</code>
     *  that will be passed
     *
     * @exception MBeanException if an object initializer throws an
     *  exception
     * @exception RuntimeOperationsException wraps IllegalArgumentException
     *  when the specified notification is <code>null</code> or invalid
     */
    public void sendAttributeChangeNotification
        (AttributeChangeNotification notification)
        throws MBeanException, RuntimeOperationsException {

        if (notification == null)
            throw new RuntimeOperationsException
                (new IllegalArgumentException("Notification is null"),
                 "Notification is null");
        if (attributeBroadcaster == null)
            return; // This means there are no registered listeners
        attributeBroadcaster.sendNotification(notification);

    }


    /**
     * Send an <code>AttributeChangeNotification</code> to all registered
     * listeners.
     *
     * @param oldValue The original value of the <code>Attribute</code>
     * @param newValue The new value of the <code>Attribute</code>
     *
     * @exception MBeanException if an object initializer throws an
     *  exception
     * @exception RuntimeOperationsException wraps IllegalArgumentException
     *  when the specified notification is <code>null</code> or invalid
     */
    public void sendAttributeChangeNotification
        (Attribute oldValue, Attribute newValue)
        throws MBeanException, RuntimeOperationsException {

        // Calculate the class name for the change notification
        String type = null;
        if (newValue.getValue() != null)
            type = newValue.getValue().getClass().getName();
        else if (oldValue.getValue() != null)
            type = oldValue.getValue().getClass().getName();
        else
            return;  // Old and new are both null == no change

        AttributeChangeNotification notification =
            new AttributeChangeNotification
            (this, 1, System.currentTimeMillis(),
             "Attribute value has changed",
             oldValue.getName(), type,
             oldValue.getValue(), newValue.getValue());
        sendAttributeChangeNotification(notification);

    }




    /**
     * Send a <code>Notification</code> to all registered listeners as a
     * <code>jmx.modelmbean.general</code> notification.
     *
     * @param notification The <code>Notification</code> that will be passed
     *
     * @exception MBeanException if an object initializer throws an
     *  exception
     * @exception RuntimeOperationsException wraps IllegalArgumentException
     *  when the specified notification is <code>null</code> or invalid
     */
    public void sendNotification(Notification notification)
        throws MBeanException, RuntimeOperationsException {

        if (notification == null)
            throw new RuntimeOperationsException
                (new IllegalArgumentException("Notification is null"),
                 "Notification is null");
        if (generalBroadcaster == null)
            return; // This means there are no registered listeners
        generalBroadcaster.sendNotification(notification);

    }


    /**
     * Send a <code>Notification</code> which contains the specified string
     * as a <code>jmx.modelmbean.generic</code> notification.
     *
     * @param message The message string to be passed
     *
     * @exception MBeanException if an object initializer throws an
     *  exception
     * @exception RuntimeOperationsException wraps IllegalArgumentException
     *  when the specified notification is <code>null</code> or invalid
     */
    public void sendNotification(String message)
        throws MBeanException, RuntimeOperationsException {

        if (message == null)
            throw new RuntimeOperationsException
                (new IllegalArgumentException("Message is null"),
                 "Message is null");
        Notification notification = new Notification
            ("jmx.modelmbean.generic", this, 1, message);
        sendNotification(notification);

    }




    // ---------------------------------------- NotificationBroadcaster Methods


    /**
     * Add a notification event listener to this MBean.
     *
     * @param listener Listener that will receive event notifications
     * @param filter Filter object used to filter event notifications
     *  actually delivered, or <code>null</code> for no filtering
     * @param handback Handback object to be sent along with event
     *  notifications
     *
     * @exception IllegalArgumentException if the listener parameter is null
     */
    public void addNotificationListener(NotificationListener listener,
                                        NotificationFilter filter,
                                        Object handback)
        throws IllegalArgumentException {

        if (listener == null)
            throw new IllegalArgumentException("Listener is null");
        if (generalBroadcaster == null)
            generalBroadcaster = new BaseNotificationBroadcaster();
        generalBroadcaster.addNotificationListener
            (listener, filter, handback);

    }


    /**
     * Return an <code>MBeanNotificationInfo</code> object describing the
     * notifications sent by this MBean.
     */
    public MBeanNotificationInfo[] getNotificationInfo() {

        // Acquire the set of application notifications
        MBeanNotificationInfo current[] = info.getNotifications();
        if (current == null)
            current = new MBeanNotificationInfo[0];
        MBeanNotificationInfo response[] =
            new MBeanNotificationInfo[current.length + 2];
        Descriptor descriptor = null;

        // Fill in entry for general notifications
        descriptor = new DescriptorSupport
            (new String[] { "name=GENERIC",
                            "descriptorType=notification",
                            "log=T",
                            "severity=5",
                            "displayName=jmx.modelmbean.generic" });
        response[0] = new ModelMBeanNotificationInfo
            (new String[] { "jmx.modelmbean.generic" },
             "GENERIC",
             "Text message notification from the managed resource",
             descriptor);

        // Fill in entry for attribute change notifications
        descriptor = new DescriptorSupport
            (new String[] { "name=ATTRIBUTE_CHANGE",
                            "descriptorType=notification",
                            "log=T",
                            "severity=5",
                            "displayName=jmx.attribute.change" });
        response[1] = new ModelMBeanNotificationInfo
            (new String[] { "jmx.attribute.change" },
             "ATTRIBUTE_CHANGE",
             "Observed MBean attribute value has changed",
             descriptor);

        // Copy remaining notifications as reported by the application
        System.arraycopy(current, 0, response, 2, current.length);
        return (response);

    }


    /**
     * Remove a notification event listener from this MBean.
     *
     * @param listener The listener to be removed (any and all registrations
     *  for this listener will be eliminated)
     *
     * @exception ListenerNotFoundException if this listener is not
     *  registered in the MBean
     */
    public void removeNotificationListener(NotificationListener listener)
        throws ListenerNotFoundException {

        if (listener == null)
            throw new IllegalArgumentException("Listener is null");
        if (generalBroadcaster == null)
            generalBroadcaster = new BaseNotificationBroadcaster();
        generalBroadcaster.removeNotificationListener(listener);


    }


    /**
     * Remove a notification event listener from this MBean.
     *
     * @param listener The listener to be removed (any and all registrations
     *  for this listener will be eliminated)
     * @param handback Handback object to be sent along with event
     *  notifications
     *
     * @exception ListenerNotFoundException if this listener is not
     *  registered in the MBean
     */
    public void removeNotificationListener(NotificationListener listener, 
                                           Object handback) 
        throws ListenerNotFoundException {

        removeNotificationListener(listener);

    }


    /**
     * Remove a notification event listener from this MBean.
     *
     * @param listener The listener to be removed (any and all registrations
     *  for this listener will be eliminated)
     * @param filter Filter object used to filter event notifications
     *  actually delivered, or <code>null</code> for no filtering
     * @param handback Handback object to be sent along with event
     *  notifications
     *
     * @exception ListenerNotFoundException if this listener is not
     *  registered in the MBean
     */
    public void removeNotificationListener(NotificationListener listener, 
                                           NotificationFilter filter, 
                                           Object handback) 
        throws ListenerNotFoundException {

        removeNotificationListener(listener);

    }


    // ------------------------------------------------ PersistentMBean Methods


    /**
     * Instantiates this MBean instance from data found in the persistent
     * store.  The data loaded could include attribute and operation values.
     * This method should be called during construction or initialization
     * of the instance, and before the MBean is registered with the
     * <code>MBeanServer</code>.
     *
     * <p><strong>IMPLEMENTATION NOTE</strong> - This implementation does
     * not support persistence.</p>
     *
     * @exception InstanceNotFoundException if the managed resource object
     *  cannot be found
     * @exception MBeanException if the initializer of the object throws
     *  an exception
     * @exception RuntimeOperationsException if an exception is reported
     *  by the persistence mechanism
     */
    public void load() throws InstanceNotFoundException,
        MBeanException, RuntimeOperationsException {

        throw new MBeanException
            (new IllegalStateException("Persistence is not supported"),
             "Persistence is not supported");

    }


    /**
     * Capture the current state of this MBean instance and write it out
     * to the persistent store.  The state stored could include attribute
     * and operation values.  If one of these methods of persistence is not
     * supported, a "service not found" exception will be thrown.
     *
     * <p><strong>IMPLEMENTATION NOTE</strong> - This implementation does
     * not support persistence.</p>
     *
     * @exception InstanceNotFoundException if the managed resource object
     *  cannot be found
     * @exception MBeanException if the initializer of the object throws
     *  an exception, or persistence is not supported
     * @exception RuntimeOperationsException if an exception is reported
     *  by the persistence mechanism
     */
    public void store() throws InstanceNotFoundException,
        MBeanException, RuntimeOperationsException {

        throw new MBeanException
            (new IllegalStateException("Persistence is not supported"),
             "Persistence is not supported");

    }


    // ------------------------------------------------------ Protected Methods


    /**
     * Create and return a default <code>ModelMBeanInfo</code> object.
     */
    protected ModelMBeanInfo createDefaultModelMBeanInfo() {

        return (new ModelMBeanInfoSupport(this.getClass().getName(),
                                          "Default ModelMBean",
                                          null, null, null, null));

    }


    /**
     * Is the specified <code>ModelMBeanInfo</code> instance valid?
     *
     * <p><strong>IMPLEMENTATION NOTE</strong> - This implementation
     * does not check anything, but this method can be overridden
     * as required.</p>
     *
     * @param info The <code>ModelMBeanInfo object to check
     */
    protected boolean isModelMBeanInfoValid(ModelMBeanInfo info) {

        return (true);

    }


}
