/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.commons.proxy.factory.util;

import java.lang.ref.WeakReference;
import java.util.HashMap;
import java.util.Map;
import java.util.WeakHashMap;

/**
 * A cache for storing implementation classes for proxies based on a specific type of {@link ProxyClassGenerator}.  A
 * proxy class cache ensures that there is only one class for every
 * {@link ProxyClassGenerator}/{@link ClassLoader}/proxy class array combination.
 *
 * @author James Carman
 * @since 1.0
 */
public class ProxyClassCache
{
//----------------------------------------------------------------------------------------------------------------------
// Fields
//----------------------------------------------------------------------------------------------------------------------

    private final Map loaderToClassCache = new WeakHashMap();
    private final ProxyClassGenerator proxyClassGenerator;

//----------------------------------------------------------------------------------------------------------------------
// Constructors
//----------------------------------------------------------------------------------------------------------------------

    public ProxyClassCache( ProxyClassGenerator proxyClassGenerator )
    {
        this.proxyClassGenerator = proxyClassGenerator;
    }

//----------------------------------------------------------------------------------------------------------------------
// Other Methods
//----------------------------------------------------------------------------------------------------------------------

    /**
     * Returns the proxy class generated by the {@link ProxyClassGenerator} using the specified {@link ClassLoader} and
     * array of proxy classes.
     * 
     * @param classLoader the classloader
     * @param proxyClasses the proxy classes
     * @return the proxy class generated by the {@link ProxyClassGenerator} using the specified {@link ClassLoader} and
     * array of proxy classes
     */
    public synchronized Class getProxyClass( ClassLoader classLoader, Class[] proxyClasses )
    {
        final Map classCache = getClassCache( classLoader );
        final String key = toClassCacheKey( proxyClasses );
        Class proxyClass;
        WeakReference proxyClassReference = ( WeakReference )classCache.get( key );
        if( proxyClassReference == null )
        {
            proxyClass = proxyClassGenerator.generateProxyClass( classLoader, proxyClasses );
            classCache.put( key, new WeakReference( proxyClass ) );
        }
        else
        {
            synchronized( proxyClassReference )
            {
                proxyClass = ( Class )proxyClassReference.get();
                if( proxyClass == null )
                {
                    proxyClass = proxyClassGenerator.generateProxyClass( classLoader, proxyClasses );
                    classCache.put( key, new WeakReference( proxyClass ) );
                }
            }
        }
        return proxyClass;
    }

    private Map getClassCache( ClassLoader classLoader )
    {
        Map cache = ( Map )loaderToClassCache.get( classLoader );
        if( cache == null )
        {
            cache = new HashMap();
            loaderToClassCache.put( classLoader, cache );
        }
        return cache;
    }

    private String toClassCacheKey( Class[] proxyClasses )
    {
        final StringBuffer sb = new StringBuffer();
        for( int i = 0; i < proxyClasses.length; i++ )
        {
            Class proxyInterface = proxyClasses[i];
            sb.append( proxyInterface.getName() );
            if( i != proxyClasses.length - 1 )
            {
                sb.append( "," );
            }
        }
        return sb.toString();
    }
}

