/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.vfs2.provider.ram.test;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.apache.commons.vfs2.FileContent;
import org.apache.commons.vfs2.FileObject;
import org.apache.commons.vfs2.FileSystemException;
import org.apache.commons.vfs2.FileSystemOptions;
import org.apache.commons.vfs2.impl.DefaultFileSystemManager;
import org.apache.commons.vfs2.provider.FileProvider;
import org.apache.commons.vfs2.provider.ram.RamFileProvider;
import org.apache.commons.vfs2.provider.ram.RamFileSystemConfigBuilder;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

public class CustomRamProviderTest {
    private static final byte[] NON_EMPTY_FILE_CONTENT = new byte[]{1, 2, 3};
    private final List<Closeable> closeables = new ArrayList<Closeable>();
    FileSystemOptions defaultRamFso = new FileSystemOptions();
    DefaultFileSystemManager manager;
    FileSystemOptions smallSizedFso = new FileSystemOptions();
    FileSystemOptions zeroSizedFso = new FileSystemOptions();

    private <C extends Closeable> C closeOnTearDown(C closeable) {
        this.closeables.add(closeable);
        return closeable;
    }

    private InputStream createEmptyFile() throws FileSystemException, IOException {
        FileObject root = this.manager.resolveFile("ram://file");
        root.createFile();
        return this.closeOnTearDown(root.getContent().getInputStream());
    }

    private InputStream createNonEmptyFile() throws FileSystemException, IOException {
        FileObject root = this.manager.resolveFile("ram://file");
        root.createFile();
        FileContent content = root.getContent();
        OutputStream output = this.closeOnTearDown(content.getOutputStream());
        output.write(1);
        output.write(2);
        output.write(3);
        output.flush();
        output.close();
        return this.closeOnTearDown(content.getInputStream());
    }

    @Before
    public void setUp() throws Exception {
        this.manager = new DefaultFileSystemManager();
        this.manager.addProvider("ram", (FileProvider)new RamFileProvider());
        this.manager.init();
        RamFileSystemConfigBuilder.getInstance().setMaxSize(this.zeroSizedFso, 0L);
        RamFileSystemConfigBuilder.getInstance().setMaxSize(this.smallSizedFso, 10L);
    }

    @After
    public void tearDown() throws Exception {
        for (Closeable closeable : this.closeables) {
            try {
                closeable.close();
            }
            catch (Exception exception) {}
        }
        this.manager.close();
    }

    @Test
    public void testReadEmptyFileByteByByte() throws FileSystemException, IOException {
        InputStream input = this.createEmptyFile();
        Assert.assertEquals((String)"Empty file didnt return EOF -1", (long)-1L, (long)input.read());
    }

    @Test
    public void testReadEmptyFileIntoBuffer() throws FileSystemException, IOException {
        InputStream input = this.createEmptyFile();
        byte[] buffer = new byte[100];
        Assert.assertEquals((String)"Empty file didnt return when filling buffer", (long)-1L, (long)input.read(buffer));
        Assert.assertArrayEquals((String)"Buffer was written too", (byte[])new byte[100], (byte[])buffer);
    }

    @Test
    public void testReadEmptyFileIntoBufferWithOffsetAndLength() throws FileSystemException, IOException {
        InputStream input = this.createEmptyFile();
        byte[] buffer = new byte[100];
        Assert.assertEquals((String)"Empty file didnt return when filling buffer", (long)-1L, (long)input.read(buffer, 10, 90));
        Assert.assertArrayEquals((String)"Buffer was written too", (byte[])new byte[100], (byte[])buffer);
    }

    @Test
    public void testReadNonEmptyFileByteByByte() throws FileSystemException, IOException {
        InputStream input = this.createNonEmptyFile();
        Assert.assertEquals((String)"Read 1st byte failed", (long)1L, (long)input.read());
        Assert.assertEquals((String)"Rread 2st byte failed", (long)2L, (long)input.read());
        Assert.assertEquals((String)"Read 3st byte failed", (long)3L, (long)input.read());
        Assert.assertEquals((String)"File should be empty", (long)-1L, (long)input.read());
    }

    @Test
    public void testReadNonEmptyFileIntoBuffer() throws FileSystemException, IOException {
        InputStream input = this.createNonEmptyFile();
        byte[] buffer = new byte[100];
        Assert.assertEquals((String)"Filling buffer failed when file is not empty", (long)NON_EMPTY_FILE_CONTENT.length, (long)input.read(buffer));
        byte[] expectedBuffer = new byte[100];
        System.arraycopy(NON_EMPTY_FILE_CONTENT, 0, expectedBuffer, 0, NON_EMPTY_FILE_CONTENT.length);
        Assert.assertArrayEquals((String)"Buffer not filled", (byte[])expectedBuffer, (byte[])buffer);
        Arrays.fill(buffer, (byte)0);
        Arrays.fill(expectedBuffer, (byte)0);
        Assert.assertEquals((String)"File should be empty after filling buffer", (long)-1L, (long)input.read(buffer));
        Assert.assertArrayEquals((String)"Buffer was written when empty", (byte[])expectedBuffer, (byte[])buffer);
    }

    @Test
    public void testReadNonEmptyFileIntoBufferWithOffsetAndLength() throws FileSystemException, IOException {
        InputStream input = this.createNonEmptyFile();
        byte[] buffer = new byte[100];
        int offset = 10;
        Assert.assertEquals((String)"Filling buffer failed when file is not empty", (long)NON_EMPTY_FILE_CONTENT.length, (long)input.read(buffer, 10, 90));
        byte[] expectedBuffer = new byte[100];
        System.arraycopy(NON_EMPTY_FILE_CONTENT, 0, expectedBuffer, 10, NON_EMPTY_FILE_CONTENT.length);
        Assert.assertArrayEquals((String)"Buffer not filled", (byte[])expectedBuffer, (byte[])buffer);
        Arrays.fill(buffer, (byte)0);
        Arrays.fill(expectedBuffer, (byte)0);
        Assert.assertEquals((String)"File should be empty after filling buffer", (long)-1L, (long)input.read(buffer, 10, 90));
        Assert.assertArrayEquals((String)"Buffer was written when empty", (byte[])expectedBuffer, (byte[])buffer);
    }

    @Test
    public void testRootFolderExists() throws FileSystemException {
        FileObject root = this.manager.resolveFile("ram:///", this.defaultRamFso);
        Assert.assertTrue((boolean)root.getType().hasChildren());
        try {
            root.delete();
            Assert.fail();
        }
        catch (FileSystemException fileSystemException) {
            // empty catch block
        }
    }

    @Test
    public void testFSOptions() throws Exception {
        FileObject fo1 = this.manager.resolveFile("ram:/");
        FileObject fo2 = this.manager.resolveFile("ram:/");
        Assert.assertTrue((String)"Both files should exist in the same fs instance.", (fo1.getFileSystem() == fo2.getFileSystem() ? 1 : 0) != 0);
        FileSystemOptions fsOptions = fo1.getFileSystem().getFileSystemOptions();
        long maxFilesystemSize = RamFileSystemConfigBuilder.getInstance().getLongMaxSize(fsOptions);
        Assert.assertEquals((String)"Filesystem option maxSize must be unlimited", (long)Long.MAX_VALUE, (long)maxFilesystemSize);
        FileObject fo3 = this.manager.resolveFile("ram:/fo3", this.smallSizedFso);
        FileObject fo4 = this.manager.resolveFile("ram:/", this.smallSizedFso);
        Assert.assertTrue((String)"Both files should exist in the same FileSystem instance.", (fo3.getFileSystem() == fo4.getFileSystem() ? 1 : 0) != 0);
        Assert.assertTrue((String)"Both files should exist in different FileSystem instance.", (fo1.getFileSystem() != fo3.getFileSystem() ? 1 : 0) != 0);
        fsOptions = fo3.getFileSystem().getFileSystemOptions();
        maxFilesystemSize = RamFileSystemConfigBuilder.getInstance().getLongMaxSize(fsOptions);
        Assert.assertEquals((String)"Filesystem option maxSize must be set", (long)10L, (long)maxFilesystemSize);
    }

    @Test
    public void testSmallFS() throws Exception {
        OutputStream os;
        FileObject fo3 = this.manager.resolveFile("ram:/fo3", this.smallSizedFso);
        fo3.createFile();
        try {
            os = fo3.getContent().getOutputStream();
            os.write(new byte[10]);
            os.close();
        }
        catch (FileSystemException e) {
            Assert.fail((String)"Test should be able to save such a small file");
        }
        try {
            os = fo3.getContent().getOutputStream();
            os.write(new byte[11]);
            os.close();
            Assert.fail((String)"It shouldn't save such a big file");
        }
        catch (FileSystemException fileSystemException) {
            // empty catch block
        }
    }

    @Test
    public void testMoveFile() throws FileSystemException {
        FileObject fileSource = this.manager.resolveFile("ram://virtual/source");
        fileSource.createFile();
        FileObject fileDest = this.manager.resolveFile("ram://virtual/dest");
        Assert.assertTrue((boolean)fileSource.canRenameTo(fileDest));
        fileSource.moveTo(fileDest);
    }
}

