/*
 * sets.c -- implement sets of tokens
 *
 * 19 Feb 1993, Bert Bos <bert@let.rug.nl>
 *
 * Sets are implemented as bit-sets, one bit for each element. The bits are
 * collected into int's. The type Set is thus defined as an array of
 * int's. Sets can only contain elements that can be typecast to integers
 * in the range 0 to N, with N arbitrarily large, but known a priori.
 * Element types that are eligible for this include enum's, char's and
 * int's.
 *
 * New sets are allocated on the heap.
 *
 * The following functions are defined:
 *
 * Set SetNew(int max)
 * -------------------
 * Create an empty set with room for elements 0..N. The set is allocated on
 * the heap and can thus be deallocated with free() when it is no longer
 * needed.
 *
 * Set SetSingleton(int max, int x)
 * --------------------------------
 * Create a set with the single element x.
 *
 * void SetAdd(Set a, Set b)
 * -------------------------
 * Add all elements of b to a.
 *
 * void SetAdd(Set a, int x)
 * -------------------------
 * Add one element to a.
 *
 * void SetSubtract(Set a, Set b)
 * ------------------------------
 * Remove all elements of b from a.
 *
 * void SetSubtract1(Set a, int x)
 * -------------------------------
 * Remove the element x from a.
 *
 * Set SetUnion(Set a, Set b)
 * --------------------------
 * Create a new Set (on the heap) consisting of the union of a and b.
 *
 * Set SetIntersection(Set a, Set b)
 * ---------------------------------
 * Create a new set (on the heap) with only the common elements of a and b.
 *
 * Set SetDiff(Set a, Set b)
 * -------------------------
 * Create a new Set with all elements of a that are not in b.
 *
 * Boolean SetMember(int x, Set a)
 * -------------------------------
 * Return TRUE if c is a member of a.
 *
 * int SetCardinality(Set a)
 * -------------------------
 * Return the number of elements of a.
 *
 * int SetEmpty(Set a)
 * -------------------------
 * Return 1 if a is empty, otherwise 0.
 *
 * void SetPrint(FILE *f, Set a, char *name)
 * -----------------------------------------
 * Write the set a on file f, in a format that the C compiler understands.
 * The output is something like: int name[] = { ... };
 *
 */

#ifndef __lint
static char rcsid[] = "$Header: sets.c,v 0.2 93/02/23 19:28:47 bert Exp $";
#endif

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include "sets.h"

#define BITS (8 * sizeof(int))

Set SetNew(max)
    int max;
{
    Set s;
    int n;
    n = (max + BITS - 1)/BITS;
    s = calloc(1 + n, sizeof(int));
    s[0] = n;
    return s;
}

void SetAdd(a, b)
    Set a;
    Set b;
{
    int i;
    assert(a != NULL && b != NULL && a[0] == b[0]);
    for (i = 1; i <= a[0]; i++) a[i] |= b[i];
}

void SetAdd1(a, x)
    Set a;
    int x;
{
    assert(a != NULL && 1 + x/BITS <= a[0]);
    a[1+x/BITS] = 1 << (x % BITS);
}
		
Set SetSingleton(max, x)
    int max;
    int x;
{
    Set s;
    s = SetNew(max);
    SetAdd1(s, x);
    return s;
}

#if notdef
void SetSubtract(a, b)
    Set a;
    Set b;
{
    int i;
    assert(a != NULL && b != NULL && a[0] == b[0]);
    for (i = 1; i <= a[0]; i++) a[i] &= ~b[i];
}
#endif

#if notdef
void SetSubtract1(a, x)
    Set a;
    int x;
{
    int i;
    unsigned int h;
    assert(a != NULL && 1 + x/BITS <= a[0]);
    i = 1 + x/BITS;
    h = 1 << (x % BITS);
    a[i] &= ~h;
}
#endif

Set SetUnion(a, b)
    Set a;
    Set b;
{
    Set s;
    int i;
    assert(a != NULL && b != NULL && a[0] == b[0]);
    s = calloc(1 + (size_t) a[0], sizeof(int));
    s[0] = a[0];
    for (i = 1; i <= a[0]; i++) s[i] = a[i] | b[i];
    return s;
}

Set SetIntersection(a, b)
    Set a;
    Set b;
{
    Set s;
    int i;
    assert(a != NULL && b != NULL && a[0] == b[0]);
    s = calloc(1 + (size_t) a[0], sizeof(int));
    s[0] = a[0];
    for (i = 1; i <= a[0]; i++) s[i] = a[i] & b[i];
    return s;
}

#if notdef
Set SetDiff(a, b)
    Set a;
    Set b;
{
    Set s;
    int i;
    assert(a != NULL && b != NULL && a[0] == b[0]);
    s = calloc(1 + (size_t) a[0], sizeof(int));
    s[0] = a[0];
    for (i = 1; i <= a[0]; i++) s[i] = a[i] & ~b[i];
    return s;
}
#endif

int SetMember(x, a)
    int x;
    Set a;
{
    int i;
    unsigned int h;
    assert(a != NULL && 1 + x/BITS <= a[0]);
    i = 1 + x/BITS;
    h = 1 << (x % BITS);
    return (a[i] & h) == h;
}

#if notdef
int SetEqual(a, b)
    Set a;
    Set b;
{
    int i;
    assert(a != NULL && b != NULL && a[0] == b[0]);
    for (i = 1; i <= a[0]; i++) if (a[i] != b[i]) return 0;
    return 1;
}
#endif

int SetSubset(a, b)
    Set a;
    Set b;
{
    int i;
    assert(a != NULL && b != NULL && a[0] == b[0]);
    for (i = 1; i <= a[0]; i++) if ((a[i] & b[i]) != a[i]) return 0;
    return 1;
}

#if notdef
int SetCardinality(a)
    Set a;
{
    int j, i, n = 0;
    unsigned int h;
    assert(a != NULL);
    for (i = 1; i <= a[0]; i++)
	for (h = a[i], j = 0; j < BITS; h = h >> 1, j++)
	    if (h & 1) n++;
    return n;
}
#endif

int SetEmpty(a)
    Set a;
{
    int i;
    assert(a != NULL);
    for (i = 1; i <= a[0]; i++) if (a[i]) return 0;
    return 1;
}

Set SetCopy(a)
    Set a;
{
    Set s;
    assert(a != NULL);
    s = SetNew(BITS * (int)a[0]);
    SetAdd(s, a);
    return s;
}

void SetPrint(f, a, name)
    FILE *f;
    Set a;
    char *name;
{
    int i, j;
    assert(a != NULL);
    fprintf(f, "int %s[] = {\n  ", name);
    for (j = 0, i = 0; i <= a[0]; i++, j++) {
	if (j == 7) { j = 0; fprintf(f, "\n  "); }
	fprintf(f, "0x%08x,", a[i]);
    }
    fprintf(f, "};\n");
}
