/*
 * (c) COPYRIGHT 1999 World Wide Web Consortium
 * (Massachusetts Institute of Technology, Institut National de Recherche
 *  en Informatique et en Automatique, Keio University).
 * All Rights Reserved. http://www.w3.org/Consortium/Legal/
 *
 * $Id: sac.h,v 1.5 2000/10/27 20:46:25 plehegar Exp $
 */
#ifndef _PARSER_H_
#define _PARSER_H_

#include "defs.h"
#include "selectors.h"
#include "lexicalUnit.h"

typedef void * SAC_Parser;

/* Handlers definition
   All SAC_STRING are NULL terminated.
 */

/*
  Receive notification of the beginning of a document.
  The SAC parser will invoke this method only once, before any other methods in
  this interface.  
*/
typedef void (*SAC_StartDocumentHandler)(void *userData);


/*
  Receive notification of the end of a document.
  The SAC parser will invoke this method only once, and it will be the last
  method invoked during the parse. The parser shall not invoke this method
  until it has either abandoned parsing (because of an unrecoverable error) or
  reached the end of input.
*/
typedef void (*SAC_EndDocumentHandler)(void *userData);

/*
  Receive notification of a comment
  This method is not required to be supported by the parser.
 */
typedef void (*SAC_CommentHandler)(void *userData,
				   const SAC_STRING data);

/*
  Receive notification of an unknown rule at-rule not supported by this parser.
 */
typedef void (*SAC_IgnorableAtRuleHandler)(void *userData,
					   const SAC_STRING atRule);

/*
  Receive notification of a namespace declaration and the default namespace
  if any (prefix will be NULL in that case). uri can never be NULL.
 */
typedef void (*SAC_NamespaceDeclarationHandler)(void *userData,
						const SAC_STRING prefix,
						const SAC_STRING uri);
/*
  Receive notification of the beginning of an import statement.
  base  - The base argument is whatever was set by SAC_SetBase.
  uri   - The unresolved URI of the imported style sheet.
          It can never be NULL.
  media - It is an array of all media for the imported style sheet.
          It terminated by NULL.
  defaultNamepaceURI - The default namespace URI for the imported style sheet.
 */
typedef void (*SAC_ImportHandler)(void *userData,
				  const SAC_STRING base,
				  const SAC_STRING uri,
				  const SAC_STRING media[],
				  const SAC_STRING defaultNamepaceURI);

/*
  Receive notification of the beginning of a media rule.

  The SAC Parser will invoke this method at the beginning of every media rule
  in the CSS document. There will be a corresponding SAC_EndMediaHandler event
  for every SAC_StartMediaHandler event. All rules inside the rule will be
  reported, in order, before the corresponding SAC_EndMediaHandler event.

  media - It is an array of all media for the media rule.
          It terminated by NULL.  
*/
typedef void (*SAC_StartMediaHandler)(void *userData,
				      const SAC_STRING media[]);
/*
  Receive notification of the end of a media rule.

  The SAC Parser will invoke this method at the end of every media rule in the
  CSS document. There will be a corresponding SAC_StartMediaHandler event for
  every SAC_EndMediaHandler event.

  media - It is an array of all media for the media rule.
          It terminated by NULL.  
*/
typedef void (*SAC_EndMediaHandler)(void *userData,
				    const SAC_STRING media[]);

/*
  Receive notification of beginning of a page rule.

  The SAC Parser will invoke this method at the beginning of every page rule in
  the CSS document. There will be a corresponding SAC_EndPageHandler event for
  every SAC_StartPageHandler event. All properties inside the rule will be
  reported, in order, before the corresponding SAC_EndPageHandler event.

  name        - The name of the page if any.
  pseudoPseudo - The pseudo page associated to the page
*/
typedef void (*SAC_StartPageHandler)(void *userData,
				     const SAC_STRING name,
				     const SAC_STRING pseudoPage);
/*
  Receive notification of the end of a page rule.

  The SAC Parser will invoke this method at the end of every page rule in the
  CSS document. There will be a corresponding SAC_StartPageHandler event for
  every SAC_EndPageHandler event.

  name        - The name of the page if any.
  pseudoPage - The pseudo page associated to the page
*/
typedef void (*SAC_EndPageHandler)(void *userData,
				   const SAC_STRING name,
				   const SAC_STRING pseudoPage);

/*
  Receive notification of a beginning of font-face rule.

  The SAC Parser will invoke this method at the beginning of every font-face
  rule in the CSS document. There will be a corresponding
  SAC_EndFontFaceHandler event for every SAC_StartFontFaceHandler event. All
  properties inside the rule will be reported, in order, before the
  corresponding SAC_EndFontFaceHandler event.  
*/
typedef void (*SAC_StartFontFaceHandler)(void *userData);

/*
  Receive notification of the end of a font-face rule.

  The SAC Parser will invoke this method at the end of every font-face rule in
  the CSS document. There will be a corresponding SAC_StartFontFaceHandler
  event for every SAC_EndFontFaceHandler event.
*/
typedef void (*SAC_EndFontFaceHandler)(void *userData);

/*
  Receive notification of the beginning of a style rule.

  The SAC Parser will invoke this method at the beginning of every style rule
  in the CSS document. There will be a corresponding SAC_EndStyleHandler event
  for every SAC_StartStyleHandler event. All properties inside the rule will be
  reported, in order, before the corresponding SAC_EndStyleHandler event.

  selectors - It is an array of all selectors for the style rule.
*/
typedef void (*SAC_StartSelectorHandler)(void *userData,
					 const SAC_Selector * selectors[]);
/*
  Receive notification of the end of a style rule.

  The SAC Parser will invoke this method at the end of every style rule in the
  CSS document. There will be a corresponding SAC_StartMediaHandler event for
  every SAC_EndStyleHandler event.

  selectors - It is an array of all selectors for the style rule.  
*/
typedef void (*SAC_EndSelectorHandler)(void *userData,
				       const SAC_Selector * selectors[]);

/*
  Receive notification of a property declaration.

  The SAC Parser will call this method to report each property inside a rule.
  Note that shorthand properties will be reported as-is.

  propertyName - The name of the property.
  value - The value of the property.
  important - true if this property is important (i.e. "!important"),
              false otherwise.
 */
typedef void (*SAC_PropertyHandler)(void *userData,
				    const SAC_STRING propertyName,
				    const SAC_LexicalUnit * value,
				    SAC_Boolean important);

/*
  Methods to set handlers on the parser.
 */

void SAC_SetDocumentHandler(SAC_Parser parser,
			    SAC_StartDocumentHandler start,
			    SAC_EndDocumentHandler end);

void SAC_SetCommentHandler(SAC_Parser parser,
			   SAC_CommentHandler handler);

void SAC_SetIgnorableAtRuleHandler(SAC_Parser parser,
				   SAC_IgnorableAtRuleHandler handler);

void SAC_SetNamespaceDeclarationHandler(SAC_Parser parser,
					SAC_NamespaceDeclarationHandler handler);

void SAC_SetImportHandler(SAC_Parser parser,
			  SAC_ImportHandler handler);

void SAC_SetPageHandler(SAC_Parser parser,
			SAC_StartPageHandler start,
			SAC_EndPageHandler end);

void SAC_SetFontFaceHandler(SAC_Parser parser,
			    SAC_StartFontFaceHandler start,
			    SAC_EndFontFaceHandler end);

void SAC_SetMediaHandler(SAC_Parser parser,
			 SAC_StartMediaHandler start,
			 SAC_EndMediaHandler end);

void SAC_SetSelectorHandler(SAC_Parser parser,
			    SAC_StartStyleHandler start,
			    SAC_EndStyleHandler end);

void SAC_SetPropertyHandler(SAC_Parser parser,
			    SAC_PropertyHandler handler);

/* This value is passed as the userData argument to callbacks. */
void SAC_SetUserData(SAC_Parser parser, void *userData);

/* Returns the last value set by SAC_SetUserData or null. */
void * SAC_GetUserData(SAC_Parser parser);

/* Sets the base to be used for resolving relative URIs in system identifiers
  in declarations.  Resolving relative identifiers is left to the application:
  this value will be passed through as the base argument to the
  SAC_ImportHandler. The base argument will be copied. Returns zero if out of
  memory, non-zero otherwise. */
int SAC_SetBase(SAC_Parser parser, const SAC_STRING base);

const SAC_STRING SAC_GetBase(SAC_Parser parser);

int SAC_ParseStyleSheet(SAC_Parser parser, const char * buffer, int len);

int SAC_ParseStyleDeclaration(SAC_Parser parser, const char * buffer, int len);

int SAC_ParseRule(SAC_Parser parser, const char * buffer, int len);

const SAC_Selector * selectors[] SAC_ParseSelectors(SAC_Parser parser,
						    const char * buffer, int len);

const SAC_LexicalUnit * value SAC_ParserPropertyValue(SAC_Parser parser, 
						      const char * buffer, 
						      int len);

SAC_Boolean SAC_ParserPriority(SAC_Parser parser, const char * buffer, int len);


SAC_Parser SAC_CreateParser();

void SAC_DisposeParser(SAC_Parser parser);

/* Returns a string about which CSS language is supported by this
  parser. For CSS Level 1, it returns "http://www.w3.org/TR/REC-CSS1", for
  CSS Level 2, it returns "http://www.w3.org/TR/REC-CSS2". Note that a
  "CSSx" parser can return lexical unit other than those allowed by CSS
  Level x but this usage is not recommended.
*/
const char * SAC_GetParserVersion(SAC_Parser parser);

typedef enum _SAC_FatalError_Code {
    SAC_FATAL_ERROR_NONE,
    SAC_FATALERROR_NO_MEMORY
    /* @@CONTINUEME */
} SAC_FatalError_Code;

typedef struct _SAC_Fatal_Error {
    signed int     line;   /* -1 if unknown */
    signed int     column; /* -1 if unknown */
    SAC_FatalError_Code code;
} SAC_Fatal_Error;

typedef void (*SAC_FatalErrorHandler)(void *userData,
				      const SAC_Fatal_Error * fatalError);

void SAC_SetFatalErrorHandler(SAC_Parser parser,
			      SAC_FatalErrorHandler handler);


typedef enum _SAC_Error_Code {
    SAC_UNSPECIFIED_ERR,
    SAC_NOT_SUPPORTED_ERR,
    SAC_SYNTAX_ERR
} SAC_Error_Code;

typedef struct _SAC_Error {
    signed int       line;   /* -1 if unknown */
    signed int       column; /* -1 if unknown */
    SAC_Error_Code   code;
    SAC_STRING data;   /* NULL if it is not relevant */
} SAC_Error;

typedef void (*SAC_ErrorHandler)(void *userData,
				 const SAC_Error * error);

void SAC_SetErrorHandler(SAC_Parser parser,
			 SAC_ErrorHandler handler);

#endif /* _PARSER_H_ */
