#!/usr/bin/gawk -f
#
# Read the list of deliverables in ini-file format and generate an
# HTML table for insertion in current-work.*.html
#
# Author: Bert Bos <bert@w3.org>
# Created: 27 Aug 2014

BEGIN {
  TEXTDOMAIN = "CSS";
  bindtextdomain(root ? root : ".");
  IGNORECASE = 1;		# Ini-files are case-insensitive

  printf "<table class=status summary=\"%s\">\n",
    esc(_"List of CSS specifications in development or maintenance, with their current status.");

  # Sections in the order of output, and their localized names.
  sections[1] = "completed"; stitle["completed"] = _"Com&shy;pleted";
  sections[2] = "stable";    stitle["stable"] = _"Stable";
  sections[3] = "testing";   stitle["testing"] = _"Testing";
  sections[4] = "refine";    stitle["refine"] = _"Refining";
  sections[5] = "revise";    stitle["revise"] = _"Revising";
  sections[6] = "explore";   stitle["explore"] = _"Exploring";
  sections[7] = "rewrite";   stitle["rewrite"] = _"Rewriting";
  sections[8] = "abandon";   stitle["abandon"] = _"Abandoned";

  # Known document statuses, used for error checking.
  statuscodes[""] = 1;
  statuscodes["cr"] = 1;
  statuscodes["fpwd"] = 1;
  statuscodes["note"] = 1;
  statuscodes["pr"] = 1;
  statuscodes["rec"] = 1;
  statuscodes["spsd"] = 1;
  statuscodes["wd"] = 1;
}

/^\[.*\][ \t]*$/ {id[++n] = gensub(/^\[(.*)\][ \t]*$/, "\\1", 1); next}
/^[ \t]*title[ \t]*=/ {title[n] = val($0); next}
/^[ \t]*href[ \t]*=/ {href[n] = val($0); next}
/^[ \t]*status[ \t]*=/ {status[n] = val($0); check_status(status[n]); next}
/^[ \t]*next[ \t]*=/ {nextstatus[n] = val($0); next}
/^[ \t]*section[ \t]*=/ {section[n] = val($0); count[section[n]]++; check_section(section[n]); next}
/^[ \t]*ed[ \t]*=/ {ed[n] = val($0); next}
/^[ \t]*info[ \t]*=/ {info[n] = val($0); next}
/^[ \t]*note[ \t]*=/ {note[n] = val($0); next}
/^[ \t]*$/ {next}		# Ignore empty line
/^#/ {next}			# Ignore comment line
{printf "%s:%d: Syntax error\n", FILENAME, FNR; exit 1}
  

END {
  for (i = 1; i in sections; i++) print_tbody(sections[i], stitle[sections[i]]);
  printf "</table>\n";
}

# print_tbody -- print the tbody and rows for a section
function print_tbody(key, label,	i, title2)
{
  if (count[key]) {
    printf "<tbody id=\"%s\">\n", key;
    printf "<tr>\n";
    printf "<th scope=rowgroup>%s\n", label;
    printf "<th scope=col>%s\n", _"Cur&shy;rent";
    printf "<th scope=col>%s\n", _"Up&shy;com&shy;ing"
    printf "<th scope=col>%s\n", _"Notes";
    printf "<th scope=col><abbr title=\"%s\">%s</abbr>\n\n",
      _"Sum&shy;mary", _"&#8505;&#8413;";

    for (i = 1; id[i]; i++) {
      if (section[i] == key) {

	title2 = title[i] ? dcgettext(title[i]) : "";
	title2 = title2 == title[i] ? "" : "(" title2 ")";

	printf "<tr id=\"%s\">\n", id[i];
	if (i in href && href[i] != "") {
	  printf "<td><a class=notranslate lang=en href=\"%s\">%s</a> %s\n",
	    esc(href[i]), title[i], title2;
	  printf "<td class=\"%s\"><a href=\"%s\">%s</a>\n",
	    status[i], esc(href[i]), toupper(status[i]);
	} else {
	  printf "<td><span class=notranslate lang=en>%s</span> %s\n",
	    title[i], title2;
	  printf "<td class=\"%s\">%s\n",
	    status[i], toupper(status[i]);
	}
	printf "<td class=\"%s\"><a", nextstatus[i];
	if (ed[i]) printf " href=\"%s\"", ed[i];
	printf ">%s</a>\n", toupper(nextstatus[i]);
	printf "<td>%s\n", note[i] ? dcgettext(note[i]) : "";
	if (i in info && info[i] != "")
	  printf "<td><a href=\"%s\" title=\"%s %s\">%s</a>\n\n",
	    info[i],  esc(_"Summary of"), esc(title[i]), _"&#8505;&#8413;";
	else
	  printf "<td>";
	# printf "<td>";
	# if (i in info && info[i] != "") {
	#   printf "<a href=\"%s\" title=\"", info[i]
	#   printf _"Summary of %s", esc(title[i])
	#   printf "\">%s</a>\n\n",  _"&#8505;&#8413;"
	# }
      }
    }
  }
}


# val -- return the value of a property (the trimmed part after the "=")
function val(s)
{
  sub(/^[^=]*=[ \t]*/, "", s);
  sub(/[ \t]+$/, "", s);
  return s;
}


# esc -- escape " (for use in double-quote-delimited attributes)
function esc(s)
{
  return gensub(/"/, "\\&quot;", "g", s);
}


# check_section -- exit with an error if the string isn't a valid section name
function check_section(s)
{
  if (! (s in stitle)) {
    printf "%s:%d: Unknown section \"%s\"\n", FILENAME, FNR, s >"/dev/stderr"
    exit 1
  }
}


# check_status -- exit with an error if the string isn't a valid status code
function check_status(s)
{
  if (! (s in statuscodes)) {
    printf "%s:%d Unknown status \"%s\"\n", FILENAME, FNR, s >"/dev/stderr"
    exit 1
  }
}
