#!/usr/bin/awk
#
# Input is http://www.w3.org/2002/01/tr-automation/tr.rdf converted to 
# N-Triples (with "rapper"). Output is a series of HTML TR elements with
# three columns: date (YYY-MM-DD), title and status (REC, WD, etc.)
#
# Set LC_ALL or LANG to a locale to get translated
# titles (if available). E.g., LC_ALL=de_DE.utf8 for German.
#
# Call with "-v status=1" (or 2 or 3) to get completed specifications
# (resp., high-priority or draft). Default is 1.
#
# Todo: make URLs relative.
# Todo: localize dates with strftime()
#
# Created: 5 December 2011
# Author: Bert Bos <bert@w3.org>

BEGIN {
  TEXTDOMAIN = "mk-tr-list";
  bindtextdomain(root ? root : ".");
  CSSWG = "<http://www.w3.org/Style/CSS/members>";
  TITLE = "<http://purl.org/dc/elements/1.1/title>";
  HOMEPAGE = "<http://www.w3.org/2000/10/swap/pim/contact#homePage>";
  TYPE = "<http://www.w3.org/1999/02/22-rdf-syntax-ns#type>";
  DATE = "<http://purl.org/dc/elements/1.1/date>";
  VERSIONOF = "<http://www.w3.org/2000/10/swap/pim/doc#versionOf>";
  DELIVEREDBY = "<http://www.w3.org/2001/04/roadmap/org#deliveredBy>";
  NOTE = "<http://www.w3.org/2001/02pd/rec54#NOTE>";
  REC = "<http://www.w3.org/2001/02pd/rec54#REC>";
  CR = "<http://www.w3.org/2001/02pd/rec54#CR>";
  PR = "<http://www.w3.org/2001/02pd/rec54#PR>";
  PER = "<http://www.w3.org/2001/02pd/rec54#PER>";
  WD = "<http://www.w3.org/2001/02pd/rec54#WD>";
  RETIRED = "<http://www.w3.org/2001/02pd/rec54#Retired>";
}

/^[ \t]*$/ {next}		# Skip empty lines
/^[ \t]*#/ {next}		# Skip comments

$2 == HOMEPAGE {homePage[$1] = getvalue($0)}
$2 == TITLE {title[$1] = getvalue($0)}
$2 == TYPE {type[$1] = getvalue($0)}
$2 == DATE {date[$1] = getvalue($0)}
$2 == VERSIONOF {versionOf[$1] = getvalue($0)}
$2 == DELIVEREDBY {deliveredBy[$1] = deliveredBy[$i] getvalue($0) "\t"}

END {
  if (status == 3) {		# Output only WDs
    statuscode[WD] = 1;
  } else if (status == 2) {	# Output CRs, PRs and PERs
    statuscode[CR] = 1;
    statuscode[PR] = 1;
    statuscode[PER] = 1;
  } else {			# Output RECs and NOTEs
    statuscode[REC] = 1;
    statuscode[NOTE] = 1;
  }

  # Store the HTML lines in an array, prefixed with the date
  # deliveredBy contains one or more tab-separated "blank nodes."
  # We check if one of those represents the CSS WG.
  # (deliveredBy ends with a tab, so ignore the last field.)
  #
  for (spec in deliveredBy) {
    if (type[spec] in statuscode) {
      i = split(deliveredBy[spec], h, /\t/) - 1;
      while (i > 0 && homePage[h[i]] != CSSWG) i--;
      if (i > 0) x[++n] = format_line(spec);
    }
  }
  # Sort the array on the date,
  # then print it (without the date prefix).
  #
  asort(x);
  for (; n > 0; n--) print substr(x[n], index(x[n], "\t") + 1);
}


# format_line -- create an HTML table row for a spec x, prefixed with the date
function format_line(x,       d, s, h, t, translated, latest, short, time)
{
  latest = strip(versionOf[x]);
  short = latest;
  sub(/\/$/, "", short);	# Remove possible trailing slash
  gsub(/[^/]*\//, "", short);	# Remove all components that end with a slash
  d = strip(date[x]);
  time = mktime(substr(d, 1, 4) " " substr(d, 6, 2) " " substr(d, 9, 2) " 0 0 0");
  s = url_to_status(type[x]);
  t = strip(title[x]);
  translated = dcgettext(t);
  h = d "\t<tr><td><a href=\"http://www.w3.org/standards/history/" short "\"";
  h = h " title=\"" _"Publication history" "\">" strftime("%e %b %Y", time);
  h = h "</a><td><a class=notranslate lang=en href=\"" strip(x) "\">" t "</a>";
  if (t != translated) h = h " (" translated ")";
  h = h "<td class=" tolower(s) ">" s;
  return h;
}


# getvalue -- isolate the value (object) of a triple
function getvalue(line)
{
  sub(/^[ \t]*[^ \t][^ \t]*[ \t][ \t]*[^ \t][^ \t]*[ \t][ \t]*/, "", line);
  sub(/[ \t]*\.[ \t]*$/, "", line);
  return line;
}


# strip -- remove <> and "", unescape, and then escape HTML delimiters
function strip(s)
{
  if (s ~ /^</) {
    s = substr(s, 2);
    sub(/>$/, "", s);
  } else if (s ~ /^"/) {
    s = substr(s, 2);
    sub(/"$/, "", s);
  }
  gsub(/\\t/, "&#9;", s);
  gsub(/\\n/, "&#10;", s);
  gsub(/\\r/, "&#13;", s);
  gsub(/\\"/, "&quot;", s);
  gsub(/\\u([0-9a-fA-F]{4})/, "&#x\1", s);
  gsub(/\\U([0-9a-fA-F]{8})/, "&#x\1", s);
  gsub(/\\\\/, "\\", s);
  gsub(/&/, "&amp;", s);
  gsub(/</, "&lt;", s);
  return s;
}


# url_to_status -- return the document status code
function url_to_status(u)
{
  if (u == WD) return "WD";
  if (u == CR) return "CR";
  if (u == REC) return "REC";
  if (u == NOTE) return "NOTE";
  if (u == PR) return "PR";
  if (u == PER) return "PER";
  if (u == RETIRED) return _"Retired";
  return _"Unknown";
}
