#!/usr/bin/gawk -f
#
# A typical entry in a .foo.var file looks like this:
#
#     # Magyar [2011-11-01]
#     # Translator: Szabolcs Varga (ClearDebt - IVA cég)
#     # Translator-e-mail: shirokuma@shirokuma.hu
#     # Translator-URI: http://www.cleardebt.co.uk/iva
#     # Downloaded from http://www.cleardebt.co.uk/learning.hu.html
#     # Rel: nofollow
#     URI: learning.hu.html
#     Content-Type: text/html; qs=0.5
#     Content-Language: hu
#
# or like this:
#
#     # &#1038;&#1079;&#1073;&#1077;&#1082; [2010-09-30]
#     # http://www.uzbek-webstyles.com/Style/CSS/learning.html
#     # Translator: Axel Friedrich
#     # Translator-e-mail: Axel Friedrich <axelfr74@googlemail.com>
#     ## Translator-URI: http://www.uzbek-webstyles.com/
#     URI: learning.uz.html.redirect
#     Content-Type: text/html; qs=0.5
#     Content-Language: uz
#
# The first line must have the name of the language in that
# language. The comment between square brackets (typically a date) is
# optional. The last three lines are are also required. The other
# lines are optional and are treated as comments, except when they
# start with one of the following (exactly, i.e., with the exact
# spaces as well):
#
#     # http:
#     # ftp:
#     # Translator:
#     # Translator-URI:
#     # Rel:
#
# If there is a line with http or ftp, then that is the link to the
# translation, replacing whatever is on the URI: line. If there is a
# Translator, then it is mentioned at the top of the generated
# page. If there is also a Translator-URI, then the translator is made
# into a link to that URL. And if there is a Rel line, the value of it
# is added as rel="..." to that link.

BEGIN {
  TEXTDOMAIN = "CSS";
  bindtextdomain(root ? root : ".");
  printf "\n";
}
/^# / {if (name == "") {name = substr($0, 3); u = ""}}
/^# *(http:|ftp:)/ {u = $2}
/^# Translator:/ {tn = substr($0, 15)}
/^# Translator-URI:/ {tu = $3}
/^# Rel: / {tr = substr($0, 8)}
/^URI:/ {if (u == "") u = $2}
/^Content-Type:/ {}
/^Content-Language:/ {
  sub(/;.*/, ""); 
  label[$2] = name;
  url[$2] = u;
  lang[n++] = $2;
  if (tn) {translator[$2] = tn; tn = ""}
  if (tu) {translatorurl[$2] = tu; tu = ""}
  if (tr) {translatorrel[$2] = tr; tr = ""}
  name = "";
}
END {
  if (n > 1) {
    printf "<div class=section id=language>\n";
    printf "<p><a href=\"#translations\">%s</a>\n", _"Languages";
    lc = tolower("LC_ALL" in ENVIRON ? ENVIRON["LC_ALL"] :
		 "LANG" in ENVIRON ? ENVIRON["LANG"] : "en_US.UTF-8");
    gsub(/\..*/, "", lc);
    gsub(/_/, "-", lc);
    # Reduce xx-yy to xx, unless there is a translation specifically for xx-yy
    if (! (lc in url)) gsub(/-.*/, "", lc);
    if (lc != "en") {
      printf "<p>%s\n%s\n<a rel=alternate hreflang=en href=\"%s\">%s</a>\n%s\n",
	_"This is a translation.",
	_"It may contain errors or be out of date with respect to",
	url["en"],
	_"the English version.",
	_"   ";
      if (lc in translator) {
	printf "%s",  _"Translator: ";
	if (lc in translatorurl) {
	  printf "<a href=\"%s\"", translatorurl[lc];
	  if (lc in translatorrel) printf " rel=\"%s\"", translatorrel[lc];
	  printf ">%s</a>\n",  translator[lc];
	} else {
	  printf "%s\n", translator[lc];
	}
      }
    }
    printf "</div>\n";
  }
}
