//-------------------------------------------------------------------------------------------------------
// Copyright (C) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.txt file in the project root for full license information.
//-------------------------------------------------------------------------------------------------------

/*
    The table below is a hash table with integratedoverflow entries, designed
    to optimize the conversion from an arbitrary 64-bit immediate value
    to a valid immediate encoding.

    To use the hash table, take the 64-bit immediate and MOD it against
    the ARM64_LOGICAL_IMMEDIATE_HASH_MODULO value defined below. Use that value
    as an index into the table below. If the corresponding entry does not
    contain a matching 64-bit value in ImmediateValue, then read the
    NextWithSameHash value and use that as an index to find the next table entry
    with the same hash, repeating until a match is found or until NextWithSameHash
    is -1. If the value is not found in the table, then the 64-bit input
    cannot be encoded as an immediate value.

    If a matching entry is found, the encoding can be found in the Encoding
    array. Entry 0 is for a 32-bit encoding, and entry 1 is for a 64-bit
    encoding. If the encoding value is ARM64_LOGICAL_IMMEDIATE_NO_ENCODING, then that value cannot be
    encoded for that bit size.

    The hash constant was obtained by finding a value that produced very
    few empty slots in the hash table (only ~20) and also resulted in a
    short average and maximum search depth for each hash table entry point.

    The content of this file is automatically generated by building under
    LogImmGen. To save build time, this code is not hooked to the
    build and the file is checked in.

    To update this header follow these steps:
      1. check out header
      2. cd LogImmGen & bcz
      3. check in header
*/

#ifndef __ARM64_LOGICAL_IMMEDIATES__
#define __ARM64_LOGICAL_IMMEDIATES__

const int ARM64_LOGICAL_IMMEDIATE_HASH_MODULO = 6214;
const unsigned short ARM64_LOGICAL_IMMEDIATE_NO_ENCODING = 0xFFFF;

typedef struct {
    unsigned __int64 ImmediateValue;
    int NextWithSameHash;
    unsigned short Encoding[2];
} Arm64LogicalImmediateEncoding;

inline unsigned short FindArm64LogicalImmediateEncoding(unsigned __int64 Value, int Size) {
    extern const Arm64LogicalImmediateEncoding g_Arm64LogicalImmediateEncodingTable[6214];
    for (int Index = int(Value % ARM64_LOGICAL_IMMEDIATE_HASH_MODULO); Index != -1; Index = g_Arm64LogicalImmediateEncodingTable[Index].NextWithSameHash) {
        const Arm64LogicalImmediateEncoding &Entry = g_Arm64LogicalImmediateEncodingTable[Index];
        if (Entry.ImmediateValue == Value) {
            return Entry.Encoding[Size/8];
        }
    }
    return ARM64_LOGICAL_IMMEDIATE_NO_ENCODING;
}

#endif

#ifdef ARM64_INCLUDE_LOGICAL_IMMEDIATE_TABLE
extern const Arm64LogicalImmediateEncoding g_Arm64LogicalImmediateEncodingTable[6214] = {
{0xfffffffc7fffffff,    5,{0xffff,0x17bc}},  //     0
{0x0000000000000001,   -1,{0x0000,0x1000}},  //     1, depth=1
{0x0000000000000002,   -1,{0x07c0,0x1fc0}},  //     2, depth=1
{0x0000000000000003,    0,{0x0001,0x1001}},  //     3, depth=3
{0x0000000000000004,   -1,{0x0780,0x1f80}},  //     4, depth=1
{0xff87ffffff87ffff,   -1,{0xffff,0x025b}},  //     5
{0x1fff80001fff8000,   10,{0xffff,0x044d}},  //     6, depth=2
{0x0000000000000007,   13,{0x0002,0x1002}},  //     7, depth=2
{0x0000000000000008,   -1,{0x0740,0x1f40}},  //     8, depth=1
{0x8000001fffffffff,   -1,{0xffff,0x1065}},  //     9, depth=1
{0x0000000000000006,   -1,{0x07c1,0x1fc1}},  //    10
{0xbfffbfffbfffbfff,   20,{0xffff,0x006e}},  //    11, depth=2
{0x3fff00003fff0000,   21,{0xffff,0x040d}},  //    12, depth=2
{0xffff00000001ffff,   -1,{0xffff,0x1420}},  //    13
{0x000000000000000e,   26,{0x07c2,0x1fc2}},  //    14, depth=2
{0x000000000000000f,   -1,{0x0003,0x1003}},  //    15, depth=1
{0x0000000000000010,   -1,{0x0700,0x1f00}},  //    16, depth=1
{0xffffffffffc3ffff,   -1,{0xffff,0x1abb}},  //    17, depth=1
{0x1fffc00000000000,   -1,{0xffff,0x148e}},  //    18, depth=1
{0x00000000e007e007,   -1,{0x00e5,0xffff}},  //    19, depth=1
{0xfffffffffe00007f,   -1,{0xffff,0x19ed}},  //    20
{0x000000000000000c,   -1,{0x0781,0x1f81}},  //    21
{0x0000fffe00000000,   -1,{0xffff,0x17ce}},  //    22, depth=1
{0xfff000000007ffff,   -1,{0xffff,0x131e}},  //    23, depth=1
{0x7ffe00007ffe0000,   27,{0xffff,0x03cd}},  //    24, depth=2
{0xc000000000000fff,   -1,{0xffff,0x108d}},  //    25, depth=1
{0x07fffe0000000000,   -1,{0xffff,0x15d1}},  //    26
{0x0000000000000018,   -1,{0x0741,0x1f41}},  //    27
{0x000000000000001c,   37,{0x0782,0x1f82}},  //    28, depth=2
{0xfffffffe0001ffff,   -1,{0xffff,0x17ef}},  //    29, depth=1
{0x000000000000001e,   40,{0x07c3,0x1fc3}},  //    30, depth=2
{0x000000000000001f,   42,{0x0004,0x1004}},  //    31, depth=2
{0x0000000000000020,   -1,{0x06c0,0x1ec0}},  //    32, depth=1
{0x00000000ffc000ff,   -1,{0x0291,0xffff}},  //    33, depth=1
{0x01fffffffffffc00,   -1,{0xffff,0x1dae}},  //    34, depth=1
{0xe7ffffffe7ffffff,   -1,{0xffff,0x00dd}},  //    35, depth=1
{0x3fff800000000000,   -1,{0xffff,0x144e}},  //    36, depth=1
{0x0ffffc0000000000,   -1,{0xffff,0x1591}},  //    37
{0x0000380000000000,   -1,{0xffff,0x1542}},  //    38, depth=1
{0xffffff0fffffffff,   -1,{0xffff,0x163b}},  //    39, depth=1
{0x001fffffffffe000,   -1,{0xffff,0x1ce7}},  //    40
{0xffffffe000001fff,   43,{0xffff,0x16e7}},  //    41, depth=2
{0xffffffffff8003ff,   -1,{0xffff,0x1a72}},  //    42
{0xfffffffc003fffff,   -1,{0xffff,0x17b3}},  //    43
{0x0001fffc00000000,   -1,{0xffff,0x178e}},  //    44, depth=1
{0xffc01fffffffffff,   -1,{0xffff,0x12b6}},  //    45
{0x0000000000000030,   -1,{0x0701,0x1f01}},  //    46
{0xfc001ffffc001fff,   45,{0xffff,0x0192}},  //    47, depth=2
{0xfffc0000fffc0000,   46,{0xffff,0x038d}},  //    48, depth=2
{0x000000009fff9fff,   -1,{0x006d,0xffff}},  //    49, depth=1
{0x000000000fe00000,   -1,{0x02c6,0x1ac6}},  //    50, depth=1
{0x00000000ff80007f,   52,{0x024f,0xffff}},  //    51, depth=2
{0xfffe00003fffffff,   -1,{0xffff,0x13ec}},  //    52
{0x01f0000000000000,   57,{0xffff,0x1304}},  //    53
{0x7000000070000000,   53,{0xffff,0x0102}},  //    54, depth=3
{0xffffffff8003ffff,   -1,{0xffff,0x1872}},  //    55, depth=1
{0x0000000000000038,   58,{0x0742,0x1f42}},  //    56, depth=2
{0x000000003fc00000,   -1,{0x0287,0x1a87}},  //    57
{0x1ffff80000000000,   -1,{0xffff,0x1551}},  //    58
{0x003fffffffffc000,   -1,{0xffff,0x1ca7}},  //    59
{0x000000000000003c,   59,{0x0783,0x1f83}},  //    60, depth=2
{0xe0000000ffffffff,   73,{0xffff,0x10e2}},  //    61, depth=2
{0x000000000000003e,   -1,{0x07c4,0x1fc4}},  //    62, depth=1
{0x000000000000003f,   75,{0x0005,0x1005}},  //    63, depth=2
{0x0000000000000040,   -1,{0x0680,0x1e80}},  //    64, depth=1
{0xfe0ffe0ffe0ffe0f,   80,{0xffff,0x01ea}},  //    65, depth=2
{0x0003800000038000,   83,{0xffff,0x0442}},  //    66, depth=3
{0x00000000f8001fff,   -1,{0x0151,0xffff}},  //    67, depth=1
{0x03fffffffffff800,   -1,{0xffff,0x1d6e}},  //    68, depth=1
{0x00000000ff3fffff,   85,{0x021d,0xffff}},  //    69, depth=2
{0x0000000fffffff00,   -1,{0xffff,0x1e1b}},  //    70, depth=1
{0x00000000800000ff,   -1,{0x0048,0xffff}},  //    71, depth=1
{0x7fff000000000000,   -1,{0xffff,0x140e}},  //    72, depth=1
{0xffff800000000fff,   -1,{0xffff,0x145c}},  //    73
{0x0000001ffffe0000,   -1,{0xffff,0x1bd3}},  //    74, depth=1
{0xffffff8000003fff,   -1,{0xffff,0x1666}},  //    75
{0x0000700000000000,   -1,{0xffff,0x1502}},  //    76, depth=1
{0x00000000fffe03ff,   -1,{0x03d8,0xffff}},  //    77, depth=1
{0x0007ff800007ff80,   -1,{0xffff,0x064b}},  //    78, depth=1
{0xffffc000ffffffff,   -1,{0xffff,0x14b1}},  //    79, depth=1
{0xffe3ffffffffffff,   -1,{0xffff,0x12fc}},  //    80
{0x8181818181818181,   86,{0xffff,0x0071}},  //    81, depth=2
{0x07f007f007f007f0,   -1,{0xffff,0x0326}},  //    82, depth=1
{0x00000f8000000000,   84,{0xffff,0x1644}},  //    83
{0x000000000001fe00,   -1,{0x05c7,0x1dc7}},  //    84
{0xfffffff8000fffff,   -1,{0xffff,0x1770}},  //    85
{0xffff001fffffffff,   -1,{0xffff,0x1434}},  //    86
{0x00000000f3f3f3f3,   89,{0x0135,0xffff}},  //    87, depth=2
{0x0003fff800000000,   -1,{0xffff,0x174e}},  //    88, depth=1
{0xfffffff000007fff,   -1,{0xffff,0x172a}},  //    89
{0xfffe00001fffffff,   -1,{0xffff,0x13eb}},  //    90
{0x03e0000000000000,   92,{0xffff,0x12c4}},  //    91
{0x000000007f800000,   -1,{0x0247,0x1a47}},  //    92
{0xffffe000000001ff,   -1,{0xffff,0x14db}},  //    93, depth=1
{0x7ffffffe00000000,   -1,{0xffff,0x17dd}},  //    94, depth=1
{0xfffffffc3fffffff,   -1,{0xffff,0x17bb}},  //    95, depth=1
{0x0000000000000060,   -1,{0x06c1,0x1ec1}},  //    96, depth=1
{0x80000fff80000fff,   90,{0xffff,0x004c}},  //    97, depth=2
{0xffffffe0000007ff,   -1,{0xffff,0x16e5}},  //    98
{0xfffc0003ffffffff,   -1,{0xffff,0x13af}},  //    99, depth=1
{0x000000001fc00000,   -1,{0x0286,0x1a86}},  //   100, depth=1
{0xf0ffffffffffffff,   -1,{0xffff,0x113b}},  //   101, depth=1
{0x3ffff00000000000,   -1,{0xffff,0x1511}},  //   102
{0xffe000000000ffff,   -1,{0xffff,0x12da}},  //   103, depth=1
{0x007fffffffff8000,   -1,{0xffff,0x1c67}},  //   104
{0xfffffe0000ffffff,   -1,{0xffff,0x15ee}},  //   105
{0x00001f0000000000,  109,{0xffff,0x1604}},  //   106
{0xffffffff000003ff,   -1,{0xffff,0x1829}},  //   107, depth=1
{0xe0000000e0000000,   91,{0xffff,0x00c2}},  //   108, depth=3
{0x000000000003fc00,   -1,{0x0587,0x1d87}},  //   109
{0xfffff00000000fff,  116,{0xffff,0x151f}},  //   110
{0xff007fffff007fff,   98,{0xffff,0x0216}},  //   111, depth=2
{0x0000000000000070,  102,{0x0702,0x1f02}},  //   112, depth=2
{0xf800000000ffffff,   -1,{0xffff,0x115c}},  //   113, depth=1
{0xffff8000ffff8000,   -1,{0xffff,0x0450}},  //   114, depth=1
{0xfffffffffffe1fff,   -1,{0xffff,0x1bfb}},  //   115, depth=1
{0xe0000000007fffff,   -1,{0xffff,0x10d9}},  //   116
{0xfffff800ffffffff,   -1,{0xffff,0x1574}},  //   117, depth=1
{0x00000003fff00000,   -1,{0xffff,0x1b0d}},  //   118, depth=1
{0x03ffffffffffe000,   -1,{0xffff,0x1cec}},  //   119
{0x0000000000000078,  104,{0x0743,0x1f43}},  //   120, depth=2
{0xfffe0ffffffe0fff,   -1,{0xffff,0x03da}},  //   121, depth=1
{0x000001fffffff000,   -1,{0xffff,0x1d1c}},  //   122, depth=1
{0x00000000fe7ffe7f,   -1,{0x01ed,0xffff}},  //   123, depth=1
{0x000000000000007c,   -1,{0x0784,0x1f84}},  //   124, depth=1
{0xff00000001ffffff,   -1,{0xffff,0x1220}},  //   125, depth=1
{0x000000000000007e,   -1,{0x07c5,0x1fc5}},  //   126, depth=1
{0x000000000000007f,  105,{0x0006,0x1006}},  //   127, depth=2
{0x0000000000000080,   -1,{0x0640,0x1e40}},  //   128, depth=1
{0xfffffffc1fffffff,   -1,{0xffff,0x17ba}},  //   129
{0xfffe000000000003,  131,{0xffff,0x13d0}},  //   130
{0xc00001ffffffffff,   -1,{0xffff,0x10aa}},  //   131
{0x0007000000070000,  106,{0xffff,0x0402}},  //   132, depth=3
{0x0000003ffffc0000,   -1,{0xffff,0x1b93}},  //   133
{0x0c00000000000000,   -1,{0xffff,0x1181}},  //   134, depth=1
{0xfffe000000000007,   -1,{0xffff,0x13d1}},  //   135
{0x07fffffffffff000,   -1,{0xffff,0x1d2e}},  //   136, depth=1
{0x00000000f80007ff,  110,{0x014f,0xffff}},  //   137, depth=3
{0x003ffffffc000000,  119,{0xffff,0x199b}},  //   138, depth=2
{0xfffffffffe0000ff,   -1,{0xffff,0x19ee}},  //   139, depth=1
{0x0000001ffffffe00,   -1,{0xffff,0x1ddb}},  //   140, depth=1
{0xffffff1fffffff1f,  129,{0xffff,0x061c}},  //   141, depth=2
{0x00007f8000000000,   -1,{0xffff,0x1647}},  //   142, depth=1
{0xfffdffffffffffff,   -1,{0xffff,0x13be}},  //   143, depth=1
{0xfffe000000000000,   -1,{0xffff,0x13ce}},  //   144, depth=1
{0xfffe000000000001,   -1,{0xffff,0x13cf}},  //   145, depth=1
{0x07f0000007f00000,   -1,{0xffff,0x0306}},  //   146, depth=1
{0x03ffffff03ffffff,  130,{0xffff,0x0019}},  //   147, depth=3
{0x000ffffe000ffffe,  133,{0xffff,0x07d2}},  //   148, depth=2
{0xffffe0000001ffff,   -1,{0xffff,0x14e3}},  //   149
{0xffffffffff83ffff,   -1,{0xffff,0x1a7a}},  //   150
{0xf0000ffff0000fff,  135,{0xffff,0x010f}},  //   151, depth=2
{0x0000e00000000000,   -1,{0xffff,0x14c2}},  //   152, depth=1
{0xffc00001ffffffff,   -1,{0xffff,0x12aa}},  //   153
{0xff9fff9fff9fff9f,   -1,{0xffff,0x026d}},  //   154
{0xffffc000007fffff,   -1,{0xffff,0x14a8}},  //   155, depth=1
{0x000fff00000fff00,   -1,{0xffff,0x060b}},  //   156, depth=1
{0xffff81ffffffffff,   -1,{0xffff,0x1479}},  //   157
{0x007fffc0007fffc0,   -1,{0xffff,0x0690}},  //   158, depth=1
{0xfffe00000000000f,  149,{0xffff,0x13d2}},  //   159, depth=2
{0xfffffffbffffffff,   -1,{0xffff,0x177e}},  //   160
{0xc7ffffffffffffff,   -1,{0xffff,0x10bc}},  //   161, depth=1
{0x0001fffffff00000,   -1,{0xffff,0x1b1c}},  //   162, depth=1
{0x00000000f1f1f1f1,  150,{0x0134,0xffff}},  //   163, depth=2
{0x0fe00fe00fe00fe0,   -1,{0xffff,0x02e6}},  //   164, depth=1
{0xffe007ffffe007ff,   -1,{0xffff,0x02d5}},  //   165, depth=1
{0xfffffffc00000001,   -1,{0xffff,0x179e}},  //   166
{0xff800007ffffffff,   -1,{0xffff,0x126b}},  //   167, depth=1
{0xfffffffc00000003,  169,{0xffff,0x179f}},  //   168
{0xfff8000000001fff,   -1,{0xffff,0x1359}},  //   169
{0x000000fffe000000,   -1,{0xffff,0x19ce}},  //   170, depth=1
{0xfffff800007fffff,  172,{0xffff,0x156b}},  //   171
{0xfe00003ffe00003f,   -1,{0xffff,0x01cc}},  //   172
{0x00000000fffc7fff,   -1,{0x039c,0xffff}},  //   173, depth=1
{0x000003fffffff800,   -1,{0xffff,0x1d5e}},  //   174, depth=1
{0xfffe00000000001f,   -1,{0xffff,0x13d3}},  //   175, depth=1
{0x0007fff000000000,   -1,{0xffff,0x170e}},  //   176, depth=1
{0xff03ffffff03ffff,  153,{0xffff,0x0219}},  //   177, depth=2
{0x3fffff8000000000,   -1,{0xffff,0x1656}},  //   178, depth=1
{0x00000000ff8000ff,  154,{0x0250,0xffff}},  //   179, depth=2
{0xffc0ffffffffffff,   -1,{0xffff,0x12b9}},  //   180
{0xfffffffc0000000f,   -1,{0xffff,0x17a1}},  //   181
{0xfffffffff81fffff,   -1,{0xffff,0x1979}},  //   182
{0xfffffffbfffffffb,  157,{0xffff,0x075e}},  //   183, depth=2
{0xfffffc03fffffc03,   -1,{0xffff,0x0597}},  //   184
{0xfffffffffffe07ff,   -1,{0xffff,0x1bf9}},  //   185, depth=1
{0x00000007f0000000,   -1,{0xffff,0x1906}},  //   186, depth=1
{0xfe07fe07fe07fe07,  160,{0xffff,0x01e9}},  //   187, depth=2
{0xfffffffc00000000,   -1,{0xffff,0x179d}},  //   188, depth=1
{0x00000000e00001ff,  166,{0x00cb,0xffff}},  //   189, depth=2
{0x07c0000000000000,   -1,{0xffff,0x1284}},  //   190
{0x00000000ffff3fff,  168,{0x041d,0xffff}},  //   191, depth=3
{0x00000000000000c0,   -1,{0x0681,0x1e81}},  //   192, depth=1
{0x0000000007ff07ff,  171,{0x002a,0xffff}},  //   193, depth=3
{0xfffffffc0000001f,   -1,{0xffff,0x17a2}},  //   194
{0xfffffffc00000007,   -1,{0xffff,0x17a0}},  //   195, depth=1
{0x7fffe00000000000,   -1,{0xffff,0x14d1}},  //   196
{0xffffff00000001ff,   -1,{0xffff,0x1620}},  //   197
{0x0000000fffffff80,   -1,{0xffff,0x1e5c}},  //   198, depth=1
{0xfffffdfffffffdff,  180,{0xffff,0x059e}},  //   199, depth=2
{0x000000003f800000,   -1,{0x0246,0x1a46}},  //   200, depth=1
{0xfcffffffffffffff,   -1,{0xffff,0x11bd}},  //   201
{0x0ffffffffffff800,   -1,{0xffff,0x1d70}},  //   202, depth=1
{0xffffff8000000fff,  181,{0xffff,0x1664}},  //   203, depth=2
{0xffffffff0003ffff,   -1,{0xffff,0x1831}},  //   204
{0x00000000fe3ffe3f,   -1,{0x01ec,0xffff}},  //   205, depth=1
{0x00ffffffffff0000,   -1,{0xffff,0x1c27}},  //   206
{0xfffe00000000003f,  182,{0xffff,0x13d4}},  //   207, depth=2
{0x0007ffffffffff00,   -1,{0xffff,0x1e2a}},  //   208
{0xffc1ffffffc1ffff,   -1,{0xffff,0x029a}},  //   209, depth=1
{0xfffffffff00fffff,  211,{0xffff,0x1937}},  //   210
{0xff83ffffffffffff,  212,{0xffff,0x127a}},  //   211
{0xe3ffffffe3ffffff,  213,{0xffff,0x00dc}},  //   212
{0x00000000e001ffff,   -1,{0x00d3,0xffff}},  //   213
{0x00000ff000000ff0,   -1,{0xffff,0x0707}},  //   214, depth=1
{0x00000000feffffff,  184,{0x01de,0xffff}},  //   215, depth=2
{0x00000000ff000000,  190,{0x0207,0x1a07}},  //   216, depth=2
{0x00000000ff000001,   -1,{0x0208,0xffff}},  //   217, depth=1
{0x0000000fffffffc0,  220,{0xffff,0x1e9d}},  //   218
{0x00000000ff000003,  194,{0x0209,0xffff}},  //   219, depth=2
{0x000001ffffffc000,   -1,{0xffff,0x1c9a}},  //   220
{0x00003e0000000000,  222,{0xffff,0x15c4}},  //   221
{0x000000000007f800,   -1,{0x0547,0x1d47}},  //   222
{0x00000000ff000007,   -1,{0x020a,0xffff}},  //   223, depth=1
{0x00000000000000e0,  196,{0x06c2,0x1ec2}},  //   224, depth=2
{0x8000001f8000001f,  197,{0xffff,0x0045}},  //   225, depth=2
{0x0000060000000000,   -1,{0xffff,0x15c1}},  //   226, depth=1
{0x0003fffff0000000,   -1,{0xffff,0x1915}},  //   227
{0x07ffffffffffc000,   -1,{0xffff,0x1cac}},  //   228
{0xffe0000fffe0000f,   -1,{0xffff,0x02ce}},  //   229, depth=1
{0x00fc000000000000,   -1,{0xffff,0x1385}},  //   230
{0x00000000ff00000f,   -1,{0x020b,0xffff}},  //   231, depth=1
{0x00000000ff00003f,   -1,{0x020d,0xffff}},  //   232
{0xffffe01fffffffff,   -1,{0xffff,0x14f7}},  //   233, depth=1
{0xff8000000000ffff,   -1,{0xffff,0x1258}},  //   234
{0x00000000c00001ff,  201,{0x008a,0xffff}},  //   235, depth=2
{0x00000007ffe00000,   -1,{0xffff,0x1acd}},  //   236, depth=1
{0x0000000fffffffe0,   -1,{0xffff,0x1ede}},  //   237
{0x00f8000000f80000,   -1,{0xffff,0x0344}},  //   238, depth=1
{0x00000000efefefef,  204,{0x00f6,0xffff}},  //   239, depth=2
{0x00000000000000f0,  206,{0x0703,0x1f03}},  //   240, depth=2
{0x0000007ffff80000,   -1,{0xffff,0x1b53}},  //   241
{0x0fe0000000000000,   -1,{0xffff,0x12c6}},  //   242, depth=1
{0x0380000000000000,   -1,{0xffff,0x1242}},  //   243
{0x000003ffffffe000,   -1,{0xffff,0x1cdc}},  //   244, depth=1
{0xe00000007fffffff,   -1,{0xffff,0x10e1}},  //   245, depth=1
{0x000000001f800000,  208,{0x0245,0x1a45}},  //   246, depth=2
{0x00000000ff00001f,   -1,{0x020c,0xffff}},  //   247, depth=1
{0x00000000000000f8,   -1,{0x0744,0x1f44}},  //   248, depth=1
{0xfffe001ffffe001f,   -1,{0xffff,0x03d3}},  //   249, depth=1
{0xfffff8007fffffff,  253,{0xffff,0x1573}},  //   250
{0xfffffffc0000003f,   -1,{0xffff,0x17a3}},  //   251, depth=1
{0x00000000000000fc,   -1,{0x0785,0x1f85}},  //   252, depth=1
{0xffffc000003fffff,   -1,{0xffff,0x14a7}},  //   253
{0x00000000000000fe,   -1,{0x07c6,0x1fc6}},  //   254, depth=1
{0x00000000000000ff,  210,{0x0007,0x1007}},  //   255, depth=5
{0x0000000000000100,   -1,{0x0600,0x1e00}},  //   256, depth=1
{0x01ffffff01ffffff,   -1,{0xffff,0x0018}},  //   257, depth=1
{0x00001ffffe000000,   -1,{0xffff,0x19d3}},  //   258
{0xfffff1ffffffffff,   -1,{0xffff,0x153c}},  //   259, depth=1
{0xffffffffe7ffffff,   -1,{0xffff,0x18fd}},  //   260
{0xfff80000000007ff,   -1,{0xffff,0x1357}},  //   261, depth=1
{0x0000000003f003f0,  218,{0x0325,0xffff}},  //   262, depth=3
{0xffffc0007fffffff,   -1,{0xffff,0x14b0}},  //   263, depth=1
{0x000e0000000e0000,  221,{0xffff,0x03c2}},  //   264, depth=3
{0xfe00007fffffffff,   -1,{0xffff,0x11ed}},  //   265, depth=1
{0x0001c00000000000,   -1,{0xffff,0x1482}},  //   266
{0xf8001ffff8001fff,   -1,{0xffff,0x0151}},  //   267, depth=1
{0x1800000000000000,   -1,{0xffff,0x1141}},  //   268, depth=1
{0xfffffffffff00003,   -1,{0xffff,0x1b2d}},  //   269
{0x0000000ffffffff0,   -1,{0xffff,0x1f1f}},  //   270
{0xfffe00000000007f,   -1,{0xffff,0x13d5}},  //   271, depth=1
{0x0fffffffffffe000,   -1,{0xffff,0x1cee}},  //   272, depth=1
{0xfff0007fffffffff,   -1,{0xffff,0x1332}},  //   273
{0x0000000038003800,  227,{0x0162,0xffff}},  //   274, depth=2
{0x0000000ffffffffc,   -1,{0xffff,0x1fa1}},  //   275
{0x007ffffff8000000,  228,{0xffff,0x195b}},  //   276, depth=2
{0x0000000ffffffffe,   -1,{0xffff,0x1fe2}},  //   277
{0x0000080000000800,  230,{0xffff,0x0540}},  //   278, depth=2
{0xf000001ff000001f,  232,{0xffff,0x0108}},  //   279, depth=2
{0x0000003ffffffc00,   -1,{0xffff,0x1d9b}},  //   280, depth=1
{0x0000000fffffffff,   -1,{0xffff,0x1023}},  //   281
{0x1ffffe0000000000,   -1,{0xffff,0x15d3}},  //   282, depth=1
{0xe0000000003fffff,   -1,{0xffff,0x10d8}},  //   283, depth=1
{0x0000ff0000000000,   -1,{0xffff,0x1607}},  //   284, depth=1
{0x0000001000000000,   -1,{0xffff,0x1700}},  //   285
{0xffc0000000ffffff,   -1,{0xffff,0x12a1}},  //   286
{0x00000000fdfffdff,   -1,{0x01ae,0xffff}},  //   287, depth=1
{0x3ffe3ffe3ffe3ffe,   -1,{0xffff,0x03ec}},  //   288, depth=1
{0x00000000ffc001ff,   -1,{0x0292,0xffff}},  //   289, depth=1
{0xfe000fffffffffff,  297,{0xffff,0x11f2}},  //   290
{0xffffff000001ffff,   -1,{0xffff,0x1628}},  //   291, depth=1
{0x0fe000000fe00000,   -1,{0xffff,0x02c6}},  //   292, depth=1
{0x00000000ffc1ffc1,  234,{0x02aa,0xffff}},  //   293, depth=2
{0x07fffffe07fffffe,  237,{0xffff,0x07d9}},  //   294, depth=2
{0xff0000ffffffffff,   -1,{0xffff,0x122f}},  //   295, depth=1
{0x001ffffc001ffffc,  241,{0xffff,0x0792}},  //   296, depth=2
{0x00000000ffff0fff,   -1,{0x041b,0xffff}},  //   297
{0x3e003e003e003e00,  243,{0xffff,0x01e4}},  //   298, depth=2
{0xfffdfffdfffdfffd,   -1,{0xffff,0x03ae}},  //   299, depth=1
{0xfffffffffff0001f,   -1,{0xffff,0x1b30}},  //   300
{0x00000000c001ffff,  250,{0x0092,0xffff}},  //   301, depth=3
{0x07ffff0007ffff00,  258,{0xffff,0x0612}},  //   302, depth=2
{0xfc00003ffc00003f,  260,{0xffff,0x018b}},  //   303, depth=2
{0xfefefefefefefefe,  266,{0xffff,0x01f6}},  //   304, depth=2
{0xffffffffffefffff,   -1,{0xffff,0x1afe}},  //   305, depth=1
{0xfffffffffff00000,   -1,{0xffff,0x1b2b}},  //   306, depth=1
{0xfffffffffff00001,   -1,{0xffff,0x1b2c}},  //   307, depth=1
{0xfffff800003fffff,   -1,{0xffff,0x156a}},  //   308
{0xfdfffdfffdfffdff,  269,{0xffff,0x01ae}},  //   309, depth=2
{0x001fffffffff8000,  270,{0xffff,0x1c65}},  //   310, depth=2
{0x0001f80000000000,   -1,{0xffff,0x1545}},  //   311
{0x001ffe00001ffe00,   -1,{0xffff,0x05cb}},  //   312, depth=1
{0xfffffffffff00007,   -1,{0xffff,0x1b2e}},  //   313, depth=1
{0xfffffe00ffffffff,  320,{0xffff,0x15f6}},  //   314
{0xfffffffc0000007f,   -1,{0xffff,0x17a4}},  //   315, depth=1
{0x00ffff8000ffff80,   -1,{0xffff,0x0650}},  //   316, depth=1
{0xffc000ffffc000ff,   -1,{0xffff,0x0291}},  //   317, depth=1
{0x0000000ffffffff8,   -1,{0xffff,0x1f60}},  //   318, depth=1
{0xfc01fc01fc01fc01,  273,{0xffff,0x01a6}},  //   319, depth=2
{0xf000000003ffffff,   -1,{0xffff,0x111d}},  //   320
{0xfffffffffff0000f,   -1,{0xffff,0x1b2f}},  //   321, depth=1
{0x000000000001ff00,  275,{0x0608,0x1e08}},  //   322, depth=2
{0xffffffffff0003ff,   -1,{0xffff,0x1a31}},  //   323, depth=1
{0x0003ffffffe00000,  277,{0xffff,0x1adc}},  //   324, depth=2
{0xffc00001ffc00001,  281,{0xffff,0x028a}},  //   325, depth=2
{0x07e007e007e007e0,  285,{0xffff,0x02e5}},  //   326, depth=2
{0x00000000800001ff,   -1,{0x0049,0xffff}},  //   327, depth=1
{0x1fc01fc01fc01fc0,   -1,{0xffff,0x02a6}},  //   328, depth=1
{0xf8000000ffffffff,  286,{0xffff,0x1164}},  //   329, depth=2
{0xffffffffc7ffffff,   -1,{0xffff,0x18bc}},  //   330
{0x803f803f803f803f,  290,{0xffff,0x0066}},  //   331, depth=3
{0x07fffffffc000000,   -1,{0xffff,0x19a0}},  //   332
{0xffffc0003fffffff,   -1,{0xffff,0x14af}},  //   333
{0x000000000007e000,   -1,{0x04c5,0x1cc5}},  //   334, depth=1
{0xffffff1fffffffff,   -1,{0xffff,0x163c}},  //   335
{0x00fffff800000000,   -1,{0xffff,0x1754}},  //   336
{0xe00000003fffffff,  300,{0xffff,0x10e0}},  //   337, depth=2
{0xffffffe00007ffff,   -1,{0xffff,0x16ed}},  //   338
{0xffffe07fffffe07f,  308,{0xffff,0x04d9}},  //   339, depth=2
{0x000001fffc000000,   -1,{0xffff,0x198e}},  //   340, depth=1
{0x00003ffffffff000,  344,{0xffff,0x1d21}},  //   341
{0x0000001000000010,  311,{0xffff,0x0700}},  //   342, depth=2
{0x00000000ff00007f,  314,{0x020e,0xffff}},  //   343, depth=3
{0x000000000ffe0ffe,   -1,{0x03ea,0xffff}},  //   344
{0xfffff8003fffffff,  347,{0xffff,0x1572}},  //   345
{0x7fe000007fe00000,   -1,{0xffff,0x02c9}},  //   346, depth=1
{0xff0001ffff0001ff,   -1,{0xffff,0x0210}},  //   347
{0x000007fffffff000,   -1,{0xffff,0x1d1e}},  //   348, depth=1
{0xfff87ffffff87fff,  330,{0xffff,0x035b}},  //   349, depth=2
{0xffffc0001fffffff,   -1,{0xffff,0x14ae}},  //   350
{0xfe7ffffffe7fffff,   -1,{0xffff,0x01dd}},  //   351, depth=1
{0x000fffe000000000,   -1,{0xffff,0x16ce}},  //   352, depth=1
{0xfe007fffffffffff,   -1,{0xffff,0x11f5}},  //   353
{0x000000007ffc7ffc,  332,{0x03ac,0xffff}},  //   354, depth=2
{0x00000000ffc1ffff,  333,{0x029a,0xffff}},  //   355, depth=2
{0x7fffff0000000000,   -1,{0xffff,0x1616}},  //   356, depth=1
{0xffff87ffffffffff,   -1,{0xffff,0x147b}},  //   357
{0x0180018001800180,   -1,{0xffff,0x0261}},  //   358, depth=1
{0xfffffffffff83fff,   -1,{0xffff,0x1b7a}},  //   359, depth=1
{0x81ffffffffffffff,   -1,{0xffff,0x1079}},  //   360
{0x800fffffffffffff,   -1,{0xffff,0x1074}},  //   361, depth=1
{0x00000ffffff80000,   -1,{0xffff,0x1b58}},  //   362, depth=1
{0x03ffffffffff8000,   -1,{0xffff,0x1c6a}},  //   363
{0xfffffe00007fffff,   -1,{0xffff,0x15ed}},  //   364
{0xe1ffffffe1ffffff,  335,{0xffff,0x00db}},  //   365, depth=2
{0x03f0000003f00000,  336,{0xffff,0x0305}},  //   366, depth=2
{0xfffffff000001fff,   -1,{0xffff,0x1728}},  //   367, depth=1
{0x00000001fe000000,   -1,{0xffff,0x19c7}},  //   368
{0xfffffffffff0003f,   -1,{0xffff,0x1b31}},  //   369, depth=1
{0x00ff000000ff0000,   -1,{0xffff,0x0407}},  //   370, depth=1
{0xfffffc0003ffffff,   -1,{0xffff,0x15af}},  //   371, depth=1
{0x0000000fe0000000,   -1,{0xffff,0x18c6}},  //   372, depth=1
{0xc00007ffffffffff,   -1,{0xffff,0x10ac}},  //   373, depth=1
{0x0007ffffffffff80,   -1,{0xffff,0x1e6b}},  //   374, depth=1
{0x00000000ff81ff81,  338,{0x0269,0xffff}},  //   375, depth=2
{0xfffffffffff0007f,   -1,{0xffff,0x1b32}},  //   376
{0x00000000fc0fffff,   -1,{0x0199,0xffff}},  //   377, depth=1
{0xfff803ffffffffff,  379,{0xffff,0x1376}},  //   378
{0xff8003ffff8003ff,   -1,{0xffff,0x0252}},  //   379
{0xfffffff0000007ff,  382,{0xffff,0x1726}},  //   380
{0x80000000000003ff,   -1,{0xffff,0x104a}},  //   381, depth=1
{0xe000000000000007,   -1,{0xffff,0x10c5}},  //   382
{0xe00000001fffffff,   -1,{0xffff,0x10df}},  //   383, depth=1
{0x0000000000000180,   -1,{0x0641,0x1e41}},  //   384, depth=1
{0xfff00fffffffffff,   -1,{0xffff,0x1337}},  //   385, depth=1
{0x0000000007fc0000,  341,{0x0388,0x1b88}},  //   386, depth=3
{0x0007ffffffffffc0,   -1,{0xffff,0x1eac}},  //   387
{0xfffffffc000000ff,   -1,{0xffff,0x17a5}},  //   388
{0xffffffc0001fffff,   -1,{0xffff,0x16ae}},  //   389
{0xffffc00000000000,   -1,{0xffff,0x1491}},  //   390
{0xfffffc0000003fff,   -1,{0xffff,0x15a3}},  //   391
{0xffff0000000003ff,  394,{0xffff,0x1419}},  //   392
{0x000000008001ffff,  345,{0x0051,0xffff}},  //   393, depth=3
{0x00000000fffc1fff,   -1,{0x039a,0xffff}},  //   394
{0xfffffffffe0001ff,   -1,{0xffff,0x19ef}},  //   395, depth=1
{0x0000001fffffff00,   -1,{0xffff,0x1e1c}},  //   396, depth=1
{0xffffffffff03ffff,   -1,{0xffff,0x1a39}},  //   397
{0xfffffffffe01ffff,  402,{0xffff,0x19f7}},  //   398
{0xfffe0000000000ff,   -1,{0xffff,0x13d6}},  //   399, depth=1
{0x000000007f000000,   -1,{0x0206,0x1a06}},  //   400, depth=1
{0x00000000f807ffff,  350,{0x0157,0xffff}},  //   401, depth=2
{0xfff07fffffffffff,  408,{0xffff,0x133a}},  //   402
{0xfffc000001ffffff,   -1,{0xffff,0x13a6}},  //   403, depth=1
{0x1ffffffffffff000,   -1,{0xffff,0x1d30}},  //   404, depth=1
{0xf8000000007fffff,   -1,{0xffff,0x115b}},  //   405, depth=1
{0x01ffffe000000000,   -1,{0xffff,0x16d3}},  //   406, depth=1
{0x00000000c00fc00f,  353,{0x00a5,0xffff}},  //   407, depth=2
{0xe00000000000001f,   -1,{0xffff,0x10c7}},  //   408
{0xff000003ff000003,  357,{0xffff,0x0209}},  //   409, depth=2
{0x07fe000007fe0000,   -1,{0xffff,0x03c9}},  //   410, depth=1
{0xc0003fffffffffff,   -1,{0xffff,0x10af}},  //   411, depth=1
{0xfffffffffcffffff,  416,{0xffff,0x19bd}},  //   412
{0xfff01ffffff01fff,   -1,{0xffff,0x0318}},  //   413, depth=1
{0x007fe000007fe000,   -1,{0xffff,0x04c9}},  //   414, depth=1
{0xfffbfffffffbffff,  360,{0xffff,0x035e}},  //   415, depth=2
{0xff800007ff800007,   -1,{0xffff,0x024b}},  //   416
{0xfffe000fffffffff,   -1,{0xffff,0x13f2}},  //   417
{0x0000000fc0000000,  363,{0xffff,0x1885}},  //   418, depth=2
{0xcfcfcfcfcfcfcfcf,  364,{0xffff,0x00b5}},  //   419, depth=2
{0x0007ffffffffffe0,   -1,{0xffff,0x1eed}},  //   420
{0xf000000000003fff,   -1,{0xffff,0x1111}},  //   421, depth=1
{0x0000000001f00000,   -1,{0x0304,0x1b04}},  //   422, depth=1
{0xff8fff8fff8fff8f,   -1,{0xffff,0x026c}},  //   423, depth=1
{0x3ffffffffff80000,   -1,{0xffff,0x1b6a}},  //   424
{0x00000000f3ffffff,   -1,{0x011d,0xffff}},  //   425, depth=1
{0x000ffffffffe0000,   -1,{0xffff,0x1be2}},  //   426, depth=1
{0xff001fffffffffff,   -1,{0xffff,0x1234}},  //   427, depth=1
{0x00001fe000001fe0,   -1,{0xffff,0x06c7}},  //   428, depth=1
{0xdfffffffffffffff,   -1,{0xffff,0x10be}},  //   429, depth=1
{0xe000000000000000,   -1,{0xffff,0x10c2}},  //   430, depth=1
{0xe000000000000001,   -1,{0xffff,0x10c3}},  //   431, depth=1
{0x0f80000000000000,  368,{0xffff,0x1244}},  //   432, depth=2
{0xe000000000000003,  376,{0xffff,0x10c4}},  //   433, depth=2
{0xfffffffe000003ff,   -1,{0xffff,0x17e8}},  //   434
{0x00000000ff8001ff,  378,{0x0251,0xffff}},  //   435, depth=3
{0x01fffffffffe0000,   -1,{0xffff,0x1be7}},  //   436
{0x0000000080078007,  380,{0x0063,0xffff}},  //   437, depth=3
{0x00007ffffff80000,  387,{0xffff,0x1b5b}},  //   438, depth=2
{0xfffff8001fffffff,   -1,{0xffff,0x1571}},  //   439, depth=1
{0xf03fffffffffffff,   -1,{0xffff,0x1139}},  //   440
{0xffffffff87ffffff,   -1,{0xffff,0x187b}},  //   441, depth=1
{0x0007fffffffffff0,   -1,{0xffff,0x1f2e}},  //   442
{0xffe00fffffe00fff,  388,{0xffff,0x02d6}},  //   443, depth=2
{0x000ffffffffffe00,   -1,{0xffff,0x1dea}},  //   444
{0xe00000000000000f,  389,{0xffff,0x10c6}},  //   445, depth=2
{0xfffff80000000007,   -1,{0xffff,0x1557}},  //   446
{0xffffbfffffffffff,   -1,{0xffff,0x147e}},  //   447, depth=1
{0x00000000000001c0,  390,{0x0682,0x1e82}},  //   448, depth=2
{0xffffc00000000001,  391,{0xffff,0x1492}},  //   449, depth=2
{0x000000000001ff80,   -1,{0x0649,0x1e49}},  //   450, depth=1
{0xffffc00000000003,   -1,{0xffff,0x1493}},  //   451, depth=1
{0x00000c0000000000,   -1,{0xffff,0x1581}},  //   452, depth=1
{0x1f1f1f1f1f1f1f1f,  392,{0xffff,0x0034}},  //   453, depth=3
{0x0000fffffe000000,   -1,{0xffff,0x19d6}},  //   454, depth=1
{0xffffc00000000007,  397,{0xffff,0x1494}},  //   455, depth=2
{0xfffff8000000000f,  457,{0xffff,0x1558}},  //   456
{0x0007ffffffffffff,   -1,{0xffff,0x1032}},  //   457
{0x0000c0000000c000,   -1,{0xffff,0x0481}},  //   458, depth=1
{0xffffffffe0007fff,   -1,{0xffff,0x18f1}},  //   459, depth=1
{0x0000000f80000000,  462,{0xffff,0x1844}},  //   460
{0xfc7ffffffc7fffff,  398,{0xffff,0x019c}},  //   461, depth=4
{0x00000000000001fe,   -1,{0x07c7,0x1fc7}},  //   462
{0xffffc0000000000f,   -1,{0xffff,0x1495}},  //   463, depth=1
{0xffffc0000000003f,   -1,{0xffff,0x1497}},  //   464
{0xf80000007fffffff,   -1,{0xffff,0x1163}},  //   465
{0x001ffffff8000000,   -1,{0xffff,0x1959}},  //   466, depth=1
{0x000000003fff3fff,  412,{0x002d,0xffff}},  //   467, depth=3
{0x007fffc000000000,  478,{0xffff,0x1690}},  //   468
{0xc0007fffc0007fff,  417,{0xffff,0x0090}},  //   469, depth=2
{0x3fffc0003fffc000,  420,{0xffff,0x048f}},  //   470, depth=2
{0x00000000ff0000ff,   -1,{0x020f,0xffff}},  //   471, depth=1
{0x0000000fffc00000,   -1,{0xffff,0x1a8d}},  //   472, depth=1
{0xffcfffffffcfffff,   -1,{0xffff,0x029d}},  //   473, depth=1
{0x000000fc00000000,  424,{0xffff,0x1785}},  //   474, depth=2
{0xdfffffffdfffffff,  434,{0xffff,0x009e}},  //   475, depth=2
{0x01f0000001f00000,   -1,{0xffff,0x0304}},  //   476, depth=1
{0xfc00000003ffffff,   -1,{0xffff,0x119f}},  //   477, depth=1
{0x03fffffe03fffffe,   -1,{0xffff,0x07d8}},  //   478
{0xffffc0000000001f,   -1,{0xffff,0x1496}},  //   479, depth=1
{0x00000000000001e0,  436,{0x06c3,0x1ec3}},  //   480, depth=2
{0xffff7fffffff7fff,  440,{0xffff,0x041e}},  //   481, depth=2
{0x0000000003ff0000,   -1,{0x0409,0x1c09}},  //   482, depth=1
{0x00000000fffc07ff,   -1,{0x0398,0xffff}},  //   483
{0x1fc0000000000000,   -1,{0xffff,0x1286}},  //   484, depth=1
{0xfffff7ffffffffff,   -1,{0xffff,0x153e}},  //   485, depth=1
{0xfffff80000000000,  442,{0xffff,0x1554}},  //   486, depth=2
{0xfffff80000000001,   -1,{0xffff,0x1555}},  //   487, depth=1
{0x000007ffffffc000,   -1,{0xffff,0x1c9c}},  //   488, depth=1
{0xfffff80000000003,   -1,{0xffff,0x1556}},  //   489, depth=1
{0x0000001fffffff80,  497,{0xffff,0x1e5d}},  //   490
{0xfffffff801ffffff,   -1,{0xffff,0x1775}},  //   491, depth=1
{0x000000003f000000,  444,{0x0205,0x1a05}},  //   492, depth=2
{0xe00000000000003f,  446,{0xffff,0x10c8}},  //   493, depth=2
{0x0007fffffffffff8,   -1,{0xffff,0x1f6f}},  //   494, depth=1
{0xffffffc00fffffff,   -1,{0xffff,0x16b5}},  //   495, depth=1
{0x00000000000001f0,   -1,{0x0704,0x1f04}},  //   496, depth=1
{0x000003ffffff8000,   -1,{0xffff,0x1c5a}},  //   497
{0x0007fffffffffffc,   -1,{0xffff,0x1fb0}},  //   498, depth=1
{0xfffffffffff80fff,   -1,{0xffff,0x1b78}},  //   499, depth=1
{0x0007fffffffffffe,   -1,{0xffff,0x1ff1}},  //   500, depth=1
{0x00000000ff81ffff,  456,{0x0259,0xffff}},  //   501, depth=3
{0x0008000000000000,   -1,{0xffff,0x1340}},  //   502, depth=1
{0x00000000f0003fff,   -1,{0x0111,0xffff}},  //   503, depth=1
{0x00000000000001f8,   -1,{0x0745,0x1f45}},  //   504, depth=1
{0xffffffffc0007fff,   -1,{0xffff,0x18b0}},  //   505, depth=1
{0x00003fffffffc000,   -1,{0xffff,0x1c9f}},  //   506
{0x00000000fe7fffff,   -1,{0x01dd,0xffff}},  //   507, depth=1
{0x00000000000001fc,   -1,{0x0786,0x1f86}},  //   508, depth=1
{0x00007c0000000000,  515,{0xffff,0x1584}},  //   509
{0x0000038000000380,  460,{0xffff,0x0642}},  //   510, depth=3
{0x00000000000001ff,  464,{0x0008,0x1008}},  //   511, depth=2
{0x0000000000000200,   -1,{0x05c0,0x1dc0}},  //   512, depth=1
{0x800000000003ffff,  465,{0xffff,0x1052}},  //   513, depth=2
{0x000000000001ffc0,  468,{0x068a,0x1e8a}},  //   514, depth=3
{0x00000000000ff000,   -1,{0x0507,0x1d07}},  //   515
{0xffc00000ffffffff,   -1,{0xffff,0x12a9}},  //   516
{0xfffff8000000001f,   -1,{0xffff,0x1559}},  //   517, depth=1
{0x0000000007fffc00,   -1,{0x0590,0x1d90}},  //   518, depth=1
{0x01fffffffffffe00,   -1,{0xffff,0x1def}},  //   519
{0x0007ffffe0000000,   -1,{0xffff,0x18d5}},  //   520
{0x0fffffffffff8000,   -1,{0xffff,0x1c6c}},  //   521
{0x03ffff0003ffff00,   -1,{0xffff,0x0611}},  //   522, depth=1
{0xf800003ff800003f,  483,{0xffff,0x014a}},  //   523, depth=2
{0x0000000007e007e0,  490,{0x02e5,0xffff}},  //   524, depth=3
{0xfff800000007ffff,   -1,{0xffff,0x135f}},  //   525, depth=1
{0x7ffffffffe000000,  506,{0xffff,0x19e5}},  //   526, depth=2
{0xfffffe007fffffff,   -1,{0xffff,0x15f5}},  //   527, depth=1
{0x001c0000001c0000,  509,{0xffff,0x0382}},  //   528, depth=3
{0xc0000000000fffff,   -1,{0xffff,0x1095}},  //   529
{0x01f8000000000000,   -1,{0xffff,0x1345}},  //   530
{0xff000001ffffffff,   -1,{0xffff,0x1228}},  //   531
{0xfffffffffff000ff,   -1,{0xffff,0x1b33}},  //   532
{0x03fffffff8000000,   -1,{0xffff,0x195e}},  //   533
{0x000000000001fffe,   -1,{0x07cf,0x1fcf}},  //   534
{0xc003ffffc003ffff,   -1,{0xffff,0x0093}},  //   535, depth=1
{0x3000000000000000,   -1,{0xffff,0x1101}},  //   536, depth=1
{0xffff00000003ffff,   -1,{0xffff,0x1421}},  //   537
{0x000000001f000000,   -1,{0x0204,0x1a04}},  //   538, depth=1
{0x00000000ff01ff01,   -1,{0x0228,0xffff}},  //   539, depth=1
{0x0000001fffffffc0,   -1,{0xffff,0x1e9e}},  //   540
{0xfffffe0ffffffe0f,   -1,{0xffff,0x05da}},  //   541, depth=1
{0xfffffc0000000fff,   -1,{0xffff,0x15a1}},  //   542
{0x00000000e7e7e7e7,   -1,{0x00f5,0xffff}},  //   543, depth=1
{0x1fffffffffffc000,   -1,{0xffff,0x1cae}},  //   544, depth=1
{0xff87ff87ff87ff87,  516,{0xffff,0x026b}},  //   545, depth=2
{0x000000000001ffe0,  519,{0x06cb,0x1ecb}},  //   546, depth=2
{0x000000fffff00000,   -1,{0xffff,0x1b13}},  //   547
{0x0000000070007000,  520,{0x0122,0xffff}},  //   548, depth=2
{0xfffff8000000003f,   -1,{0xffff,0x155a}},  //   549, depth=1
{0x0007ffffc0000000,   -1,{0xffff,0x1894}},  //   550
{0xf8000000003fffff,   -1,{0xffff,0x115a}},  //   551, depth=1
{0x00fffffff0000000,  521,{0xffff,0x191b}},  //   552, depth=2
{0x00000000ffe03fff,  529,{0x02d8,0xffff}},  //   553, depth=2
{0x0700000000000000,   -1,{0xffff,0x1202}},  //   554
{0xfc00000000003fff,   -1,{0xffff,0x1193}},  //   555, depth=1
{0x0000100000001000,  530,{0xffff,0x0500}},  //   556, depth=2
{0xe00000000000007f,  531,{0xffff,0x10c9}},  //   557, depth=2
{0x00003ffffc000000,   -1,{0xffff,0x1993}},  //   558
{0xffffffffffc07fff,   -1,{0xffff,0x1ab8}},  //   559, depth=1
{0x0000007ffffff800,   -1,{0xffff,0x1d5b}},  //   560, depth=1
{0xf000000000000fff,  532,{0xffff,0x110f}},  //   561, depth=2
{0x000000000001fff0,   -1,{0x070c,0x1f0c}},  //   562, depth=1
{0xffffe000000003ff,   -1,{0xffff,0x14dc}},  //   563
{0x3ffffc0000000000,   -1,{0xffff,0x1593}},  //   564, depth=1
{0xfffffe00003fffff,   -1,{0xffff,0x15ec}},  //   565, depth=1
{0x000007fffc000000,   -1,{0xffff,0x1990}},  //   566, depth=1
{0x0003800000000000,   -1,{0xffff,0x1442}},  //   567
{0x0001fe0000000000,   -1,{0xffff,0x15c7}},  //   568, depth=1
{0xfffffffff800ffff,   -1,{0xffff,0x1974}},  //   569, depth=1
{0x000000000001fff8,   -1,{0x074d,0x1f4d}},  //   570, depth=1
{0xffe0001fffe0001f,   -1,{0xffff,0x02cf}},  //   571, depth=1
{0xfc3fffffffffffff,   -1,{0xffff,0x11bb}},  //   572
{0xfffc00ffffffffff,   -1,{0xffff,0x13b5}},  //   573, depth=1
{0x000000000001fffc,  533,{0x078e,0x1f8e}},  //   574, depth=2
{0xffffc0000000007f,   -1,{0xffff,0x1498}},  //   575, depth=1
{0x7ffc7ffc7ffc7ffc,  534,{0xffff,0x03ac}},  //   576, depth=2
{0x000000000001ffff,   -1,{0x0010,0x1010}},  //   577, depth=1
{0x0000000000020000,   -1,{0x03c0,0x1bc0}},  //   578, depth=1
{0xff00ffffffffffff,   -1,{0xffff,0x1237}},  //   579, depth=1
{0x0000000000020002,   -1,{0x03e0,0xffff}},  //   580, depth=1
{0x0000000fff800000,   -1,{0xffff,0x1a4c}},  //   581
{0x0000001fffffffe0,   -1,{0xffff,0x1edf}},  //   582
{0x01fffffe01fffffe,   -1,{0xffff,0x07d7}},  //   583
{0x1fc000001fc00000,   -1,{0xffff,0x0286}},  //   584, depth=1
{0xffe0000003ffffff,  537,{0xffff,0x12e4}},  //   585, depth=2
{0xffffffff07ffffff,   -1,{0xffff,0x183a}},  //   586
{0x0000007fffffe000,   -1,{0xffff,0x1cd9}},  //   587
{0x0ffffffc0ffffffc,  540,{0xffff,0x0799}},  //   588, depth=2
{0xffff8000000fffff,  542,{0xffff,0x1464}},  //   589, depth=2
{0x0007c0000007c000,   -1,{0xffff,0x0484}},  //   590, depth=1
{0x0000001ffffffff8,   -1,{0xffff,0x1f61}},  //   591
{0x003ffff8003ffff8,  547,{0xffff,0x0752}},  //   592, depth=2
{0x0000001ffffffffc,   -1,{0xffff,0x1fa2}},  //   593
{0x001f8000001f8000,  550,{0xffff,0x0445}},  //   594, depth=2
{0xf80000001fffffff,   -1,{0xffff,0x1161}},  //   595
{0x7c007c007c007c00,  554,{0xffff,0x01a4}},  //   596, depth=2
{0xffffffff80007fff,   -1,{0xffff,0x186f}},  //   597, depth=1
{0x0000002000000000,   -1,{0xffff,0x16c0}},  //   598
{0x00000000fe3fffff,   -1,{0x01dc,0xffff}},  //   599
{0x003c000000000000,   -1,{0xffff,0x1383}},  //   600
{0xfffffe001fffffff,   -1,{0xffff,0x15f3}},  //   601
{0x00ffffffffe00000,   -1,{0xffff,0x1ae2}},  //   602, depth=1
{0xfff0003ffff0003f,   -1,{0xffff,0x0311}},  //   603
{0x0ffffe000ffffe00,  558,{0xffff,0x05d2}},  //   604, depth=2
{0xf80000003fffffff,  563,{0xffff,0x1162}},  //   605, depth=2
{0x00001ffc00000000,   -1,{0xffff,0x178a}},  //   606, depth=1
{0xfffffffe0003ffff,   -1,{0xffff,0x17f0}},  //   607, depth=1
{0x003e003e003e003e,  567,{0xffff,0x03e4}},  //   608, depth=2
{0xfffffffff1ffffff,   -1,{0xffff,0x193c}},  //   609
{0xffffffc7ffffffff,  611,{0xffff,0x16bc}},  //   610
{0x00000000f80ff80f,   -1,{0x0168,0xffff}},  //   611
{0x0003f00000000000,   -1,{0xffff,0x1505}},  //   612
{0xfffff8000000007f,   -1,{0xffff,0x155b}},  //   613, depth=1
{0x0000000000078000,   -1,{0x0443,0x1c43}},  //   614, depth=1
{0x00000000fcfffcff,  572,{0x01ad,0xffff}},  //   615, depth=2
{0xfffc07fffffc07ff,   -1,{0xffff,0x0398}},  //   616
{0xfc00000000000fff,   -1,{0xffff,0x1191}},  //   617
{0x0001fffc0001fffc,  581,{0xffff,0x078e}},  //   618, depth=2
{0xfffffe003fffffff,   -1,{0xffff,0x15f4}},  //   619, depth=1
{0x003fffffffff0000,  582,{0xffff,0x1c25}},  //   620, depth=2
{0xffc00000007fffff,   -1,{0xffff,0x12a0}},  //   621, depth=1
{0x1ffffffffc000000,   -1,{0xffff,0x19a2}},  //   622, depth=1
{0x0000001fe0000000,   -1,{0xffff,0x18c7}},  //   623
{0x003ffc00003ffc00,  583,{0xffff,0x058b}},  //   624, depth=2
{0xffffffc7ffffffc7,  586,{0xffff,0x069c}},  //   625, depth=2
{0x00000ffffffff800,   -1,{0xffff,0x1d60}},  //   626, depth=1
{0xfffffffc000001ff,   -1,{0xffff,0x17a6}},  //   627
{0xfffffff00007ffff,  629,{0xffff,0x172e}},  //   628
{0xf800000000000003,   -1,{0xffff,0x1146}},  //   629
{0x03fe000003fe0000,  587,{0xffff,0x03c8}},  //   630, depth=2
{0xfff801fffff801ff,   -1,{0xffff,0x0355}},  //   631, depth=1
{0x01ffff0001ffff00,   -1,{0xffff,0x0610}},  //   632, depth=1
{0xffffc000000000ff,   -1,{0xffff,0x1499}},  //   633
{0x0001fe000001fe00,   -1,{0xffff,0x05c7}},  //   634, depth=1
{0xffffffffff807fff,  637,{0xffff,0x1a77}},  //   635
{0x0000001ffffffff0,   -1,{0xffff,0x1f20}},  //   636, depth=1
{0xfffc1fffffffffff,  638,{0xffff,0x13ba}},  //   637
{0xf800000000000007,   -1,{0xffff,0x1147}},  //   638
{0xffff8003ffffffff,   -1,{0xffff,0x1472}},  //   639
{0x0ffffffff8000000,   -1,{0xffff,0x1960}},  //   640
{0xff00000000ffffff,   -1,{0xffff,0x121f}},  //   641
{0x7ffffffc7ffffffc,   -1,{0xffff,0x079c}},  //   642, depth=1
{0x00000000f0000fff,   -1,{0x010f,0xffff}},  //   643, depth=1
{0x000000000003fe00,  591,{0x05c8,0x1dc8}},  //   644, depth=2
{0x000007ff000007ff,   -1,{0xffff,0x000a}},  //   645, depth=1
{0x00000003fffff000,   -1,{0xffff,0x1d15}},  //   646, depth=1
{0xfff80003fff80003,   -1,{0xffff,0x034e}},  //   647, depth=1
{0x0007ffffffc00000,  593,{0xffff,0x1a9c}},  //   648, depth=2
{0xfffffdffffffffff,   -1,{0xffff,0x15be}},  //   649
{0x0000001ffffffffe,   -1,{0xffff,0x1fe3}},  //   650, depth=1
{0x0000001fffffffff,  595,{0xffff,0x1024}},  //   651, depth=2
{0x0fc00fc00fc00fc0,  598,{0xffff,0x02a5}},  //   652, depth=2
{0x80001fff80001fff,  599,{0xffff,0x004d}},  //   653, depth=2
{0xfffffe0000000001,  657,{0xffff,0x15d7}},  //   654
{0xfffe0000000001ff,   -1,{0xffff,0x13d7}},  //   655, depth=1
{0x3f803f803f803f80,   -1,{0xffff,0x0266}},  //   656, depth=1
{0xf80000000000000f,   -1,{0xffff,0x1148}},  //   657
{0x7ffffe007ffffe00,  600,{0xffff,0x05d5}},  //   658, depth=2
{0x9fff9fff9fff9fff,   -1,{0xffff,0x006d}},  //   659, depth=1
{0xfffffe0000000003,  664,{0xffff,0x15d8}},  //   660
{0x8000003fffffffff,   -1,{0xffff,0x1066}},  //   661, depth=1
{0x007ffffffffc0000,   -1,{0xffff,0x1ba4}},  //   662, depth=1
{0xffe0000000003fff,   -1,{0xffff,0x12d8}},  //   663, depth=1
{0xc001ffffffffffff,   -1,{0xffff,0x10b2}},  //   664
{0xfffff000000fffff,  601,{0xffff,0x1527}},  //   665, depth=2
{0x0000003ffffe0000,   -1,{0xffff,0x1bd4}},  //   666
{0xff7fff7fff7fff7f,   -1,{0xffff,0x022e}},  //   667, depth=1
{0x00000000000fc000,   -1,{0x0485,0x1c85}},  //   668, depth=1
{0x00000000c3ffc3ff,  603,{0x00ab,0xffff}},  //   669, depth=2
{0x3c00000000000000,   -1,{0xffff,0x1183}},  //   670, depth=1
{0xc0000003ffffffff,   -1,{0xffff,0x10a3}},  //   671, depth=1
{0xfffffe000000000f,   -1,{0xffff,0x15da}},  //   672
{0xffc000007fffffff,   -1,{0xffff,0x12a8}},  //   673
{0x0000000000007c00,   -1,{0x0584,0x1d84}},  //   674, depth=1
{0xffffff80000fffff,   -1,{0xffff,0x166c}},  //   675
{0x01fffff000000000,   -1,{0xffff,0x1714}},  //   676
{0xfffe1ffffffe1fff,  609,{0xffff,0x03db}},  //   677, depth=2
{0x000ffff000000000,   -1,{0xffff,0x170f}},  //   678, depth=1
{0xffffff00000003ff,   -1,{0xffff,0x1621}},  //   679
{0x000003fff8000000,   -1,{0xffff,0x194e}},  //   680, depth=1
{0xf87ffffff87fffff,  610,{0xffff,0x015b}},  //   681, depth=3
{0xfffff800000000ff,   -1,{0xffff,0x155c}},  //   682
{0xf00001ffffffffff,   -1,{0xffff,0x112c}},  //   683, depth=1
{0x0000002000000020,  612,{0xffff,0x06c0}},  //   684, depth=2
{0xe0000000000000ff,   -1,{0xffff,0x10ca}},  //   685, depth=1
{0x0007ffff80000000,   -1,{0xffff,0x1853}},  //   686, depth=1
{0x003ffff800000000,   -1,{0xffff,0x1752}},  //   687
{0x00000000fffe7fff,   -1,{0x03dd,0xffff}},  //   688
{0xfffffffffc7fffff,   -1,{0xffff,0x19bc}},  //   689
{0x0001fffffffff000,   -1,{0xffff,0x1d24}},  //   690, depth=1
{0xfe000000001fffff,   -1,{0xffff,0x11db}},  //   691, depth=1
{0xffc00000ffc00000,   -1,{0xffff,0x0289}},  //   692, depth=1
{0x00000000ffe00fff,  616,{0x02d6,0xffff}},  //   693, depth=2
{0x0000000f00000000,   -1,{0xffff,0x1803}},  //   694, depth=1
{0x00000000e3e3e3e3,  617,{0x00f4,0xffff}},  //   695, depth=2
{0x00000fffffffe000,   -1,{0xffff,0x1cde}},  //   696, depth=1
{0xf7ffffffffffffff,   -1,{0xffff,0x113e}},  //   697, depth=1
{0xf800000000000000,  623,{0xffff,0x1144}},  //   698, depth=2
{0xf800000000000001,  627,{0xffff,0x1145}},  //   699, depth=2
{0xfffffffc0001ffff,   -1,{0xffff,0x17ae}},  //   700
{0x00000000e00003ff,  628,{0x00cc,0xffff}},  //   701, depth=3
{0x00007ffffffff800,   -1,{0xffff,0x1d63}},  //   702, depth=1
{0xc7c7c7c7c7c7c7c7,  633,{0xffff,0x00b4}},  //   703, depth=2
{0x001fffc000000000,   -1,{0xffff,0x168e}},  //   704, depth=1
{0xff1fffffff1fffff,  635,{0xffff,0x021c}},  //   705, depth=4
{0xffc00000003fffff,   -1,{0xffff,0x129f}},  //   706
{0xf0001ffff0001fff,  639,{0xffff,0x0110}},  //   707, depth=2
{0x00000000fff8fff8,  640,{0x036c,0xffff}},  //   708, depth=2
{0x0000000f0000000f,  641,{0xffff,0x0003}},  //   709, depth=2
{0x00007fffffffe000,  714,{0xffff,0x1ce1}},  //   710
{0x03ff03ff03ff03ff,  649,{0xffff,0x0029}},  //   711, depth=2
{0xfffffe0000000000,   -1,{0xffff,0x15d6}},  //   712, depth=1
{0x000000003f3f3f3f,  654,{0x0035,0xffff}},  //   713, depth=3
{0x000000001ffc1ffc,   -1,{0x03aa,0xffff}},  //   714
{0x00000000fff8ffff,  660,{0x035c,0xffff}},  //   715, depth=3
{0x0300030003000300,   -1,{0xffff,0x0221}},  //   716, depth=1
{0xfffffe000000003f,   -1,{0xffff,0x15dc}},  //   717
{0xffff0003ffff0003,   -1,{0xffff,0x0411}},  //   718
{0xfffffe0000000007,   -1,{0xffff,0x15d9}},  //   719, depth=1
{0x00000003ffffc000,   -1,{0xffff,0x1c93}},  //   720
{0xfffe00000001ffff,   -1,{0xffff,0x13df}},  //   721, depth=1
{0x0000000000180000,   -1,{0x0341,0x1b41}},  //   722, depth=1
{0x007ffffffffffc00,   -1,{0xffff,0x1dac}},  //   723
{0x00001ffffff00000,   -1,{0xffff,0x1b18}},  //   724, depth=1
{0xffffff83ffffff83,   -1,{0xffff,0x065a}},  //   725, depth=1
{0x000000fc000000fc,  666,{0xffff,0x0785}},  //   726, depth=2
{0x00000000ff0001ff,  672,{0x0210,0xffff}},  //   727, depth=2
{0xfffe001fffffffff,   -1,{0xffff,0x13f3}},  //   728
{0xf80000000000001f,  673,{0xffff,0x1149}},  //   729, depth=2
{0x0000001fffc00000,   -1,{0xffff,0x1a8e}},  //   730
{0x0003000300030003,  675,{0xffff,0x0021}},  //   731, depth=2
{0x07e0000007e00000,  676,{0xffff,0x02c5}},  //   732, depth=2
{0xffff003fffffffff,   -1,{0xffff,0x1435}},  //   733, depth=1
{0x00000000ffc003ff,   -1,{0x0293,0xffff}},  //   734
{0xffe0000000000fff,   -1,{0xffff,0x12d6}},  //   735
{0xffffffc00000ffff,   -1,{0xffff,0x16a9}},  //   736
{0xffffe0000003ffff,  679,{0xffff,0x14e4}},  //   737, depth=2
{0x3ffffffffffff800,   -1,{0xffff,0x1d72}},  //   738, depth=1
{0xffffffffe0001fff,   -1,{0xffff,0x18ef}},  //   739, depth=1
{0x01fe000001fe0000,   -1,{0xffff,0x03c7}},  //   740, depth=1
{0xfe0000000fffffff,  682,{0xffff,0x11e2}},  //   741, depth=2
{0x0070000000000000,   -1,{0xffff,0x1302}},  //   742
{0xfffffe000000001f,   -1,{0xffff,0x15db}},  //   743, depth=1
{0x0000001fc0000000,   -1,{0xffff,0x1886}},  //   744, depth=1
{0xfff00000001fffff,   -1,{0xffff,0x1320}},  //   745, depth=1
{0x0000000000180018,  687,{0x0361,0xffff}},  //   746, depth=2
{0x00000000c00003ff,   -1,{0x008b,0xffff}},  //   747, depth=1
{0x000fffffffffff00,   -1,{0xffff,0x1e2b}},  //   748, depth=1
{0xfffffffffff001ff,  752,{0xffff,0x1b34}},  //   749
{0x0003ff8000000000,   -1,{0xffff,0x164a}},  //   750, depth=1
{0xc00001ffc00001ff,  688,{0xffff,0x008a}},  //   751, depth=2
{0xfc0001ffffffffff,   -1,{0xffff,0x11ae}},  //   752
{0xe00007ffe00007ff,   -1,{0xffff,0x00cd}},  //   753, depth=1
{0xfff8007fffffffff,   -1,{0xffff,0x1373}},  //   754
{0xe000000fffffffff,   -1,{0xffff,0x10e6}},  //   755, depth=1
{0xf80000000000007f,   -1,{0xffff,0x114b}},  //   756
{0xfffc000ffffc000f,   -1,{0xffff,0x0391}},  //   757, depth=1
{0x0000fffc00000000,   -1,{0xffff,0x178d}},  //   758, depth=1
{0xffff87ffffff87ff,  689,{0xffff,0x045b}},  //   759, depth=2
{0x0001ffffffffc000,   -1,{0xffff,0x1ca2}},  //   760
{0xf80000000000003f,   -1,{0xffff,0x114a}},  //   761, depth=1
{0xffffffffffffefff,   -1,{0xffff,0x1cfe}},  //   762
{0xffff01ffffff01ff,   -1,{0xffff,0x0418}},  //   763, depth=1
{0xfffffffffffff000,   -1,{0xffff,0x1d33}},  //   764
{0x00000000e01fe01f,  700,{0x00e7,0xffff}},  //   765, depth=2
{0xfffffffffffff001,   -1,{0xffff,0x1d34}},  //   766
{0xc0000003c0000003,  706,{0xffff,0x0083}},  //   767, depth=2
{0x0000000000000300,   -1,{0x0601,0x1e01}},  //   768, depth=1
{0x07ffffffffff0000,   -1,{0xffff,0x1c2a}},  //   769
{0x00001ff000001ff0,   -1,{0xffff,0x0708}},  //   770, depth=1
{0x00000000e1e1e1e1,   -1,{0x00f3,0xffff}},  //   771, depth=1
{0x000000000ff80000,  710,{0x0348,0x1b48}},  //   772, depth=3
{0xffffc00fffffffff,   -1,{0xffff,0x14b5}},  //   773, depth=1
{0xffffffffffc01fff,  776,{0xffff,0x1ab6}},  //   774
{0xff80000003ffffff,  717,{0xffff,0x1262}},  //   775, depth=2
{0xfffffe000000007f,   -1,{0xffff,0x15dd}},  //   776
{0xfffffffffffff00f,  780,{0xffff,0x1d37}},  //   777
{0x3fe0000000000000,   -1,{0xffff,0x12c8}},  //   778, depth=1
{0x00000000fc7ffc7f,  718,{0x01ac,0xffff}},  //   779, depth=2
{0xffffff83ffffffff,   -1,{0xffff,0x167a}},  //   780
{0xffffffff00007fff,   -1,{0xffff,0x182e}},  //   781, depth=1
{0x00000003fc000000,   -1,{0xffff,0x1987}},  //   782
{0xfffff003ffffffff,   -1,{0xffff,0x1535}},  //   783, depth=1
{0xffc000001fffffff,   -1,{0xffff,0x12a6}},  //   784
{0xffffffffc0001fff,   -1,{0xffff,0x18ae}},  //   785, depth=1
{0x7ffe000000000000,  720,{0xffff,0x13cd}},  //   786, depth=2
{0x07fc000000000000,   -1,{0xffff,0x1388}},  //   787
{0x000fffffffffff80,   -1,{0xffff,0x1e6c}},  //   788
{0xfffffffffff1ffff,   -1,{0xffff,0x1b3c}},  //   789
{0x00ff800000000000,   -1,{0xffff,0x1448}},  //   790
{0xffffffffffc007ff,  796,{0xffff,0x1ab4}},  //   791
{0x0000003ffffffe00,   -1,{0xffff,0x1ddc}},  //   792, depth=1
{0x00000000ff01ffff,   -1,{0x0218,0xffff}},  //   793, depth=1
{0x0007ffffff800000,  723,{0xffff,0x1a5b}},  //   794, depth=2
{0xfff000000fffffff,  728,{0xffff,0x1327}},  //   795, depth=2
{0xf00007ffffffffff,   -1,{0xffff,0x112e}},  //   796
{0x0000007fffff8000,   -1,{0xffff,0x1c57}},  //   797
{0x000000003fe03fe0,  730,{0x02e8,0xffff}},  //   798, depth=2
{0x00000000fdffffff,   -1,{0x019e,0xffff}},  //   799, depth=1
{0x00000000fe000000,   -1,{0x01c6,0x19c6}},  //   800, depth=1
{0x00000000fe000001,  734,{0x01c7,0xffff}},  //   801, depth=2
{0x01ffffffffffff00,   -1,{0xffff,0x1e30}},  //   802, depth=1
{0x00000000fe000003,  735,{0x01c8,0xffff}},  //   803, depth=2
{0xffc0000000000003,   -1,{0xffff,0x128b}},  //   804
{0x000003c000000000,   -1,{0xffff,0x1683}},  //   805
{0xfffffff9ffffffff,  810,{0xffff,0x177d}},  //   806
{0x00000000fe000007,  736,{0x01c9,0xffff}},  //   807, depth=2
{0x3fffffffffffe000,   -1,{0xffff,0x1cf0}},  //   808, depth=1
{0xffffffffe00007ff,   -1,{0xffff,0x18ed}},  //   809, depth=1
{0x00000000dddddddd,   -1,{0x00ba,0xffff}},  //   810
{0xfffff80fffffffff,   -1,{0xffff,0x1578}},  //   811, depth=1
{0x03ffffc000000000,   -1,{0xffff,0x1693}},  //   812, depth=1
{0xffe001ffffffffff,   -1,{0xffff,0x12f3}},  //   813
{0x07c007c007c007c0,  742,{0xffff,0x02a4}},  //   814, depth=2
{0x00000000fe00000f,   -1,{0x01ca,0xffff}},  //   815, depth=1
{0xfe000007ffffffff,  818,{0xffff,0x11e9}},  //   816
{0xe000000fe000000f,  749,{0xffff,0x00c6}},  //   817, depth=3
{0x00000000fe00007f,   -1,{0x01cd,0xffff}},  //   818
{0x801f801f801f801f,   -1,{0xffff,0x0065}},  //   819, depth=1
{0x0ffc00000ffc0000,   -1,{0xffff,0x0389}},  //   820, depth=1
{0xffc000003fffffff,  754,{0xffff,0x12a7}},  //   821, depth=2
{0x000001ffffff0000,   -1,{0xffff,0x1c18}},  //   822, depth=1
{0xffff800fffff800f,   -1,{0xffff,0x0454}},  //   823, depth=1
{0xfffe003ffffe003f,   -1,{0xffff,0x03d4}},  //   824
{0xffffff3fffffff3f,  756,{0xffff,0x061d}},  //   825, depth=2
{0x000000007ffe7ffe,   -1,{0x03ed,0xffff}},  //   826
{0x000fffffffffffc0,   -1,{0xffff,0x1ead}},  //   827
{0x00ffc00000ffc000,   -1,{0xffff,0x0489}},  //   828, depth=1
{0xe0000000000001ff,   -1,{0xffff,0x10cb}},  //   829
{0x7ffffffc00000000,  760,{0xffff,0x179c}},  //   830, depth=2
{0x00000000fe00001f,   -1,{0x01cb,0xffff}},  //   831, depth=1
{0x0fffffc000000000,   -1,{0xffff,0x1695}},  //   832
{0x00000000e003ffff,  762,{0x00d4,0xffff}},  //   833, depth=2
{0x0000180000001800,  764,{0xffff,0x0541}},  //   834, depth=2
{0xfffc0001ffffffff,  766,{0xffff,0x13ae}},  //   835, depth=2
{0x0000001f80000000,  769,{0xffff,0x1845}},  //   836, depth=2
{0xfffffffffffff003,   -1,{0xffff,0x1d35}},  //   837, depth=1
{0xffffffff800007ff,  840,{0xffff,0x186b}},  //   838
{0x00000000800003ff,  774,{0x004a,0xffff}},  //   839, depth=3
{0xf0003fffffffffff,   -1,{0xffff,0x1131}},  //   840
{0xfffffffffffff007,   -1,{0xffff,0x1d36}},  //   841, depth=1
{0x007fffe0007fffe0,   -1,{0xffff,0x06d1}},  //   842, depth=1
{0x7ffffffffff00000,   -1,{0xffff,0x1b2a}},  //   843
{0x0000000003e00000,   -1,{0x02c4,0x1ac4}},  //   844, depth=1
{0xc3c3c3c3c3c3c3c3,   -1,{0xffff,0x00b3}},  //   845, depth=1
{0xfffffffffffcffff,   -1,{0xffff,0x1bbd}},  //   846
{0x00000000dfdfdfdf,   -1,{0x00b6,0xffff}},  //   847, depth=1
{0x03fffffffffc0000,   -1,{0xffff,0x1ba7}},  //   848
{0xffffe3ffffffe3ff,  777,{0xffff,0x04dc}},  //   849, depth=3
{0x07fffff800000000,   -1,{0xffff,0x1757}},  //   850
{0xfff87fffffffffff,   -1,{0xffff,0x137b}},  //   851
{0x001ffffffffc0000,   -1,{0xffff,0x1ba2}},  //   852, depth=1
{0xff80000000003fff,   -1,{0xffff,0x1256}},  //   853, depth=1
{0x00000001fff00000,   -1,{0xffff,0x1b0c}},  //   854, depth=1
{0xffffffffc00007ff,   -1,{0xffff,0x18ac}},  //   855, depth=1
{0x00003fc000003fc0,   -1,{0xffff,0x0687}},  //   856, depth=1
{0xfffcffffffffffff,   -1,{0xffff,0x13bd}},  //   857, depth=1
{0xffff000007ffffff,  859,{0xffff,0x142a}},  //   858
{0xfffc0ffffffc0fff,   -1,{0xffff,0x0399}},  //   859
{0xffffffffffffc001,   -1,{0xffff,0x1cb2}},  //   860
{0x00000000fc3ffc3f,   -1,{0x01ab,0xffff}},  //   861, depth=1
{0xffffffffffffc003,   -1,{0xffff,0x1cb3}},  //   862
{0x00000000fe00003f,   -1,{0x01cc,0xffff}},  //   863, depth=1
{0x1f00000000000000,  782,{0xffff,0x1204}},  //   864, depth=2
{0xfffffffffffff01f,   -1,{0xffff,0x1d38}},  //   865, depth=1
{0xffffffffffffc007,   -1,{0xffff,0x1cb4}},  //   866
{0x00000000fe01fe01,  784,{0x01e7,0xffff}},  //   867, depth=2
{0x001ffffffffffc00,   -1,{0xffff,0x1daa}},  //   868
{0xffffff000003ffff,   -1,{0xffff,0x1629}},  //   869, depth=1
{0x0007ffff00000000,   -1,{0xffff,0x1812}},  //   870, depth=1
{0xffffffffff801fff,   -1,{0xffff,0x1a75}},  //   871
{0xeeeeeeeeeeeeeeee,   -1,{0xffff,0x00fa}},  //   872, depth=1
{0x00000000ffc3ffc3,   -1,{0x02ab,0xffff}},  //   873, depth=1
{0x7ffc00007ffc0000,  787,{0xffff,0x038c}},  //   874, depth=2
{0xfffc000000ffffff,   -1,{0xffff,0x13a5}},  //   875
{0x0000fffffff00000,  788,{0xffff,0x1b1b}},  //   876, depth=2
{0xffffffff80001fff,   -1,{0xffff,0x186d}},  //   877, depth=1
{0xfffffffe07ffffff,   -1,{0xffff,0x17f9}},  //   878
{0x00000000c003ffff,   -1,{0x0093,0xffff}},  //   879, depth=1
{0xffffffffff007fff,   -1,{0xffff,0x1a36}},  //   880
{0xffffe007ffffe007,   -1,{0xffff,0x04d5}},  //   881, depth=1
{0x000ffffc000ffffc,   -1,{0xffff,0x0791}},  //   882, depth=1
{0xfffffe1ffffffe1f,  789,{0xffff,0x05db}},  //   883, depth=2
{0x0000001f00000000,  888,{0xffff,0x1804}},  //   884
{0xffffe03fffffffff,   -1,{0xffff,0x14f8}},  //   885, depth=1
{0x0fff80000fff8000,  790,{0xffff,0x044c}},  //   886, depth=2
{0xfff80fffffffffff,   -1,{0xffff,0x1378}},  //   887, depth=1
{0x00000000000003fc,   -1,{0x0787,0x1f87}},  //   888
{0xf800000fffffffff,   -1,{0xffff,0x1168}},  //   889
{0x01ffffffffffffe0,   -1,{0xffff,0x1ef3}},  //   890
{0x00ffff8000000000,  892,{0xffff,0x1650}},  //   891
{0x07fffffc07fffffc,   -1,{0xffff,0x0798}},  //   892
{0x00000000fffe1fff,   -1,{0x03db,0xffff}},  //   893
{0xffffffffffffc03f,  901,{0xffff,0x1cb7}},  //   894
{0xfffff07ffffff07f,   -1,{0xffff,0x051a}},  //   895, depth=1
{0x0000000000000380,   -1,{0x0642,0x1e42}},  //   896, depth=1
{0xfffffffffffff03f,   -1,{0xffff,0x1d39}},  //   897, depth=1
{0x000ffe00000ffe00,   -1,{0xffff,0x05ca}},  //   898, depth=1
{0x8000000007ffffff,   -1,{0xffff,0x105b}},  //   899, depth=1
{0x000000000003ff00,   -1,{0x0609,0x1e09}},  //   900, depth=1
{0xfffffe0fffffffff,   -1,{0xffff,0x15fa}},  //   901
{0x0003fffffff80000,   -1,{0xffff,0x1b5e}},  //   902
{0x0000003fffffff00,  911,{0xffff,0x1e1d}},  //   903
{0x0000180000000000,   -1,{0xffff,0x1541}},  //   904, depth=1
{0xfffffffffc3fffff,   -1,{0xffff,0x19bb}},  //   905, depth=1
{0x3e3e3e3e3e3e3e3e,   -1,{0xffff,0x01f4}},  //   906, depth=1
{0xfffffffffe0003ff,   -1,{0xffff,0x19f0}},  //   907, depth=1
{0x0001fffffc000000,   -1,{0xffff,0x1996}},  //   908, depth=1
{0x8000003f8000003f,  791,{0xffff,0x0046}},  //   909, depth=3
{0x0000000fff000000,  797,{0xffff,0x1a0b}},  //   910, depth=2
{0x000007ffffff0000,   -1,{0xffff,0x1c1a}},  //   911
{0x01fffffffffffff8,   -1,{0xffff,0x1f75}},  //   912
{0xffbfffffffffffff,   -1,{0xffff,0x127e}},  //   913, depth=1
{0xffc0000000000000,   -1,{0xffff,0x1289}},  //   914, depth=1
{0xffc0000000000001,   -1,{0xffff,0x128a}},  //   915, depth=1
{0x0001800000018000,   -1,{0xffff,0x0441}},  //   916, depth=1
{0xfffffff9fffffff9,  804,{0xffff,0x075d}},  //   917, depth=2
{0x0000000001f001f0,  805,{0x0324,0xffff}},  //   918, depth=2
{0xfffffffffbffffff,   -1,{0xffff,0x197e}},  //   919
{0x00007fffffff8000,   -1,{0xffff,0x1c5f}},  //   920
{0xffc0000000000007,   -1,{0xffff,0x128c}},  //   921, depth=1
{0xfffffffffc000001,   -1,{0xffff,0x19a6}},  //   922
{0x00000fff00000fff,  806,{0xffff,0x000b}},  //   923, depth=3
{0xffffffffff8007ff,  926,{0xffff,0x1a73}},  //   924
{0xff000000ffffffff,  813,{0xffff,0x1227}},  //   925, depth=2
{0xfffffffffc000003,  928,{0xffff,0x19a7}},  //   926
{0xfc3ffc3ffc3ffc3f,  816,{0xffff,0x01ab}},  //   927, depth=3
{0x8000000000007fff,   -1,{0xffff,0x104f}},  //   928
{0xffc000000000000f,   -1,{0xffff,0x128d}},  //   929, depth=1
{0x01ffffffffffff80,   -1,{0xffff,0x1e71}},  //   930, depth=1
{0xe007ffffffffffff,   -1,{0xffff,0x10f5}},  //   931, depth=1
{0x003ffffff0000000,   -1,{0xffff,0x1919}},  //   932, depth=1
{0x00000000ffc3ffff,  824,{0x029b,0xffff}},  //   933, depth=2
{0x6000000060000000,  826,{0xffff,0x00c1}},  //   934, depth=2
{0x01fffffffffffffe,  937,{0xffff,0x1ff7}},  //   935
{0x1ffe1ffe1ffe1ffe,   -1,{0xffff,0x03eb}},  //   936, depth=1
{0x0000f80000000000,  938,{0xffff,0x1544}},  //   937
{0x00000000001fe000,   -1,{0x04c7,0x1cc7}},  //   938
{0xfffffffffc000007,   -1,{0xffff,0x19a8}},  //   939
{0x7fff80007fff8000,  827,{0xffff,0x044f}},  //   940, depth=2
{0xfffffcfffffffcff,  829,{0xffff,0x059d}},  //   941, depth=2
{0x0000000e00000000,   -1,{0xffff,0x17c2}},  //   942
{0x00000000fbfffbff,   -1,{0x016e,0xffff}},  //   943, depth=1
{0x0000001fff800000,   -1,{0xffff,0x1a4d}},  //   944, depth=1
{0xffc000000000001f,   -1,{0xffff,0x128e}},  //   945, depth=1
{0x1e001e001e001e00,  832,{0xffff,0x01e3}},  //   946, depth=2
{0x00000000ff8003ff,  838,{0x0252,0xffff}},  //   947, depth=3
{0x000001f800000000,  843,{0xffff,0x1745}},  //   948, depth=2
{0xfffffc01fffffc01,   -1,{0xffff,0x0596}},  //   949, depth=1
{0x0000000001fff000,   -1,{0x050c,0x1d0c}},  //   950, depth=1
{0xfffff8000001ffff,  956,{0xffff,0x1565}},  //   951
{0x03e0000003e00000,   -1,{0xffff,0x02c4}},  //   952, depth=1
{0xf8000000000000ff,   -1,{0xffff,0x114c}},  //   953, depth=1
{0x0000000000001c00,   -1,{0x0582,0x1d82}},  //   954, depth=1
{0x00000000ff83ff83,  846,{0x026a,0xffff}},  //   955, depth=2
{0xc01fffffc01fffff,   -1,{0xffff,0x0096}},  //   956
{0x000003fffffc0000,   -1,{0xffff,0x1b97}},  //   957
{0x00000007fff80000,   -1,{0xffff,0x1b4f}},  //   958, depth=1
{0xffffc000000001ff,   -1,{0xffff,0x149a}},  //   959, depth=1
{0x00000000000003c0,  848,{0x0683,0x1e83}},  //   960, depth=2
{0xfffffffffffff07f,   -1,{0xffff,0x1d3a}},  //   961, depth=1
{0x00fff00000000000,  850,{0xffff,0x150b}},  //   962, depth=2
{0xf000003ff000003f,  851,{0xffff,0x0109}},  //   963, depth=2
{0x0000000007fe0000,   -1,{0x03c9,0x1bc9}},  //   964, depth=1
{0xfffffffffc00000f,   -1,{0xffff,0x19a9}},  //   965
{0x0000000010001000,   -1,{0x0120,0xffff}},  //   966, depth=1
{0xfffffe00000000ff,   -1,{0xffff,0x15de}},  //   967, depth=1
{0x3f80000000000000,   -1,{0xffff,0x1246}},  //   968, depth=1
{0x0000000e0000000e,   -1,{0xffff,0x07c2}},  //   969
{0x007ff000007ff000,   -1,{0xffff,0x050a}},  //   970, depth=1
{0x000000008003ffff,  858,{0x0052,0xffff}},  //   971, depth=3
{0x000fffffffffffe0,   -1,{0xffff,0x1eee}},  //   972, depth=1
{0xffffffffffffbfff,   -1,{0xffff,0x1c7e}},  //   973, depth=1
{0xffffffffffffc000,   -1,{0xffff,0x1cb1}},  //   974, depth=1
{0xc000000001ffffff,  860,{0xffff,0x109a}},  //   975, depth=2
{0x00000fffffff8000,   -1,{0xffff,0x1c5c}},  //   976, depth=1
{0xffc000000000003f,  862,{0xffff,0x128f}},  //   977, depth=2
{0x0000003000000000,   -1,{0xffff,0x1701}},  //   978, depth=1
{0xfc003fffffffffff,   -1,{0xffff,0x11b3}},  //   979
{0x00003fffffff0000,   -1,{0xffff,0x1c1d}},  //   980
{0xfff00007ffffffff,  866,{0xffff,0x132e}},  //   981, depth=2
{0x003fffffffe00000,   -1,{0xffff,0x1ae0}},  //   982, depth=1
{0x8fff8fff8fff8fff,   -1,{0xffff,0x006c}},  //   983, depth=1
{0x000000007e000000,  868,{0x01c5,0x19c5}},  //   984, depth=2
{0x00000000f007f007,  871,{0x0126,0xffff}},  //   985, depth=2
{0x00000000001ff800,   -1,{0x0549,0x1d49}},  //   986, depth=1
{0xbfbfbfbfbfbfbfbf,  875,{0xffff,0x0076}},  //   987, depth=2
{0x000ffffffffffff0,   -1,{0xffff,0x1f2f}},  //   988, depth=1
{0xffffffffffffc00f,   -1,{0xffff,0x1cb5}},  //   989, depth=1
{0xfffffffffffff0ff,   -1,{0xffff,0x1d3b}},  //   990
{0x7fffc00000000000,   -1,{0xffff,0x1490}},  //   991
{0x00000000000003e0,   -1,{0x06c4,0x1ec4}},  //   992, depth=1
{0xff80000000000fff,  878,{0xffff,0x1254}},  //   993, depth=2
{0x01ffffffffffffc0,   -1,{0xffff,0x1eb2}},  //   994, depth=1
{0x03fffffffffffc00,   -1,{0xffff,0x1daf}},  //   995
{0x000ffffffffffff8,   -1,{0xffff,0x1f70}},  //   996, depth=1
{0xfffff800000001ff,  880,{0xffff,0x155d}},  //   997, depth=2
{0x000fffffc0000000, 1009,{0xffff,0x1895}},  //   998
{0xffe01fffffe01fff,   -1,{0xffff,0x02d7}},  //   999, depth=1
{0x000ffffffffffffc,   -1,{0xffff,0x1fb1}},  //  1000, depth=1
{0xff000000007fffff,   -1,{0xffff,0x121e}},  //  1001, depth=1
{0x000ffffffffffffe,   -1,{0xffff,0x1ff2}},  //  1002, depth=1
{0x000fffffffffffff,   -1,{0xffff,0x1033}},  //  1003, depth=1
{0x0010000000000000,   -1,{0xffff,0x1300}},  //  1004, depth=1
{0xffffffffffffc01f,   -1,{0xffff,0x1cb6}},  //  1005, depth=1
{0x0000000000f00000,   -1,{0x0303,0x1b03}},  //  1006, depth=1
{0xe00000000001ffff,   -1,{0xffff,0x10d3}},  //  1007, depth=1
{0x00000000000003f0,   -1,{0x0705,0x1f05}},  //  1008, depth=1
{0x001e001e001e001e,   -1,{0xffff,0x03e3}},  //  1009
{0xffffc01fffffffff,   -1,{0xffff,0x14b6}},  //  1010
{0xffff800001ffffff,   -1,{0xffff,0x1469}},  //  1011, depth=1
{0x000007fe000007fe,   -1,{0xffff,0x07c9}},  //  1012, depth=1
{0xffffffffffffc07f,   -1,{0xffff,0x1cb8}},  //  1013
{0x1fffffffffff0000,   -1,{0xffff,0x1c2c}},  //  1014
{0xfe03ffffffffffff,   -1,{0xffff,0x11f8}},  //  1015, depth=1
{0x00000000000003f8,   -1,{0x0746,0x1f46}},  //  1016, depth=1
{0xffffff3fffffffff,   -1,{0xffff,0x163d}},  //  1017, depth=1
{0x0000c00000000000,   -1,{0xffff,0x1481}},  //  1018, depth=1
{0x03f0000000000000,   -1,{0xffff,0x1305}},  //  1019
{0x0000070000000700,  884,{0xffff,0x0602}},  //  1020, depth=3
{0x0f0f0f0f0f0f0f0f,   -1,{0xffff,0x0033}},  //  1021, depth=1
{0x00000000000003fe,   -1,{0x07c8,0x1fc8}},  //  1022, depth=1
{0x00000000000003ff,  889,{0x0009,0x1009}},  //  1023, depth=2
{0x0000000000000400,   -1,{0x0580,0x1d80}},  //  1024, depth=1
{0xffffc0000001ffff,   -1,{0xffff,0x14a2}},  //  1025, depth=1
{0x0000003000000030,  890,{0xffff,0x0701}},  //  1026, depth=2
{0xff01ffffff01ffff,   -1,{0xffff,0x0218}},  //  1027, depth=1
{0x000000000003ff80,  891,{0x064a,0x1e4a}},  //  1028, depth=3
{0xfffffffffc00003f,   -1,{0xffff,0x19ab}},  //  1029
{0x0000000007fffe00,   -1,{0x05d1,0x1dd1}},  //  1030, depth=1
{0xe0000fffe0000fff,  893,{0xffff,0x00ce}},  //  1031, depth=2
{0xffffffffc007ffff,   -1,{0xffff,0x18b4}},  //  1032
{0xffffe00007ffffff,   -1,{0xffff,0x14ed}},  //  1033
{0xfffffffff8003fff,   -1,{0xffff,0x1972}},  //  1034
{0xff01ff01ff01ff01,   -1,{0xffff,0x0228}},  //  1035, depth=1
{0x000000000ffff800,   -1,{0x0550,0x1d50}},  //  1036, depth=1
{0xffff8fffffff8fff,  894,{0xffff,0x045c}},  //  1037, depth=3
{0x00001e0000000000,   -1,{0xffff,0x15c3}},  //  1038
{0xfffffffffe03ffff,   -1,{0xffff,0x19f8}},  //  1039, depth=1
{0x000001fff0000000,   -1,{0xffff,0x190c}},  //  1040
{0xffc000000000007f,   -1,{0xffff,0x1290}},  //  1041, depth=1
{0x01fffffffffffff0,   -1,{0xffff,0x1f34}},  //  1042, depth=1
{0xfc0007ffffffffff,   -1,{0xffff,0x11b0}},  //  1043, depth=1
{0x07fffe0007fffe00,   -1,{0xffff,0x05d1}},  //  1044, depth=1
{0xc00000ffffffffff,   -1,{0xffff,0x10a9}},  //  1045
{0x0000000088888888,  902,{0x0078,0xffff}},  //  1046, depth=2
{0xffffffe00fffffff,   -1,{0xffff,0x16f6}},  //  1047
{0x000000000fc00fc0,  903,{0x02a5,0xffff}},  //  1048, depth=3
{0xfe003ffffe003fff,   -1,{0xffff,0x01d4}},  //  1049, depth=1
{0x000fffffe0000000,  912,{0xffff,0x18d6}},  //  1050, depth=2
{0x0000001f0000001f,  919,{0xffff,0x0004}},  //  1051, depth=2
{0xfffffffffc000000,  920,{0xffff,0x19a5}},  //  1052, depth=2
{0xfe0000000000ffff,  922,{0xffff,0x11d6}},  //  1053, depth=2
{0x01fffffffffffffc,   -1,{0xffff,0x1fb6}},  //  1054, depth=1
{0x00000000fe0000ff,  924,{0x01ce,0xffff}},  //  1055, depth=4
{0x0038000000380000,  935,{0xffff,0x0342}},  //  1056, depth=4
{0x01ffffffffffffff,   -1,{0xffff,0x1038}},  //  1057, depth=1
{0x0200000000000000,   -1,{0xffff,0x11c0}},  //  1058, depth=1
{0xffbfffffffbfffff,  939,{0xffff,0x025e}},  //  1059, depth=2
{0x07fffffff0000000,   -1,{0xffff,0x191e}},  //  1060
{0xffffffff00001fff,   -1,{0xffff,0x182c}},  //  1061, depth=1
{0x01e0000001e00000,  942,{0xffff,0x02c3}},  //  1062, depth=2
{0x800f800f800f800f,  951,{0xffff,0x0064}},  //  1063, depth=3
{0x000fffffffc00000,   -1,{0xffff,0x1a9d}},  //  1064
{0x000000000003fffc,   -1,{0x078f,0x1f8f}},  //  1065
{0x0000007ff8000000,  957,{0xffff,0x194b}},  //  1066, depth=2
{0x807fffff807fffff,  965,{0xffff,0x0057}},  //  1067, depth=2
{0xfc003ffffc003fff,   -1,{0xffff,0x0193}},  //  1068
{0xfff3ffffffffffff,   -1,{0xffff,0x133d}},  //  1069, depth=1
{0xffc00000000000ff,   -1,{0xffff,0x1291}},  //  1070
{0xffffffffffc7ffff,   -1,{0xffff,0x1abc}},  //  1071
{0x6000000000000000,   -1,{0xffff,0x10c1}},  //  1072, depth=1
{0xffffffffe007ffff,   -1,{0xffff,0x18f5}},  //  1073, depth=1
{0x0000003fffffff80,   -1,{0xffff,0x1e5e}},  //  1074
{0xfffffff800ffffff,   -1,{0xffff,0x1774}},  //  1075, depth=1
{0x000000003e000000,  969,{0x01c4,0x19c4}},  //  1076, depth=2
{0x000001ffffe00000,   -1,{0xffff,0x1ad3}},  //  1077
{0x000fffff80000000,   -1,{0xffff,0x1854}},  //  1078
{0x00000000ff83ffff,   -1,{0x025a,0xffff}},  //  1079, depth=1
{0x0e00000000000000,   -1,{0xffff,0x11c2}},  //  1080
{0xe000001fffffffff,  979,{0xffff,0x10e7}},  //  1081, depth=2
{0x000003fffffffc00,   -1,{0xffff,0x1d9f}},  //  1082
{0xfffffffffc00001f,   -1,{0xffff,0x19aa}},  //  1083, depth=1
{0x00007ffff8000000,   -1,{0xffff,0x1953}},  //  1084
{0xffffffff8007ffff,   -1,{0xffff,0x1873}},  //  1085
{0x0007ffffff000000,  980,{0xffff,0x1a1a}},  //  1086, depth=2
{0xfffff00001ffffff,   -1,{0xffff,0x152c}},  //  1087, depth=1
{0x3fffffffffff8000,   -1,{0xffff,0x1c6e}},  //  1088, depth=1
{0xf0000000000fffff,  990,{0xffff,0x1117}},  //  1089, depth=2
{0x0000000001ffc000,  991,{0x048a,0x1c8a}},  //  1090, depth=2
{0x0007000000000000,   -1,{0xffff,0x1402}},  //  1091
{0x000000000003ffc0,  995,{0x068b,0x1e8b}},  //  1092, depth=2
{0xffffc1ffffffc1ff,   -1,{0xffff,0x049a}},  //  1093
{0x0ffffffffffc0000,   -1,{0xffff,0x1ba9}},  //  1094, depth=1
{0x00000007fffff800,   -1,{0xffff,0x1d57}},  //  1095
{0x00000000e000e000,  998,{0x00e2,0xffff}},  //  1096, depth=3
{0xffffffe0001fffff,   -1,{0xffff,0x16ef}},  //  1097, depth=1
{0xfffffe00000001ff, 1100,{0xffff,0x15df}},  //  1098
{0xfffc001ffffc001f, 1010,{0xffff,0x0392}},  //  1099, depth=2
{0xfc000000000fffff,   -1,{0xffff,0x1199}},  //  1100
{0x00000000fffe07ff, 1013,{0x03d9,0xffff}},  //  1101, depth=2
{0xffffffffffffc0ff,   -1,{0xffff,0x1cb9}},  //  1102
{0x0000001fff000000,   -1,{0xffff,0x1a0c}},  //  1103
{0x01ffffffe0000000, 1014,{0xffff,0x18db}},  //  1104, depth=2
{0xffffc003ffffc003,   -1,{0xffff,0x0493}},  //  1105
{0x000000000fffe000,   -1,{0x04ce,0x1cce}},  //  1106, depth=1
{0xfff000000000ffff,   -1,{0xffff,0x131b}},  //  1107, depth=1
{0xffc0000fffffffff,   -1,{0xffff,0x12ad}},  //  1108
{0xff0000007fffffff,   -1,{0xffff,0x1226}},  //  1109, depth=1
{0x0000003fffffffc0,   -1,{0xffff,0x1e9f}},  //  1110
{0x0ff0000000000000,   -1,{0xffff,0x1307}},  //  1111
{0x0000200000002000, 1019,{0xffff,0x04c0}},  //  1112, depth=2
{0xffffffffffff8fff,   -1,{0xffff,0x1c7c}},  //  1113, depth=1
{0x000c0000000c0000,   -1,{0xffff,0x0381}},  //  1114, depth=1
{0xff8001ffffffffff, 1029,{0xffff,0x1271}},  //  1115, depth=2
{0xfff80000001fffff,   -1,{0xffff,0x1361}},  //  1116
{0xfffffc00000fffff,   -1,{0xffff,0x15a9}},  //  1117, depth=1
{0x03fffffc03fffffc,   -1,{0xffff,0x0797}},  //  1118
{0x00000000ff03ff03, 1032,{0x0229,0xffff}},  //  1119, depth=2
{0x000000fffffff000,   -1,{0xffff,0x1d1b}},  //  1120, depth=1
{0xf07ffffff07fffff,   -1,{0xffff,0x011a}},  //  1121, depth=1
{0x000000001e000000,   -1,{0x01c3,0x19c3}},  //  1122, depth=1
{0x00000000f81fffff, 1033,{0x0159,0xffff}},  //  1123, depth=2
{0x000000000003ffe0,   -1,{0x06cc,0x1ecc}},  //  1124, depth=1
{0xffffff0007ffffff,   -1,{0xffff,0x1632}},  //  1125
{0x000000ffffffc000, 1133,{0xffff,0x1c99}},  //  1126
{0xffff000000007fff,   -1,{0xffff,0x141e}},  //  1127, depth=1
{0x7ffff80000000000,   -1,{0xffff,0x1553}},  //  1128, depth=1
{0xbbbbbbbbbbbbbbbb, 1034,{0xffff,0x007a}},  //  1129, depth=2
{0x000000000f800f80, 1038,{0x0264,0xffff}},  //  1130, depth=2
{0xffffffff000007ff,   -1,{0xffff,0x182a}},  //  1131, depth=1
{0x00000ffff8000000,   -1,{0xffff,0x1950}},  //  1132, depth=1
{0x0ffe0ffe0ffe0ffe,   -1,{0xffff,0x03ea}},  //  1133
{0x003fff80003fff80, 1040,{0xffff,0x064e}},  //  1134, depth=2
{0xff0000ffff0000ff,   -1,{0xffff,0x020f}},  //  1135, depth=1
{0x0003fc0000000000,   -1,{0xffff,0x1587}},  //  1136, depth=1
{0xfffff81fffffffff,   -1,{0xffff,0x1579}},  //  1137, depth=1
{0xfff9fffffff9ffff,   -1,{0xffff,0x035d}},  //  1138
{0x0000000007f807f8,   -1,{0x0367,0xffff}},  //  1139
{0x000000000003fff0,   -1,{0x070d,0x1f0d}},  //  1140, depth=1
{0xf80000000001ffff,   -1,{0xffff,0x1155}},  //  1141
{0x1ffc000000000000,   -1,{0xffff,0x138a}},  //  1142, depth=1
{0xff000001ff000001,   -1,{0xffff,0x0208}},  //  1143, depth=1
{0xffffffffff001fff,   -1,{0xffff,0x1a34}},  //  1144
{0x87ff87ff87ff87ff, 1045,{0xffff,0x006b}},  //  1145, depth=2
{0xffffe00000007fff,   -1,{0xffff,0x14e1}},  //  1146
{0xff000000003fffff, 1047,{0xffff,0x121d}},  //  1147, depth=2
{0x000000000003fff8, 1060,{0x074e,0x1f4e}},  //  1148, depth=2
{0xfffffffe00007fff,   -1,{0xffff,0x17ed}},  //  1149, depth=1
{0x01ffffffc0000000, 1064,{0xffff,0x189a}},  //  1150, depth=2
{0xffe007ffffffffff,   -1,{0xffff,0x12f5}},  //  1151, depth=1
{0xfff8fff8fff8fff8, 1065,{0xffff,0x036c}},  //  1152, depth=2
{0xffffff8001ffffff,   -1,{0xffff,0x1671}},  //  1153, depth=1
{0x000000000003fffe,   -1,{0x07d0,0x1fd0}},  //  1154, depth=1
{0x000000000003ffff,   -1,{0x0011,0x1011}},  //  1155, depth=1
{0x0000000000040000,   -1,{0x0380,0x1b80}},  //  1156, depth=1
{0x0000003ffffffff0,   -1,{0xffff,0x1f21}},  //  1157
{0xffffffc003ffffff, 1161,{0xffff,0x16b3}},  //  1158
{0xe000001fe000001f, 1068,{0xffff,0x00c7}},  //  1159, depth=2
{0x0000000000040004,   -1,{0x03a0,0xffff}},  //  1160, depth=1
{0xfffffe000001ffff,   -1,{0xffff,0x15e7}},  //  1161
{0xfffffff8ffffffff,   -1,{0xffff,0x177c}},  //  1162
{0xff00000000000000,   -1,{0xffff,0x1207}},  //  1163
{0x0000003ffffffff8,   -1,{0xffff,0x1f62}},  //  1164
{0xffff801fffff801f,   -1,{0xffff,0x0455}},  //  1165, depth=1
{0x007fffe000000000,   -1,{0xffff,0x16d1}},  //  1166, depth=1
{0xfffe0000000003ff,   -1,{0xffff,0x13d8}},  //  1167, depth=1
{0x3f8000003f800000,   -1,{0xffff,0x0246}},  //  1168, depth=1
{0x0000000033333333, 1070,{0x0039,0xffff}},  //  1169, depth=2
{0x00007ff800007ff8,   -1,{0xffff,0x074b}},  //  1170, depth=1
{0x00000000f00fffff,   -1,{0x0117,0xffff}},  //  1171, depth=1
{0xfff800000fffffff,   -1,{0xffff,0x1368}},  //  1172
{0xfffff87ffffff87f, 1071,{0xffff,0x055b}},  //  1173, depth=2
{0x000007f800000000,   -1,{0xffff,0x1747}},  //  1174, depth=1
{0x807fffffffffffff,   -1,{0xffff,0x1077}},  //  1175, depth=1
{0x1ffffff81ffffff8, 1074,{0xffff,0x0759}},  //  1176, depth=2
{0x00000f0000000f00,   -1,{0xffff,0x0603}},  //  1177
{0x0000004000000000,   -1,{0xffff,0x1680}},  //  1178
{0xfffffffffc00007f,   -1,{0xffff,0x19ac}},  //  1179, depth=1
{0x000f8000000f8000,   -1,{0xffff,0x0444}},  //  1180, depth=1
{0xffff80ffffffffff,   -1,{0xffff,0x1478}},  //  1181, depth=1
{0xfffffffffc0000ff,   -1,{0xffff,0x19ad}},  //  1182
{0x00000000801f801f,   -1,{0x0065,0xffff}},  //  1183, depth=1
{0x007ffff0007ffff0, 1077,{0xffff,0x0712}},  //  1184, depth=2
{0x8007800780078007,   -1,{0xffff,0x0063}},  //  1185, depth=1
{0x0000000000007e00,   -1,{0x05c5,0x1dc5}},  //  1186, depth=1
{0xf8fff8fff8fff8ff,   -1,{0xffff,0x016c}},  //  1187, depth=1
{0x003f0000003f0000, 1078,{0xffff,0x0405}},  //  1188, depth=2
{0xffe03fffffffffff,   -1,{0xffff,0x12f8}},  //  1189, depth=1
{0xff0000000000000f,   -1,{0xffff,0x120b}},  //  1190
{0x0078000000000000,   -1,{0xffff,0x1343}},  //  1191
{0xf800f800f800f800, 1080,{0xffff,0x0164}},  //  1192, depth=2
{0xff0000000000001f,   -1,{0xffff,0x120c}},  //  1193
{0x3fffffc03fffffc0,   -1,{0xffff,0x0697}},  //  1194
{0x7800000000000000,   -1,{0xffff,0x1143}},  //  1195
{0xff8007ffffffffff,   -1,{0xffff,0x1273}},  //  1196
{0xf800001fffffffff, 1200,{0xffff,0x1169}},  //  1197
{0x07ffffffffe00000, 1082,{0xffff,0x1ae5}},  //  1198, depth=2
{0xf807ffffffffffff,   -1,{0xffff,0x1177}},  //  1199, depth=1
{0x00000000fff0007f,   -1,{0x0312,0xffff}},  //  1200
{0xff0000003fffffff,   -1,{0xffff,0x1225}},  //  1201, depth=1
{0x00007c0000007c00,   -1,{0xffff,0x0584}},  //  1202, depth=1
{0xfffc0000ffffffff,   -1,{0xffff,0x13ad}},  //  1203, depth=1
{0x01ffffffffc00000,   -1,{0xffff,0x1aa2}},  //  1204, depth=1
{0x00000000ffefffef,   -1,{0x02ee,0xffff}},  //  1205, depth=1
{0x01ffffffff800000, 1210,{0xffff,0x1a61}},  //  1206
{0xf0000003ffffffff,   -1,{0xffff,0x1125}},  //  1207, depth=1
{0x1ffffc001ffffc00, 1084,{0xffff,0x0592}},  //  1208, depth=2
{0xf8000000000001ff,   -1,{0xffff,0x114d}},  //  1209, depth=1
{0x0000003fe0000000,   -1,{0xffff,0x18c8}},  //  1210
{0xfffffffc000003ff, 1085,{0xffff,0x17a7}},  //  1211, depth=2
{0x00003ff800000000,   -1,{0xffff,0x174a}},  //  1212, depth=1
{0x7ffffe0000000000,   -1,{0xffff,0x15d5}},  //  1213
{0xff0000000000003f,   -1,{0xffff,0x120d}},  //  1214
{0xfffffff8007fffff,   -1,{0xffff,0x1773}},  //  1215
{0x007c007c007c007c, 1091,{0xffff,0x03a4}},  //  1216, depth=2
{0x0007e00000000000,   -1,{0xffff,0x14c5}},  //  1217
{0xff803fffffffffff,   -1,{0xffff,0x1276}},  //  1218
{0x00000000e7ffffff,   -1,{0x00dd,0xffff}},  //  1219, depth=1
{0xfc01ffffffffffff,   -1,{0xffff,0x11b6}},  //  1220
{0x00000000ffefffff, 1093,{0x02de,0xffff}},  //  1221, depth=2
{0x00000000fff00000, 1095,{0x030b,0x1b0b}},  //  1222, depth=2
{0x00000000fff00001, 1098,{0x030c,0xffff}},  //  1223, depth=3
{0x000000000fff8000,   -1,{0x044c,0x1c4c}},  //  1224
{0x00000000fff00003,   -1,{0x030d,0xffff}},  //  1225, depth=1
{0x0ffffffffffffc00,   -1,{0xffff,0x1db1}},  //  1226, depth=1
{0x80000000000007ff,   -1,{0xffff,0x104b}},  //  1227
{0x00000000000f0000,   -1,{0x0403,0x1c03}},  //  1228, depth=1
{0x00000000fff00007, 1102,{0x030e,0xffff}},  //  1229, depth=2
{0x1ffffff800000000,   -1,{0xffff,0x1759}},  //  1230, depth=1
{0xffff000000001fff,   -1,{0xffff,0x141c}},  //  1231
{0x000007ffffe00000,   -1,{0xffff,0x1ad5}},  //  1232
{0x00fffc0000000000,   -1,{0xffff,0x158d}},  //  1233
{0xffc7ffffffffffff, 1241,{0xffff,0x12bc}},  //  1234
{0xfffffc03ffffffff,   -1,{0xffff,0x15b7}},  //  1235, depth=1
{0x0003fff80003fff8, 1103,{0xffff,0x074e}},  //  1236, depth=2
{0x00000000fff0000f, 1105,{0x030f,0xffff}},  //  1237, depth=2
{0x0007fffe00000000,   -1,{0xffff,0x17d1}},  //  1238, depth=1
{0x00000000ff0003ff, 1108,{0x0211,0xffff}},  //  1239, depth=2
{0x007ffffffffe0000, 1110,{0xffff,0x1be5}},  //  1240, depth=2
{0x00000000fff80fff,   -1,{0x0358,0xffff}},  //  1241
{0x01ffffff80000000,   -1,{0xffff,0x1859}},  //  1242, depth=1
{0x00000000000f000f,   -1,{0x0023,0xffff}},  //  1243, depth=1
{0x3ffffffff8000000,   -1,{0xffff,0x1962}},  //  1244, depth=1
{0x0000001e0000001e,   -1,{0xffff,0x07c3}},  //  1245
{0x0000000000f000f0, 1111,{0x0323,0xffff}},  //  1246, depth=2
{0xff0000001fffffff, 1116,{0xffff,0x1224}},  //  1247, depth=2
{0x007ff800007ff800, 1118,{0xffff,0x054b}},  //  1248, depth=2
{0xff0000000000007f,   -1,{0xffff,0x120e}},  //  1249
{0xfffc0000003fffff,   -1,{0xffff,0x13a3}},  //  1250
{0xf001ffffffffffff,   -1,{0xffff,0x1134}},  //  1251, depth=1
{0x00001ffffffff000,   -1,{0xffff,0x1d20}},  //  1252, depth=1
{0x00000000fff0001f,   -1,{0x0310,0xffff}},  //  1253, depth=1
{0x01ffffff00000000,   -1,{0xffff,0x1818}},  //  1254
{0xffe0003fffe0003f, 1125,{0xffff,0x02d0}},  //  1255, depth=2
{0xfffffffe00001fff,   -1,{0xffff,0x17eb}},  //  1256
{0xfffff0ffffffffff,   -1,{0xffff,0x153b}},  //  1257, depth=1
{0x00007fc000000000,   -1,{0xffff,0x1688}},  //  1258
{0xfffe003fffffffff, 1262,{0xffff,0x13f4}},  //  1259
{0x07fc000007fc0000, 1126,{0xffff,0x0388}},  //  1260, depth=3
{0xfffffffff0ffffff,   -1,{0xffff,0x193b}},  //  1261, depth=1
{0x00000000fe03fe03,   -1,{0x01e8,0xffff}},  //  1262
{0x0000007ffffc0000,   -1,{0xffff,0x1b94}},  //  1263
{0x03fffe0003fffe00,   -1,{0xffff,0x05d0}},  //  1264, depth=1
{0x00000000c7ffffff, 1138,{0x009c,0xffff}},  //  1265, depth=2
{0x003f800000000000,   -1,{0xffff,0x1446}},  //  1266, depth=1
{0xffffffe00000ffff,   -1,{0xffff,0x16ea}},  //  1267
{0x0003fc000003fc00,   -1,{0xffff,0x0587}},  //  1268, depth=1
{0xfffffffff8000fff,   -1,{0xffff,0x1970}},  //  1269, depth=1
{0x0e000e000e000e00, 1139,{0xffff,0x01e2}},  //  1270, depth=2
{0x03ffffe000000000,   -1,{0xffff,0x16d4}},  //  1271
{0x0000003fffffffe0,   -1,{0xffff,0x1ee0}},  //  1272, depth=1
{0x1fffffffffe00000,   -1,{0xffff,0x1ae7}},  //  1273
{0xfff0fff0fff0fff0,   -1,{0xffff,0x032b}},  //  1274, depth=1
{0x00000000fff83fff, 1141,{0x035a,0xffff}},  //  1275, depth=2
{0xfffffff00fffffff,   -1,{0xffff,0x1737}},  //  1276
{0xc0001fffffffffff, 1144,{0xffff,0x10ae}},  //  1277, depth=2
{0xfffffff87fffffff,   -1,{0xffff,0x177b}},  //  1278
{0xfffc0000007fffff, 1146,{0xffff,0x13a4}},  //  1279, depth=2
{0xffff0000000007ff,   -1,{0xffff,0x141a}},  //  1280
{0xfffc00003fffffff,   -1,{0xffff,0x13ab}},  //  1281
{0x0000fe0000000000,   -1,{0xffff,0x15c6}},  //  1282, depth=1
{0x3fffffc000000000,   -1,{0xffff,0x1697}},  //  1283
{0xfffffff8fffffff8,   -1,{0xffff,0x075c}},  //  1284, depth=1
{0x00000000fff0003f,   -1,{0x0311,0xffff}},  //  1285, depth=1
{0x0000000007ffff00,   -1,{0x0612,0x1e12}},  //  1286, depth=1
{0xffffffffffffc1ff,   -1,{0xffff,0x1cba}},  //  1287
{0x000000000007fc00, 1157,{0x0588,0x1d88}},  //  1288, depth=2
{0x00000000e7ffe7ff, 1158,{0x00ed,0xffff}},  //  1289, depth=3
{0x00000ffe00000ffe,   -1,{0xffff,0x07ca}},  //  1290, depth=1
{0xff0fffffff0fffff, 1162,{0xffff,0x021b}},  //  1291, depth=2
{0x00000007ffffe000,   -1,{0xffff,0x1cd5}},  //  1292, depth=1
{0xfeffffffffffffff,   -1,{0xffff,0x11fe}},  //  1293, depth=1
{0x000000000f000f00, 1163,{0x0223,0xffff}},  //  1294, depth=2
{0xff00000000000001,   -1,{0xffff,0x1208}},  //  1295, depth=1
{0x000fffffff800000, 1164,{0xffff,0x1a5c}},  //  1296, depth=2
{0xff00000000000003, 1172,{0xffff,0x1209}},  //  1297, depth=2
{0x0000000000700000, 1177,{0x0302,0x1b02}},  //  1298, depth=2
{0xfffe00000003ffff,   -1,{0xffff,0x13e0}},  //  1299, depth=1
{0x0000003ffffffffc,   -1,{0xffff,0x1fa3}},  //  1300, depth=1
{0xff00000000000007,   -1,{0xffff,0x120a}},  //  1301, depth=1
{0x0000003ffffffffe,   -1,{0xffff,0x1fe4}},  //  1302, depth=1
{0x0000003fffffffff,   -1,{0xffff,0x1025}},  //  1303, depth=1
{0x1f801f801f801f80, 1178,{0xffff,0x0265}},  //  1304, depth=2
{0x0707070707070707,   -1,{0xffff,0x0032}},  //  1305, depth=1
{0x007ffffffffffe00,   -1,{0xffff,0x1ded}},  //  1306, depth=1
{0x7fff7fff7fff7fff, 1182,{0xffff,0x002e}},  //  1307, depth=2
{0x007ffff000000000,   -1,{0xffff,0x1712}},  //  1308
{0xffffffe7ffffffe7, 1190,{0xffff,0x06dd}},  //  1309, depth=2
{0x0003fffffffff800,   -1,{0xffff,0x1d66}},  //  1310, depth=1
{0x00000000fe0001ff,   -1,{0x01cf,0xffff}},  //  1311, depth=1
{0x7f007f007f007f00,   -1,{0xffff,0x0226}},  //  1312, depth=1
{0xffff9fffffffffff,   -1,{0xffff,0x147d}},  //  1313, depth=1
{0x0000ffc00000ffc0,   -1,{0xffff,0x0689}},  //  1314, depth=1
{0xffe1ffffffffffff,   -1,{0xffff,0x12fb}},  //  1315
{0xfffffc00fffffc00, 1191,{0xffff,0x0595}},  //  1316, depth=2
{0xffff00ffffff00ff,   -1,{0xffff,0x0417}},  //  1317
{0xffffffc000000fff,   -1,{0xffff,0x16a5}},  //  1318
{0xffffffffff87ffff,   -1,{0xffff,0x1a7b}},  //  1319, depth=1
{0x0000000007ffffe0,   -1,{0x06d5,0x1ed5}},  //  1320
{0xfffffff8003fffff, 1322,{0xffff,0x1772}},  //  1321
{0xffff0001ffff0001,   -1,{0xffff,0x0410}},  //  1322
{0x00000ffffffc0000, 1327,{0xffff,0x1b99}},  //  1323
{0x00fffffffff80000,   -1,{0xffff,0x1b64}},  //  1324, depth=1
{0x00000000c1ffc1ff, 1193,{0x00aa,0xffff}},  //  1325, depth=2
{0x07ff800007ff8000, 1194,{0xffff,0x044b}},  //  1326, depth=2
{0xffe0ffe0ffe0ffe0,   -1,{0xffff,0x02ea}},  //  1327
{0xfffc00001fffffff,   -1,{0xffff,0x13aa}},  //  1328
{0xfffffffffff003ff,   -1,{0xffff,0x1b35}},  //  1329, depth=1
{0x0000000007fffff0,   -1,{0x0716,0x1f16}},  //  1330
{0xffe00000000fffff,   -1,{0xffff,0x12de}},  //  1331, depth=1
{0xffffffffffff0000, 1334,{0xffff,0x1c2f}},  //  1332
{0xffffffe7ffffffff,   -1,{0xffff,0x16fd}},  //  1333, depth=1
{0x0000000007fffff8,   -1,{0x0757,0x1f57}},  //  1334
{0x8000000000001fff,   -1,{0xffff,0x104d}},  //  1335, depth=1
{0x00000000001f8000,   -1,{0x0445,0x1c45}},  //  1336, depth=1
{0xffc007ffffc007ff,   -1,{0xffff,0x0294}},  //  1337, depth=1
{0x000000fffc000000,   -1,{0xffff,0x198d}},  //  1338, depth=1
{0xffffffffffff0001,   -1,{0xffff,0x1c30}},  //  1339
{0x000e000e000e000e, 1195,{0xffff,0x03e2}},  //  1340, depth=2
{0xfc000003ffffffff, 1196,{0xffff,0x11a7}},  //  1341, depth=2
{0x003ff00000000000,   -1,{0xffff,0x1509}},  //  1342, depth=1
{0xfffffffc0003ffff,   -1,{0xffff,0x17af}},  //  1343, depth=1
{0x03fffffffffe0000,   -1,{0xffff,0x1be8}},  //  1344
{0xfffffffffffff1ff,   -1,{0xffff,0x1d3c}},  //  1345, depth=1
{0x00000003ffff0000,   -1,{0xffff,0x1c11}},  //  1346, depth=1
{0xffffffffff0007ff,   -1,{0xffff,0x1a32}},  //  1347, depth=1
{0x000000000000f800,   -1,{0x0544,0x1d44}},  //  1348, depth=1
{0xf0fff0fff0fff0ff, 1197,{0xffff,0x012b}},  //  1349, depth=3
{0x0000000080808080, 1206,{0x0070,0xffff}},  //  1350, depth=3
{0x0000000007fffffe,   -1,{0x07d9,0x1fd9}},  //  1351
{0x000000001ffe1ffe,   -1,{0x03eb,0xffff}},  //  1352, depth=1
{0xffffffffffff0007,   -1,{0xffff,0x1c32}},  //  1353
{0xf000f000f000f000, 1213,{0xffff,0x0123}},  //  1354, depth=2
{0xcfffffffcfffffff,   -1,{0xffff,0x009d}},  //  1355, depth=1
{0x001fffe000000000,   -1,{0xffff,0x16cf}},  //  1356, depth=1
{0x0000000087ffffff, 1214,{0x005b,0xffff}},  //  1357, depth=2
{0x01fffffc01fffffc,   -1,{0xffff,0x0796}},  //  1358, depth=1
{0x0000ffffffffc000, 1361,{0xffff,0x1ca1}},  //  1359
{0x000007fff0000000,   -1,{0xffff,0x190e}},  //  1360, depth=1
{0x000000003ff83ff8,   -1,{0x036a,0xffff}},  //  1361
{0xffffffffffff000f,   -1,{0xffff,0x1c33}},  //  1362
{0xfffffe1fffffffff,   -1,{0xffff,0x15fb}},  //  1363, depth=1
{0xffffe00000001fff,   -1,{0xffff,0x14df}},  //  1364
{0x801fffffffffffff,   -1,{0xffff,0x1075}},  //  1365, depth=1
{0xffffffffffff001f,   -1,{0xffff,0x1c34}},  //  1366
{0xffffffc000003fff, 1215,{0xffff,0x16a7}},  //  1367, depth=2
{0x0000004000000040, 1217,{0xffff,0x0680}},  //  1368, depth=2
{0xfffffff83fffffff,   -1,{0xffff,0x177a}},  //  1369
{0x00000fc000000000,   -1,{0xffff,0x1685}},  //  1370, depth=1
{0x00000000ff03ffff,   -1,{0x0219,0xffff}},  //  1371, depth=1
{0x000fffff00000000,   -1,{0xffff,0x1813}},  //  1372, depth=1
{0xfff800fffff800ff,   -1,{0xffff,0x0354}},  //  1373, depth=1
{0x01fffe0001fffe00,   -1,{0xffff,0x05cf}},  //  1374, depth=1
{0x00000000e0007fff,   -1,{0x00d1,0xffff}},  //  1375, depth=1
{0x000003e000000000, 1378,{0xffff,0x16c4}},  //  1376
{0x00000000fe01ffff,   -1,{0x01d7,0xffff}},  //  1377, depth=1
{0x0000000000007f80,   -1,{0x0647,0x1e47}},  //  1378
{0xf8003ffff8003fff, 1218,{0xffff,0x0152}},  //  1379, depth=2
{0x0003ffffffffe000,   -1,{0xffff,0x1ce4}},  //  1380, depth=1
{0xfff80001fff80001,   -1,{0xffff,0x034d}},  //  1381, depth=1
{0x00000001fffff000,   -1,{0xffff,0x1d14}},  //  1382, depth=1
{0x00000000fcffffff,   -1,{0x019d,0xffff}},  //  1383, depth=1
{0x00000007ffff8000,   -1,{0xffff,0x1c53}},  //  1384
{0xc001ffffc001ffff, 1220,{0xffff,0x0092}},  //  1385, depth=2
{0x8888888888888888, 1224,{0xffff,0x0078}},  //  1386, depth=2
{0xfffc00007fffffff,   -1,{0xffff,0x13ac}},  //  1387, depth=1
{0x0000001e00000000,   -1,{0xffff,0x17c3}},  //  1388, depth=1
{0xfffc000000000007,   -1,{0xffff,0x1390}},  //  1389
{0x0001ffffffff0000,   -1,{0xffff,0x1c20}},  //  1390, depth=1
{0xffffff81ffffffff,   -1,{0xffff,0x1679}},  //  1391
{0x00001fffffffc000,   -1,{0xffff,0x1c9e}},  //  1392, depth=1
{0xfffc00000000000f,   -1,{0xffff,0x1391}},  //  1393
{0x00fffffffffff800,   -1,{0xffff,0x1d6c}},  //  1394
{0xffffffff0007ffff,   -1,{0xffff,0x1832}},  //  1395, depth=1
{0x0000003fc0000000,   -1,{0xffff,0x1887}},  //  1396, depth=1
{0xffff9fffffff9fff,   -1,{0xffff,0x045d}},  //  1397
{0x00003ffff0000000,   -1,{0xffff,0x1911}},  //  1398, depth=1
{0x0000003fff800000,   -1,{0xffff,0x1a4e}},  //  1399
{0xffffc0000003ffff,   -1,{0xffff,0x14a3}},  //  1400
{0xffc0000fffc0000f,   -1,{0xffff,0x028d}},  //  1401, depth=1
{0x1f0000001f000000,   -1,{0xffff,0x0204}},  //  1402, depth=1
{0xfffffff81fffffff,   -1,{0xffff,0x1779}},  //  1403
{0x0000fffffffff000,   -1,{0xffff,0x1d23}},  //  1404, depth=1
{0xfffff3fffffff3ff, 1227,{0xffff,0x051d}},  //  1405, depth=2
{0x000000007ff00000,   -1,{0x030a,0x1b0a}},  //  1406, depth=1
{0xc0000001ffffffff, 1231,{0xffff,0x10a2}},  //  1407, depth=2
{0x003fff8000000000,   -1,{0xffff,0x164e}},  //  1408, depth=1
{0x0000fffffc000000,   -1,{0xffff,0x1995}},  //  1409
{0x0000000000700070, 1232,{0x0322,0xffff}},  //  1410, depth=2
{0xffffff0000007fff,   -1,{0xffff,0x1626}},  //  1411, depth=1
{0x00e0000000000000,   -1,{0xffff,0x12c2}},  //  1412
{0xfffffffffc01ffff,   -1,{0xffff,0x19b6}},  //  1413
{0x0000000007ffff80, 1233,{0x0653,0x1e53}},  //  1414, depth=2
{0xfc1ffc1ffc1ffc1f, 1234,{0xffff,0x01aa}},  //  1415, depth=3
{0x1ffffffff0000000,   -1,{0xffff,0x1920}},  //  1416, depth=1
{0xff800003ffffffff,   -1,{0xffff,0x126a}},  //  1417
{0x000000000000e000, 1245,{0x04c2,0x1cc2}},  //  1418, depth=2
{0xff3fff3fff3fff3f,   -1,{0xffff,0x022d}},  //  1419
{0x00000ffffffffc00,   -1,{0xffff,0x1da1}},  //  1420
{0x00000000c0007fff, 1249,{0x0090,0xffff}},  //  1421, depth=2
{0x07fe07fe07fe07fe,   -1,{0xffff,0x03e9}},  //  1422, depth=1
{0xfffffff0001fffff,   -1,{0xffff,0x1730}},  //  1423, depth=1
{0xffff03ffffffffff,   -1,{0xffff,0x1439}},  //  1424
{0xffc00000000001ff, 1250,{0xffff,0x1292}},  //  1425, depth=2
{0x000000007e7e7e7e, 1254,{0x01f5,0xffff}},  //  1426, depth=2
{0xfffffffffffc0fff,   -1,{0xffff,0x1bb9}},  //  1427
{0x00ffffffffffe000,   -1,{0xffff,0x1cea}},  //  1428
{0xc000ffffffffffff, 1256,{0xffff,0x10b1}},  //  1429, depth=2
{0x001ffffffffe0000,   -1,{0xffff,0x1be3}},  //  1430, depth=1
{0xfffffff7ffffffff,   -1,{0xffff,0x173e}},  //  1431
{0x0600060006000600,   -1,{0xffff,0x01e1}},  //  1432, depth=1
{0xe00fffffffffffff,   -1,{0xffff,0x10f6}},  //  1433, depth=1
{0x0003ffffffff8000,   -1,{0xffff,0x1c62}},  //  1434
{0xffffffffffff007f,   -1,{0xffff,0x1c36}},  //  1435
{0xfffffff800000007,   -1,{0xffff,0x175f}},  //  1436
{0xfffff007fffff007,   -1,{0xffff,0x0516}},  //  1437, depth=1
{0x0007fffffe000000,   -1,{0xffff,0x19d9}},  //  1438
{0xfffff00000ffffff,   -1,{0xffff,0x152b}},  //  1439
{0x0ffffffffffe0000,   -1,{0xffff,0x1bea}},  //  1440
{0xfe3fffffffffffff,   -1,{0xffff,0x11fc}},  //  1441
{0x0000000000007f00,   -1,{0x0606,0x1e06}},  //  1442, depth=1
{0xffff8001ffffffff,   -1,{0xffff,0x1471}},  //  1443, depth=1
{0x0000000000300000,   -1,{0x0301,0x1b01}},  //  1444, depth=1
{0xfffffff80000000f,   -1,{0xffff,0x1760}},  //  1445
{0x0007ffc00007ffc0, 1258,{0xffff,0x068c}},  //  1446, depth=2
{0x0303030303030303, 1259,{0xffff,0x0031}},  //  1447, depth=3
{0x00003fffffe00000,   -1,{0xffff,0x1ad8}},  //  1448, depth=1
{0xffe00003ffffffff,   -1,{0xffff,0x12ec}},  //  1449, depth=1
{0x0000003fffc00000,   -1,{0xffff,0x1a8f}},  //  1450, depth=1
{0x000000007f807f80,   -1,{0x0267,0xffff}},  //  1451
{0x000001f8000001f8, 1263,{0xffff,0x0745}},  //  1452, depth=2
{0xe0000000000003ff,   -1,{0xffff,0x10cc}},  //  1453, depth=1
{0x001ffff000000000,   -1,{0xffff,0x1710}},  //  1454
{0x00000000cfcfcfcf,   -1,{0x00b5,0xffff}},  //  1455, depth=1
{0xff81ffffffffffff,   -1,{0xffff,0x1279}},  //  1456
{0x007fffffffffff80,   -1,{0xffff,0x1e6f}},  //  1457
{0xfffffff80000001f,   -1,{0xffff,0x1761}},  //  1458
{0xffffff81ffffff81, 1267,{0xffff,0x0659}},  //  1459, depth=2
{0x7ffffffffc000000,   -1,{0xffff,0x19a4}},  //  1460
{0xfffffffffff3ffff,   -1,{0xffff,0x1b3d}},  //  1461, depth=1
{0x0006000600060006,   -1,{0xffff,0x03e1}},  //  1462, depth=1
{0xfffffffff03fffff,   -1,{0xffff,0x1939}},  //  1463
{0x0fc000000fc00000, 1271,{0xffff,0x0285}},  //  1464, depth=2
{0xf80fffffffffffff,   -1,{0xffff,0x1178}},  //  1465
{0x0000000000001e00, 1273,{0x05c3,0x1dc3}},  //  1466, depth=2
{0x0001fffff0000000,   -1,{0xffff,0x1914}},  //  1467
{0x000f000000000000,   -1,{0xffff,0x1403}},  //  1468
{0x00000007f8000000,   -1,{0xffff,0x1947}},  //  1469
{0x00001ffffc000000,   -1,{0xffff,0x1992}},  //  1470, depth=1
{0xffffc000000003ff,   -1,{0xffff,0x149b}},  //  1471, depth=1
{0xe000003fffffffff,   -1,{0xffff,0x10e8}},  //  1472
{0x83ffffffffffffff, 1276,{0xffff,0x107a}},  //  1473, depth=2
{0xfffffe00000003ff,   -1,{0xffff,0x15e0}},  //  1474
{0x00000000ffc07fff, 1278,{0x0298,0xffff}},  //  1475, depth=2
{0x7ffffffffffff000,   -1,{0xffff,0x1d32}},  //  1476, depth=1
{0x00000000fff000ff, 1280,{0x0313,0xffff}},  //  1477, depth=2
{0x0000000007ffffc0,   -1,{0x0694,0x1e94}},  //  1478, depth=1
{0x00001fff00001fff, 1281,{0xffff,0x000c}},  //  1479, depth=2
{0x03fc000003fc0000,   -1,{0xffff,0x0387}},  //  1480, depth=1
{0x000ffffe00000000,   -1,{0xffff,0x17d2}},  //  1481
{0x07ff800000000000, 1283,{0xffff,0x144b}},  //  1482, depth=2
{0xfff80007ffffffff,   -1,{0xffff,0x136f}},  //  1483, depth=1
{0x0ff8000000000000,   -1,{0xffff,0x1348}},  //  1484
{0x00000000f800ffff, 1287,{0x0154,0xffff}},  //  1485, depth=2
{0x00000000ffffefff,   -1,{0x04de,0xffff}},  //  1486
{0xe1ffffffffffffff,   -1,{0xffff,0x10fb}},  //  1487, depth=1
{0x0000003f80000000,   -1,{0xffff,0x1846}},  //  1488, depth=1
{0x1fff800000000000,   -1,{0xffff,0x144d}},  //  1489
{0xffffc03fffffffff,   -1,{0xffff,0x14b7}},  //  1490
{0xffc000000001ffff,   -1,{0xffff,0x129a}},  //  1491, depth=1
{0x0000000000300030, 1308,{0x0321,0xffff}},  //  1492, depth=2
{0xc00000ffc00000ff, 1315,{0xffff,0x0089}},  //  1493, depth=2
{0x0000000ffe000000,   -1,{0xffff,0x19ca}},  //  1494, depth=1
{0x001fffffffffff00,   -1,{0xffff,0x1e2c}},  //  1495
{0x001ffffffffffe00,   -1,{0xffff,0x1deb}},  //  1496, depth=1
{0xffe0000001ffffff, 1503,{0xffff,0x12e3}},  //  1497
{0x000000003ff00000,   -1,{0x0309,0x1b09}},  //  1498, depth=1
{0xfffffffe000007ff,   -1,{0xffff,0x17e9}},  //  1499, depth=1
{0x0007ff0000000000,   -1,{0xffff,0x160a}},  //  1500, depth=1
{0xc0000001c0000001,   -1,{0xffff,0x0082}},  //  1501, depth=1
{0x000000007c7c7c7c,   -1,{0x01b4,0xffff}},  //  1502, depth=1
{0xfe00000000000fff,   -1,{0xffff,0x11d2}},  //  1503
{0xf9ffffffffffffff,   -1,{0xffff,0x117d}},  //  1504
{0x0000000003ff03ff, 1317,{0x0029,0xffff}},  //  1505, depth=2
{0x0018000000000000,   -1,{0xffff,0x1341}},  //  1506, depth=1
{0xfe00fffffe00ffff, 1318,{0xffff,0x01d6}},  //  1507, depth=2
{0x3ffffffffffffc00,   -1,{0xffff,0x1db3}},  //  1508
{0xfffff800000003ff,   -1,{0xffff,0x155e}},  //  1509, depth=1
{0xff000000ff000000, 1320,{0xffff,0x0207}},  //  1510, depth=2
{0xf000000001ffffff,   -1,{0xffff,0x111c}},  //  1511, depth=1
{0x00000000fffff00f,   -1,{0x0517,0xffff}},  //  1512
{0x0000000080007fff, 1321,{0x004f,0xffff}},  //  1513, depth=3
{0x0000000080008000,   -1,{0x0060,0xffff}},  //  1514, depth=1
{0xfffffffffffc3fff,   -1,{0xffff,0x1bbb}},  //  1515, depth=1
{0x0001fff800000000,   -1,{0xffff,0x174d}},  //  1516, depth=1
{0x01ff000000000000,   -1,{0xffff,0x1408}},  //  1517
{0x7fc000007fc00000, 1323,{0xffff,0x0288}},  //  1518, depth=3
{0xfffff001ffffffff,   -1,{0xffff,0x1534}},  //  1519, depth=1
{0x00000000fffff01f,   -1,{0x0518,0xffff}},  //  1520
{0xff800000000fffff,   -1,{0xffff,0x125c}},  //  1521, depth=1
{0x00000001ffffc000,   -1,{0xffff,0x1c92}},  //  1522, depth=1
{0x00000000fc01fc01,   -1,{0x01a6,0xffff}},  //  1523, depth=1
{0xfffffff00000ffff,   -1,{0xffff,0x172b}},  //  1524
{0xfff03ffffff03fff, 1328,{0xffff,0x0319}},  //  1525, depth=2
{0x0080000000800000, 1330,{0xffff,0x0240}},  //  1526, depth=2
{0xfffc1ffffffc1fff,   -1,{0xffff,0x039a}},  //  1527, depth=1
{0xffffffffffff00ff, 1529,{0xffff,0x1c37}},  //  1528
{0xfffff83fffffffff,   -1,{0xffff,0x157a}},  //  1529
{0x03fffff000000000,   -1,{0xffff,0x1715}},  //  1530
{0x01fffffe00000000,   -1,{0xffff,0x17d7}},  //  1531
{0xffffffffe01fffff, 1543,{0xffff,0x18f7}},  //  1532
{0xfffffffffffeffff,   -1,{0xffff,0x1bfe}},  //  1533, depth=1
{0x0000000000fff000, 1332,{0x050b,0x1d0b}},  //  1534, depth=3
{0xfe00000003ffffff, 1339,{0xffff,0x11e0}},  //  1535, depth=2
{0x0000000000000600,   -1,{0x05c1,0x1dc1}},  //  1536, depth=1
{0xffffffffffff0003,   -1,{0xffff,0x1c31}},  //  1537, depth=1
{0x0000000007fffffc, 1344,{0x0798,0x1f98}},  //  1538, depth=2
{0xfffffc0001ffffff,   -1,{0xffff,0x15ae}},  //  1539, depth=1
{0x00003fe000003fe0, 1351,{0xffff,0x06c8}},  //  1540, depth=2
{0x0000000007ffffff, 1353,{0x001a,0x101a}},  //  1541, depth=2
{0x0000000008000000,   -1,{0x0140,0x1940}},  //  1542, depth=1
{0xff07ffffffffffff,   -1,{0xffff,0x123a}},  //  1543
{0x000000001ff00000, 1359,{0x0308,0x1b08}},  //  1544, depth=3
{0x007ffffffffffff8,   -1,{0xffff,0x1f73}},  //  1545
{0x03ff800003ff8000,   -1,{0xffff,0x044a}},  //  1546, depth=1
{0xf0001fffffffffff,   -1,{0xffff,0x1130}},  //  1547
{0x007ffffffffffffe,   -1,{0xffff,0x1ff5}},  //  1548
{0xff000000000000ff, 1362,{0xffff,0x120f}},  //  1549, depth=2
{0x007fffffffffffff,   -1,{0xffff,0x1036}},  //  1550
{0x000000ffffff0000,   -1,{0xffff,0x1c17}},  //  1551
{0x1ffff00000000000, 1554,{0xffff,0x1510}},  //  1552
{0xfffffffff07fffff,   -1,{0xffff,0x193a}},  //  1553, depth=1
{0xffffff80ffffff80,   -1,{0xffff,0x0658}},  //  1554
{0x0000078000000000,   -1,{0xffff,0x1643}},  //  1555
{0x7fc0000000000000,   -1,{0xffff,0x1288}},  //  1556, depth=1
{0xffffffffc01fffff,   -1,{0xffff,0x18b6}},  //  1557
{0x07fff00000000000,   -1,{0xffff,0x150e}},  //  1558, depth=1
{0xc000000000ffffff, 1364,{0xffff,0x1099}},  //  1559, depth=2
{0xfffffffff0000000,   -1,{0xffff,0x1923}},  //  1560
{0xfe00007ffe00007f,   -1,{0xffff,0x01cd}},  //  1561, depth=1
{0x007fffffffffff00,   -1,{0xffff,0x1e2e}},  //  1562, depth=1
{0xfffffffffc0001ff,   -1,{0xffff,0x19ae}},  //  1563, depth=1
{0xfffffffff0000003,   -1,{0xffff,0x1925}},  //  1564
{0xffc0001fffffffff, 1366,{0xffff,0x12ae}},  //  1565, depth=2
{0x0000000003f80000,   -1,{0x0346,0x1b46}},  //  1566, depth=1
{0xfffffe3ffffffe3f, 1369,{0xffff,0x05dc}},  //  1567, depth=2
{0xfffffffff0000007,   -1,{0xffff,0x1926}},  //  1568
{0xfffffffffe001fff,   -1,{0xffff,0x19f3}},  //  1569
{0x0000e0000000e000, 1376,{0xffff,0x04c2}},  //  1570, depth=3
{0xfffbffffffffffff,   -1,{0xffff,0x137e}},  //  1571, depth=1
{0xfffc000000000000, 1384,{0xffff,0x138d}},  //  1572, depth=2
{0xfffc000000000001,   -1,{0xffff,0x138e}},  //  1573, depth=1
{0x01fc01fc01fc01fc,   -1,{0xffff,0x03a6}},  //  1574, depth=1
{0xfffc000000000003,   -1,{0xffff,0x138f}},  //  1575, depth=1
{0x00000000fffcfffc,   -1,{0x03ad,0xffff}},  //  1576
{0xff0000000001ffff,   -1,{0xffff,0x1218}},  //  1577
{0xfffffffff000001f, 1580,{0xffff,0x1928}},  //  1578
{0x800003ffffffffff, 1389,{0xffff,0x106a}},  //  1579, depth=2
{0x00000000fffff07f,   -1,{0x051a,0xffff}},  //  1580
{0xfffffffffe007fff,   -1,{0xffff,0x19f5}},  //  1581, depth=1
{0x3fffe0003fffe000,   -1,{0xffff,0x04d0}},  //  1582, depth=1
{0x000003c0000003c0,   -1,{0xffff,0x0683}},  //  1583
{0x0000007ffffffc00,   -1,{0xffff,0x1d9c}},  //  1584, depth=1
{0xe00000000003ffff, 1391,{0xffff,0x10d4}},  //  1585, depth=2
{0x001fffffffffff80,   -1,{0xffff,0x1e6d}},  //  1586
{0xe0001fffe0001fff, 1393,{0xffff,0x00cf}},  //  1587, depth=2
{0x000fffffff000000, 1394,{0xffff,0x1a1b}},  //  1588, depth=2
{0xfff0000003ffffff,   -1,{0xffff,0x1325}},  //  1589, depth=1
{0x01fc000001fc0000,   -1,{0xffff,0x0386}},  //  1590, depth=1
{0xfffff000ffffffff,   -1,{0xffff,0x1533}},  //  1591
{0xffffffffcfffffff, 1601,{0xffff,0x18bd}},  //  1592
{0x00000000f1ffffff, 1397,{0x011c,0xffff}},  //  1593, depth=2
{0x00007ffffffc0000,   -1,{0xffff,0x1b9c}},  //  1594, depth=1
{0xffff800000ffffff,   -1,{0xffff,0x1468}},  //  1595, depth=1
{0x000000007fc07fc0, 1399,{0x02a8,0xffff}},  //  1596, depth=2
{0xffffffffffff003f,   -1,{0xffff,0x1c35}},  //  1597, depth=1
{0x0fffffe000000000,   -1,{0xffff,0x16d6}},  //  1598, depth=1
{0x00000000f9fff9ff,   -1,{0x016d,0xffff}},  //  1599, depth=1
{0x00000001fc000000,   -1,{0xffff,0x1986}},  //  1600, depth=1
{0x00000000ffffc001,   -1,{0x0492,0xffff}},  //  1601
{0x1fffff8000000000,   -1,{0xffff,0x1655}},  //  1602
{0xfffc00000000001f, 1400,{0xffff,0x1392}},  //  1603, depth=2
{0x03fffffffffffe00,   -1,{0xffff,0x1df0}},  //  1604, depth=1
{0xffffe0000007ffff, 1606,{0xffff,0x14e5}},  //  1605
{0x00000000ffffc003, 1608,{0x0493,0xffff}},  //  1606
{0x8fffffffffffffff,   -1,{0xffff,0x107c}},  //  1607, depth=1
{0xfdfdfdfdfdfdfdfd,   -1,{0xffff,0x01b6}},  //  1608
{0xfff800000000ffff,   -1,{0xffff,0x135c}},  //  1609, depth=1
{0x0000018000000000,   -1,{0xffff,0x1641}},  //  1610, depth=1
{0xffffffffff07ffff,   -1,{0xffff,0x1a3a}},  //  1611, depth=1
{0xffffffffffe00000,   -1,{0xffff,0x1aea}},  //  1612
{0xfe00000000003fff, 1403,{0xffff,0x11d4}},  //  1613, depth=2
{0x03f8000000000000,   -1,{0xffff,0x1346}},  //  1614, depth=1
{0xffc00fffffc00fff,   -1,{0xffff,0x0295}},  //  1615, depth=1
{0x7fffffffffffc000,   -1,{0xffff,0x1cb0}},  //  1616, depth=1
{0xfc00fffffc00ffff,   -1,{0xffff,0x0195}},  //  1617, depth=1
{0xffffffffffe00001, 1620,{0xffff,0x1aeb}},  //  1618
{0xff00000fffffffff,   -1,{0xffff,0x122b}},  //  1619, depth=1
{0xfffc000fffffffff,   -1,{0xffff,0x13b1}},  //  1620
{0x00000000ff807fff,   -1,{0x0257,0xffff}},  //  1621, depth=1
{0x000000000e000e00, 1409,{0x01e2,0xffff}},  //  1622, depth=2
{0xffffffffffe00003,   -1,{0xffff,0x1aec}},  //  1623
{0x07ffff8000000000,   -1,{0xffff,0x1653}},  //  1624, depth=1
{0x0000000003ffe000,   -1,{0x04cc,0x1ccc}},  //  1625
{0xffffffffffe00007,   -1,{0xffff,0x1aed}},  //  1626
{0x000001fffff80000,   -1,{0xffff,0x1b55}},  //  1627
{0x0f800f800f800f80, 1412,{0xffff,0x0264}},  //  1628, depth=2
{0xffffe000000007ff, 1413,{0xffff,0x14dd}},  //  1629, depth=2
{0x3ffffffffffc0000,   -1,{0xffff,0x1bab}},  //  1630, depth=1
{0xffffffffffe0000f,   -1,{0xffff,0x1aee}},  //  1631
{0x07fffffffff80000,   -1,{0xffff,0x1b67}},  //  1632
{0xffe000ffffffffff,   -1,{0xffff,0x12f2}},  //  1633
{0x0000000000007fc0,   -1,{0x0688,0x1e88}},  //  1634, depth=1
{0xfffc00000000003f,   -1,{0xffff,0x1393}},  //  1635, depth=1
{0x0ffffff000000000,   -1,{0xffff,0x1717}},  //  1636
{0xff807fffff807fff,   -1,{0xffff,0x0257}},  //  1637, depth=1
{0xffffffffffe0001f,   -1,{0xffff,0x1aef}},  //  1638
{0xfe07fffffe07ffff, 1417,{0xffff,0x01d9}},  //  1639, depth=2
{0x1ff800001ff80000,   -1,{0xffff,0x0349}},  //  1640, depth=1
{0xfffff8000003ffff,   -1,{0xffff,0x1566}},  //  1641, depth=1
{0x01ffffffff000000,   -1,{0xffff,0x1a20}},  //  1642, depth=1
{0x5555555555555555, 1419,{0xffff,0x003c}},  //  1643, depth=2
{0x000003fffffe0000,   -1,{0xffff,0x1bd8}},  //  1644, depth=1
{0xfc00000001ffffff,   -1,{0xffff,0x119e}},  //  1645, depth=1
{0x003fc000003fc000,   -1,{0xffff,0x0487}},  //  1646, depth=1
{0xfff001ffffffffff,   -1,{0xffff,0x1334}},  //  1647
{0xff80000001ffffff,   -1,{0xffff,0x1261}},  //  1648
{0xc00fffffc00fffff,   -1,{0xffff,0x0095}},  //  1649, depth=1
{0x0000000001ff0000, 1420,{0x0408,0x1c08}},  //  1650, depth=2
{0xfffffff7fffffff7, 1424,{0xffff,0x071e}},  //  1651, depth=2
{0x000000f800000000, 1657,{0xffff,0x1744}},  //  1652
{0x00000000c0ffc0ff,   -1,{0x00a9,0xffff}},  //  1653, depth=1
{0x0000000078787878,   -1,{0x0173,0xffff}},  //  1654, depth=1
{0x00000000e0001fff, 1427,{0x00cf,0xffff}},  //  1655, depth=2
{0x01ff800001ff8000,   -1,{0xffff,0x0449}},  //  1656, depth=1
{0x0000000000001fe0,   -1,{0x06c7,0x1ec7}},  //  1657
{0x00000003f0000000, 1428,{0xffff,0x1905}},  //  1658, depth=2
{0xfc0ffc0ffc0ffc0f, 1431,{0xffff,0x01a9}},  //  1659, depth=2
{0xfffffff800000000, 1434,{0xffff,0x175c}},  //  1660, depth=2
{0xfffffff800000001, 1435,{0xffff,0x175d}},  //  1661, depth=2
{0xfc001fffffffffff,   -1,{0xffff,0x11b2}},  //  1662
{0xfffffff800000003,   -1,{0xffff,0x175e}},  //  1663, depth=1
{0xffffff80ffffffff,   -1,{0xffff,0x1678}},  //  1664
{0xffffffffffe0003f,   -1,{0xffff,0x1af0}},  //  1665
{0x0000000000007fe0,   -1,{0x06c9,0x1ec9}},  //  1666, depth=1
{0xfff0000000003fff, 1436,{0xffff,0x1319}},  //  1667, depth=2
{0x0000300000003000,   -1,{0xffff,0x0501}},  //  1668, depth=1
{0x800000000007ffff,   -1,{0xffff,0x1053}},  //  1669, depth=1
{0x8080808080808080, 1438,{0xffff,0x0070}},  //  1670, depth=2
{0xfc00007ffc00007f, 1439,{0xffff,0x018c}},  //  1671, depth=2
{0x0000003f00000000, 1440,{0xffff,0x1805}},  //  1672, depth=2
{0xe0ffe0ffe0ffe0ff, 1441,{0xffff,0x00ea}},  //  1673, depth=2
{0x0000000000ffc000,   -1,{0x0489,0x1c89}},  //  1674, depth=1
{0x00000000fc7fffff, 1445,{0x019c,0xffff}},  //  1675, depth=2
{0xfffff000007fffff,   -1,{0xffff,0x152a}},  //  1676
{0xf000ffffffffffff,   -1,{0xffff,0x1133}},  //  1677
{0xe000e000e000e000, 1451,{0xffff,0x00e2}},  //  1678, depth=2
{0x000003ffffffff00,   -1,{0xffff,0x1e21}},  //  1679
{0x003ffffffffff800,   -1,{0xffff,0x1d6a}},  //  1680
{0xf00000ffffffffff,   -1,{0xffff,0x112b}},  //  1681, depth=1
{0x0000000000007ff0, 1454,{0x070a,0x1f0a}},  //  1682, depth=2
{0xfffffbfffffffbff, 1456,{0xffff,0x055e}},  //  1683, depth=2
{0x00ffffc000ffffc0,   -1,{0xffff,0x0691}},  //  1684, depth=1
{0xfffe000007ffffff,   -1,{0xffff,0x13e9}},  //  1685, depth=1
{0x00000000fc000001,   -1,{0x0186,0xffff}},  //  1686
{0xfffffc01ffffffff, 1689,{0xffff,0x15b6}},  //  1687
{0x0000000007c00000,   -1,{0x0284,0x1a84}},  //  1688, depth=1
{0xe000000007ffffff, 1692,{0xffff,0x10dd}},  //  1689
{0x0000000000007ff8, 1457,{0x074b,0x1f4b}},  //  1690, depth=2
{0xffffff0000001fff, 1458,{0xffff,0x1624}},  //  1691, depth=2
{0xffffc001ffffc001, 1693,{0xffff,0x0492}},  //  1692
{0x00000000ff8007ff,   -1,{0x0253,0xffff}},  //  1693
{0x0000000000007ffc, 1460,{0x078c,0x1f8c}},  //  1694, depth=2
{0x0007fffc00000000,   -1,{0xffff,0x1790}},  //  1695
{0x0000000000007ffe,   -1,{0x07cd,0x1fcd}},  //  1696, depth=1
{0x0000000000007fff,   -1,{0x000e,0x100e}},  //  1697, depth=1
{0x0000000000008000,   -1,{0x0440,0x1c40}},  //  1698, depth=1
{0xfffc00000000007f, 1463,{0xffff,0x1394}},  //  1699, depth=2
{0x001fffffffffffe0,   -1,{0xffff,0x1eef}},  //  1700
{0x00000000c0001fff, 1465,{0x008e,0xffff}},  //  1701, depth=2
{0x0007e0000007e000, 1467,{0xffff,0x04c5}},  //  1702, depth=2
{0xff83ffffff83ffff,   -1,{0xffff,0x025a}},  //  1703, depth=1
{0x003ffffffff80000,   -1,{0xffff,0x1b62}},  //  1704, depth=1
{0xffffffff8fffffff, 1706,{0xffff,0x187c}},  //  1705
{0xfff0fffffff0ffff,   -1,{0xffff,0x031b}},  //  1706
{0x0000000087ff87ff,   -1,{0x006b,0xffff}},  //  1707, depth=1
{0x00000003ffe00000,   -1,{0xffff,0x1acc}},  //  1708, depth=1
{0xfffffcffffffffff,   -1,{0xffff,0x15bd}},  //  1709, depth=1
{0x000003fffffffe00,   -1,{0xffff,0x1de0}},  //  1710, depth=1
{0xfcfffffffcffffff,   -1,{0xffff,0x019d}},  //  1711
{0x00007f8000007f80,   -1,{0xffff,0x0647}},  //  1712, depth=1
{0xffffc00007ffffff,   -1,{0xffff,0x14ac}},  //  1713
{0x0fffffffffffff00,   -1,{0xffff,0x1e33}},  //  1714
{0xfffff807fffff807,   -1,{0xffff,0x0557}},  //  1715, depth=1
{0xffffffffffffc3ff,   -1,{0xffff,0x1cbb}},  //  1716
{0xf800003fffffffff,   -1,{0xffff,0x116a}},  //  1717
{0x1fffff801fffff80, 1468,{0xffff,0x0655}},  //  1718, depth=2
{0x007fffff80000000,   -1,{0xffff,0x1857}},  //  1719
{0x00000000fffff0ff,   -1,{0x051b,0xffff}},  //  1720
{0xf8000000000003ff,   -1,{0xffff,0x114e}},  //  1721, depth=1
{0x0000000000001f00,   -1,{0x0604,0x1e04}},  //  1722, depth=1
{0xfffffff80000003f,   -1,{0xffff,0x1762}},  //  1723, depth=1
{0x001ffffffffffffe,   -1,{0xffff,0x1ff3}},  //  1724
{0x00000000e00007ff,   -1,{0x00cd,0xffff}},  //  1725, depth=1
{0x00007ffffffffc00,   -1,{0xffff,0x1da4}},  //  1726, depth=1
{0x07fffc0000000000, 1730,{0xffff,0x1590}},  //  1727
{0x3e00000000000000, 1469,{0xffff,0x11c4}},  //  1728, depth=2
{0xfffffffc07ffffff,   -1,{0xffff,0x17b8}},  //  1729, depth=1
{0x3fffffe03fffffe0,   -1,{0xffff,0x06d8}},  //  1730
{0x00000000f01ff01f,   -1,{0x0128,0xffff}},  //  1731, depth=1
{0xfffffe3fffffffff,   -1,{0xffff,0x15fc}},  //  1732
{0x00000000fff001ff, 1472,{0x0314,0xffff}},  //  1733, depth=2
{0x0000000000060000,   -1,{0x03c1,0x1bc1}},  //  1734, depth=1
{0x0000003f0000003f, 1474,{0xffff,0x0005}},  //  1735, depth=2
{0xffffff000007ffff,   -1,{0xffff,0x162a}},  //  1736
{0xffffff8000ffffff,   -1,{0xffff,0x1670}},  //  1737, depth=1
{0x0ffffffffffffe00,   -1,{0xffff,0x1df2}},  //  1738, depth=1
{0xffffff80007fffff,   -1,{0xffff,0x166f}},  //  1739
{0x0000000000060006, 1481,{0x03e1,0xffff}},  //  1740, depth=2
{0xffff00000007ffff,   -1,{0xffff,0x1422}},  //  1741, depth=1
{0x80000fffffffffff,   -1,{0xffff,0x106c}},  //  1742
{0xffc0001fffc0001f,   -1,{0xffff,0x028e}},  //  1743, depth=1
{0xffff8000003fffff,   -1,{0xffff,0x1466}},  //  1744
{0xffffffc0000fffff, 1747,{0xffff,0x16ad}},  //  1745
{0x000007fc000007fc,   -1,{0xffff,0x0788}},  //  1746, depth=1
{0xf9f9f9f9f9f9f9f9,   -1,{0xffff,0x0175}},  //  1747
{0xfff80000fff80000, 1484,{0xffff,0x034c}},  //  1748, depth=2
{0xfc07fffffc07ffff, 1486,{0xffff,0x0198}},  //  1749, depth=2
{0x00000000fffff000, 1489,{0x0513,0x1d13}},  //  1750, depth=2
{0x00000000fffff001, 1490,{0x0514,0xffff}},  //  1751, depth=2
{0x0001ffffffe00000, 1495,{0xffff,0x1adb}},  //  1752, depth=2
{0x00000000fffff003, 1497,{0x0515,0xffff}},  //  1753, depth=3
{0x007fffffffffffc0,   -1,{0xffff,0x1eb0}},  //  1754, depth=1
{0x00000000ffc01fff, 1504,{0x0296,0xffff}},  //  1755, depth=2
{0xffff007fffffffff,   -1,{0xffff,0x1436}},  //  1756
{0x00000000fffff007,   -1,{0x0516,0xffff}},  //  1757, depth=1
{0x0000003e00000000, 1760,{0xffff,0x17c4}},  //  1758
{0x00000000c7c7c7c7,   -1,{0x00b4,0xffff}},  //  1759, depth=1
{0x00000000000007f8,   -1,{0x0747,0x1f47}},  //  1760
{0xffffff00000007ff,   -1,{0xffff,0x1622}},  //  1761, depth=1
{0x0000060000000600, 1508,{0xffff,0x05c1}},  //  1762, depth=2
{0xfffffffe0007ffff,   -1,{0xffff,0x17f1}},  //  1763, depth=1
{0x001ffff8001ffff8,   -1,{0xffff,0x0751}},  //  1764, depth=1
{0x80003fff80003fff, 1512,{0xffff,0x004e}},  //  1765, depth=2
{0xffffffffffff01ff,   -1,{0xffff,0x1c38}},  //  1766
{0x03ffffffffffffc0,   -1,{0xffff,0x1eb3}},  //  1767
{0x001fffffe0000000,   -1,{0xffff,0x18d7}},  //  1768
{0x01ffff0000000000, 1770,{0xffff,0x1610}},  //  1769
{0x0ffffff80ffffff8,   -1,{0xffff,0x0758}},  //  1770
{0x00000000c00007ff,   -1,{0x008c,0xffff}},  //  1771, depth=1
{0x1fff00001fff0000, 1517,{0xffff,0x040c}},  //  1772, depth=2
{0xfffff0007fffffff,   -1,{0xffff,0x1532}},  //  1773
{0xfff8000003ffffff,   -1,{0xffff,0x1366}},  //  1774
{0xc0000000ffffffff,   -1,{0xffff,0x10a1}},  //  1775, depth=1
{0xfffffffffffe3fff,   -1,{0xffff,0x1bfc}},  //  1776
{0x000003ffffffff80,   -1,{0xffff,0x1e62}},  //  1777
{0x0003ff800003ff80,   -1,{0xffff,0x064a}},  //  1778, depth=1
{0xf000000000ffffff,   -1,{0xffff,0x111b}},  //  1779
{0x0000007ffffffe00, 1784,{0xffff,0x1ddd}},  //  1780
{0xfc07fc07fc07fc07, 1520,{0xffff,0x01a8}},  //  1781, depth=2
{0x3ff0000000000000,   -1,{0xffff,0x1309}},  //  1782, depth=1
{0xfffc003ffffc003f,   -1,{0xffff,0x0393}},  //  1783, depth=1
{0x00000ffffffe0000,   -1,{0xffff,0x1bda}},  //  1784
{0x00000000fff1fff1, 1524,{0x032c,0xffff}},  //  1785, depth=2
{0x007fffffffffffe0,   -1,{0xffff,0x1ef1}},  //  1786, depth=1
{0xfffffff80000007f,   -1,{0xffff,0x1763}},  //  1787, depth=1
{0x03fffffffffffff0,   -1,{0xffff,0x1f35}},  //  1788
{0xfffe3ffffffe3fff, 1528,{0xffff,0x03dc}},  //  1789, depth=3
{0x0780078007800780, 1530,{0xffff,0x0263}},  //  1790, depth=2
{0xfffffffff00000ff,   -1,{0xffff,0x192b}},  //  1791
{0x0000000000000700,   -1,{0x0602,0x1e02}},  //  1792, depth=1
{0x0000000080001fff,   -1,{0x004d,0xffff}},  //  1793, depth=1
{0x003ffc0000000000, 1531,{0xffff,0x158b}},  //  1794, depth=2
{0xc7ffffffc7ffffff, 1532,{0xffff,0x009c}},  //  1795, depth=3
{0x001ffc00001ffc00,   -1,{0xffff,0x058a}},  //  1796, depth=1
{0xfffffffff80fffff,   -1,{0xffff,0x1978}},  //  1797, depth=1
{0x0000000c00000000,   -1,{0xffff,0x1781}},  //  1798, depth=1
{0x00000000fff1ffff,   -1,{0x031c,0xffff}},  //  1799, depth=1
{0x000000000007fe00,   -1,{0x05c9,0x1dc9}},  //  1800, depth=1
{0xffff0fffffffffff, 1804,{0xffff,0x143b}},  //  1801
{0x007ffffffffffff0,   -1,{0xffff,0x1f32}},  //  1802, depth=1
{0xffffe3ffffffffff,   -1,{0xffff,0x14fc}},  //  1803, depth=1
{0xfe000007fe000007,   -1,{0xffff,0x01c9}},  //  1804
{0xff000000000001ff,   -1,{0xffff,0x1210}},  //  1805, depth=1
{0x007fffffff800000,   -1,{0xffff,0x1a5f}},  //  1806
{0xfff0000000000fff,   -1,{0xffff,0x1317}},  //  1807, depth=1
{0x0000300000000000,   -1,{0xffff,0x1501}},  //  1808, depth=1
{0x03fffffffffffffc, 1822,{0xffff,0x1fb7}},  //  1809
{0x0000000c0000000c, 1545,{0xffff,0x0781}},  //  1810, depth=2
{0xffff8000ffffffff,   -1,{0xffff,0x1470}},  //  1811, depth=1
{0x7c7c7c7c7c7c7c7c,   -1,{0xffff,0x01b4}},  //  1812, depth=1
{0x00000000fffff03f, 1547,{0x0519,0xffff}},  //  1813, depth=2
{0x007ffffffffffffc,   -1,{0xffff,0x1fb4}},  //  1814, depth=1
{0xfc0000ffffffffff,   -1,{0xffff,0x11ad}},  //  1815, depth=1
{0x0003fffff8000000, 1548,{0xffff,0x1956}},  //  1816, depth=2
{0x00000000c07fc07f, 1550,{0x00a8,0xffff}},  //  1817, depth=2
{0x0080000000000000,   -1,{0xffff,0x1240}},  //  1818, depth=1
{0xf0003ffff0003fff,   -1,{0xffff,0x0111}},  //  1819, depth=1
{0x0000001ffe000000, 1551,{0xffff,0x19cb}},  //  1820, depth=2
{0x00000000fc3fffff,   -1,{0x019b,0xffff}},  //  1821, depth=1
{0x0001f00000000000, 1824,{0xffff,0x1504}},  //  1822
{0x00000000fe0003ff,   -1,{0x01d0,0xffff}},  //  1823, depth=1
{0x00000000003fc000,   -1,{0x0487,0x1c87}},  //  1824
{0x00000000ffc007ff,   -1,{0x0294,0xffff}},  //  1825, depth=1
{0x00000000007ff000, 1552,{0x050a,0x1d0a}},  //  1826, depth=3
{0xfffc0000000000ff,   -1,{0xffff,0x1395}},  //  1827, depth=1
{0x03fffffffffffffe,   -1,{0xffff,0x1ff8}},  //  1828
{0x03ffffffffffffff, 1831,{0xffff,0x1039}},  //  1829
{0x0000000003fff800,   -1,{0x054e,0x1d4e}},  //  1830, depth=1
{0xfff7fffffff7ffff,   -1,{0xffff,0x031e}},  //  1831
{0x0003000000030000,   -1,{0xffff,0x0401}},  //  1832, depth=1
{0xfffffc0000ffffff,   -1,{0xffff,0x15ad}},  //  1833
{0x0000000007800000,   -1,{0x0243,0x1a43}},  //  1834, depth=1
{0x0000001c00000000,   -1,{0xffff,0x1782}},  //  1835
{0x0000000003e003e0, 1555,{0x02e4,0xffff}},  //  1836, depth=2
{0xf800fffff800ffff,   -1,{0xffff,0x0154}},  //  1837, depth=1
{0x0000f0000000f000,   -1,{0xffff,0x0503}},  //  1838
{0xe000000000007fff,   -1,{0xffff,0x10d1}},  //  1839
{0x000007fffff80000,   -1,{0xffff,0x1b57}},  //  1840
{0xfffe000000007fff, 1557,{0xffff,0x13dd}},  //  1841, depth=2
{0x000000007f007f00,   -1,{0x0226,0xffff}},  //  1842, depth=1
{0xe000003fe000003f,   -1,{0xffff,0x00c8}},  //  1843, depth=1
{0xffffc00000007fff, 1848,{0xffff,0x14a0}},  //  1844
{0xffffffffefffffff,   -1,{0xffff,0x18fe}},  //  1845, depth=1
{0x00001ffe00001ffe, 1560,{0xffff,0x07cb}},  //  1846, depth=2
{0xfffffffff0000001,   -1,{0xffff,0x1924}},  //  1847, depth=1
{0xc000000000000001,   -1,{0xffff,0x1082}},  //  1848
{0xffff803fffff803f, 1564,{0xffff,0x0456}},  //  1849, depth=2
{0x0000000000001f80,   -1,{0x0645,0x1e45}},  //  1850, depth=1
{0xc0000000007fffff,   -1,{0xffff,0x1098}},  //  1851, depth=1
{0xffffffffffe000ff,   -1,{0xffff,0x1af2}},  //  1852
{0xf80000000003ffff, 1568,{0xffff,0x1156}},  //  1853, depth=2
{0x00001f0000001f00,   -1,{0xffff,0x0604}},  //  1854, depth=1
{0x0000001c0000001c,   -1,{0xffff,0x0782}},  //  1855
{0xfff007ffffffffff,   -1,{0xffff,0x1336}},  //  1856
{0xfffffffffffff3ff,   -1,{0xffff,0x1d3d}},  //  1857, depth=1
{0xffffffe000000fff,   -1,{0xffff,0x16e6}},  //  1858
{0x00000ffffffffe00,   -1,{0xffff,0x1de2}},  //  1859
{0x03ffffffffffff00,   -1,{0xffff,0x1e31}},  //  1860, depth=1
{0xfffffffff000000f, 1569,{0xffff,0x1927}},  //  1861, depth=2
{0x000001fffffff800, 1865,{0xffff,0x1d5d}},  //  1862
{0x00000000800007ff,   -1,{0x004b,0xffff}},  //  1863, depth=1
{0x007fffffe0000000,   -1,{0xffff,0x18d9}},  //  1864, depth=1
{0x00003ffffff80000,   -1,{0xffff,0x1b5a}},  //  1865
{0xfffffff8000001ff,   -1,{0xffff,0x1765}},  //  1866
{0xfffffe000003ffff,   -1,{0xffff,0x15e8}},  //  1867, depth=1
{0xc0000000c0000000, 1576,{0xffff,0x0081}},  //  1868, depth=2
{0x00007ffffffe0000,   -1,{0xffff,0x1bdd}},  //  1869
{0x000fe00000000000,   -1,{0xffff,0x14c6}},  //  1870, depth=1
{0x00000000fffcffff, 1577,{0x039d,0xffff}},  //  1871, depth=2
{0x3ffc3ffc3ffc3ffc,   -1,{0xffff,0x03ab}},  //  1872, depth=1
{0xffffff80003fffff,   -1,{0xffff,0x166e}},  //  1873
{0x00003f8000000000,   -1,{0xffff,0x1646}},  //  1874, depth=1
{0xfe0001ffffffffff,   -1,{0xffff,0x11ef}},  //  1875, depth=1
{0xffff800000000000,   -1,{0xffff,0x1450}},  //  1876
{0xfefffffffeffffff, 1578,{0xffff,0x01de}},  //  1877, depth=3
{0x00000000001c0000, 1583,{0x0382,0x1b82}},  //  1878, depth=2
{0xffff800000000001,   -1,{0xffff,0x1451}},  //  1879
{0xffff0000ffff0000, 1586,{0xffff,0x040f}},  //  1880, depth=2
{0xfffff80000007fff,   -1,{0xffff,0x1563}},  //  1881
{0x00003ff000003ff0,   -1,{0xffff,0x0709}},  //  1882, depth=1
{0xfffbfffbfffbfffb,   -1,{0xffff,0x036e}},  //  1883, depth=1
{0x07fffffffffff800,   -1,{0xffff,0x1d6f}},  //  1884
{0xfffffffc00007fff,   -1,{0xffff,0x17ac}},  //  1885, depth=1
{0xffe00001ffffffff,   -1,{0xffff,0x12eb}},  //  1886
{0xffff8000007fffff, 1591,{0xffff,0x1467}},  //  1887, depth=2
{0x0000003fff000000,   -1,{0xffff,0x1a0d}},  //  1888, depth=1
{0x00000000ffffbfff,   -1,{0x045e,0xffff}},  //  1889, depth=1
{0x00000000ffffc000,   -1,{0x0491,0x1c91}},  //  1890, depth=1
{0xf800007ff800007f, 1592,{0xffff,0x014b}},  //  1891, depth=3
{0x3c003c003c003c00, 1602,{0xffff,0x01a3}},  //  1892, depth=2
{0x007fffff007fffff, 1605,{0xffff,0x0016}},  //  1893, depth=4
{0x0fffffffffffffc0,   -1,{0xffff,0x1eb5}},  //  1894
{0xffffffffffdfffff,   -1,{0xffff,0x1abe}},  //  1895, depth=1
{0x000003f000000000, 1612,{0xffff,0x1705}},  //  1896, depth=2
{0x00000000ffffc007, 1618,{0x0494,0xffff}},  //  1897, depth=3
{0x001fffff80000000,   -1,{0xffff,0x1855}},  //  1898
{0x00000000c03fc03f, 1623,{0x00a7,0xffff}},  //  1899, depth=2
{0x2222222222222222, 1625,{0xffff,0x00f8}},  //  1900, depth=2
{0x00000000ff801fff,   -1,{0x0255,0xffff}},  //  1901, depth=1
{0xfff03fffffffffff, 1907,{0xffff,0x1339}},  //  1902
{0xfbfffbfffbfffbff, 1626,{0xffff,0x016e}},  //  1903, depth=2
{0x07c0000007c00000,   -1,{0xffff,0x0284}},  //  1904, depth=1
{0x00000000ffffc00f,   -1,{0x0495,0xffff}},  //  1905, depth=1
{0x00000000001c001c, 1627,{0x03a2,0xffff}},  //  1906, depth=2
{0xffffff7fffffff7f,   -1,{0xffff,0x061e}},  //  1907
{0x0000000000003800,   -1,{0x0542,0x1d42}},  //  1908, depth=1
{0xfffffffff000003f,   -1,{0xffff,0x1929}},  //  1909, depth=1
{0x007fffffc0000000,   -1,{0xffff,0x1898}},  //  1910, depth=1
{0x00000000c3c3c3c3, 1631,{0x00b3,0xffff}},  //  1911, depth=2
{0xfffffffffe07ffff,   -1,{0xffff,0x19f9}},  //  1912
{0x00000000ff007fff,   -1,{0x0216,0xffff}},  //  1913, depth=1
{0x0000000000001fc0,   -1,{0x0686,0x1e86}},  //  1914, depth=1
{0xfffffff8000000ff,   -1,{0xffff,0x1764}},  //  1915, depth=1
{0x0000000ffff00000,   -1,{0xffff,0x1b0f}},  //  1916, depth=1
{0xffcfffffffffffff,   -1,{0xffff,0x12bd}},  //  1917, depth=1
{0xfffffffffc03ffff, 1922,{0xffff,0x19b7}},  //  1918
{0xff8001ffff8001ff,   -1,{0xffff,0x0251}},  //  1919, depth=1
{0x0000000000000780, 1632,{0x0643,0x1e43}},  //  1920, depth=2
{0x00000000ffffc01f,   -1,{0x0496,0xffff}},  //  1921, depth=1
{0xffe0ffffffffffff, 1925,{0xffff,0x12fa}},  //  1922
{0xf803f803f803f803, 1633,{0xffff,0x0166}},  //  1923, depth=2
{0x01ffe00000000000, 1636,{0xffff,0x14cb}},  //  1924, depth=2
{0xc00000000000003f,   -1,{0xffff,0x1087}},  //  1925
{0x3ffffffffffe0000,   -1,{0xffff,0x1bec}},  //  1926
{0x00000000f8fff8ff, 1638,{0x016c,0xffff}},  //  1927, depth=2
{0x000000000ffc0000,   -1,{0x0389,0x1b89}},  //  1928, depth=1
{0x000000001f1f1f1f, 1647,{0x0034,0xffff}},  //  1929, depth=2
{0x0006000000000000,   -1,{0xffff,0x13c1}},  //  1930, depth=1
{0xfffffffffe0007ff,   -1,{0xffff,0x19f1}},  //  1931, depth=1
{0x0000000020002000,   -1,{0x00e0,0xffff}},  //  1932, depth=1
{0xffffffff801fffff,   -1,{0xffff,0x1875}},  //  1933, depth=1
{0x000000007ffff000,   -1,{0x0512,0x1d12}},  //  1934, depth=1
{0xff800003ff800003,   -1,{0xffff,0x024a}},  //  1935, depth=1
{0x7f00000000000000,   -1,{0xffff,0x1206}},  //  1936, depth=1
{0xffc00000000003ff,   -1,{0xffff,0x1293}},  //  1937, depth=1
{0x00ffffffffff8000,   -1,{0xffff,0x1c68}},  //  1938, depth=1
{0x000003fffffffff8,   -1,{0xffff,0x1f66}},  //  1939
{0x00ffe00000ffe000,   -1,{0xffff,0x04ca}},  //  1940, depth=1
{0xffffffe003ffffff,   -1,{0xffff,0x16f4}},  //  1941, depth=1
{0x000003fffffffffc,   -1,{0xffff,0x1fa7}},  //  1942
{0xf0000001ffffffff, 1648,{0xffff,0x1124}},  //  1943, depth=2
{0x001fffffffffffc0,   -1,{0xffff,0x1eae}},  //  1944, depth=1
{0xff00001fffffffff,   -1,{0xffff,0x122c}},  //  1945, depth=1
{0x0000380000003800, 1652,{0xffff,0x0542}},  //  1946, depth=3
{0x007f007f007f007f, 1662,{0xffff,0x0026}},  //  1947, depth=2
{0xfffffffff9ffffff,   -1,{0xffff,0x197d}},  //  1948
{0xfffc001fffffffff, 1950,{0xffff,0x13b2}},  //  1949
{0x00000000fc0000ff,   -1,{0x018d,0xffff}},  //  1950
{0x07e0000000000000,   -1,{0xffff,0x12c5}},  //  1951
{0x00001fffffff0000,   -1,{0xffff,0x1c1c}},  //  1952, depth=1
{0x00000000ffffc03f, 1664,{0x0497,0xffff}},  //  1953, depth=2
{0x7878787878787878,   -1,{0xffff,0x0173}},  //  1954, depth=1
{0x00000000fe03ffff,   -1,{0x01d8,0xffff}},  //  1955, depth=1
{0x0000006000000000,   -1,{0xffff,0x16c1}},  //  1956, depth=1
{0xffffff803fffffff,   -1,{0xffff,0x1676}},  //  1957
{0x0000000070707070,   -1,{0x0132,0xffff}},  //  1958, depth=1
{0xc00000007fffffff, 1665,{0xffff,0x10a0}},  //  1959, depth=2
{0x0000fff800000000,   -1,{0xffff,0x174c}},  //  1960
{0x0ffffffffffffff0, 1979,{0xffff,0x1f37}},  //  1961
{0x0000000000001ff0,   -1,{0x0708,0x1f08}},  //  1962, depth=1
{0xffffc0ffffffc0ff, 1676,{0xffff,0x0499}},  //  1963, depth=2
{0x007fffffffc00000,   -1,{0xffff,0x1aa0}},  //  1964, depth=1
{0x8000007fffffffff, 1677,{0xffff,0x1067}},  //  1965, depth=2
{0x00000000007fc000, 1679,{0x0488,0x1c88}},  //  1966, depth=2
{0x00000000fbffffff,   -1,{0x015e,0xffff}},  //  1967, depth=1
{0x00000000fc000000, 1680,{0x0185,0x1985}},  //  1968, depth=2
{0xf807fffff807ffff, 1686,{0xffff,0x0157}},  //  1969, depth=2
{0x0000000000001ff8,   -1,{0x0749,0x1f49}},  //  1970, depth=1
{0x00000000fc000003, 1687,{0x0187,0xffff}},  //  1971, depth=5
{0x00000000003ff000,   -1,{0x0509,0x1d09}},  //  1972, depth=1
{0xfffffffff000007f,   -1,{0xffff,0x192a}},  //  1973, depth=1
{0x0000000000001ffc, 1695,{0x078a,0x1f8a}},  //  1974, depth=2
{0x00000000fc000007,   -1,{0x0188,0xffff}},  //  1975, depth=1
{0x0000000000001ffe, 1700,{0x07cb,0x1fcb}},  //  1976, depth=2
{0x0000000000001fff,   -1,{0x000c,0x100c}},  //  1977, depth=1
{0x0000000000002000,   -1,{0x04c0,0x1cc0}},  //  1978, depth=1
{0x0007c00000000000, 1980,{0xffff,0x1484}},  //  1979
{0x0000000000ff0000,   -1,{0x0407,0x1c07}},  //  1980
{0x00000000bfffbfff,   -1,{0x006e,0xffff}},  //  1981, depth=1
{0xfffffffc000007ff, 1985,{0xffff,0x17a8}},  //  1982
{0x00000000fc00000f, 1705,{0x0189,0xffff}},  //  1983, depth=3
{0x00000000000007c0,   -1,{0x0684,0x1e84}},  //  1984, depth=1
{0x00000000ff87ffff,   -1,{0x025b,0xffff}},  //  1985
{0x00007ffffffffe00,   -1,{0xffff,0x1de5}},  //  1986
{0x00000000c1c1c1c1, 1711,{0x00b2,0xffff}},  //  1987, depth=2
{0x03ffffffffffff80,   -1,{0xffff,0x1e72}},  //  1988, depth=1
{0x00000000e007ffff, 1713,{0x00d5,0xffff}},  //  1989, depth=2
{0x0ffffffffffffff8,   -1,{0xffff,0x1f78}},  //  1990
{0xff801fffffffffff,   -1,{0xffff,0x1275}},  //  1991
{0x001ffffffffffff0,   -1,{0xffff,0x1f30}},  //  1992, depth=1
{0x0ffffffffffffffc,   -1,{0xffff,0x1fb9}},  //  1993
{0x0000018000000180, 1714,{0xffff,0x0641}},  //  1994, depth=2
{0xffff80007fffffff,   -1,{0xffff,0x146f}},  //  1995, depth=1
{0xe07fffffffffffff,   -1,{0xffff,0x10f9}},  //  1996
{0xc0000000003fffff, 1716,{0xffff,0x1097}},  //  1997, depth=2
{0x0000ffe00000ffe0,   -1,{0xffff,0x06ca}},  //  1998, depth=1
{0x00000000fc00001f,   -1,{0x018a,0xffff}},  //  1999, depth=1
{0x001ffffffffffff8,   -1,{0xffff,0x1f71}},  //  2000, depth=1
{0xe07fffffe07fffff, 1717,{0xffff,0x00d9}},  //  2001, depth=2
{0x000fff0000000000, 1719,{0xffff,0x160b}},  //  2002, depth=2
{0xfffffffffff07fff,   -1,{0xffff,0x1b3a}},  //  2003, depth=1
{0x001ffffffffffffc,   -1,{0xffff,0x1fb2}},  //  2004, depth=1
{0xfff3fff3fff3fff3, 1720,{0xffff,0x032d}},  //  2005, depth=2
{0xffc0ffc0ffc0ffc0, 1724,{0xffff,0x02a9}},  //  2006, depth=2
{0x001fffffffffffff,   -1,{0xffff,0x1034}},  //  2007, depth=1
{0x0020000000000000,   -1,{0xffff,0x12c0}},  //  2008, depth=1
{0x0ffffffffffffffe,   -1,{0xffff,0x1ffa}},  //  2009
{0x00000000001ffc00, 1727,{0x058a,0x1d8a}},  //  2010, depth=3
{0x0fffffffffffffff,   -1,{0xffff,0x103b}},  //  2011
{0x0000000001e00000,   -1,{0x02c3,0x1ac3}},  //  2012, depth=1
{0xffffefffffffffff,   -1,{0xffff,0x14fe}},  //  2013
{0x000000001fc01fc0,   -1,{0x02a6,0xffff}},  //  2014, depth=1
{0xc3ffffffc3ffffff, 1732,{0xffff,0x009b}},  //  2015, depth=2
{0x00000000000007e0,   -1,{0x06c5,0x1ec5}},  //  2016, depth=1
{0x00000000ffffc07f,   -1,{0x0498,0xffff}},  //  2017, depth=1
{0xffffffffffc0ffff,   -1,{0xffff,0x1ab9}},  //  2018
{0xffffffe000003fff,   -1,{0xffff,0x16e8}},  //  2019, depth=1
{0x00003c0000000000, 2021,{0xffff,0x1583}},  //  2020
{0x007ffffe007ffffe,   -1,{0xffff,0x07d5}},  //  2021
{0x00000fe000000000,   -1,{0xffff,0x16c6}},  //  2022, depth=1
{0xffffffffffe0007f,   -1,{0xffff,0x1af1}},  //  2023, depth=1
{0x00000ffc00000ffc,   -1,{0xffff,0x0789}},  //  2024, depth=1
{0xfffffc0ffffffc0f, 1736,{0xffff,0x0599}},  //  2025, depth=2
{0x000000003ffff000,   -1,{0x0511,0x1d11}},  //  2026, depth=1
{0xfffff80007ffffff,   -1,{0xffff,0x156f}},  //  2027, depth=1
{0xfffff00000000007,   -1,{0xffff,0x1516}},  //  2028
{0x00000000ffff8fff, 1739,{0x045c,0xffff}},  //  2029, depth=2
{0x0000007ffffe0000,   -1,{0xffff,0x1bd5}},  //  2030, depth=1
{0x00000000fc00003f, 1742,{0x018b,0xffff}},  //  2031, depth=2
{0x00000000000007f0,   -1,{0x0706,0x1f06}},  //  2032, depth=1
{0x00000000ffc7ffc7, 1744,{0x02ac,0xffff}},  //  2033, depth=2
{0x000003ffe0000000,   -1,{0xffff,0x18cc}},  //  2034
{0x00000000c007ffff, 1745,{0x0094,0xffff}},  //  2035, depth=3
{0x0001800000000000,   -1,{0xffff,0x1441}},  //  2036, depth=1
{0xfff0007ffff0007f, 1756,{0xffff,0x0312}},  //  2037, depth=2
{0x003fffffffffe000,   -1,{0xffff,0x1ce8}},  //  2038, depth=1
{0xfffff0000000000f,   -1,{0xffff,0x1517}},  //  2039
{0x00000e0000000e00, 1758,{0xffff,0x05c2}},  //  2040, depth=3
{0xffffffffff9fffff,   -1,{0xffff,0x1a7d}},  //  2041, depth=1
{0x1e1e1e1e1e1e1e1e,   -1,{0xffff,0x01f3}},  //  2042, depth=1
{0xf000fffff000ffff,   -1,{0xffff,0x0113}},  //  2043
{0x00000000000007fc,   -1,{0x0788,0x1f88}},  //  2044, depth=1
{0x00000000f8003fff, 1766,{0x0152,0xffff}},  //  2045, depth=2
{0x00000000000007fe,   -1,{0x07c9,0x1fc9}},  //  2046, depth=1
{0x00000000000007ff,   -1,{0x000a,0x100a}},  //  2047, depth=1
{0x0000000000000800,   -1,{0x0540,0x1d40}},  //  2048, depth=1
{0x1ffffc0000000000,   -1,{0xffff,0x1592}},  //  2049
{0x0000007000000000,   -1,{0xffff,0x1702}},  //  2050
{0xc00000003fffffff,   -1,{0xffff,0x109f}},  //  2051, depth=1
{0x0000006000000060, 1767,{0xffff,0x06c1}},  //  2052, depth=2
{0x0003c0000003c000,   -1,{0xffff,0x0483}},  //  2053
{0x0003ffc000000000, 1768,{0xffff,0x168b}},  //  2054, depth=2
{0xffe01fffffffffff,   -1,{0xffff,0x12f7}},  //  2055, depth=1
{0x000000000007ff00, 1769,{0x060a,0x1e0a}},  //  2056, depth=3
{0x0fffffffe0000000,   -1,{0xffff,0x18de}},  //  2057
{0x001fffffff800000,   -1,{0xffff,0x1a5d}},  //  2058
{0x000000000007fffe,   -1,{0x07d1,0x1fd1}},  //  2059
{0x000000000ffffc00,   -1,{0x0591,0x1d91}},  //  2060, depth=1
{0xf0000000ffffffff,   -1,{0xffff,0x1123}},  //  2061
{0x00000000f000f000,   -1,{0x0123,0xffff}},  //  2062, depth=1
{0x00000000bfbfbfbf,   -1,{0x0076,0xffff}},  //  2063, depth=1
{0x00001fffffe00000,   -1,{0xffff,0x1ad7}},  //  2064
{0xf3fff3fff3fff3ff,   -1,{0xffff,0x012d}},  //  2065, depth=1
{0xfffff0000000003f,   -1,{0xffff,0x1519}},  //  2066
{0xffffff7fffffffff,   -1,{0xffff,0x163e}},  //  2067
{0xffffffffff1fffff,   -1,{0xffff,0x1a3c}},  //  2068
{0xffc000000003ffff,   -1,{0xffff,0x129b}},  //  2069, depth=1
{0xffffff800000000f,   -1,{0xffff,0x165c}},  //  2070
{0xfe0003fffe0003ff, 1773,{0xffff,0x01d0}},  //  2071, depth=2
{0x000000001ffff000,   -1,{0x0510,0x1d10}},  //  2072, depth=1
{0xffffc000ffffc000,   -1,{0xffff,0x0491}},  //  2073
{0x000000007fffc000,   -1,{0x0490,0x1c90}},  //  2074, depth=1
{0xfffffffffc0003ff,   -1,{0xffff,0x19af}},  //  2075, depth=1
{0xfffffc00ffffffff,   -1,{0xffff,0x15b5}},  //  2076
{0xfc000001ffffffff,   -1,{0xffff,0x11a6}},  //  2077, depth=1
{0x00001ff800000000,   -1,{0xffff,0x1749}},  //  2078, depth=1
{0x0000007fffffff00,   -1,{0xffff,0x1e1e}},  //  2079
{0xfffffffffff007ff,   -1,{0xffff,0x1b36}},  //  2080
{0xfffe03ffffffffff,   -1,{0xffff,0x13f8}},  //  2081
{0x00000001ffff0000,   -1,{0xffff,0x1c10}},  //  2082, depth=1
{0xfffc0000000001ff,   -1,{0xffff,0x1396}},  //  2083, depth=1
{0x03ffffffffffffe0,   -1,{0xffff,0x1ef4}},  //  2084, depth=1
{0xf003f003f003f003,   -1,{0xffff,0x0125}},  //  2085, depth=1
{0x0001fff80001fff8,   -1,{0xffff,0x074d}},  //  2086, depth=1
{0xffff80003fffffff,   -1,{0xffff,0x146e}},  //  2087, depth=1
{0x0ffffc000ffffc00,   -1,{0xffff,0x0591}},  //  2088, depth=1
{0x00000000ffc7ffff,   -1,{0x029c,0xffff}},  //  2089, depth=1
{0x000003ffffc00000,   -1,{0xffff,0x1a93}},  //  2090
{0x00000000f87ff87f, 1774,{0x016b,0xffff}},  //  2091, depth=2
{0x0007fffffff00000,   -1,{0xffff,0x1b1e}},  //  2092, depth=1
{0xffffffc3ffffffc3, 1776,{0xffff,0x069b}},  //  2093, depth=2
{0x00fffffff8000000, 1777,{0xffff,0x195c}},  //  2094, depth=2
{0x00000000fc00007f, 1779,{0x018c,0xffff}},  //  2095, depth=2
{0x000000001f801f80, 1780,{0x0265,0xffff}},  //  2096, depth=3
{0xc00000001fffffff,   -1,{0xffff,0x109e}},  //  2097, depth=1
{0x00007fe000000000,   -1,{0xffff,0x16c9}},  //  2098, depth=1
{0xfc00ffffffffffff,   -1,{0xffff,0x11b5}},  //  2099, depth=1
{0x001fffffc0000000, 1788,{0xffff,0x1896}},  //  2100, depth=2
{0xfe0007ffffffffff, 1791,{0xffff,0x11f1}},  //  2101, depth=2
{0x0000003e0000003e,   -1,{0xffff,0x07c4}},  //  2102, depth=1
{0x00000000fc03fc03, 1801,{0x01a7,0xffff}},  //  2103, depth=3
{0x0000ffffffff0000,   -1,{0xffff,0x1c1f}},  //  2104, depth=1
{0xff800001ffffffff,   -1,{0xffff,0x1269}},  //  2105
{0x001fffff00000000,   -1,{0xffff,0x1814}},  //  2106
{0x80007fffffffffff,   -1,{0xffff,0x106f}},  //  2107, depth=1
{0x03fffffffffffff8,   -1,{0xffff,0x1f76}},  //  2108, depth=1
{0xfffff000003fffff,   -1,{0xffff,0x1529}},  //  2109, depth=1
{0x7ffffff87ffffff8, 1806,{0xffff,0x075b}},  //  2110, depth=2
{0xffe03fffffe03fff,   -1,{0xffff,0x02d8}},  //  2111, depth=1
{0x0070000000700000, 1809,{0xffff,0x0302}},  //  2112, depth=4
{0xff8000ffffffffff,   -1,{0xffff,0x1270}},  //  2113, depth=1
{0x00000007fffc0000, 1828,{0xffff,0x1b90}},  //  2114, depth=2
{0x00000000ff87ff87, 1829,{0x026b,0xffff}},  //  2115, depth=3
{0x0400000000000000,   -1,{0xffff,0x1180}},  //  2116, depth=1
{0xffffffff001fffff,   -1,{0xffff,0x1834}},  //  2117, depth=1
{0x0001fc000001fc00,   -1,{0xffff,0x0586}},  //  2118, depth=1
{0x00000001f0000000,   -1,{0xffff,0x1904}},  //  2119
{0x000007fffffff800,   -1,{0xffff,0x1d5f}},  //  2120
{0xfffe000000001fff,   -1,{0xffff,0x13db}},  //  2121, depth=1
{0x0fffffffffffff80,   -1,{0xffff,0x1e74}},  //  2122, depth=1
{0x9f9f9f9f9f9f9f9f, 1833,{0xffff,0x0075}},  //  2123, depth=2
{0x03c0000003c00000, 1835,{0xffff,0x0283}},  //  2124, depth=2
{0xc1ffffffc1ffffff,   -1,{0xffff,0x009a}},  //  2125, depth=1
{0x0000000007000000, 1838,{0x0202,0x1a02}},  //  2126, depth=2
{0x000000008007ffff, 1839,{0x0053,0xffff}},  //  2127, depth=2
{0xfe000003ffffffff,   -1,{0xffff,0x11e8}},  //  2128
{0x00000000ffffc1ff,   -1,{0x049a,0xffff}},  //  2129
{0x0007ffe00007ffe0,   -1,{0xffff,0x06cd}},  //  2130, depth=1
{0xff1fff1fff1fff1f,   -1,{0xffff,0x022c}},  //  2131, depth=1
{0x000000fff0000000, 1840,{0xffff,0x190b}},  //  2132, depth=2
{0xffff80001fffffff,   -1,{0xffff,0x146d}},  //  2133, depth=1
{0x3fffffe000000000,   -1,{0xffff,0x16d8}},  //  2134, depth=1
{0xffffffffffe001ff,   -1,{0xffff,0x1af3}},  //  2135
{0xfffffe0000007fff, 2140,{0xffff,0x15e5}},  //  2136
{0xffffff807fffffff,   -1,{0xffff,0x1677}},  //  2137, depth=1
{0x3ffff0003ffff000,   -1,{0xffff,0x0511}},  //  2138, depth=1
{0xfe003fffffffffff,   -1,{0xffff,0x11f4}},  //  2139, depth=1
{0xf007fffff007ffff,   -1,{0xffff,0x0116}},  //  2140
{0x0000fffff0000000,   -1,{0xffff,0x1913}},  //  2141
{0x7ffffc007ffffc00,   -1,{0xffff,0x0594}},  //  2142, depth=1
{0xbfffffffffffffff,   -1,{0xffff,0x107e}},  //  2143, depth=1
{0xc000000000000000,   -1,{0xffff,0x1081}},  //  2144, depth=1
{0x00000000ffffc0ff, 1844,{0x0499,0xffff}},  //  2145, depth=3
{0x00000003fff80000,   -1,{0xffff,0x1b4e}},  //  2146
{0xc000000000000003,   -1,{0xffff,0x1083}},  //  2147, depth=1
{0xfff801ffffffffff,   -1,{0xffff,0x1375}},  //  2148
{0xfffc00000001ffff,   -1,{0xffff,0x139e}},  //  2149, depth=1
{0x000e000000000000,   -1,{0xffff,0x13c2}},  //  2150
{0xc000000000000007, 1852,{0xffff,0x1084}},  //  2151, depth=2
{0x000000007c000000, 1855,{0x0184,0x1984}},  //  2152, depth=2
{0xffffffc3ffffffff,   -1,{0xffff,0x16bb}},  //  2153, depth=1
{0x001fffffffc00000,   -1,{0xffff,0x1a9e}},  //  2154, depth=1
{0xff0007ffff0007ff, 1856,{0xffff,0x0212}},  //  2155, depth=2
{0xff000fffff000fff,   -1,{0xffff,0x0213}},  //  2156
{0xffffffffe000ffff,   -1,{0xffff,0x18f2}},  //  2157, depth=1
{0x000003ffffffffc0,   -1,{0xffff,0x1ea3}},  //  2158, depth=1
{0xc00000000000000f, 1858,{0xffff,0x1085}},  //  2159, depth=2
{0x00003ffffffff800,   -1,{0xffff,0x1d62}},  //  2160
{0x0000000001ff01ff,   -1,{0x0028,0xffff}},  //  2161, depth=1
{0x03ffffffe0000000, 1859,{0xffff,0x18dc}},  //  2162, depth=2
{0xfffff0003fffffff,   -1,{0xffff,0x1531}},  //  2163, depth=1
{0x0000000ffffff000,   -1,{0xffff,0x1d17}},  //  2164
{0xfffffffc00001fff,   -1,{0xffff,0x17aa}},  //  2165, depth=1
{0x000000003fffc000,   -1,{0x048f,0x1c8f}},  //  2166, depth=1
{0xffffffff0fffffff,   -1,{0xffff,0x183b}},  //  2167, depth=1
{0xffffff800000007f, 2174,{0xffff,0x165f}},  //  2168
{0xfff8000000003fff,   -1,{0xffff,0x135a}},  //  2169, depth=1
{0x000000007e007e00, 1862,{0x01e5,0xffff}},  //  2170, depth=3
{0xffc01fffffc01fff, 1866,{0xffff,0x0296}},  //  2171, depth=2
{0x000ffffffe000000, 1869,{0xffff,0x19da}},  //  2172, depth=2
{0x00000000f83ff83f,   -1,{0x016a,0xffff}},  //  2173, depth=1
{0xff0000000003ffff,   -1,{0xffff,0x1219}},  //  2174
{0xc00000000000001f, 1873,{0xffff,0x1086}},  //  2175, depth=2
{0x7fffffffffff0000,   -1,{0xffff,0x1c2e}},  //  2176, depth=1
{0x00000000f0ffffff,   -1,{0x011b,0xffff}},  //  2177, depth=1
{0x0001ff8000000000,   -1,{0xffff,0x1649}},  //  2178, depth=1
{0xffff7fffffffffff,   -1,{0xffff,0x143e}},  //  2179, depth=1
{0x0000000003ff8000, 1876,{0x044a,0x1c4a}},  //  2180, depth=2
{0xfc0003fffc0003ff, 1879,{0xffff,0x018f}},  //  2181, depth=2
{0x000000ffffe00000,   -1,{0xffff,0x1ad2}},  //  2182, depth=1
{0xffff800000000003, 1881,{0xffff,0x1452}},  //  2183, depth=2
{0x000000000007ff80, 1884,{0x064b,0x1e4b}},  //  2184, depth=2
{0x00000000f8000fff, 1886,{0x0150,0xffff}},  //  2185, depth=2
{0x007fffff00000000, 1894,{0xffff,0x1816}},  //  2186, depth=2
{0xffff800000000007,   -1,{0xffff,0x1453}},  //  2187, depth=1
{0x1ffffffffff80000,   -1,{0xffff,0x1b69}},  //  2188, depth=1
{0xffffe07fffffffff,   -1,{0xffff,0x14f9}},  //  2189, depth=1
{0x000003ffffffffe0,   -1,{0xffff,0x1ee4}},  //  2190, depth=1
{0xfffe0000000007ff,   -1,{0xffff,0x13d9}},  //  2191, depth=1
{0x003c003c003c003c, 1898,{0xffff,0x03a3}},  //  2192, depth=2
{0x00000000ff001fff, 1902,{0x0214,0xffff}},  //  2193, depth=3
{0x000000f8000000f8,   -1,{0xffff,0x0744}},  //  2194, depth=1
{0xffff80000000000f, 1912,{0xffff,0x1454}},  //  2195, depth=2
{0xfff00003ffffffff,   -1,{0xffff,0x132d}},  //  2196
{0xfffe00000007ffff,   -1,{0xffff,0x13e1}},  //  2197
{0xffffffc001ffffff, 2199,{0xffff,0x16b2}},  //  2198
{0x00000000ffff0007,   -1,{0x0412,0xffff}},  //  2199
{0xfffff80000001fff,   -1,{0xffff,0x1561}},  //  2200
{0x000007ffffffe000,   -1,{0xffff,0x1cdd}},  //  2201
{0x0003fffffffc0000,   -1,{0xffff,0x1b9f}},  //  2202, depth=1
{0xffffffffc000ffff,   -1,{0xffff,0x18b1}},  //  2203, depth=1
{0xc000000fffffffff,   -1,{0xffff,0x10a5}},  //  2204
{0x0000003ffe000000,   -1,{0xffff,0x19cc}},  //  2205
{0x000003fffffffff0,   -1,{0xffff,0x1f25}},  //  2206, depth=1
{0xf8fffffff8ffffff, 1918,{0xffff,0x015c}},  //  2207, depth=4
{0x03ffffffc0000000, 1926,{0xffff,0x189b}},  //  2208, depth=2
{0xfffff0001fffffff,   -1,{0xffff,0x1530}},  //  2209, depth=1
{0x0001fffffff80000,   -1,{0xffff,0x1b5d}},  //  2210
{0xffff80000000001f,   -1,{0xffff,0x1455}},  //  2211, depth=1
{0x000000001fffc000,   -1,{0x048e,0x1c8e}},  //  2212, depth=1
{0xfc000007fc000007,   -1,{0xffff,0x0188}},  //  2213, depth=1
{0x7ffc000000000000, 1939,{0xffff,0x138c}},  //  2214, depth=2
{0x00000000bbbbbbbb,   -1,{0x007a,0xffff}},  //  2215, depth=1
{0xffffffff0000ffff,   -1,{0xffff,0x182f}},  //  2216
{0xffc0003fffffffff,   -1,{0xffff,0x12af}},  //  2217, depth=1
{0x0fffffffffffffe0, 1942,{0xffff,0x1ef6}},  //  2218, depth=2
{0xff00000fff00000f, 1948,{0xffff,0x020b}},  //  2219, depth=2
{0x000003fffffffffe,   -1,{0xffff,0x1fe8}},  //  2220, depth=1
{0x000003ffffffffff,   -1,{0xffff,0x1029}},  //  2221, depth=1
{0x0000040000000000,   -1,{0xffff,0x1580}},  //  2222, depth=1
{0x8000ffff8000ffff, 1949,{0xffff,0x0050}},  //  2223, depth=3
{0x0000400000004000, 1951,{0xffff,0x0480}},  //  2224, depth=2
{0x0000007fffffff80,   -1,{0xffff,0x1e5f}},  //  2225
{0x01fffffffe000000,   -1,{0xffff,0x19df}},  //  2226, depth=1
{0xfffffffe001fffff,   -1,{0xffff,0x17f3}},  //  2227
{0x0018000000180000,   -1,{0xffff,0x0341}},  //  2228, depth=1
{0xfc00000000ffffff, 1957,{0xffff,0x119d}},  //  2229, depth=2
{0x1fffc0001fffc000, 1960,{0xffff,0x048e}},  //  2230, depth=2
{0xff9fffffff9fffff,   -1,{0xffff,0x025d}},  //  2231, depth=1
{0x01fe000000000000,   -1,{0xffff,0x13c7}},  //  2232
{0xfffffffffffe0fff,   -1,{0xffff,0x1bfa}},  //  2233, depth=1
{0x01c0000001c00000, 1961,{0xffff,0x0282}},  //  2234, depth=4
{0xbfffffffbfffffff, 1982,{0xffff,0x005e}},  //  2235, depth=3
{0x1fe0000000000000,   -1,{0xffff,0x12c7}},  //  2236
{0xfffffff80001ffff,   -1,{0xffff,0x176d}},  //  2237, depth=1
{0x7070707070707070, 1986,{0xffff,0x0132}},  //  2238, depth=2
{0xf800000007ffffff,   -1,{0xffff,0x115f}},  //  2239, depth=1
{0x000001ffffffe000,   -1,{0xffff,0x1cdb}},  //  2240, depth=1
{0xffffc000000007ff,   -1,{0xffff,0x149c}},  //  2241
{0x0000001ffff00000, 1990,{0xffff,0x1b10}},  //  2242, depth=2
{0xffff80000000003f,   -1,{0xffff,0x1456}},  //  2243, depth=1
{0x000000003c000000,   -1,{0x0183,0x1983}},  //  2244, depth=1
{0x00000000fff003ff, 1991,{0x0315,0xffff}},  //  2245, depth=2
{0x00000007fffffc00, 1993,{0xffff,0x1d98}},  //  2246, depth=2
{0xfffefffffffeffff, 1996,{0xffff,0x03de}},  //  2247, depth=2
{0x000000000007ffc0, 2009,{0x068c,0x1e8c}},  //  2248, depth=2
{0xffe3ffe3ffe3ffe3, 2011,{0xffff,0x02ec}},  //  2249, depth=2
{0x1000000000000000,   -1,{0xffff,0x1100}},  //  2250, depth=1
{0x07fffff807fffff8,   -1,{0xffff,0x0757}},  //  2251
{0xfff1ffffffffffff, 2265,{0xffff,0x133c}},  //  2252
{0xfffffe0007ffffff,   -1,{0xffff,0x15f1}},  //  2253, depth=1
{0x07ffffffffffe000,   -1,{0xffff,0x1ced}},  //  2254, depth=1
{0x00000000f7fff7ff, 2013,{0x012e,0xffff}},  //  2255, depth=2
{0xfffff00000000000,   -1,{0xffff,0x1513}},  //  2256, depth=1
{0xfffff00000000001, 2018,{0xffff,0x1514}},  //  2257, depth=2
{0x0007f0000007f000,   -1,{0xffff,0x0506}},  //  2258, depth=1
{0xfffff00000000003,   -1,{0xffff,0x1515}},  //  2259, depth=1
{0x000000001f001f00, 2020,{0x0224,0xffff}},  //  2260, depth=3
{0x00000000fffff1ff,   -1,{0x051c,0xffff}},  //  2261, depth=1
{0x03ffffffffc00000,   -1,{0xffff,0x1aa3}},  //  2262, depth=1
{0x00000000ff0007ff, 2028,{0x0212,0xffff}},  //  2263, depth=2
{0x00001ffff0000000,   -1,{0xffff,0x1910}},  //  2264, depth=1
{0xff07ff07ff07ff07,   -1,{0xffff,0x022a}},  //  2265
{0xffff00000fffffff, 2269,{0xffff,0x142b}},  //  2266
{0xfffffff003ffffff,   -1,{0xffff,0x1735}},  //  2267, depth=1
{0x007fff00007fff00, 2034,{0xffff,0x060e}},  //  2268, depth=2
{0x00000000ffff003f,   -1,{0x0415,0xffff}},  //  2269
{0x000001ffffff8000, 2273,{0xffff,0x1c59}},  //  2270
{0xc00000000000007f, 2039,{0xffff,0x1088}},  //  2271, depth=2
{0x0007f80000000000,   -1,{0xffff,0x1547}},  //  2272, depth=1
{0x1ffc1ffc1ffc1ffc,   -1,{0xffff,0x03aa}},  //  2273
{0x3ffffffffffffe00,   -1,{0xffff,0x1df4}},  //  2274, depth=1
{0xffffff801fffffff,   -1,{0xffff,0x1675}},  //  2275, depth=1
{0x1fffffffffffe000,   -1,{0xffff,0x1cef}},  //  2276
{0x8000007f8000007f, 2043,{0xffff,0x0047}},  //  2277, depth=2
{0x000000000c000c00, 2049,{0x01a1,0xffff}},  //  2278, depth=2
{0x00000000ff07ff07,   -1,{0x022a,0xffff}},  //  2279, depth=1
{0x000000000007ffe0,   -1,{0x06cd,0x1ecd}},  //  2280, depth=1
{0xfe000000000fffff,   -1,{0xffff,0x11da}},  //  2281, depth=1
{0x0f0000000f000000, 2050,{0xffff,0x0203}},  //  2282, depth=2
{0xfffffffffff01fff,   -1,{0xffff,0x1b38}},  //  2283, depth=1
{0x3ff8000000000000,   -1,{0xffff,0x134a}},  //  2284, depth=1
{0xff80000000ffffff,   -1,{0xffff,0x1260}},  //  2285
{0x3fffffffffffff00,   -1,{0xffff,0x1e35}},  //  2286
{0xfffff0000000001f,   -1,{0xffff,0x1518}},  //  2287, depth=1
{0xfffff800000007ff,   -1,{0xffff,0x155f}},  //  2288
{0xfffffffe0fffffff,   -1,{0xffff,0x17fa}},  //  2289
{0x000000001c000000, 2053,{0x0182,0x1982}},  //  2290, depth=2
{0x000000000ff00ff0,   -1,{0x0327,0xffff}},  //  2291
{0xe3ffffffffffffff,   -1,{0xffff,0x10fc}},  //  2292
{0xffe00000ffffffff,   -1,{0xffff,0x12ea}},  //  2293
{0x007ffffe00000000,   -1,{0xffff,0x17d5}},  //  2294
{0xffffffff8000ffff,   -1,{0xffff,0x1870}},  //  2295, depth=1
{0x000000000007fff0, 2057,{0x070e,0x1f0e}},  //  2296, depth=2
{0xff0fffffffffffff,   -1,{0xffff,0x123b}},  //  2297, depth=1
{0xfffffc00003fffff, 2299,{0xffff,0x15ab}},  //  2298
{0xff00001fff00001f,   -1,{0xffff,0x020c}},  //  2299
{0x03ffffff80000000, 2058,{0xffff,0x185a}},  //  2300, depth=2
{0xfffe007ffffe007f,   -1,{0xffff,0x03d5}},  //  2301, depth=1
{0x7ffffff800000000,   -1,{0xffff,0x175b}},  //  2302, depth=1
{0x0000007fffffffe0,   -1,{0xffff,0x1ee1}},  //  2303
{0x000000000007fff8,   -1,{0x074f,0x1f4f}},  //  2304, depth=1
{0x01ff01ff01ff01ff,   -1,{0xffff,0x0028}},  //  2305, depth=1
{0x001fff80001fff80,   -1,{0xffff,0x064d}},  //  2306, depth=1
{0xffff80000000007f,   -1,{0xffff,0x1457}},  //  2307, depth=1
{0x000000000007fffc,   -1,{0x0790,0x1f90}},  //  2308, depth=1
{0xfff8000000000fff,   -1,{0xffff,0x1358}},  //  2309, depth=1
{0x0100010001000100, 2059,{0xffff,0x0220}},  //  2310, depth=2
{0x000000000007ffff, 2061,{0x0012,0x1012}},  //  2311, depth=2
{0x0000000000080000,   -1,{0x0340,0x1b40}},  //  2312, depth=1
{0xffffff80000000ff, 2316,{0xffff,0x1660}},  //  2313
{0x000003ffc0000000, 2064,{0xffff,0x188b}},  //  2314, depth=2
{0x0001000100010001,   -1,{0xffff,0x0020}},  //  2315, depth=1
{0x00000000ffff007f,   -1,{0x0416,0xffff}},  //  2316
{0xff000000000003ff,   -1,{0xffff,0x1211}},  //  2317, depth=1
{0x003fffffffff8000,   -1,{0xffff,0x1c66}},  //  2318, depth=1
{0xf1f1f1f1f1f1f1f1, 2066,{0xffff,0x0134}},  //  2319, depth=2
{0x0000000000080008,   -1,{0x0360,0xffff}},  //  2320, depth=1
{0xc0ffc0ffc0ffc0ff, 2067,{0xffff,0x00a9}},  //  2321, depth=2
{0xffffff8000000000,   -1,{0xffff,0x1658}},  //  2322, depth=1
{0xffffff8000000001,   -1,{0xffff,0x1659}},  //  2323, depth=1
{0x0000007ffffffff0,   -1,{0xffff,0x1f22}},  //  2324
{0xffffff8000000003,   -1,{0xffff,0x165a}},  //  2325, depth=1
{0xc000c000c000c000,   -1,{0xffff,0x00a1}},  //  2326, depth=1
{0x00001e0000001e00,   -1,{0xffff,0x05c3}},  //  2327
{0x0000007fffffffff,   -1,{0xffff,0x1026}},  //  2328
{0xffffff8000000007,   -1,{0xffff,0x165b}},  //  2329, depth=1
{0x0000008000000000,   -1,{0xffff,0x1640}},  //  2330
{0xf000007ff000007f,   -1,{0xffff,0x010a}},  //  2331, depth=1
{0x00ffffc000000000,   -1,{0xffff,0x1691}},  //  2332, depth=1
{0xffffe1ffffffe1ff, 2068,{0xffff,0x04db}},  //  2333, depth=2
{0x0003fffffffffc00,   -1,{0xffff,0x1da7}},  //  2334, depth=1
{0xfff00000000fffff,   -1,{0xffff,0x131f}},  //  2335, depth=1
{0x7f0000007f000000,   -1,{0xffff,0x0206}},  //  2336, depth=1
{0xffe0000000ffffff, 2070,{0xffff,0x12e2}},  //  2337, depth=2
{0x0000000066666666, 2073,{0x00f9,0xffff}},  //  2338, depth=2
{0x00000000e00fe00f, 2076,{0x00e6,0xffff}},  //  2339, depth=2
{0x0000fff00000fff0,   -1,{0xffff,0x070b}},  //  2340, depth=1
{0x00000fffffffffc0,   -1,{0xffff,0x1ea5}},  //  2341
{0x0fffffffc0000000,   -1,{0xffff,0x189d}},  //  2342, depth=1
{0xfffffc003fffffff, 2344,{0xffff,0x15b3}},  //  2343
{0xffffe000001fffff,   -1,{0xffff,0x14e7}},  //  2344
{0xfffffff000003fff,   -1,{0xffff,0x1729}},  //  2345, depth=1
{0xfffffffffff80000,   -1,{0xffff,0x1b6c}},  //  2346
{0xfffffffffff80001,   -1,{0xffff,0x1b6d}},  //  2347
{0x00000ff000000000,   -1,{0xffff,0x1707}},  //  2348, depth=1
{0xffffe003ffffe003,   -1,{0xffff,0x04d4}},  //  2349, depth=1
{0x000ffff8000ffff8,   -1,{0xffff,0x0750}},  //  2350, depth=1
{0xffe00000007fffff,   -1,{0xffff,0x12e1}},  //  2351
{0x3ffffff03ffffff0, 2079,{0xffff,0x0719}},  //  2352, depth=2
{0xffffff800000001f, 2080,{0xffff,0x165d}},  //  2353, depth=2
{0xfffffffffff8000f,   -1,{0xffff,0x1b70}},  //  2354
{0x8007ffff8007ffff,   -1,{0xffff,0x0053}},  //  2355, depth=1
{0xffffffffe3ffffff,   -1,{0xffff,0x18fc}},  //  2356
{0xfffffffff00001ff,   -1,{0xffff,0x192c}},  //  2357, depth=1
{0xffffff8fffffffff,   -1,{0xffff,0x167c}},  //  2358
{0x7fffff807fffff80,   -1,{0xffff,0x0657}},  //  2359
{0x001f0000001f0000,   -1,{0xffff,0x0404}},  //  2360, depth=1
{0x00000ffffffffff0,   -1,{0xffff,0x1f27}},  //  2361
{0xfffffc001fffffff,   -1,{0xffff,0x15b2}},  //  2362
{0xffffc3ffffffffff,   -1,{0xffff,0x14bb}},  //  2363
{0x00000ffffffffffe,   -1,{0xffff,0x1fea}},  //  2364
{0x00000000fff3fff3, 2081,{0x032d,0xffff}},  //  2365, depth=2
{0xf000000000000000,   -1,{0xffff,0x1103}},  //  2366
{0xfffffc1ffffffc1f,   -1,{0xffff,0x059a}},  //  2367, depth=1
{0x00ffffe000ffffe0, 2090,{0xffff,0x06d2}},  //  2368, depth=2
{0xfffffffffff8003f,   -1,{0xffff,0x1b72}},  //  2369
{0x1f80000000000000,   -1,{0xffff,0x1245}},  //  2370
{0xfffffff8000003ff,   -1,{0xffff,0x1766}},  //  2371
{0x000000000000fc00,   -1,{0x0585,0x1d85}},  //  2372, depth=1
{0xffffffe0000fffff,   -1,{0xffff,0x16ee}},  //  2373
{0xffff8000000001ff,   -1,{0xffff,0x1459}},  //  2374
{0xff0fff0fff0fff0f, 2105,{0xffff,0x022b}},  //  2375, depth=2
{0x007e0000007e0000, 2106,{0xffff,0x03c5}},  //  2376, depth=2
{0x00000000fff3ffff,   -1,{0x031d,0xffff}},  //  2377, depth=1
{0x00000003fffff800,   -1,{0xffff,0x1d56}},  //  2378
{0xffffffffff00ffff, 2380,{0xffff,0x1a37}},  //  2379
{0xfff83fffffffffff, 2381,{0xffff,0x137a}},  //  2380
{0xf00000000000000f,   -1,{0xffff,0x1107}},  //  2381
{0x000ffc00000ffc00,   -1,{0xffff,0x0589}},  //  2382, depth=1
{0xfffff0000000007f,   -1,{0xffff,0x151a}},  //  2383, depth=1
{0x1c00000000000000,   -1,{0xffff,0x1182}},  //  2384, depth=1
{0xffffff800000003f,   -1,{0xffff,0x165e}},  //  2385, depth=1
{0xffff0007ffffffff,   -1,{0xffff,0x1432}},  //  2386
{0xf0000000007fffff,   -1,{0xffff,0x111a}},  //  2387, depth=1
{0x03ffffffff000000,   -1,{0xffff,0x1a21}},  //  2388
{0xe3ffe3ffe3ffe3ff,   -1,{0xffff,0x00ec}},  //  2389, depth=1
{0xf81fffffffffffff, 2392,{0xffff,0x1179}},  //  2390
{0x80000000001fffff,   -1,{0xffff,0x1055}},  //  2391, depth=1
{0xffffbfffffffbfff,   -1,{0xffff,0x045e}},  //  2392
{0xfff807fffff807ff,   -1,{0xffff,0x0357}},  //  2393, depth=1
{0x0000007000000070, 2119,{0xffff,0x0702}},  //  2394, depth=2
{0xf800000000007fff,   -1,{0xffff,0x1153}},  //  2395, depth=1
{0x0fffffffffc00000, 2120,{0xffff,0x1aa5}},  //  2396, depth=2
{0xff80ffffffffffff,   -1,{0xffff,0x1278}},  //  2397, depth=1
{0x0003ffe000000000,   -1,{0xffff,0x16cc}},  //  2398
{0xc0000000000000ff, 2128,{0xffff,0x1089}},  //  2399, depth=2
{0x00fffff000fffff0,   -1,{0xffff,0x0713}},  //  2400
{0xf80003fff80003ff, 2129,{0xffff,0x014e}},  //  2401, depth=2
{0x007fffffff000000,   -1,{0xffff,0x1a1e}},  //  2402, depth=1
{0xffffffffff80ffff,   -1,{0xffff,0x1a78}},  //  2403, depth=1
{0x0000f8000000f800,   -1,{0xffff,0x0544}},  //  2404, depth=1
{0xfffffc0000000003, 2410,{0xffff,0x1597}},  //  2405
{0x000003ff80000000,   -1,{0xffff,0x184a}},  //  2406, depth=1
{0xe000000000001fff, 2135,{0xffff,0x10cf}},  //  2407, depth=2
{0x03ffffffff800000,   -1,{0xffff,0x1a62}},  //  2408, depth=1
{0xe003e003e003e003, 2136,{0xffff,0x00e4}},  //  2409, depth=3
{0xf00000000000001f,   -1,{0xffff,0x1108}},  //  2410
{0xfffffc0000000007,   -1,{0xffff,0x1598}},  //  2411
{0xfc0000003fffffff,   -1,{0xffff,0x11a3}},  //  2412
{0x3fffffffffffffc0, 2417,{0xffff,0x1eb7}},  //  2413
{0x0780000000000000,   -1,{0xffff,0x1243}},  //  2414, depth=1
{0xfffffc00007fffff,   -1,{0xffff,0x15ac}},  //  2415, depth=1
{0x3ffff8003ffff800, 2141,{0xffff,0x0552}},  //  2416, depth=2
{0x001f000000000000, 2419,{0xffff,0x1404}},  //  2417
{0x00000fffffffff00,   -1,{0xffff,0x1e23}},  //  2418, depth=1
{0x0000000003fc0000,   -1,{0x0387,0x1b87}},  //  2419
{0x00007ffffffffff0,   -1,{0xffff,0x1f2a}},  //  2420
{0xffffff07ffffff07,   -1,{0xffff,0x061a}},  //  2421
{0x000fc00000000000,   -1,{0xffff,0x1485}},  //  2422
{0xfffffffff001ffff,   -1,{0xffff,0x1934}},  //  2423, depth=1
{0x00007ff000000000,   -1,{0xffff,0x170a}},  //  2424, depth=1
{0xffffc00000001fff,   -1,{0xffff,0x149e}},  //  2425, depth=1
{0xffe000007fffffff,   -1,{0xffff,0x12e9}},  //  2426
{0xffc0003fffc0003f,   -1,{0xffff,0x028f}},  //  2427, depth=1
{0xff800000ffffffff,   -1,{0xffff,0x1268}},  //  2428
{0xfffffffffff8007f,   -1,{0xffff,0x1b73}},  //  2429
{0x0000000007fc07fc, 2146,{0x03a8,0xffff}},  //  2430, depth=2
{0x00000000fffc3fff, 2148,{0x039b,0xffff}},  //  2431, depth=2
{0x00f800f800f800f8, 2150,{0xffff,0x0364}},  //  2432, depth=2
{0xf8000007f8000007, 2156,{0xffff,0x0147}},  //  2433, depth=2
{0x0fffffff80000000, 2160,{0xffff,0x185c}},  //  2434, depth=2
{0xffff8000000000ff,   -1,{0xffff,0x1458}},  //  2435, depth=1
{0x00007ffffffffffc,   -1,{0xffff,0x1fac}},  //  2436
{0xe01fffffffffffff,   -1,{0xffff,0x10f7}},  //  2437, depth=1
{0x00007ffffffffffe,   -1,{0xffff,0x1fed}},  //  2438
{0xfffffffffc007fff,   -1,{0xffff,0x19b4}},  //  2439
{0x3fffffffffffffe0,   -1,{0xffff,0x1ef8}},  //  2440
{0x800000000fffffff,   -1,{0xffff,0x105c}},  //  2441, depth=1
{0xffff80000001ffff, 2446,{0xffff,0x1461}},  //  2442
{0xfe01ffffffffffff,   -1,{0xffff,0x11f7}},  //  2443, depth=1
{0x00000001ffe00000, 2164,{0xffff,0x1acb}},  //  2444, depth=2
{0xfc000000ffffffff,   -1,{0xffff,0x11a5}},  //  2445, depth=1
{0xfff0000001ffffff, 2447,{0xffff,0x1324}},  //  2446
{0x00000000c01fffff,   -1,{0x0096,0xffff}},  //  2447
{0x00000000fffefffe,   -1,{0x03ee,0xffff}},  //  2448, depth=1
{0x00000000fffeffff, 2168,{0x03de,0xffff}},  //  2449, depth=3
{0x00000000ffff0000,   -1,{0x040f,0x1c0f}},  //  2450, depth=1
{0x00000000ffff0001,   -1,{0x0410,0xffff}},  //  2451, depth=1
{0x1ffffffffffff800,   -1,{0xffff,0x1d71}},  //  2452, depth=1
{0x00000000ffff0003, 2196,{0x0411,0xffff}},  //  2453, depth=2
{0x00000007fffffe00,   -1,{0xffff,0x1dd9}},  //  2454
{0xffc0000007ffffff, 2197,{0xffff,0x12a4}},  //  2455, depth=2
{0x00000000001e0000,   -1,{0x03c3,0x1bc3}},  //  2456, depth=1
{0xfff8000ffff8000f, 2198,{0xffff,0x0350}},  //  2457, depth=3
{0xfffffe00000007ff,   -1,{0xffff,0x15e1}},  //  2458
{0xffffffffc0003fff,   -1,{0xffff,0x18af}},  //  2459
{0x3ffffff000000000,   -1,{0xffff,0x1719}},  //  2460, depth=1
{0x0000000011111111,   -1,{0x0038,0xffff}},  //  2461, depth=1
{0xfffff000000001ff,   -1,{0xffff,0x151c}},  //  2462
{0xffff0000001fffff, 2200,{0xffff,0x1424}},  //  2463, depth=2
{0x3ffffffffffffff0,   -1,{0xffff,0x1f39}},  //  2464
{0x00000000ffff000f,   -1,{0x0413,0xffff}},  //  2465, depth=1
{0x0000ffffffe00000, 2201,{0xffff,0x1ada}},  //  2466, depth=2
{0x000ff80000000000,   -1,{0xffff,0x1548}},  //  2467
{0x0000003c00000000,   -1,{0xffff,0x1783}},  //  2468
{0x00000000f07fffff, 2204,{0x011a,0xffff}},  //  2469, depth=2
{0xffffffff83ffffff, 2471,{0xffff,0x187a}},  //  2470
{0xffffffe3ffffffe3, 2475,{0xffff,0x06dc}},  //  2471
{0x0007fff00007fff0, 2205,{0xffff,0x070e}},  //  2472, depth=2
{0xffffffffffe1ffff,   -1,{0xffff,0x1afb}},  //  2473, depth=1
{0x003ffffff8000000, 2210,{0xffff,0x195a}},  //  2474, depth=2
{0x00000000fe001fff,   -1,{0x01d3,0xffff}},  //  2475
{0x000ffffc00000000,   -1,{0xffff,0x1791}},  //  2476, depth=1
{0xe0000000000007ff,   -1,{0xffff,0x10cd}},  //  2477, depth=1
{0x3ffffffffffffff8,   -1,{0xffff,0x1f7a}},  //  2478
{0x00000000fc0001ff, 2216,{0x018e,0xffff}},  //  2479, depth=2
{0x00fffffffffc0000, 2225,{0xffff,0x1ba5}},  //  2480, depth=2
{0x00000000ffff001f,   -1,{0x0414,0xffff}},  //  2481, depth=1
{0x3ffffffffffffffe,   -1,{0xffff,0x1ffc}},  //  2482
{0xffffff000fffffff,   -1,{0xffff,0x1633}},  //  2483
{0x03ffffff00000000,   -1,{0xffff,0x1819}},  //  2484, depth=1
{0xfffffff000000fff, 2227,{0xffff,0x1727}},  //  2485, depth=2
{0x00000000001e001e, 2232,{0x03e3,0xffff}},  //  2486, depth=2
{0xfffffffff8ffffff,   -1,{0xffff,0x197c}},  //  2487
{0x7ffffffff0000000,   -1,{0xffff,0x1922}},  //  2488, depth=1
{0x0000000000ff00ff,   -1,{0x0027,0xffff}},  //  2489, depth=1
{0xffffffe3ffffffff,   -1,{0xffff,0x16fc}},  //  2490
{0x000003ffff000000,   -1,{0xffff,0x1a11}},  //  2491
{0x0000000001e001e0, 2236,{0x02e3,0xffff}},  //  2492, depth=2
{0xf00000000000007f,   -1,{0xffff,0x110a}},  //  2493
{0x00007fffffffff00,   -1,{0xffff,0x1e26}},  //  2494, depth=1
{0xf00000007fffffff, 2241,{0xffff,0x1122}},  //  2495, depth=2
{0x00fff00000fff000, 2251,{0xffff,0x050b}},  //  2496, depth=2
{0x00000000fe007fff, 2252,{0x01d5,0xffff}},  //  2497, depth=3
{0x0007fe0000000000,   -1,{0xffff,0x15c9}},  //  2498, depth=1
{0xfffffffc0007ffff,   -1,{0xffff,0x17b0}},  //  2499, depth=1
{0xfffe03fffffe03ff,   -1,{0xffff,0x03d8}},  //  2500
{0xfbffffffffffffff,   -1,{0xffff,0x117e}},  //  2501
{0x003fffc0003fffc0,   -1,{0xffff,0x068f}},  //  2502, depth=1
{0xfc00000000000000,   -1,{0xffff,0x1185}},  //  2503
{0x00003fffffffe000,   -1,{0xffff,0x1ce0}},  //  2504, depth=1
{0xffff800fffffffff,   -1,{0xffff,0x1474}},  //  2505, depth=1
{0x000000000ffe0000,   -1,{0x03ca,0x1bca}},  //  2506, depth=1
{0xfffffff80003ffff, 2508,{0xffff,0x176e}},  //  2507
{0xfc00000000000001,   -1,{0xffff,0x1186}},  //  2508
{0xffffffffffc03fff, 2510,{0xffff,0x1ab7}},  //  2509
{0xfffe0fffffffffff, 2512,{0xffff,0x13fa}},  //  2510
{0xfffff000000000ff,   -1,{0xffff,0x151b}},  //  2511, depth=1
{0xfc00000000000003,   -1,{0xffff,0x1187}},  //  2512
{0xc00007ffc00007ff, 2266,{0xffff,0x008c}},  //  2513, depth=3
{0x000003ffff800000,   -1,{0xffff,0x1a52}},  //  2514, depth=1
{0xf87fffffffffffff,   -1,{0xffff,0x117b}},  //  2515, depth=1
{0xff800000007fffff, 2517,{0xffff,0x125f}},  //  2516
{0x00000000ffe00007,   -1,{0x02cd,0xffff}},  //  2517
{0x00000fffffc00000,   -1,{0xffff,0x1a95}},  //  2518
{0xffff00000000ffff,   -1,{0xffff,0x141f}},  //  2519
{0x0ff800000ff80000, 2270,{0xffff,0x0348}},  //  2520, depth=3
{0xfc000000007fffff,   -1,{0xffff,0x119c}},  //  2521, depth=1
{0x00000000001ffe00, 2276,{0x05cb,0x1dcb}},  //  2522, depth=2
{0xfffffc007fffffff,   -1,{0xffff,0x15b4}},  //  2523, depth=1
{0x0000f00000000000, 2529,{0xffff,0x1503}},  //  2524
{0xffff07ffffff07ff,   -1,{0xffff,0x041a}},  //  2525, depth=1
{0x00000000007f0000,   -1,{0x0406,0x1c06}},  //  2526, depth=1
{0x00000000ff07ffff, 2285,{0x021a,0xffff}},  //  2527, depth=2
{0x07fffc0007fffc00,   -1,{0xffff,0x0590}},  //  2528, depth=1
{0x01fffff801fffff8,   -1,{0xffff,0x0755}},  //  2529
{0x01fffffc00000000, 2286,{0xffff,0x1796}},  //  2530, depth=2
{0xffffffc1ffffffc1,   -1,{0xffff,0x069a}},  //  2531
{0x007f000000000000,   -1,{0xffff,0x1406}},  //  2532, depth=1
{0xf0000000003fffff, 2288,{0xffff,0x1119}},  //  2533, depth=2
{0x07ffffffffff8000,   -1,{0xffff,0x1c6b}},  //  2534, depth=1
{0xffffff8fffffff8f, 2289,{0xffff,0x065c}},  //  2535, depth=2
{0x0007f8000007f800,   -1,{0xffff,0x0547}},  //  2536, depth=1
{0x000000000f0f0f0f,   -1,{0x0033,0xffff}},  //  2537, depth=1
{0x7fffffffffe00000,   -1,{0xffff,0x1ae9}},  //  2538, depth=1
{0x01fff80000000000,   -1,{0xffff,0x154d}},  //  2539
{0x1c001c001c001c00, 2291,{0xffff,0x01a2}},  //  2540, depth=2
{0xffe000003fffffff,   -1,{0xffff,0x12e8}},  //  2541
{0x0fffffffff800000,   -1,{0xffff,0x1a64}},  //  2542, depth=1
{0xffffff80000001ff,   -1,{0xffff,0x1661}},  //  2543
{0x0000007fffffffc0,   -1,{0xffff,0x1ea0}},  //  2544, depth=1
{0x00000000fc01ffff, 2292,{0x0196,0xffff}},  //  2545, depth=2
{0x00000fffffffff80,   -1,{0xffff,0x1e64}},  //  2546, depth=1
{0xfff003fffff003ff,   -1,{0xffff,0x0315}},  //  2547, depth=1
{0xfffffc000000007f, 2549,{0xffff,0x159c}},  //  2548
{0xff00000007ffffff,   -1,{0xffff,0x1222}},  //  2549
{0x0000003c0000003c,   -1,{0xffff,0x0783}},  //  2550
{0xf00003fff00003ff,   -1,{0xffff,0x010d}},  //  2551
{0xfffe0007fffe0007,   -1,{0xffff,0x03d1}},  //  2552
{0xff81ffffff81ffff, 2293,{0xffff,0x0259}},  //  2553, depth=2
{0x00f000f000f000f0, 2294,{0xffff,0x0323}},  //  2554, depth=2
{0xfffffffe0000ffff,   -1,{0xffff,0x17ee}},  //  2555
{0x03fffffe00000000,   -1,{0xffff,0x17d8}},  //  2556
{0xffffffffffff03ff,   -1,{0xffff,0x1c39}},  //  2557, depth=1
{0x00000fff80000000,   -1,{0xffff,0x184c}},  //  2558
{0xfffffffffff800ff, 2560,{0xffff,0x1b74}},  //  2559
{0xfe0000ffffffffff,   -1,{0xffff,0x11ee}},  //  2560
{0x00000000c001c001, 2298,{0x00a2,0xffff}},  //  2561, depth=3
{0x000003e0000003e0,   -1,{0xffff,0x06c4}},  //  2562, depth=1
{0xfffc003fffffffff, 2565,{0xffff,0x13b3}},  //  2563
{0x0001fc0000000000,   -1,{0xffff,0x1586}},  //  2564, depth=1
{0xffe000001fffffff,   -1,{0xffff,0x12e7}},  //  2565
{0x0000000060606060,   -1,{0x00f1,0xffff}},  //  2566, depth=1
{0xfc0000000000003f,   -1,{0xffff,0x118b}},  //  2567
{0xfffffffffffff807, 2570,{0xffff,0x1d77}},  //  2568
{0xffe0ffffffe0ffff,   -1,{0xffff,0x02da}},  //  2569, depth=1
{0xffffffc1ffffffff, 2573,{0xffff,0x16ba}},  //  2570
{0x00000000fffc0fff,   -1,{0x0399,0xffff}},  //  2571, depth=1
{0x000000000ffffe00,   -1,{0x05d2,0x1dd2}},  //  2572, depth=1
{0xfffff1fffffff1ff,   -1,{0xffff,0x051c}},  //  2573
{0x7ff800007ff80000,   -1,{0xffff,0x034b}},  //  2574, depth=1
{0x0000ff8000000000,   -1,{0xffff,0x1648}},  //  2575
{0x00000000000ff800, 2303,{0x0548,0x1d48}},  //  2576, depth=2
{0xc000000fc000000f, 2313,{0xffff,0x0085}},  //  2577, depth=3
{0x0000003ffff00000,   -1,{0xffff,0x1b11}},  //  2578
{0xfff00007fff00007,   -1,{0xffff,0x030e}},  //  2579, depth=1
{0x00001ffc00001ffc,   -1,{0xffff,0x078a}},  //  2580, depth=1
{0xfffff00fffffffff,   -1,{0xffff,0x1537}},  //  2581, depth=1
{0x0000000003f803f8,   -1,{0x0366,0xffff}},  //  2582, depth=1
{0x7ffff00000000000,   -1,{0xffff,0x1512}},  //  2583
{0x0000000fffffc000,   -1,{0xffff,0x1c95}},  //  2584, depth=1
{0xffffff800001ffff,   -1,{0xffff,0x1669}},  //  2585
{0x000000fffff80000,   -1,{0xffff,0x1b54}},  //  2586
{0xf00000003fffffff,   -1,{0xffff,0x1121}},  //  2587, depth=1
{0x000000001e001e00,   -1,{0x01e3,0xffff}},  //  2588, depth=1
{0xffff000fffff000f,   -1,{0xffff,0x0413}},  //  2589, depth=1
{0x000003ff00000000,   -1,{0xffff,0x1809}},  //  2590, depth=1
{0x00003fff00003fff,   -1,{0xffff,0x000d}},  //  2591, depth=1
{0x001fffffff000000, 2324,{0xffff,0x1a1c}},  //  2592, depth=2
{0x80ffffff80ffffff,   -1,{0xffff,0x0058}},  //  2593, depth=1
{0x000ffffffff00000,   -1,{0xffff,0x1b1f}},  //  2594, depth=1
{0xfffc0000000003ff,   -1,{0xffff,0x1397}},  //  2595, depth=1
{0x0000000000e00000, 2327,{0x02c2,0x1ac2}},  //  2596, depth=2
{0xff00003fffffffff,   -1,{0xffff,0x122d}},  //  2597, depth=1
{0x0fffff800fffff80,   -1,{0xffff,0x0654}},  //  2598, depth=1
{0xfffffffffffff81f,   -1,{0xffff,0x1d79}},  //  2599
{0x0000007ffffffff8,   -1,{0xffff,0x1f63}},  //  2600, depth=1
{0x00000000e3ffe3ff,   -1,{0x00ec,0xffff}},  //  2601, depth=1
{0x000001c000000000,   -1,{0xffff,0x1682}},  //  2602
{0x3fff3fff3fff3fff,   -1,{0xffff,0x002d}},  //  2603, depth=1
{0x0000007ffffffffc,   -1,{0xffff,0x1fa4}},  //  2604, depth=1
{0xffffffffffe003ff, 2609,{0xffff,0x1af4}},  //  2605
{0x0000007ffffffffe,   -1,{0xffff,0x1fe5}},  //  2606, depth=1
{0x00000000fe07fe07, 2328,{0x01e9,0xffff}},  //  2607, depth=2
{0x3f003f003f003f00, 2330,{0xffff,0x0225}},  //  2608, depth=2
{0xf80003ffffffffff, 2613,{0xffff,0x116e}},  //  2609
{0x0e0e0e0e0e0e0e0e, 2341,{0xffff,0x01f2}},  //  2610, depth=2
{0xffc0000000007fff,   -1,{0xffff,0x1298}},  //  2611, depth=1
{0x00fffffffffffc00,   -1,{0xffff,0x1dad}},  //  2612, depth=1
{0x00000000fff07fff,   -1,{0x031a,0xffff}},  //  2613
{0xfffefffefffefffe,   -1,{0xffff,0x03ee}},  //  2614, depth=1
{0x00000000f03fffff, 2343,{0x0119,0xffff}},  //  2615, depth=3
{0xffe0000000000001,   -1,{0xffff,0x12cb}},  //  2616
{0xfffffffffff7ffff,   -1,{0xffff,0x1b3e}},  //  2617, depth=1
{0x0fffffff00000000, 2346,{0xffff,0x181b}},  //  2618, depth=2
{0xfefffefffefffeff, 2347,{0xffff,0x01ee}},  //  2619, depth=2
{0x0007fffffffff000,   -1,{0xffff,0x1d26}},  //  2620, depth=1
{0xfffffffffff80003,   -1,{0xffff,0x1b6e}},  //  2621, depth=1
{0x00007fffffffff80,   -1,{0xffff,0x1e67}},  //  2622, depth=1
{0xffe0007fffe0007f,   -1,{0xffff,0x02d1}},  //  2623, depth=1
{0xfe00fe00fe00fe00,   -1,{0xffff,0x01e6}},  //  2624, depth=1
{0xfffffffffff80007,   -1,{0xffff,0x1b6f}},  //  2625, depth=1
{0xfff000ffffffffff,   -1,{0xffff,0x1333}},  //  2626
{0x80000007ffffffff,   -1,{0xffff,0x1063}},  //  2627, depth=1
{0x0001ff800001ff80,   -1,{0xffff,0x0649}},  //  2628, depth=1
{0xfc0000007fffffff, 2351,{0xffff,0x11a4}},  //  2629, depth=2
{0x07ffffc000000000,   -1,{0xffff,0x1694}},  //  2630
{0x3fffffffffc00000,   -1,{0xffff,0x1aa7}},  //  2631
{0x00f0000000000000,   -1,{0xffff,0x1303}},  //  2632, depth=1
{0xf00000001fffffff, 2354,{0xffff,0x1120}},  //  2633, depth=2
{0x000000007fff0000,   -1,{0x040e,0x1c0e}},  //  2634, depth=1
{0xf0000000000000ff,   -1,{0xffff,0x110b}},  //  2635
{0x000f0000000f0000,   -1,{0xffff,0x0403}},  //  2636
{0xffffffffffffe001,   -1,{0xffff,0x1cf3}},  //  2637
{0xffffffffffffe003,   -1,{0xffff,0x1cf4}},  //  2638
{0xfffc3ffffffc3fff, 2356,{0xffff,0x039b}},  //  2639, depth=2
{0xffffffffffc00fff,   -1,{0xffff,0x1ab5}},  //  2640
{0x1ffffffff8000000, 2643,{0xffff,0x1961}},  //  2641
{0x00000fffffffffe0,   -1,{0xffff,0x1ee6}},  //  2642, depth=1
{0x000003fe00000000,   -1,{0xffff,0x17c8}},  //  2643
{0xffffffff03ffffff, 2650,{0xffff,0x1839}},  //  2644
{0xc007ffffffffffff,   -1,{0xffff,0x10b4}},  //  2645, depth=1
{0x00000000000fe000,   -1,{0x04c6,0x1cc6}},  //  2646, depth=1
{0xf0fffffff0ffffff, 2358,{0xffff,0x011b}},  //  2647, depth=2
{0x01fffffffff00000,   -1,{0xffff,0x1b24}},  //  2648, depth=1
{0xfffffffffff8001f,   -1,{0xffff,0x1b71}},  //  2649, depth=1
{0xffc00000000007ff,   -1,{0xffff,0x1294}},  //  2650
{0xe00003ffffffffff,   -1,{0xffff,0x10ec}},  //  2651, depth=1
{0x0fff00000fff0000, 2359,{0xffff,0x040b}},  //  2652, depth=2
{0x00000000007f007f,   -1,{0x0026,0xffff}},  //  2653, depth=1
{0x0000007fe0000000,   -1,{0xffff,0x18c9}},  //  2654, depth=1
{0xc0000000000001ff,   -1,{0xffff,0x108a}},  //  2655, depth=1
{0xffffffffff803fff,   -1,{0xffff,0x1a76}},  //  2656
{0xfff807ffffffffff,   -1,{0xffff,0x1377}},  //  2657, depth=1
{0x3fffffffffffff80, 2361,{0xffff,0x1e76}},  //  2658, depth=2
{0x7fffff8000000000,   -1,{0xffff,0x1657}},  //  2659
{0x00000fffff800000,   -1,{0xffff,0x1a54}},  //  2660
{0xff8000ffff8000ff, 2362,{0xffff,0x0250}},  //  2661, depth=2
{0x0000000ffc000000,   -1,{0xffff,0x1989}},  //  2662, depth=1
{0x3fffffff80000000,   -1,{0xffff,0x185e}},  //  2663
{0x03ffe00000000000,   -1,{0xffff,0x14cc}},  //  2664
{0xffffe0000fffffff,   -1,{0xffff,0x14ee}},  //  2665, depth=1
{0x00000ffffffffff8,   -1,{0xffff,0x1f68}},  //  2666, depth=1
{0xfc000000003fffff,   -1,{0xffff,0x119b}},  //  2667, depth=1
{0xffffffffffffe01f, 2677,{0xffff,0x1cf7}},  //  2668
{0xff800001ff800001, 2363,{0xffff,0x0249}},  //  2669, depth=2
{0x00000ffffffffffc,   -1,{0xffff,0x1fa9}},  //  2670, depth=1
{0xfff80ffffff80fff,   -1,{0xffff,0x0358}},  //  2671, depth=1
{0x00000000003f0000, 2364,{0x0405,0x1c05}},  //  2672, depth=2
{0x00000fffffffffff,   -1,{0xffff,0x102b}},  //  2673, depth=1
{0x0000100000000000,   -1,{0xffff,0x1500}},  //  2674, depth=1
{0xf800000000001fff,   -1,{0xffff,0x1151}},  //  2675, depth=1
{0x000001fff8000000,   -1,{0xffff,0x194d}},  //  2676, depth=1
{0xffffff07ffffffff,   -1,{0xffff,0x163a}},  //  2677
{0x00ffffe000000000,   -1,{0xffff,0x16d2}},  //  2678
{0xefffffffffffffff,   -1,{0xffff,0x10fe}},  //  2679, depth=1
{0x001c001c001c001c, 2366,{0xffff,0x03a2}},  //  2680, depth=2
{0xf000000000000001, 2369,{0xffff,0x1104}},  //  2681, depth=2
{0x0001000000010000, 2370,{0xffff,0x0400}},  //  2682, depth=2
{0xf000000000000003, 2371,{0xffff,0x1105}},  //  2683, depth=2
{0x007fe00000000000,   -1,{0xffff,0x14c9}},  //  2684, depth=1
{0xffffffffc3ffffff,   -1,{0xffff,0x18bb}},  //  2685, depth=1
{0x00007fffffffffc0,   -1,{0xffff,0x1ea8}},  //  2686, depth=1
{0xf000000000000007, 2373,{0xffff,0x1106}},  //  2687, depth=2
{0x007ffffffe000000,   -1,{0xffff,0x19dd}},  //  2688
{0xfffffe0000001fff,   -1,{0xffff,0x15e3}},  //  2689, depth=1
{0x07fffffff8000000,   -1,{0xffff,0x195f}},  //  2690, depth=1
{0x8f8f8f8f8f8f8f8f, 2374,{0xffff,0x0074}},  //  2691, depth=2
{0x00000007fffe0000,   -1,{0xffff,0x1bd1}},  //  2692, depth=1
{0xff00000000007fff,   -1,{0xffff,0x1216}},  //  2693
{0x4000000040000000, 2378,{0xffff,0x0080}},  //  2694, depth=2
{0xfe3ffffffe3fffff, 2379,{0xffff,0x01dc}},  //  2695, depth=4
{0x000000000001f000,   -1,{0x0504,0x1d04}},  //  2696, depth=1
{0x80ffffffffffffff,   -1,{0xffff,0x1078}},  //  2697
{0x0060000000600000,   -1,{0xffff,0x02c1}},  //  2698, depth=1
{0xe0003fffe0003fff, 2386,{0xffff,0x00d0}},  //  2699, depth=2
{0x0000007fc0000000, 2388,{0xffff,0x1888}},  //  2700, depth=2
{0xffe7ffffffe7ffff,   -1,{0xffff,0x02dd}},  //  2701, depth=1
{0xfe001fffffffffff,   -1,{0xffff,0x11f3}},  //  2702
{0xfe00000001ffffff,   -1,{0xffff,0x11df}},  //  2703, depth=1
{0x000000003ffc3ffc,   -1,{0x03ab,0xffff}},  //  2704, depth=1
{0x00000000ffff00ff, 2390,{0x0417,0xffff}},  //  2705, depth=3
{0x7fff00007fff0000, 2398,{0xffff,0x040e}},  //  2706, depth=2
{0xfffffbffffffffff,   -1,{0xffff,0x157e}},  //  2707, depth=1
{0xfffffc0000000000,   -1,{0xffff,0x1595}},  //  2708, depth=1
{0xfffffc0000000001,   -1,{0xffff,0x1596}},  //  2709, depth=1
{0x0000000006000000, 2400,{0x01c1,0x19c1}},  //  2710, depth=2
{0x00000000e01fffff, 2405,{0x00d7,0xffff}},  //  2711, depth=3
{0x003fffc000000000,   -1,{0xffff,0x168f}},  //  2712, depth=1
{0x00000000f80fffff,   -1,{0x0158,0xffff}},  //  2713, depth=1
{0x001fff0000000000,   -1,{0xffff,0x160c}},  //  2714
{0x8003ffffffffffff, 2411,{0xffff,0x1072}},  //  2715, depth=2
{0x03fffff803fffff8,   -1,{0xffff,0x0756}},  //  2716, depth=1
{0xffffffffe0003fff,   -1,{0xffff,0x18f0}},  //  2717, depth=1
{0x00007fffffffffe0,   -1,{0xffff,0x1ee9}},  //  2718, depth=1
{0xfffffffffffff87f, 2724,{0xffff,0x1d7b}},  //  2719
{0x00000fffe0000000,   -1,{0xffff,0x18ce}},  //  2720, depth=1
{0xc00000000001ffff, 2412,{0xffff,0x1092}},  //  2721, depth=2
{0x0700000007000000, 2413,{0xffff,0x0202}},  //  2722, depth=4
{0xfffffc000000000f,   -1,{0xffff,0x1599}},  //  2723, depth=1
{0xf80000000007ffff,   -1,{0xffff,0x1157}},  //  2724
{0x007ffe00007ffe00,   -1,{0xffff,0x05cd}},  //  2725
{0x000000003fff0000,   -1,{0x040d,0x1c0d}},  //  2726, depth=1
{0xfffc00000003ffff,   -1,{0xffff,0x139f}},  //  2727, depth=1
{0xff8000003fffffff,   -1,{0xffff,0x1266}},  //  2728
{0x00000000ff1fff1f,   -1,{0x022c,0xffff}},  //  2729
{0x00000000001ffff8,   -1,{0x0751,0x1f51}},  //  2730
{0xfffffffffc001fff,   -1,{0xffff,0x19b2}},  //  2731
{0x0003fff000000000,   -1,{0xffff,0x170d}},  //  2732
{0xfffffc0fffffffff,   -1,{0xffff,0x15b9}},  //  2733
{0x0000000001f80000, 2420,{0x0345,0x1b45}},  //  2734, depth=2
{0x00000000003f003f, 2421,{0x0025,0xffff}},  //  2735, depth=2
{0x0000008000000080, 2422,{0xffff,0x0640}},  //  2736, depth=2
{0xffc3ffc3ffc3ffc3, 2426,{0xffff,0x02ab}},  //  2737, depth=2
{0x0001fffffffff800,   -1,{0xffff,0x1d65}},  //  2738, depth=1
{0xfffffc000000001f,   -1,{0xffff,0x159a}},  //  2739, depth=1
{0x00001f8000000000,   -1,{0xffff,0x1645}},  //  2740, depth=1
{0xe00000000007ffff,   -1,{0xffff,0x10d5}},  //  2741, depth=1
{0x00007ffffffffff8,   -1,{0xffff,0x1f6b}},  //  2742, depth=1
{0xf00000000000003f, 2428,{0xffff,0x1109}},  //  2743, depth=2
{0x001ffffe00000000,   -1,{0xffff,0x17d3}},  //  2744, depth=1
{0xf8000000000007ff, 2429,{0xffff,0x114f}},  //  2745, depth=2
{0x000007ffffff8000, 2436,{0xffff,0x1c5b}},  //  2746, depth=2
{0xffffff00001fffff,   -1,{0xffff,0x162c}},  //  2747, depth=1
{0x03fffc0003fffc00, 2438,{0xffff,0x058f}},  //  2748, depth=2
{0x00007fffffffffff, 2439,{0xffff,0x102e}},  //  2749, depth=2
{0x0000800000000000,   -1,{0xffff,0x1440}},  //  2750, depth=1
{0xfffe007fffffffff,   -1,{0xffff,0x13f5}},  //  2751, depth=1
{0x00001ffffff80000, 2755,{0xffff,0x1b59}},  //  2752
{0x800000000000ffff,   -1,{0xffff,0x1050}},  //  2753, depth=1
{0x0000007fffc00000, 2440,{0xffff,0x1a90}},  //  2754, depth=2
{0xff800000ff800000,   -1,{0xffff,0x0248}},  //  2755
{0xe000007fffffffff,   -1,{0xffff,0x10e9}},  //  2756
{0x00000000800f800f, 2442,{0x0064,0xffff}},  //  2757, depth=4
{0x003fe000003fe000, 2454,{0xffff,0x04c8}},  //  2758, depth=2
{0xffffc0000007ffff, 2458,{0xffff,0x14a4}},  //  2759, depth=2
{0x0007ffffffffc000,   -1,{0xffff,0x1ca4}},  //  2760, depth=1
{0x00000000efffffff,   -1,{0x00de,0xffff}},  //  2761, depth=1
{0x00000000f0000000,   -1,{0x0103,0x1903}},  //  2762, depth=1
{0x00000000f0000001, 2459,{0x0104,0xffff}},  //  2763, depth=2
{0x00000003ffffe000,   -1,{0xffff,0x1cd4}},  //  2764, depth=1
{0x00000000f0000003,   -1,{0x0105,0xffff}},  //  2765, depth=1
{0x00000fffc0000000,   -1,{0xffff,0x188d}},  //  2766, depth=1
{0xfc0000001fffffff, 2462,{0xffff,0x11a2}},  //  2767, depth=2
{0x07ffff8007ffff80,   -1,{0xffff,0x0653}},  //  2768
{0x00000000f0000007,   -1,{0x0106,0xffff}},  //  2769, depth=1
{0x0000001ffffff000, 2464,{0xffff,0x1d18}},  //  2770, depth=2
{0xfffffc000000003f,   -1,{0xffff,0x159b}},  //  2771, depth=1
{0x000000001fff0000,   -1,{0x040c,0x1c0c}},  //  2772, depth=1
{0x00000000fffff3ff,   -1,{0x051d,0xffff}},  //  2773, depth=1
{0x00fff80000fff800, 2467,{0xffff,0x054c}},  //  2774, depth=2
{0xffe00000003fffff,   -1,{0xffff,0x12e0}},  //  2775, depth=1
{0x00000000fefefefe, 2468,{0x01f6,0xffff}},  //  2776, depth=2
{0x00000000f000000f, 2470,{0x0107,0xffff}},  //  2777, depth=4
{0x00000000001fff00, 2478,{0x060c,0x1e0c}},  //  2778, depth=2
{0x00000000ffdfffdf,   -1,{0x02ae,0xffff}},  //  2779, depth=1
{0x0003fffffffe0000,   -1,{0xffff,0x1be0}},  //  2780, depth=1
{0x00007fffff800000,   -1,{0xffff,0x1a57}},  //  2781
{0x3ffffffffffffffc,   -1,{0xffff,0x1fbb}},  //  2782, depth=1
{0xffe000000000007f,   -1,{0xffff,0x12d1}},  //  2783
{0x00003fffffff8000, 2482,{0xffff,0x1c5e}},  //  2784, depth=2
{0x3fffffffffffffff,   -1,{0xffff,0x103d}},  //  2785, depth=1
{0x4000000000000000,   -1,{0xffff,0x1080}},  //  2786, depth=1
{0x800003ff800003ff,   -1,{0xffff,0x004a}},  //  2787, depth=1
{0xffffffffffe3ffff,   -1,{0xffff,0x1afc}},  //  2788
{0x000000000fffffe0,   -1,{0x06d6,0x1ed6}},  //  2789
{0x01ffc00000000000,   -1,{0xffff,0x148a}},  //  2790, depth=1
{0xc0000fffc0000fff,   -1,{0xffff,0x008d}},  //  2791, depth=1
{0x0000007f80000000,   -1,{0xffff,0x1847}},  //  2792, depth=1
{0x00000000f000001f,   -1,{0x0108,0xffff}},  //  2793, depth=1
{0x000000000ffffff0,   -1,{0x0717,0x1f17}},  //  2794
{0xc000001fffffffff,   -1,{0xffff,0x10a6}},  //  2795, depth=1
{0x00007fffe0000000,   -1,{0xffff,0x18d1}},  //  2796, depth=1
{0xfffff8000007ffff, 2483,{0xffff,0x1567}},  //  2797, depth=2
{0x00e000e000e000e0,   -1,{0xffff,0x02e2}},  //  2798, depth=1
{0xfff8001ffff8001f,   -1,{0xffff,0x0351}},  //  2799, depth=1
{0xfc000000000000ff,   -1,{0xffff,0x118d}},  //  2800
{0x07fffffffffc0000,   -1,{0xffff,0x1ba8}},  //  2801
{0x1fffffffffff8000,   -1,{0xffff,0x1c6d}},  //  2802, depth=1
{0xffff0fffffff0fff, 2487,{0xffff,0x041b}},  //  2803, depth=2
{0x3e0000003e000000,   -1,{0xffff,0x01c4}},  //  2804, depth=1
{0xfc3ffffffc3fffff, 2490,{0xffff,0x019b}},  //  2805, depth=2
{0x6060606060606060, 2491,{0xffff,0x00f1}},  //  2806, depth=2
{0x00000000cfffffff, 2493,{0x009d,0xffff}},  //  2807, depth=2
{0x0001ffffffffe000,   -1,{0xffff,0x1ce3}},  //  2808, depth=1
{0x000000000ffffffc,   -1,{0x0799,0x1f99}},  //  2809
{0xffffe001ffffe001,   -1,{0xffff,0x04d3}},  //  2810
{0x00000000ffdfffff, 2500,{0x029e,0xffff}},  //  2811, depth=2
{0x00000000ffe00000,   -1,{0x02ca,0x1aca}},  //  2812, depth=1
{0x00000000ffe00001, 2501,{0x02cb,0xffff}},  //  2813, depth=2
{0x0008000000080000, 2503,{0xffff,0x0340}},  //  2814, depth=2
{0x00000000ffe00003, 2507,{0x02cc,0xffff}},  //  2815, depth=3
{0x007fff0000000000,   -1,{0xffff,0x160e}},  //  2816, depth=1
{0xff8fffffff8fffff, 2509,{0xffff,0x025c}},  //  2817, depth=4
{0x001fc000001fc000,   -1,{0xffff,0x0486}},  //  2818, depth=1
{0x8000000780000007, 2516,{0xffff,0x0043}},  //  2819, depth=3
{0x0000000000e000e0, 2518,{0x02e2,0xffff}},  //  2820, depth=2
{0xfc00000000000007,   -1,{0xffff,0x1188}},  //  2821, depth=1
{0x7fff800000000000,   -1,{0xffff,0x144f}},  //  2822, depth=1
{0x0001ffffffff8000, 2824,{0xffff,0x1c61}},  //  2823
{0x000000007ff07ff0,   -1,{0x032a,0xffff}},  //  2824
{0x00000000f000003f, 2519,{0x0109,0xffff}},  //  2825, depth=2
{0x000000007c007c00, 2524,{0x01a4,0xffff}},  //  2826, depth=3
{0x00000000ffe0000f, 2531,{0x02ce,0xffff}},  //  2827, depth=2
{0x000000000fffff00, 2539,{0x0613,0x1e13}},  //  2828, depth=2
{0xfc0000000000000f, 2541,{0xffff,0x1189}},  //  2829, depth=2
{0x03ff000000000000,   -1,{0xffff,0x1409}},  //  2830, depth=1
{0xffff801fffffffff,   -1,{0xffff,0x1475}},  //  2831, depth=1
{0x3fffffffe0000000,   -1,{0xffff,0x18e0}},  //  2832, depth=1
{0xfffff0000001ffff, 2543,{0xffff,0x1524}},  //  2833, depth=2
{0x0fffffffff000000,   -1,{0xffff,0x1a23}},  //  2834, depth=1
{0x00000000f801f801, 2548,{0x0165,0xffff}},  //  2835, depth=3
{0x000000000001c000, 2550,{0x0482,0x1c82}},  //  2836, depth=2
{0xfff80000000fffff,   -1,{0xffff,0x1360}},  //  2837, depth=1
{0x0007fffffc000000,   -1,{0xffff,0x1998}},  //  2838, depth=1
{0xffffffffff800fff, 2840,{0xffff,0x1a74}},  //  2839
{0xe0000fffffffffff,   -1,{0xffff,0x10ee}},  //  2840
{0x0000000007070707, 2551,{0x0032,0xffff}},  //  2841, depth=2
{0x00007fffc0000000,   -1,{0xffff,0x1890}},  //  2842, depth=1
{0x00000000ffe0001f, 2552,{0x02cf,0xffff}},  //  2843, depth=2
{0x0ffc0ffc0ffc0ffc,   -1,{0xffff,0x03a9}},  //  2844, depth=1
{0xfc0000000000001f,   -1,{0xffff,0x118a}},  //  2845, depth=1
{0x0003fffffffffe00,   -1,{0xffff,0x1de8}},  //  2846, depth=1
{0x00000000fe0007ff, 2555,{0x01d1,0xffff}},  //  2847, depth=2
{0xffffffe001ffffff,   -1,{0xffff,0x16f3}},  //  2848
{0x00000000801fffff,   -1,{0x0055,0xffff}},  //  2849, depth=1
{0x0fffc0000fffc000,   -1,{0xffff,0x048d}},  //  2850
{0xffffe007ffffffff,   -1,{0xffff,0x14f5}},  //  2851, depth=1
{0x00000000fcfcfcfc, 2556,{0x01b5,0xffff}},  //  2852, depth=2
{0xff7fffffffffffff,   -1,{0xffff,0x123e}},  //  2853
{0x0000000003fffc00,   -1,{0x058f,0x1d8f}},  //  2854, depth=1
{0xffffffff80003fff,   -1,{0xffff,0x186e}},  //  2855, depth=1
{0xfffc000007ffffff,   -1,{0xffff,0x13a8}},  //  2856
{0xffffffffe0000fff,   -1,{0xffff,0x18ee}},  //  2857, depth=1
{0x01fffc0001fffc00, 2558,{0xffff,0x058e}},  //  2858, depth=2
{0xff80000000000007, 2862,{0xffff,0x124b}},  //  2859
{0x003ffffffffc0000,   -1,{0xffff,0x1ba3}},  //  2860, depth=1
{0x00000000ff9fff9f,   -1,{0x026d,0xffff}},  //  2861, depth=1
{0xfffffff3fffffff3,   -1,{0xffff,0x071d}},  //  2862
{0x00f8000000000000, 2865,{0xffff,0x1344}},  //  2863
{0x0c000c000c000c00,   -1,{0xffff,0x01a1}},  //  2864, depth=1
{0x000000001fe00000,   -1,{0x02c7,0x1ac7}},  //  2865
{0xffc003ffffffffff, 2867,{0xffff,0x12b3}},  //  2866
{0xfe000001ffffffff,   -1,{0xffff,0x11e7}},  //  2867
{0xfc00000fffffffff,   -1,{0xffff,0x11a9}},  //  2868
{0xfffffffff00003ff,   -1,{0xffff,0x192d}},  //  2869, depth=1
{0x000007c000000000, 2871,{0xffff,0x1684}},  //  2870
{0x000000000000ff00,   -1,{0x0607,0x1e07}},  //  2871
{0xfe00ffffffffffff,   -1,{0xffff,0x11f6}},  //  2872
{0xf0000007f0000007, 2559,{0xffff,0x0106}},  //  2873, depth=3
{0x001fe00000000000,   -1,{0xffff,0x14c7}},  //  2874, depth=1
{0x00000000ffe0003f, 2563,{0x02d0,0xffff}},  //  2875, depth=3
{0xc0000000000003ff,   -1,{0xffff,0x108b}},  //  2876
{0xffffff9fffffff9f, 2567,{0xffff,0x065d}},  //  2877, depth=2
{0x3fffffffc0000000,   -1,{0xffff,0x189f}},  //  2878, depth=1
{0x01fffffffffff000,   -1,{0xffff,0x1d2c}},  //  2879
{0xfffe0000001fffff,   -1,{0xffff,0x13e3}},  //  2880
{0xfffffffffffff7ff,   -1,{0xffff,0x1d3e}},  //  2881, depth=1
{0xfffffffffffff800,   -1,{0xffff,0x1d74}},  //  2882, depth=1
{0xfffffffffffff801,   -1,{0xffff,0x1d75}},  //  2883, depth=1
{0x000000000000fe00,   -1,{0x05c6,0x1dc6}},  //  2884, depth=1
{0xfffffffffffff803,   -1,{0xffff,0x1d76}},  //  2885, depth=1
{0xffffffff00000fff,   -1,{0xffff,0x182b}},  //  2886
{0xe1e1e1e1e1e1e1e1,   -1,{0xffff,0x00f3}},  //  2887, depth=1
{0x0000000000600000,   -1,{0x02c1,0x1ac1}},  //  2888, depth=1
{0x00000000f000007f, 2568,{0x010a,0xffff}},  //  2889, depth=4
{0xf007ffffffffffff,   -1,{0xffff,0x1136}},  //  2890
{0xffc0000000001fff,   -1,{0xffff,0x1296}},  //  2891, depth=1
{0x000fff80000fff80, 2575,{0xffff,0x064c}},  //  2892, depth=2
{0xfffe7fffffffffff,   -1,{0xffff,0x13fd}},  //  2893, depth=1
{0x0606060606060606, 2578,{0xffff,0x01f1}},  //  2894, depth=2
{0x0000007fff000000,   -1,{0xffff,0x1a0e}},  //  2895
{0x00007fffffc00000,   -1,{0xffff,0x1a98}},  //  2896, depth=1
{0xfffffffffffff80f,   -1,{0xffff,0x1d78}},  //  2897, depth=1
{0x0000000030003000, 2583,{0x0121,0xffff}},  //  2898, depth=2
{0x000000008fffffff, 2585,{0x005c,0xffff}},  //  2899, depth=2
{0x0000007fff800000,   -1,{0xffff,0x1a4f}},  //  2900, depth=1
{0x000001fffffffc00,   -1,{0xffff,0x1d9e}},  //  2901
{0x000007fff8000000,   -1,{0xffff,0x194f}},  //  2902, depth=1
{0xffffffffc0000fff,   -1,{0xffff,0x18ad}},  //  2903, depth=1
{0x000003f0000003f0, 2586,{0xffff,0x0705}},  //  2904, depth=2
{0xfffff003fffff003,   -1,{0xffff,0x0515}},  //  2905, depth=1
{0x00000000001fff80,   -1,{0x064d,0x1e4d}},  //  2906, depth=1
{0xfffff01fffffffff,   -1,{0xffff,0x1538}},  //  2907, depth=1
{0xfffffff807ffffff,   -1,{0xffff,0x1777}},  //  2908
{0xfff0ffffffffffff, 2911,{0xffff,0x133b}},  //  2909
{0x0000001fffffc000,   -1,{0xffff,0x1c96}},  //  2910, depth=1
{0xe000007fe000007f,   -1,{0xffff,0x00c9}},  //  2911
{0xfffe00000fffffff,   -1,{0xffff,0x13ea}},  //  2912
{0x00000000ffffc3ff, 2599,{0x049b,0xffff}},  //  2913, depth=2
{0x3c0000003c000000, 2602,{0xffff,0x0183}},  //  2914, depth=2
{0x0003ffffffffff80,   -1,{0xffff,0x1e6a}},  //  2915
{0xffffffffffff8003,   -1,{0xffff,0x1c72}},  //  2916
{0xfffffffffe1fffff,   -1,{0xffff,0x19fb}},  //  2917, depth=1
{0x00000007ffffffe0, 2920,{0xffff,0x1edd}},  //  2918
{0xc000001fc000001f, 2605,{0xffff,0x0086}},  //  2919, depth=4
{0x000000ffffffe000,   -1,{0xffff,0x1cda}},  //  2920
{0xffdfffffffffffff,   -1,{0xffff,0x12be}},  //  2921, depth=1
{0xffe0000000000000,   -1,{0xffff,0x12ca}},  //  2922, depth=1
{0x003f003f003f003f, 2616,{0xffff,0x0025}},  //  2923, depth=2
{0x000c000c000c000c,   -1,{0xffff,0x03a1}},  //  2924, depth=1
{0xffe0000000000003,   -1,{0xffff,0x12cc}},  //  2925, depth=1
{0x00000000000f8000,   -1,{0x0444,0x1c44}},  //  2926, depth=1
{0xff8000007fffffff, 2626,{0xffff,0x1267}},  //  2927, depth=2
{0x1f8000001f800000, 2630,{0xffff,0x0245}},  //  2928, depth=2
{0xffe0000000000007,   -1,{0xffff,0x12cd}},  //  2929, depth=1
{0xffffffffffff8007,   -1,{0xffff,0x1c73}},  //  2930
{0xffff001fffff001f,   -1,{0xffff,0x0414}},  //  2931, depth=1
{0x0000000000003c00, 2631,{0x0583,0x1d83}},  //  2932, depth=2
{0x0001fffff8000000,   -1,{0xffff,0x1955}},  //  2933
{0x00007fff80000000,   -1,{0xffff,0x184f}},  //  2934, depth=1
{0xfffffe7ffffffe7f, 2635,{0xffff,0x05dd}},  //  2935, depth=2
{0x007e000000000000,   -1,{0xffff,0x13c5}},  //  2936
{0xffe000000000000f,   -1,{0xffff,0x12ce}},  //  2937, depth=1
{0xffffffffffff800f,   -1,{0xffff,0x1c74}},  //  2938
{0x00000000ffe0007f,   -1,{0x02d1,0xffff}},  //  2939, depth=1
{0x00003ffff8000000,   -1,{0xffff,0x1952}},  //  2940, depth=1
{0xfc0000000000007f,   -1,{0xffff,0x118c}},  //  2941, depth=1
{0x0300000003000000,   -1,{0xffff,0x0201}},  //  2942, depth=1
{0x01c0000000000000,   -1,{0xffff,0x1282}},  //  2943
{0xffffffc0ffffffff, 2948,{0xffff,0x16b9}},  //  2944
{0xfffffffffffff83f,   -1,{0xffff,0x1d7a}},  //  2945, depth=1
{0x0000000070000000, 2636,{0x0102,0x1902}},  //  2946, depth=2
{0x00000000cfffcfff,   -1,{0x00ad,0xffff}},  //  2947, depth=1
{0xf00000000001ffff,   -1,{0xffff,0x1114}},  //  2948
{0x00000fffff000000,   -1,{0xffff,0x1a13}},  //  2949
{0xfff8000001ffffff,   -1,{0xffff,0x1365}},  //  2950
{0xffffffffffffdfff,   -1,{0xffff,0x1cbe}},  //  2951, depth=1
{0xffffffffffffe000,   -1,{0xffff,0x1cf2}},  //  2952, depth=1
{0xffe000000000001f, 2637,{0xffff,0x12cf}},  //  2953, depth=2
{0x00000007fffffff8,   -1,{0xffff,0x1f5f}},  //  2954
{0xfff80003ffffffff, 2638,{0xffff,0x136e}},  //  2955, depth=2
{0x000000000fffff80,   -1,{0x0654,0x1e54}},  //  2956, depth=1
{0x00000000ff9fffff, 2640,{0x025d,0xffff}},  //  2957, depth=2
{0x00003ffe00003ffe, 2641,{0xffff,0x07cc}},  //  2958, depth=3
{0xffffffffffffe007,   -1,{0xffff,0x1cf5}},  //  2959, depth=1
{0x07f8000007f80000,   -1,{0xffff,0x0347}},  //  2960, depth=1
{0x00000000ffff01ff, 2644,{0x0418,0xffff}},  //  2961, depth=3
{0xffffffffffff801f,   -1,{0xffff,0x1c75}},  //  2962
{0xfffffc00000000ff, 2656,{0xffff,0x159d}},  //  2963, depth=2
{0x0fff000000000000, 2659,{0xffff,0x140b}},  //  2964, depth=2
{0xff800000003fffff,   -1,{0xffff,0x125e}},  //  2965, depth=1
{0x00003f0000003f00, 2660,{0xffff,0x0605}},  //  2966, depth=2
{0xffffffffffffe00f,   -1,{0xffff,0x1cf6}},  //  2967, depth=1
{0x00000007fffffffe,   -1,{0xffff,0x1fe1}},  //  2968
{0xfffc000000007fff, 2971,{0xffff,0x139c}},  //  2969
{0x00000000001fffc0, 2663,{0x068e,0x1e8e}},  //  2970, depth=2
{0x00000007ffffffff,   -1,{0xffff,0x1022}},  //  2971
{0x0000000800000000,   -1,{0xffff,0x1740}},  //  2972
{0xffffff9fffffffff,   -1,{0xffff,0x167d}},  //  2973, depth=1
{0xfffffffc0fffffff,   -1,{0xffff,0x17b9}},  //  2974
{0x8787878787878787,   -1,{0xffff,0x0073}},  //  2975, depth=1
{0x0000007f00000000,   -1,{0xffff,0x1806}},  //  2976, depth=1
{0xffffe0000000ffff,   -1,{0xffff,0x14e2}},  //  2977, depth=1
{0xfe007ffffe007fff,   -1,{0xffff,0x01d5}},  //  2978
{0x0000fc0000000000,   -1,{0xffff,0x1585}},  //  2979
{0xfffffffffe00ffff, 2981,{0xffff,0x19f6}},  //  2980
{0xfffff000000003ff,   -1,{0xffff,0x151d}},  //  2981
{0xff80ff80ff80ff80, 2664,{0xffff,0x0268}},  //  2982, depth=2
{0xffffc7ffffffc7ff, 2668,{0xffff,0x049c}},  //  2983, depth=3
{0x0000000000600060, 2678,{0x02e1,0xffff}},  //  2984, depth=2
{0xffe000000000003f,   -1,{0xffff,0x12d0}},  //  2985, depth=1
{0x0ffffffe00000000, 2688,{0xffff,0x17da}},  //  2986, depth=2
{0x03fffffffe000000,   -1,{0xffff,0x19e0}},  //  2987
{0x0000001ffc000000,   -1,{0xffff,0x198a}},  //  2988, depth=1
{0xff001fffff001fff,   -1,{0xffff,0x0214}},  //  2989, depth=1
{0x00c000c000c000c0,   -1,{0xffff,0x02a1}},  //  2990
{0x00000000fc0003ff, 2693,{0x018f,0xffff}},  //  2991, depth=2
{0x003ffffffffffc00,   -1,{0xffff,0x1dab}},  //  2992, depth=1
{0x0000000003030303, 2697,{0x0031,0xffff}},  //  2993, depth=2
{0x007ffffc007ffffc,   -1,{0xffff,0x0794}},  //  2994, depth=1
{0xffffffff80000fff,   -1,{0xffff,0x186c}},  //  2995, depth=1
{0x000000007fe00000,   -1,{0x02c9,0x1ac9}},  //  2996, depth=1
{0x007ffffc00000000,   -1,{0xffff,0x1794}},  //  2997
{0x0000000001fff800,   -1,{0x054d,0x1d4d}},  //  2998, depth=1
{0xfff9ffffffffffff,   -1,{0xffff,0x137d}},  //  2999, depth=1
{0x000ffe0000000000,   -1,{0xffff,0x15ca}},  //  3000, depth=1
{0xfffffffff003ffff,   -1,{0xffff,0x1935}},  //  3001, depth=1
{0x00000000001fffe0,   -1,{0x06cf,0x1ecf}},  //  3002, depth=1
{0x00001ffffffff800,   -1,{0xffff,0x1d61}},  //  3003
{0x00000000f8f8f8f8,   -1,{0x0174,0xffff}},  //  3004, depth=1
{0xf000000fffffffff, 2702,{0xffff,0x1127}},  //  3005, depth=2
{0x07fc07fc07fc07fc, 2714,{0xffff,0x03a8}},  //  3006, depth=2
{0xff1fffffffffffff,   -1,{0xffff,0x123c}},  //  3007
{0xf800007fffffffff,   -1,{0xffff,0x116b}},  //  3008
{0x00000000fffe3fff, 2719,{0x03dc,0xffff}},  //  3009, depth=3
{0x0000000007fe07fe, 2725,{0x03e9,0xffff}},  //  3010, depth=2
{0x01ffffffffffc000,   -1,{0xffff,0x1caa}},  //  3011
{0x0030000000000000,   -1,{0xffff,0x1301}},  //  3012, depth=1
{0xfffffff0000fffff,   -1,{0xffff,0x172f}},  //  3013, depth=1
{0x00000007ffffff00,   -1,{0xffff,0x1e1a}},  //  3014, depth=1
{0xffffffffffffe03f,   -1,{0xffff,0x1cf8}},  //  3015, depth=1
{0xfffffe7fffffffff,   -1,{0xffff,0x15fd}},  //  3016
{0x00000000f00000ff,   -1,{0x010b,0xffff}},  //  3017, depth=1
{0x00000000001ffff0,   -1,{0x0710,0x1f10}},  //  3018, depth=1
{0x0000000083ff83ff, 2728,{0x006a,0xffff}},  //  3019, depth=2
{0x000000000fffffc0,   -1,{0x0695,0x1e95}},  //  3020, depth=1
{0xffffffffffffc7ff,   -1,{0xffff,0x1cbc}},  //  3021, depth=1
{0x0003ffffffffffe0,   -1,{0xffff,0x1eec}},  //  3022
{0xfffffe000007ffff,   -1,{0xffff,0x15e9}},  //  3023, depth=1
{0xfe03fffffe03ffff,   -1,{0xffff,0x01d8}},  //  3024
{0xf83ffffff83fffff, 2729,{0xffff,0x015a}},  //  3025, depth=2
{0x0400040004000400, 2730,{0xffff,0x01a0}},  //  3026, depth=2
{0xfff0000000ffffff,   -1,{0xffff,0x1323}},  //  3027
{0x0003fffffffffff0,   -1,{0xffff,0x1f2d}},  //  3028
{0xdddddddddddddddd, 2731,{0xffff,0x00ba}},  //  3029, depth=2
{0x00000000001ffffc,   -1,{0x0792,0x1f92}},  //  3030, depth=1
{0x1ffffffffffc0000,   -1,{0xffff,0x1baa}},  //  3031
{0x00000000001ffffe, 2732,{0x07d3,0x1fd3}},  //  3032, depth=2
{0x00000000001fffff, 2733,{0x0014,0x1014}},  //  3033, depth=2
{0x0000000000200000,   -1,{0x02c0,0x1ac0}},  //  3034, depth=1
{0x0003fffffffffff8,   -1,{0xffff,0x1f6e}},  //  3035
{0x0202020202020202, 2752,{0xffff,0x01f0}},  //  3036, depth=3
{0xc3ffc3ffc3ffc3ff, 2756,{0xffff,0x00ab}},  //  3037, depth=2
{0x0000000030000000, 2768,{0x0101,0x1901}},  //  3038, depth=2
{0xffffffff00003fff,   -1,{0xffff,0x182d}},  //  3039, depth=1
{0x0003fffffffffffe,   -1,{0xffff,0x1ff0}},  //  3040
{0xffffffc000ffffff,   -1,{0xffff,0x16b1}},  //  3041, depth=1
{0x00000fff00000000, 2781,{0xffff,0x180b}},  //  3042, depth=2
{0xfffffff800007fff,   -1,{0xffff,0x176b}},  //  3043
{0x00000003ffff8000,   -1,{0xffff,0x1c52}},  //  3044, depth=1
{0xffffffffffff807f, 3048,{0xffff,0x1c77}},  //  3045
{0x0004000400040004,   -1,{0xffff,0x03a0}},  //  3046, depth=1
{0xffffc00fffffc00f,   -1,{0xffff,0x0495}},  //  3047, depth=1
{0xfffffc1fffffffff, 3050,{0xffff,0x15ba}},  //  3048
{0x0000ffff0000ffff, 2783,{0xffff,0x000f}},  //  3049, depth=2
{0xffff1fffffff1fff,   -1,{0xffff,0x041c}},  //  3050
{0xfffffc3ffffffc3f, 2788,{0xffff,0x059b}},  //  3051, depth=2
{0x0100000001000000, 2789,{0xffff,0x0200}},  //  3052, depth=2
{0x00000007c0000000,   -1,{0xffff,0x1884}},  //  3053
{0x01fffe0000000000,   -1,{0xffff,0x15cf}},  //  3054, depth=1
{0xffffc07fffffffff,   -1,{0xffff,0x14b8}},  //  3055, depth=1
{0x003fffe000000000,   -1,{0xffff,0x16d0}},  //  3056
{0xc003c003c003c003,   -1,{0xffff,0x00a3}},  //  3057, depth=1
{0x00fffffffffe0000,   -1,{0xffff,0x1be6}},  //  3058, depth=1
{0xfff01fffffffffff,   -1,{0xffff,0x1338}},  //  3059, depth=1
{0x00001fffffffe000,   -1,{0xffff,0x1cdf}},  //  3060
{0x3fffffff3fffffff,   -1,{0xffff,0x001d}},  //  3061, depth=1
{0x000001fffffc0000,   -1,{0xffff,0x1b96}},  //  3062, depth=1
{0x00ffffffffffff00,   -1,{0xffff,0x1e2f}},  //  3063
{0xffffffffe00fffff,   -1,{0xffff,0x18f6}},  //  3064
{0xff8000001fffffff,   -1,{0xffff,0x1265}},  //  3065, depth=1
{0x0000000000200020,   -1,{0x02e0,0xffff}},  //  3066, depth=1
{0x00000000ffe000ff,   -1,{0x02d2,0xffff}},  //  3067, depth=1
{0x0000000001ffe000, 2794,{0x04cb,0x1ccb}},  //  3068, depth=2
{0x0000000001010101, 2800,{0x0030,0xffff}},  //  3069, depth=2
{0xfffffffff8000000,   -1,{0xffff,0x1964}},  //  3070
{0xfffffffff8000001, 3074,{0xffff,0x1965}},  //  3071
{0x0000000000000c00,   -1,{0x0581,0x1d81}},  //  3072, depth=1
{0x00000000e000ffff,   -1,{0x00d2,0xffff}},  //  3073, depth=1
{0xffffc01fffffc01f,   -1,{0xffff,0x0496}},  //  3074
{0xffffe0ffffffe0ff,   -1,{0xffff,0x04da}},  //  3075, depth=1
{0x000000000ffffff8, 2801,{0x0758,0x1f58}},  //  3076, depth=2
{0xfffffffff8000003,   -1,{0xffff,0x1966}},  //  3077
{0x3fffffffff800000,   -1,{0xffff,0x1a66}},  //  3078, depth=1
{0xffffffffffffe07f,   -1,{0xffff,0x1cf9}},  //  3079, depth=1
{0x00007fc000007fc0, 2809,{0xffff,0x0688}},  //  3080, depth=2
{0x01fffffffc000000,   -1,{0xffff,0x199e}},  //  3081
{0x000000000ffffffe,   -1,{0x07da,0x1fda}},  //  3082, depth=1
{0x000000000fffffff, 2810,{0x001b,0x101b}},  //  3083, depth=2
{0x0000000010000000,   -1,{0x0100,0x1900}},  //  3084, depth=1
{0xffffffffff000fff, 3087,{0xffff,0x1a33}},  //  3085
{0x1fffffc01fffffc0,   -1,{0xffff,0x0696}},  //  3086, depth=1
{0xfffffffff8000007,   -1,{0xffff,0x1967}},  //  3087
{0x000000003fe00000, 2823,{0x02c8,0x1ac8}},  //  3088, depth=3
{0xfffffffff800000f,   -1,{0xffff,0x1968}},  //  3089
{0x0003ffffe0000000,   -1,{0xffff,0x18d4}},  //  3090
{0xfffff0000003ffff, 3094,{0xffff,0x1525}},  //  3091
{0x07ff000007ff0000,   -1,{0xffff,0x040a}},  //  3092, depth=1
{0xfffff87fffffffff,   -1,{0xffff,0x157b}},  //  3093, depth=1
{0x803fffff803fffff,   -1,{0xffff,0x0056}},  //  3094
{0xfffffffff87fffff,   -1,{0xffff,0x197b}},  //  3095, depth=1
{0x00000007ffc00000,   -1,{0xffff,0x1a8c}},  //  3096
{0xfffcfffffffcffff,   -1,{0xffff,0x039d}},  //  3097, depth=1
{0xfffffffff800001f,   -1,{0xffff,0x1969}},  //  3098
{0xfffffffffc0007ff,   -1,{0xffff,0x19b0}},  //  3099, depth=1
{0x0000003ffffff000,   -1,{0xffff,0x1d19}},  //  3100
{0x001e000000000000,   -1,{0xffff,0x13c3}},  //  3101
{0x0003ffffffffff00,   -1,{0xffff,0x1e29}},  //  3102, depth=1
{0x0000007f0000007f, 2839,{0xffff,0x0006}},  //  3103, depth=3
{0xf8007fffffffffff,   -1,{0xffff,0x1173}},  //  3104
{0x0003ffffc0000000,   -1,{0xffff,0x1893}},  //  3105
{0x0000000ff0000000,   -1,{0xffff,0x1907}},  //  3106
{0xffffffffffffe1ff,   -1,{0xffff,0x1cfb}},  //  3107
{0xfc00001fffffffff,   -1,{0xffff,0x11aa}},  //  3108
{0xffffff000000ffff, 2848,{0xffff,0x1627}},  //  3109, depth=2
{0x00000000ffc0ffc0, 2850,{0x02a9,0xffff}},  //  3110, depth=2
{0x00000000fe07ffff, 2853,{0x01d9,0xffff}},  //  3111, depth=2
{0xff80000000000000,   -1,{0xffff,0x1248}},  //  3112, depth=1
{0xff80000000000001, 2856,{0xffff,0x1249}},  //  3113, depth=2
{0x03ffff0000000000,   -1,{0xffff,0x1611}},  //  3114, depth=1
{0xff80000000000003,   -1,{0xffff,0x124a}},  //  3115, depth=1
{0x0fffe00000000000,   -1,{0xffff,0x14ce}},  //  3116, depth=1
{0x8383838383838383,   -1,{0xffff,0x0072}},  //  3117, depth=1
{0x00007fff00000000,   -1,{0xffff,0x180e}},  //  3118, depth=1
{0x00000000c000ffff, 2859,{0x0091,0xffff}},  //  3119, depth=3
{0x0000001fffff0000,   -1,{0xffff,0x1c14}},  //  3120
{0xffffcfffffffffff,   -1,{0xffff,0x14bd}},  //  3121, depth=1
{0x000ffffffffff000,   -1,{0xffff,0x1d27}},  //  3122, depth=1
{0x00000000fc03ffff,   -1,{0x0197,0xffff}},  //  3123, depth=1
{0x00fffffffffffe00,   -1,{0xffff,0x1dee}},  //  3124, depth=1
{0xffffff800003ffff,   -1,{0xffff,0x166a}},  //  3125
{0x00001c0000000000,   -1,{0xffff,0x1582}},  //  3126, depth=1
{0xff8000000000000f,   -1,{0xffff,0x124c}},  //  3127, depth=1
{0x001ffffc00000000,   -1,{0xffff,0x1792}},  //  3128
{0xfffffffffff801ff,   -1,{0xffff,0x1b75}},  //  3129, depth=1
{0x0001ffc000000000,   -1,{0xffff,0x168a}},  //  3130
{0xfffffff3ffffffff,   -1,{0xffff,0x173d}},  //  3131, depth=1
{0x0000000007f00000,   -1,{0x0306,0x1b06}},  //  3132, depth=1
{0xffe003ffffe003ff,   -1,{0xffff,0x02d4}},  //  3133, depth=1
{0x3800000038000000, 2863,{0xffff,0x0142}},  //  3134, depth=3
{0x00000000f9ffffff, 2866,{0x015d,0xffff}},  //  3135, depth=3
{0xffff803fffffffff,   -1,{0xffff,0x1476}},  //  3136
{0xfffffffffffff8ff,   -1,{0xffff,0x1d7c}},  //  3137, depth=1
{0x00000001ffffe000,   -1,{0xffff,0x1cd3}},  //  3138
{0xf87ff87ff87ff87f, 2868,{0xffff,0x016b}},  //  3139, depth=2
{0x0001c0000001c000, 2870,{0xffff,0x0482}},  //  3140, depth=3
{0x00000000c003c003,   -1,{0x00a3,0xffff}},  //  3141, depth=1
{0x00000007ffffff80,   -1,{0xffff,0x1e5b}},  //  3142, depth=1
{0xff8000000000001f,   -1,{0xffff,0x124d}},  //  3143, depth=1
{0x0000000fffff0000,   -1,{0xffff,0x1c13}},  //  3144, depth=1
{0x003ffffffffffe00,   -1,{0xffff,0x1dec}},  //  3145
{0x0003ffc00003ffc0,   -1,{0xffff,0x068b}},  //  3146, depth=1
{0xc00fffffffffffff,   -1,{0xffff,0x10b5}},  //  3147, depth=1
{0x03f803f803f803f8,   -1,{0xffff,0x0366}},  //  3148, depth=1
{0x00000000fffe0fff,   -1,{0x03da,0xffff}},  //  3149, depth=1
{0x0000000fffe00000,   -1,{0xffff,0x1ace}},  //  3150
{0xfffc007ffffc007f,   -1,{0xffff,0x0394}},  //  3151, depth=1
{0xf801f801f801f801,   -1,{0xffff,0x0165}},  //  3152
{0xffffffffffff1fff,   -1,{0xffff,0x1c3c}},  //  3153
{0x3fffffff00000000,   -1,{0xffff,0x181d}},  //  3154, depth=1
{0x0038000000000000,   -1,{0xffff,0x1342}},  //  3155
{0xfffffffff800007f,   -1,{0xffff,0x196b}},  //  3156
{0xe000ffffe000ffff, 2872,{0xffff,0x00d2}},  //  3157, depth=2
{0x0000ffffffffe000,   -1,{0xffff,0x1ce2}},  //  3158
{0x7ffffffffffff800,   -1,{0xffff,0x1d73}},  //  3159
{0xffff000000000fff,   -1,{0xffff,0x141b}},  //  3160
{0xffffffc03fffffff,   -1,{0xffff,0x16b7}},  //  3161
{0xffffe00000003fff,   -1,{0xffff,0x14e0}},  //  3162
{0xffffffe1ffffffff,   -1,{0xffff,0x16fb}},  //  3163
{0x7fffc0007fffc000,   -1,{0xffff,0x0490}},  //  3164, depth=1
{0xffe00007ffe00007,   -1,{0xffff,0x02cd}},  //  3165, depth=1
{0x00000001fff80000,   -1,{0xffff,0x1b4d}},  //  3166, depth=1
{0xfffff9fffffff9ff, 2876,{0xffff,0x055d}},  //  3167, depth=2
{0x000000fffffff800,   -1,{0xffff,0x1d5c}},  //  3168, depth=1
{0xfffffffff0007fff,   -1,{0xffff,0x1932}},  //  3169
{0x03fe000000000000,   -1,{0xffff,0x13c8}},  //  3170
{0xfffc000000001fff,   -1,{0xffff,0x139a}},  //  3171
{0x007fc00000000000,   -1,{0xffff,0x1488}},  //  3172
{0x00000000ffc0ffff,   -1,{0x0299,0xffff}},  //  3173, depth=1
{0x0600000000000000,   -1,{0xffff,0x11c1}},  //  3174, depth=1
{0xff8000000000003f,   -1,{0xffff,0x124e}},  //  3175, depth=1
{0x001ffffffe000000, 2879,{0xffff,0x19db}},  //  3176, depth=2
{0xffe00000000000ff, 2880,{0xffff,0x12d2}},  //  3177, depth=2
{0x00003fc000000000,   -1,{0xffff,0x1687}},  //  3178, depth=1
{0xe0007fffffffffff, 2886,{0xffff,0x10f1}},  //  3179, depth=2
{0x03f8000003f80000,   -1,{0xffff,0x0346}},  //  3180, depth=1
{0x0007ffff0007ffff, 2890,{0xffff,0x0012}},  //  3181, depth=2
{0xfe0001fffe0001ff,   -1,{0xffff,0x01cf}},  //  3182
{0xfffff803fffff803,   -1,{0xffff,0x0556}},  //  3183, depth=1
{0xfff81fffffffffff,   -1,{0xffff,0x1379}},  //  3184
{0x0000003fffffc000,   -1,{0xffff,0x1c97}},  //  3185
{0x003fffe0003fffe0,   -1,{0xffff,0x06d0}},  //  3186, depth=1
{0x000001e000000000,   -1,{0xffff,0x16c3}},  //  3187
{0x0000fffffff80000,   -1,{0xffff,0x1b5c}},  //  3188, depth=1
{0xfff00001ffffffff,   -1,{0xffff,0x132c}},  //  3189, depth=1
{0x800001ffffffffff,   -1,{0xffff,0x1069}},  //  3190
{0xf0000000000001ff,   -1,{0xffff,0x110c}},  //  3191, depth=1
{0x00000000ff80ff80, 2895,{0x0268,0xffff}},  //  3192, depth=2
{0xffffffc01fffffff,   -1,{0xffff,0x16b6}},  //  3193
{0xffffffc0ffffffc0, 2901,{0xffff,0x0699}},  //  3194, depth=2
{0xfffffffffff9ffff,   -1,{0xffff,0x1b7d}},  //  3195, depth=1
{0x1fffffc000000000,   -1,{0xffff,0x1696}},  //  3196, depth=1
{0x07ffffe000000000,   -1,{0xffff,0x16d5}},  //  3197
{0x03fffffc00000000,   -1,{0xffff,0x1797}},  //  3198
{0x00000000fff01fff,   -1,{0x0318,0xffff}},  //  3199, depth=1
{0x00000003f8000000,   -1,{0xffff,0x1946}},  //  3200, depth=1
{0xc1ffffffffffffff, 2908,{0xffff,0x10ba}},  //  3201, depth=2
{0x87ffffffffffffff,   -1,{0xffff,0x107b}},  //  3202
{0xffff8000000003ff,   -1,{0xffff,0x145a}},  //  3203, depth=1
{0xf8007ffff8007fff,   -1,{0xffff,0x0153}},  //  3204
{0xff87ffffffffffff,   -1,{0xffff,0x127b}},  //  3205
{0x00000007ffffffc0,   -1,{0xffff,0x1e9c}},  //  3206, depth=1
{0xffffffffffffe0ff,   -1,{0xffff,0x1cfa}},  //  3207, depth=1
{0x07fffffffffffc00,   -1,{0xffff,0x1db0}},  //  3208, depth=1
{0x00fffffffffffff0,   -1,{0xffff,0x1f33}},  //  3209
{0x0000000003ffff00,   -1,{0x0611,0x1e11}},  //  3210
{0x000000008000ffff, 2909,{0x0050,0xffff}},  //  3211, depth=3
{0xffffffbfffffffff,   -1,{0xffff,0x167e}},  //  3212
{0x0001fffffffc0000,   -1,{0xffff,0x1b9e}},  //  3213
{0x000007f8000007f8,   -1,{0xffff,0x0747}},  //  3214, depth=1
{0xffffffffff8fffff,   -1,{0xffff,0x1a7c}},  //  3215
{0x00fffffffffffffc,   -1,{0xffff,0x1fb5}},  //  3216
{0xffff807fffff807f,   -1,{0xffff,0x0457}},  //  3217, depth=1
{0xffffffc000000007, 3222,{0xffff,0x169c}},  //  3218
{0xfffffc00000001ff,   -1,{0xffff,0x159e}},  //  3219, depth=1
{0x0000030000000000,   -1,{0xffff,0x1601}},  //  3220, depth=1
{0xfffffffc001fffff,   -1,{0xffff,0x17b2}},  //  3221, depth=1
{0xfff00000007fffff,   -1,{0xffff,0x1322}},  //  3222
{0x0000000700000000,   -1,{0xffff,0x1802}},  //  3223
{0x000001fffffe0000,   -1,{0xffff,0x1bd7}},  //  3224
{0xffc000000007ffff,   -1,{0xffff,0x129c}},  //  3225, depth=1
{0x007c0000007c0000,   -1,{0xffff,0x0384}},  //  3226, depth=1
{0xfff81ffffff81fff, 2912,{0xffff,0x0359}},  //  3227, depth=2
{0x07f0000000000000,   -1,{0xffff,0x1306}},  //  3228, depth=1
{0xfffffffffff803ff,   -1,{0xffff,0x1b76}},  //  3229
{0x000000000fc00000, 2915,{0x0285,0x1a85}},  //  3230, depth=2
{0xffffffffffff7fff,   -1,{0xffff,0x1c3e}},  //  3231, depth=1
{0xffffffffffff8000,   -1,{0xffff,0x1c70}},  //  3232, depth=1
{0xffffffffffff8001,   -1,{0xffff,0x1c71}},  //  3233, depth=1
{0x00001fffffff8000,   -1,{0xffff,0x1c5d}},  //  3234
{0x8000000003ffffff, 2916,{0xffff,0x105a}},  //  3235, depth=2
{0x3fffe00000000000,   -1,{0xffff,0x14d0}},  //  3236
{0xfffc03fffffc03ff,   -1,{0xffff,0x0397}},  //  3237
{0x0000000001f801f8, 2918,{0x0365,0xffff}},  //  3238, depth=3
{0xff8000000000007f, 2930,{0xffff,0x124f}},  //  3239, depth=2
{0x00000000fc007fff,   -1,{0x0194,0xffff}},  //  3240
{0xfffffffff83fffff,   -1,{0xffff,0x197a}},  //  3241, depth=1
{0xffffff0000003fff, 3243,{0xffff,0x1625}},  //  3242
{0xf803fffff803ffff,   -1,{0xffff,0x0156}},  //  3243
{0x000000001c001c00, 2933,{0x01a2,0xffff}},  //  3244, depth=2
{0xff00003fff00003f,   -1,{0xffff,0x020d}},  //  3245, depth=1
{0x0000040000000400, 2936,{0xffff,0x0580}},  //  3246, depth=2
{0xffe0000fffffffff, 2938,{0xffff,0x12ee}},  //  3247, depth=2
{0x0fffff0000000000,   -1,{0xffff,0x1613}},  //  3248, depth=1
{0x00000000ff1fffff,   -1,{0x021c,0xffff}},  //  3249, depth=1
{0x00000f0000000000,   -1,{0xffff,0x1603}},  //  3250
{0x1fff1fff1fff1fff,   -1,{0xffff,0x002c}},  //  3251, depth=1
{0x000000fff8000000,   -1,{0xffff,0x194c}},  //  3252
{0xff8007ffff8007ff,   -1,{0xffff,0x0253}},  //  3253
{0x00000007fffffff0,   -1,{0xffff,0x1f1e}},  //  3254, depth=1
{0xffffffffff003fff,   -1,{0xffff,0x1a35}},  //  3255, depth=1
{0x1f001f001f001f00, 2943,{0xffff,0x0224}},  //  3256, depth=2
{0x00000000e1ffe1ff, 2944,{0x00eb,0xffff}},  //  3257, depth=3
{0x03ffff8003ffff80, 2949,{0xffff,0x0652}},  //  3258, depth=2
{0x7f7f7f7f7f7f7f7f, 2950,{0xffff,0x0036}},  //  3259, depth=2
{0x7ffffffffff80000,   -1,{0xffff,0x1b6b}},  //  3260, depth=1
{0xffffcfffffffcfff,   -1,{0xffff,0x049d}},  //  3261, depth=1
{0x000fffffffffc000, 2954,{0xffff,0x1ca5}},  //  3262, depth=2
{0x00000000fc07fc07, 2962,{0x01a8,0xffff}},  //  3263, depth=2
{0x0007ffffffe00000,   -1,{0xffff,0x1add}},  //  3264
{0xffffffc00000003f,   -1,{0xffff,0x169f}},  //  3265
{0x00000007fffffffc,   -1,{0xffff,0x1fa0}},  //  3266, depth=1
{0xffffffe000ffffff, 3272,{0xffff,0x16f2}},  //  3267
{0x000000000000ff80, 2968,{0x0648,0x1e48}},  //  3268, depth=2
{0x00000000fff007ff, 2969,{0x0316,0xffff}},  //  3269, depth=3
{0x03f003f003f003f0, 2972,{0xffff,0x0325}},  //  3270, depth=2
{0xff00000000001fff, 2974,{0xffff,0x1214}},  //  3271, depth=2
{0xfffc0007fffc0007,   -1,{0xffff,0x0390}},  //  3272
{0x00000000f00001ff, 2978,{0x010c,0xffff}},  //  3273, depth=2
{0x000000000003f000,   -1,{0x0505,0x1d05}},  //  3274, depth=1
{0x000001fffffffe00,   -1,{0xffff,0x1ddf}},  //  3275
{0xffffffff000fffff,   -1,{0xffff,0x1833}},  //  3276
{0xfff8ffffffffffff,   -1,{0xffff,0x137c}},  //  3277
{0x0000000800000008, 2979,{0xffff,0x0740}},  //  3278, depth=2
{0x000000009f9f9f9f, 2980,{0x0075,0xffff}},  //  3279, depth=3
{0x3ff000003ff00000,   -1,{0xffff,0x0309}},  //  3280, depth=1
{0x00000003fffffc00,   -1,{0xffff,0x1d97}},  //  3281
{0xffff800007ffffff,   -1,{0xffff,0x146b}},  //  3282
{0xffc03fffffc03fff,   -1,{0xffff,0x0297}},  //  3283, depth=1
{0x000000003ffe3ffe, 2987,{0x03ec,0xffff}},  //  3284, depth=2
{0xfffffc000001ffff,   -1,{0xffff,0x15a6}},  //  3285, depth=1
{0xaaaaaaaaaaaaaaaa, 2990,{0xffff,0x007c}},  //  3286, depth=2
{0xfe00000000ffffff,   -1,{0xffff,0x11de}},  //  3287, depth=1
{0x000007fffffc0000,   -1,{0xffff,0x1b98}},  //  3288, depth=1
{0xe007ffffe007ffff,   -1,{0xffff,0x00d5}},  //  3289, depth=1
{0x01f8000001f80000, 2997,{0xffff,0x0345}},  //  3290, depth=2
{0x07f8000000000000,   -1,{0xffff,0x1347}},  //  3291
{0x007f8000007f8000,   -1,{0xffff,0x0447}},  //  3292, depth=1
{0xfffffc00000003ff,   -1,{0xffff,0x159f}},  //  3293
{0x0003ffffffffffc0,   -1,{0xffff,0x1eab}},  //  3294, depth=1
{0xffffffffffff803f,   -1,{0xffff,0x1c76}},  //  3295, depth=1
{0xf1ffffffffffffff,   -1,{0xffff,0x113c}},  //  3296
{0xfff000007fffffff,   -1,{0xffff,0x132a}},  //  3297
{0x0000000003fc03fc,   -1,{0x03a7,0xffff}},  //  3298
{0xc00000000003ffff,   -1,{0xffff,0x1093}},  //  3299, depth=1
{0x0000000003fe0000, 3003,{0x03c8,0x1bc8}},  //  3300, depth=2
{0xf07ff07ff07ff07f, 3007,{0xffff,0x012a}},  //  3301, depth=2
{0xfffffe00001fffff,   -1,{0xffff,0x15eb}},  //  3302
{0xfff07ffffff07fff,   -1,{0xffff,0x031a}},  //  3303
{0xffffffc00000007f,   -1,{0xffff,0x16a0}},  //  3304
{0x00000000f00ff00f, 3008,{0x0127,0xffff}},  //  3305, depth=2
{0x7f80000000000000,   -1,{0xffff,0x1247}},  //  3306
{0xffff000003ffffff,   -1,{0xffff,0x1429}},  //  3307, depth=1
{0x00000000f0f0f0f0,   -1,{0x0133,0xffff}},  //  3308, depth=1
{0x0000078000000780,   -1,{0xffff,0x0643}},  //  3309
{0x00fffff000000000,   -1,{0xffff,0x1713}},  //  3310, depth=1
{0x1fffffe01fffffe0,   -1,{0xffff,0x06d7}},  //  3311
{0x03ff000003ff0000,   -1,{0xffff,0x0409}},  //  3312, depth=1
{0x8000000000003fff,   -1,{0xffff,0x104e}},  //  3313, depth=1
{0x003ff000003ff000,   -1,{0xffff,0x0509}},  //  3314, depth=1
{0xfc07ffffffffffff,   -1,{0xffff,0x11b8}},  //  3315, depth=1
{0x00000007e0000000, 3011,{0xffff,0x18c5}},  //  3316, depth=2
{0xfffffffffffbffff,   -1,{0xffff,0x1b7e}},  //  3317
{0x0000000000f80000,   -1,{0x0344,0x1b44}},  //  3318, depth=1
{0x00000000ff80ffff, 3016,{0x0258,0xffff}},  //  3319, depth=2
{0x0007ffffffff0000,   -1,{0xffff,0x1c22}},  //  3320, depth=1
{0xfffffffffffc0003,   -1,{0xffff,0x1baf}},  //  3321
{0x7000000000000000,   -1,{0xffff,0x1102}},  //  3322, depth=1
{0x00000000ffe001ff,   -1,{0x02d3,0xffff}},  //  3323, depth=1
{0xfff00000003fffff,   -1,{0xffff,0x1321}},  //  3324
{0xfc000000000001ff,   -1,{0xffff,0x118e}},  //  3325, depth=1
{0x00003ffffffc0000, 3022,{0xffff,0x1b9b}},  //  3326, depth=2
{0xfffffffffffc0007,   -1,{0xffff,0x1bb0}},  //  3327
{0x00ffffffffc00000, 3330,{0xffff,0x1aa1}},  //  3328
{0xfffffffe03ffffff,   -1,{0xffff,0x17f8}},  //  3329, depth=1
{0x0000001ff0000000,   -1,{0xffff,0x1908}},  //  3330
{0xf000001fffffffff,   -1,{0xffff,0x1128}},  //  3331, depth=1
{0x000000000000ffc0,   -1,{0x0689,0x1e89}},  //  3332, depth=1
{0xffffffc0007fffff,   -1,{0xffff,0x16b0}},  //  3333, depth=1
{0x00007fffff000000,   -1,{0xffff,0x1a16}},  //  3334, depth=1
{0xffff80000003ffff,   -1,{0xffff,0x1462}},  //  3335, depth=1
{0x0000600000006000,   -1,{0xffff,0x04c1}},  //  3336, depth=1
{0x3fffff0000000000,   -1,{0xffff,0x1615}},  //  3337
{0xfffffffffffc000f,   -1,{0xffff,0x1bb1}},  //  3338
{0x00000000f001ffff, 3024,{0x0114,0xffff}},  //  3339, depth=2
{0x000ffffffc000000,   -1,{0xffff,0x1999}},  //  3340, depth=1
{0xff000000000007ff, 3027,{0xffff,0x1212}},  //  3341, depth=2
{0x1fffe0001fffe000, 3028,{0xffff,0x04cf}},  //  3342, depth=2
{0xfffffe000fffffff,   -1,{0xffff,0x15f2}},  //  3343
{0x0000007e00000000, 3031,{0xffff,0x17c5}},  //  3344, depth=2
{0xffffff80000003ff,   -1,{0xffff,0x1662}},  //  3345, depth=1
{0xfffff00007ffffff,   -1,{0xffff,0x152e}},  //  3346
{0xc0001fffc0001fff,   -1,{0xffff,0x008e}},  //  3347, depth=1
{0x0000000001ff8000,   -1,{0x0449,0x1c49}},  //  3348, depth=1
{0xffffe1ffffffffff,   -1,{0xffff,0x14fb}},  //  3349
{0x7ffffc0000000000, 3035,{0xffff,0x1594}},  //  3350, depth=2
{0x0000000007ffc000,   -1,{0x048c,0x1c8c}},  //  3351
{0x800007ffffffffff,   -1,{0xffff,0x106b}},  //  3352
{0x00000000fe1ffe1f,   -1,{0x01eb,0xffff}},  //  3353, depth=1
{0x0003fffffffffffc,   -1,{0xffff,0x1faf}},  //  3354, depth=1
{0xfe03fe03fe03fe03,   -1,{0xffff,0x01e8}},  //  3355, depth=1
{0x00000000ff00ff00, 3040,{0x0227,0xffff}},  //  3356, depth=2
{0x0003ffffffffffff, 3043,{0xffff,0x1031}},  //  3357, depth=2
{0x0004000000000000,   -1,{0xffff,0x1380}},  //  3358, depth=1
{0x00000000ffe1ffe1, 3045,{0x02eb,0xffff}},  //  3359, depth=4
{0xfffffffffffc001f,   -1,{0xffff,0x1bb2}},  //  3360
{0x000003fffff00000,   -1,{0xffff,0x1b15}},  //  3361
{0x000001c0000001c0, 3053,{0xffff,0x0682}},  //  3362, depth=2
{0xfffffffffc07ffff,   -1,{0xffff,0x19b8}},  //  3363, depth=1
{0x000000000000ffe0, 3056,{0x06ca,0x1eca}},  //  3364, depth=2
{0x007ffe0000000000,   -1,{0xffff,0x15cd}},  //  3365
{0x0000000003fffe00,   -1,{0x05d0,0x1dd0}},  //  3366, depth=1
{0xff800000000000ff,   -1,{0xffff,0x1250}},  //  3367, depth=1
{0x01ffff8001ffff80,   -1,{0xffff,0x0651}},  //  3368, depth=1
{0xfc0fffffffffffff,   -1,{0xffff,0x11b9}},  //  3369
{0x3fffffffff000000, 3060,{0xffff,0x1a25}},  //  3370, depth=2
{0xf8000fffffffffff,   -1,{0xffff,0x1170}},  //  3371, depth=1
{0xfe0000001fffffff,   -1,{0xffff,0x11e3}},  //  3372
{0x803fffffffffffff,   -1,{0xffff,0x1076}},  //  3373, depth=1
{0x00003fe000000000,   -1,{0xffff,0x16c8}},  //  3374
{0xfff000003fffffff, 3377,{0xffff,0x1329}},  //  3375
{0x000000000f800000,   -1,{0x0244,0x1a44}},  //  3376, depth=1
{0xffe1ffe1ffe1ffe1,   -1,{0xffff,0x02eb}},  //  3377
{0xf00000000003ffff,   -1,{0xffff,0x1115}},  //  3378
{0xfffffffffffc003f,   -1,{0xffff,0x1bb3}},  //  3379
{0x000000000000fff0, 3063,{0x070b,0x1f0b}},  //  3380, depth=2
{0xfffffffffe003fff,   -1,{0xffff,0x19f4}},  //  3381
{0x0ffffffffff00000,   -1,{0xffff,0x1b27}},  //  3382
{0xfc007ffffc007fff,   -1,{0xffff,0x0194}},  //  3383, depth=1
{0x00000000eeeeeeee,   -1,{0x00fa,0xffff}},  //  3384, depth=1
{0xffff000000003fff, 3064,{0xffff,0x141d}},  //  3385, depth=2
{0xffffc0000000ffff,   -1,{0xffff,0x14a1}},  //  3386
{0xfffffffff7ffffff,   -1,{0xffff,0x193e}},  //  3387, depth=1
{0x000000000000fff8, 3070,{0x074c,0x1f4c}},  //  3388, depth=2
{0x00000000ffe1ffff, 3071,{0x02db,0xffff}},  //  3389, depth=3
{0x000003fffe000000,   -1,{0xffff,0x19d0}},  //  3390, depth=1
{0xfc0000000001ffff, 3077,{0xffff,0x1196}},  //  3391, depth=2
{0x000000000000fffc,   -1,{0x078d,0x1f8d}},  //  3392, depth=1
{0xfffffffffffff9ff,   -1,{0xffff,0x1d7d}},  //  3393, depth=1
{0x000000000000fffe, 3081,{0x07ce,0x1fce}},  //  3394, depth=2
{0x000000000000ffff, 3085,{0x000f,0x100f}},  //  3395, depth=3
{0x0000000000010000,   -1,{0x0400,0x1c00}},  //  3396, depth=1
{0x0000000000010001,   -1,{0x0020,0xffff}},  //  3397, depth=1
{0x1fffffe000000000,   -1,{0xffff,0x16d7}},  //  3398
{0xffffffffc1ffffff,   -1,{0xffff,0x18ba}},  //  3399
{0x0000000003fffff0,   -1,{0x0715,0x1f15}},  //  3400
{0xffffff8007ffffff,   -1,{0xffff,0x1673}},  //  3401
{0x0003e0000003e000,   -1,{0xffff,0x04c4}},  //  3402, depth=1
{0xff7fffffff7fffff, 3089,{0xffff,0x021e}},  //  3403, depth=2
{0x000fc000000fc000, 3090,{0xffff,0x0485}},  //  3404, depth=2
{0x000007fffffe0000, 3406,{0xffff,0x1bd9}},  //  3405
{0x7ff07ff07ff07ff0,   -1,{0xffff,0x032a}},  //  3406
{0xfffffffe00003fff,   -1,{0xffff,0x17ec}},  //  3407, depth=1
{0x007ffffffff00000,   -1,{0xffff,0x1b22}},  //  3408, depth=1
{0xfffffff1ffffffff, 3412,{0xffff,0x173c}},  //  3409
{0x00000ffe00000000,   -1,{0xffff,0x17ca}},  //  3410, depth=1
{0x001f001f001f001f, 3091,{0xffff,0x0024}},  //  3411, depth=3
{0x00000000ffffdfff,   -1,{0x049e,0xffff}},  //  3412
{0xfffe0007ffffffff,   -1,{0xffff,0x13f1}},  //  3413, depth=1
{0x000000000003c000,   -1,{0x0483,0x1c83}},  //  3414, depth=1
{0x00000000f803f803,   -1,{0x0166,0xffff}},  //  3415, depth=1
{0x0000fffe0000fffe, 3096,{0xffff,0x07ce}},  //  3416, depth=2
{0x0000000003fffff8,   -1,{0x0756,0x1f56}},  //  3417
{0x0ffffffffe000000,   -1,{0xffff,0x19e2}},  //  3418, depth=1
{0x00ffffff00ffffff, 3098,{0xffff,0x0017}},  //  3419, depth=2
{0x000007fffffffc00,   -1,{0xffff,0x1da0}},  //  3420, depth=1
{0xfdffffffffffffff,   -1,{0xffff,0x11be}},  //  3421
{0x01ff000001ff0000, 3100,{0xffff,0x0408}},  //  3422, depth=2
{0xffe7ffffffffffff,   -1,{0xffff,0x12fd}},  //  3423, depth=1
{0x0000ff000000ff00,   -1,{0xffff,0x0607}},  //  3424, depth=1
{0xffffffffffe01fff, 3426,{0xffff,0x1af7}},  //  3425
{0xffff07ffffffffff, 3427,{0xffff,0x143a}},  //  3426
{0xfe00000000000001, 3432,{0xffff,0x11c7}},  //  3427
{0x3ffffffe3ffffffe,   -1,{0xffff,0x07dc}},  //  3428, depth=1
{0xfff800ffffffffff,   -1,{0xffff,0x1374}},  //  3429, depth=1
{0x00000001fffff800,   -1,{0xffff,0x1d55}},  //  3430, depth=1
{0xffffffffc00fffff,   -1,{0xffff,0x18b5}},  //  3431, depth=1
{0xffc7ffffffc7ffff,   -1,{0xffff,0x029c}},  //  3432
{0xffe00000000001ff,   -1,{0xffff,0x12d3}},  //  3433, depth=1
{0x7fffffffffff8000, 3439,{0xffff,0x1c6f}},  //  3434
{0xfffffff001ffffff,   -1,{0xffff,0x1734}},  //  3435, depth=1
{0x3fffff003fffff00, 3101,{0xffff,0x0615}},  //  3436, depth=2
{0xff80000fffffffff,   -1,{0xffff,0x126c}},  //  3437, depth=1
{0x003ffffffffe0000,   -1,{0xffff,0x1be4}},  //  3438, depth=1
{0x0000000003fffffc,   -1,{0x0797,0x1f97}},  //  3439
{0xfe00000000000003,   -1,{0xffff,0x11c8}},  //  3440
{0xffffffc07fffffff,   -1,{0xffff,0x16b8}},  //  3441, depth=1
{0x1e00000000000000,   -1,{0xffff,0x11c3}},  //  3442, depth=1
{0x01ffffffffff0000,   -1,{0xffff,0x1c28}},  //  3443
{0x0000000000003e00,   -1,{0x05c4,0x1dc4}},  //  3444, depth=1
{0xffff800000007fff,   -1,{0xffff,0x145f}},  //  3445
{0x0007fff800000000,   -1,{0xffff,0x174f}},  //  3446, depth=1
{0xc000003fffffffff, 3104,{0xffff,0x10a7}},  //  3447, depth=2
{0xffffffe0ffffffe0,   -1,{0xffff,0x06da}},  //  3448
{0xfc03fffffc03ffff,   -1,{0xffff,0x0197}},  //  3449, depth=1
{0xffffe000ffffe000, 3105,{0xffff,0x04d2}},  //  3450, depth=2
{0xfffffffff800003f,   -1,{0xffff,0x196a}},  //  3451, depth=1
{0x0000fffffffff800,   -1,{0xffff,0x1d64}},  //  3452, depth=1
{0x8000000000000fff,   -1,{0xffff,0x104c}},  //  3453, depth=1
{0x0000000780000000,   -1,{0xffff,0x1843}},  //  3454, depth=1
{0xfff000001fffffff, 3457,{0xffff,0x1328}},  //  3455
{0x7c00000000000000, 3106,{0xffff,0x1184}},  //  3456, depth=2
{0xfe00000000000007,   -1,{0xffff,0x11c9}},  //  3457
{0x00003ffffffffc00,   -1,{0xffff,0x1da3}},  //  3458, depth=1
{0xffffe00003ffffff,   -1,{0xffff,0x14ec}},  //  3459, depth=1
{0xffffffc0000000ff,   -1,{0xffff,0x16a1}},  //  3460
{0xfffc00000007ffff,   -1,{0xffff,0x13a0}},  //  3461
{0xfe0000000000000f,   -1,{0xffff,0x11ca}},  //  3462
{0xe0000000001fffff, 3107,{0xffff,0x10d7}},  //  3463, depth=2
{0x000001f000000000, 3466,{0xffff,0x1704}},  //  3464
{0xf03ffffff03fffff, 3108,{0xffff,0x0119}},  //  3465, depth=2
{0x0000000000003fc0,   -1,{0x0687,0x1e87}},  //  3466
{0xfff9fff9fff9fff9,   -1,{0xffff,0x036d}},  //  3467, depth=1
{0x00000000000c0000,   -1,{0x0381,0x1b81}},  //  3468, depth=1
{0xfffffffff80001ff,   -1,{0xffff,0x196d}},  //  3469
{0x0000007e0000007e, 3120,{0xffff,0x07c5}},  //  3470, depth=2
{0xfffffffffffc007f, 3472,{0xffff,0x1bb4}},  //  3471
{0xff00007fffffffff,   -1,{0xffff,0x122e}},  //  3472
{0x00000000ffff03ff,   -1,{0x0419,0xffff}},  //  3473, depth=1
{0xfe0000000000001f,   -1,{0xffff,0x11cb}},  //  3474
{0xfffffffffffffc00,   -1,{0xffff,0x1db5}},  //  3475
{0x1ffffffffffffc00,   -1,{0xffff,0x1db2}},  //  3476, depth=1
{0xfffffe07fffffe07, 3125,{0xffff,0x05d9}},  //  3477, depth=2
{0xfffffffffffffc03, 3484,{0xffff,0x1db7}},  //  3478
{0xffffffc0003fffff,   -1,{0xffff,0x16af}},  //  3479, depth=1
{0x00000000000c000c, 3128,{0x03a1,0xffff}},  //  3480, depth=2
{0xffffc000001fffff,   -1,{0xffff,0x14a6}},  //  3481, depth=1
{0xfcfcfcfcfcfcfcfc, 3130,{0xffff,0x01b5}},  //  3482, depth=2
{0xfff8003ffff8003f, 3136,{0xffff,0x0352}},  //  3483, depth=2
{0xffffffe0ffffffff,   -1,{0xffff,0x16fa}},  //  3484
{0xffffffffffcfffff,   -1,{0xffff,0x1abd}},  //  3485, depth=1
{0x00007ffe00000000,   -1,{0xffff,0x17cd}},  //  3486, depth=1
{0xffffffffffff80ff,   -1,{0xffff,0x1c78}},  //  3487, depth=1
{0xffff3fffffffffff,   -1,{0xffff,0x143d}},  //  3488
{0xfffffffffffffc07,   -1,{0xffff,0x1db8}},  //  3489
{0xfffff801fffff801,   -1,{0xffff,0x0555}},  //  3490
{0x00007ffffe000000,   -1,{0xffff,0x19d5}},  //  3491
{0x00000ff800000ff8,   -1,{0xffff,0x0748}},  //  3492, depth=1
{0xffefffffffffffff,   -1,{0xffff,0x12fe}},  //  3493
{0x000007fffffffe00,   -1,{0xffff,0x1de1}},  //  3494
{0x007ffffffffff000,   -1,{0xffff,0x1d2a}},  //  3495
{0x1ff0000000000000,   -1,{0xffff,0x1308}},  //  3496, depth=1
{0xf9fff9fff9fff9ff,   -1,{0xffff,0x016d}},  //  3497, depth=1
{0xfffffffffffffc1f,   -1,{0xffff,0x1dba}},  //  3498
{0xffe000000001ffff,   -1,{0xffff,0x12db}},  //  3499, depth=1
{0x3fff000000000000, 3138,{0xffff,0x140d}},  //  3500, depth=2
{0x0003ffffff000000,   -1,{0xffff,0x1a19}},  //  3501
{0xffffffff81ffffff, 3505,{0xffff,0x1879}},  //  3502
{0xfe000000ffffffff,   -1,{0xffff,0x11e6}},  //  3503, depth=1
{0x0003ffffffc00000, 3145,{0xffff,0x1a9b}},  //  3504, depth=2
{0xffe00000000003ff,   -1,{0xffff,0x12d4}},  //  3505
{0x000000001ff01ff0, 3150,{0x0328,0xffff}},  //  3506, depth=2
{0x0000000099999999, 3152,{0x0079,0xffff}},  //  3507, depth=2
{0x00ffffffffffff80,   -1,{0xffff,0x1e70}},  //  3508, depth=1
{0x000000003fc03fc0,   -1,{0x02a7,0xffff}},  //  3509
{0x000ffff800000000,   -1,{0xffff,0x1750}},  //  3510
{0xffffffe1ffffffe1, 3153,{0xffff,0x06db}},  //  3511, depth=2
{0x003fffffffffffc0,   -1,{0xffff,0x1eaf}},  //  3512
{0xe00000000fffffff,   -1,{0xffff,0x10de}},  //  3513, depth=1
{0x03e003e003e003e0, 3155,{0xffff,0x02e4}},  //  3514, depth=2
{0xff0003ffffffffff, 3156,{0xffff,0x1231}},  //  3515, depth=2
{0x3ffffffffe000000,   -1,{0xffff,0x19e4}},  //  3516
{0x0000fffff8000000,   -1,{0xffff,0x1954}},  //  3517
{0x000000ffffff8000,   -1,{0xffff,0x1c58}},  //  3518, depth=1
{0xfffff800001fffff,   -1,{0xffff,0x1569}},  //  3519, depth=1
{0x0007800000000000,   -1,{0xffff,0x1443}},  //  3520
{0xffc0007fffffffff,   -1,{0xffff,0x12b0}},  //  3521, depth=1
{0x3ffffffe00000000, 3158,{0xffff,0x17dc}},  //  3522, depth=2
{0xffffffff800fffff,   -1,{0xffff,0x1874}},  //  3523, depth=1
{0x00000c0000000c00, 3159,{0xffff,0x0581}},  //  3524, depth=2
{0x00000000fff7fff7, 3160,{0x032e,0xffff}},  //  3525, depth=2
{0xff000fffffffffff, 3529,{0xffff,0x1233}},  //  3526
{0xe0ffffffe0ffffff,   -1,{0xffff,0x00da}},  //  3527, depth=1
{0x003ffff0003ffff0,   -1,{0xffff,0x0711}},  //  3528, depth=1
{0xf8000007ffffffff,   -1,{0xffff,0x1167}},  //  3529
{0xfffffffffffffc3f,   -1,{0xffff,0x1dbb}},  //  3530
{0xffffc0000fffffff,   -1,{0xffff,0x14ad}},  //  3531, depth=1
{0xfc1ffffffc1fffff,   -1,{0xffff,0x019a}},  //  3532
{0x00000000fff7ffff, 3161,{0x031e,0xffff}},  //  3533, depth=2
{0x00000000fff80000,   -1,{0x034c,0x1b4c}},  //  3534, depth=1
{0x00000000fff80001,   -1,{0x034d,0xffff}},  //  3535, depth=1
{0xfffffe07ffffffff,   -1,{0xffff,0x15f9}},  //  3536
{0x00000000fff80003, 3162,{0x034e,0xffff}},  //  3537, depth=2
{0x0fffc00000000000,   -1,{0xffff,0x148d}},  //  3538
{0xfffe00000000ffff,   -1,{0xffff,0x13de}},  //  3539, depth=1
{0xf000003fffffffff,   -1,{0xffff,0x1129}},  //  3540
{0x00000000fff80007, 3163,{0x034f,0xffff}},  //  3541, depth=2
{0x0003ffff80000000,   -1,{0xffff,0x1852}},  //  3542, depth=1
{0x7777777777777777, 3169,{0xffff,0x003a}},  //  3543, depth=2
{0x3ffe00003ffe0000, 3170,{0xffff,0x03cc}},  //  3544, depth=2
{0x1ffffffffffffe00,   -1,{0xffff,0x1df3}},  //  3545
{0xfff000000000003f,   -1,{0xffff,0x1311}},  //  3546
{0xfffffffe00000fff,   -1,{0xffff,0x17ea}},  //  3547, depth=1
{0x0007fffe0007fffe,   -1,{0xffff,0x07d1}},  //  3548, depth=1
{0x00000000fff8000f, 3171,{0x0350,0xffff}},  //  3549, depth=2
{0x07ffc00007ffc000, 3172,{0xffff,0x048c}},  //  3550, depth=2
{0x00000000ffffe07f,   -1,{0x04d9,0xffff}},  //  3551
{0x01fffff800000000,   -1,{0xffff,0x1755}},  //  3552
{0x00ffffff00000000,   -1,{0xffff,0x1817}},  //  3553
{0x0000000600000000,   -1,{0xffff,0x17c1}},  //  3554
{0x00000000e3ffffff, 3182,{0x00dc,0xffff}},  //  3555, depth=2
{0x0007ff000007ff00,   -1,{0xffff,0x060a}},  //  3556, depth=1
{0xfff00000ffffffff,   -1,{0xffff,0x132b}},  //  3557, depth=1
{0xfffffc3fffffffff,   -1,{0xffff,0x15bb}},  //  3558
{0xfffff03fffffffff,   -1,{0xffff,0x1539}},  //  3559, depth=1
{0x003ffffffffffffc,   -1,{0xffff,0x1fb3}},  //  3560
{0xffffffbfffffffbf, 3184,{0xffff,0x065e}},  //  3561, depth=2
{0x00000007ff800000, 3185,{0xffff,0x1a4b}},  //  3562, depth=2
{0x003fffffffffffff,   -1,{0xffff,0x1035}},  //  3563
{0x7fe0000000000000,   -1,{0xffff,0x12c9}},  //  3564, depth=1
{0x00000000fff8001f,   -1,{0x0351,0xffff}},  //  3565, depth=1
{0x0000000000f800f8, 3187,{0x0364,0xffff}},  //  3566, depth=2
{0x00000000f3fff3ff,   -1,{0x012d,0xffff}},  //  3567, depth=1
{0xfffffff000ffffff, 3570,{0xffff,0x1733}},  //  3568
{0xfffff8000fffffff,   -1,{0xffff,0x1570}},  //  3569, depth=1
{0xffffff8000007fff,   -1,{0xffff,0x1667}},  //  3570
{0xfe000003fe000003,   -1,{0xffff,0x01c8}},  //  3571, depth=1
{0x00ffffffffffffc0,   -1,{0xffff,0x1eb1}},  //  3572, depth=1
{0xffe0001fffffffff,   -1,{0xffff,0x12ef}},  //  3573, depth=1
{0x3000000030000000,   -1,{0xffff,0x0101}},  //  3574, depth=1
{0x0fff0fff0fff0fff, 3190,{0xffff,0x002b}},  //  3575, depth=2
{0x0ffff80000000000, 3577,{0xffff,0x1550}},  //  3576
{0x7fffffc07fffffc0,   -1,{0xffff,0x0698}},  //  3577
{0xff007fffffffffff,   -1,{0xffff,0x1236}},  //  3578
{0xfe000000007fffff, 3193,{0xffff,0x11dd}},  //  3579, depth=2
{0x0f000f000f000f00, 3197,{0xffff,0x0223}},  //  3580, depth=2
{0xffffffffffff07ff,   -1,{0xffff,0x1c3a}},  //  3581, depth=1
{0x0000000000fff800,   -1,{0x054c,0x1d4c}},  //  3582, depth=1
{0xfffffffc0000ffff,   -1,{0xffff,0x17ad}},  //  3583, depth=1
{0x0000000000000e00,   -1,{0x05c2,0x1dc2}},  //  3584, depth=1
{0x00000000e0ffe0ff,   -1,{0x00ea,0xffff}},  //  3585, depth=1
{0x00000003fffc0000,   -1,{0xffff,0x1b8f}},  //  3586, depth=1
{0xffc00fffffffffff,   -1,{0xffff,0x12b5}},  //  3587, depth=1
{0x007ff80000000000, 3198,{0xffff,0x154b}},  //  3588, depth=2
{0xfff1fff1fff1fff1, 3202,{0xffff,0x032c}},  //  3589, depth=2
{0x0000000008000800,   -1,{0x0160,0xffff}},  //  3590, depth=1
{0xffffff0003ffffff,   -1,{0xffff,0x1631}},  //  3591, depth=1
{0x003ff800003ff800,   -1,{0xffff,0x054a}},  //  3592, depth=1
{0xffffffffffe07fff,   -1,{0xffff,0x1af9}},  //  3593, depth=1
{0x7fffffffffffe000,   -1,{0xffff,0x1cf1}},  //  3594, depth=1
{0x000001e0000001e0,   -1,{0xffff,0x06c3}},  //  3595
{0x0000001800000000,   -1,{0xffff,0x1741}},  //  3596, depth=1
{0x00000000fff8003f,   -1,{0x0352,0xffff}},  //  3597, depth=1
{0x001ffffffff00000,   -1,{0xffff,0x1b20}},  //  3598, depth=1
{0xffff0000000fffff, 3602,{0xffff,0x1423}},  //  3599
{0x00000000000ffc00,   -1,{0x0589,0x1d89}},  //  3600, depth=1
{0x00000000c3ffffff,   -1,{0x009b,0xffff}},  //  3601, depth=1
{0xfff000000000007f,   -1,{0xffff,0x1312}},  //  3602
{0xc000003fc000003f, 3204,{0xffff,0x0087}},  //  3603, depth=2
{0x00ffffffffffffe0,   -1,{0xffff,0x1ef2}},  //  3604, depth=1
{0xff0000000007ffff,   -1,{0xffff,0x121a}},  //  3605, depth=1
{0xf803ffffffffffff,   -1,{0xffff,0x1176}},  //  3606
{0x000000fffffc0000,   -1,{0xffff,0x1b95}},  //  3607
{0xfffffffc03ffffff,   -1,{0xffff,0x17b7}},  //  3608
{0x0000007c00000000, 3612,{0xffff,0x1784}},  //  3609
{0x0000000000780000,   -1,{0x0343,0x1b43}},  //  3610, depth=1
{0x00000000ff00ffff,   -1,{0x0217,0xffff}},  //  3611, depth=1
{0x0000000000000ff0,   -1,{0x0707,0x1f07}},  //  3612
{0x000003ff000003ff, 3205,{0xffff,0x0009}},  //  3613, depth=2
{0x000001ffffffff80,   -1,{0xffff,0x1e61}},  //  3614
{0xffff003fffff003f,   -1,{0xffff,0x0415}},  //  3615, depth=1
{0x0000600000000000,   -1,{0xffff,0x14c1}},  //  3616, depth=1
{0x80ff80ff80ff80ff,   -1,{0xffff,0x0068}},  //  3617, depth=1
{0xffffffe07fffffff,   -1,{0xffff,0x16f9}},  //  3618
{0xfffc0000000007ff,   -1,{0xffff,0x1398}},  //  3619, depth=1
{0x0000001800000018, 3209,{0xffff,0x0741}},  //  3620, depth=2
{0x0003fffe00000000,   -1,{0xffff,0x17d0}},  //  3621
{0x8000800080008000, 3210,{0xffff,0x0060}},  //  3622, depth=2
{0xff800000000001ff,   -1,{0xffff,0x1251}},  //  3623, depth=1
{0xf8f8f8f8f8f8f8f8,   -1,{0xffff,0x0174}},  //  3624, depth=1
{0xe07fe07fe07fe07f, 3212,{0xffff,0x00e9}},  //  3625, depth=2
{0xffffffc000000000,   -1,{0xffff,0x1699}},  //  3626, depth=1
{0xffffffc000000001,   -1,{0xffff,0x169a}},  //  3627, depth=1
{0x00fffffffffffff8,   -1,{0xffff,0x1f74}},  //  3628, depth=1
{0xffffffc000000003,   -1,{0xffff,0x169b}},  //  3629, depth=1
{0x0000000044444444, 3213,{0x00b8,0xffff}},  //  3630, depth=2
{0xfffff0fffffff0ff, 3215,{0xffff,0x051b}},  //  3631, depth=2
{0x0007fffff0000000, 3216,{0xffff,0x1916}},  //  3632, depth=2
{0x00000000e0003fff, 3218,{0x00d0,0xffff}},  //  3633, depth=3
{0x00fffffffffffffe,   -1,{0xffff,0x1ff6}},  //  3634, depth=1
{0x00ffffffffffffff,   -1,{0xffff,0x1037}},  //  3635, depth=1
{0x0100000000000000,   -1,{0xffff,0x1200}},  //  3636, depth=1
{0xfffffff800001fff,   -1,{0xffff,0x1769}},  //  3637, depth=1
{0x00f0000000f00000, 3223,{0xffff,0x0303}},  //  3638, depth=2
{0xfffff001fffff001,   -1,{0xffff,0x0514}},  //  3639, depth=1
{0x0000003ffc000000, 3224,{0xffff,0x198b}},  //  3640, depth=2
{0xffffffc00000000f, 3229,{0xffff,0x169d}},  //  3641, depth=2
{0xffffe7ffffffe7ff,   -1,{0xffff,0x04dd}},  //  3642
{0xfffffffff80000ff,   -1,{0xffff,0x196c}},  //  3643, depth=1
{0xfffe000000003fff,   -1,{0xffff,0x13dc}},  //  3644
{0x0000000700000007,   -1,{0xffff,0x0002}},  //  3645, depth=1
{0x07ffffffffffff80,   -1,{0xffff,0x1e73}},  //  3646
{0xffff01ffffffffff,   -1,{0xffff,0x1438}},  //  3647, depth=1
{0x003fffffc0000000,   -1,{0xffff,0x1897}},  //  3648
{0x03fffe0000000000, 3651,{0xffff,0x15d0}},  //  3649
{0x0003ffffff800000, 3234,{0xffff,0x1a5a}},  //  3650, depth=2
{0x1ffffff01ffffff0,   -1,{0xffff,0x0718}},  //  3651
{0x0000000000ffe000, 3236,{0x04ca,0x1cca}},  //  3652, depth=2
{0xff8fffffffffffff,   -1,{0xffff,0x127c}},  //  3653
{0x07fffffffffe0000,   -1,{0xffff,0x1be9}},  //  3654, depth=1
{0x000f000f000f000f,   -1,{0xffff,0x0023}},  //  3655, depth=1
{0x000ffffff0000000,   -1,{0xffff,0x1917}},  //  3656
{0xffffffc00000001f,   -1,{0xffff,0x169e}},  //  3657, depth=1
{0xfff80000007fffff,   -1,{0xffff,0x1363}},  //  3658
{0xf1fff1fff1fff1ff,   -1,{0xffff,0x012c}},  //  3659, depth=1
{0x0000000007fff000,   -1,{0x050e,0x1d0e}},  //  3660, depth=1
{0x00000000fff8007f, 3237,{0x0353,0xffff}},  //  3661, depth=2
{0xfffffffc00003fff,   -1,{0xffff,0x17ab}},  //  3662
{0xffc07fffffffffff,   -1,{0xffff,0x12b8}},  //  3663, depth=1
{0x0006000000060000,   -1,{0xffff,0x03c1}},  //  3664, depth=1
{0xfc0001fffc0001ff, 3240,{0xffff,0x018e}},  //  3665, depth=2
{0xc01fffffffffffff,   -1,{0xffff,0x10b6}},  //  3666
{0x000001ffffffffc0,   -1,{0xffff,0x1ea2}},  //  3667
{0x000000000f000000,   -1,{0x0203,0x1a03}},  //  3668, depth=1
{0xf001f001f001f001, 3242,{0xffff,0x0124}},  //  3669, depth=3
{0xfffffffff007ffff, 3671,{0xffff,0x1936}},  //  3670
{0xffff800000001fff,   -1,{0xffff,0x145d}},  //  3671
{0x0000000007c007c0, 3250,{0x02a4,0xffff}},  //  3672, depth=2
{0x003fffffffc00000,   -1,{0xffff,0x1a9f}},  //  3673
{0x001fffc0001fffc0, 3252,{0xffff,0x068e}},  //  3674, depth=2
{0x0000000081ff81ff,   -1,{0x0069,0xffff}},  //  3675, depth=1
{0x0000780000007800,   -1,{0xffff,0x0543}},  //  3676
{0xffffe00000000fff,   -1,{0xffff,0x14de}},  //  3677, depth=1
{0x0ffe000000000000,   -1,{0xffff,0x13ca}},  //  3678, depth=1
{0x00000000c0003fff,   -1,{0x008f,0xffff}},  //  3679, depth=1
{0xffff0003ffffffff,   -1,{0xffff,0x1431}},  //  3680
{0xfc000003fc000003, 3253,{0xffff,0x0187}},  //  3681, depth=2
{0x00ffffffe0000000, 3264,{0xffff,0x18da}},  //  3682, depth=2
{0xf00fffffffffffff,   -1,{0xffff,0x1137}},  //  3683, depth=1
{0x00000000fe00fe00,   -1,{0x01e6,0xffff}},  //  3684, depth=1
{0xc000000007ffffff,   -1,{0xffff,0x109c}},  //  3685, depth=1
{0xfffc007fffffffff,   -1,{0xffff,0x13b4}},  //  3686
{0xfe0000007fffffff,   -1,{0xffff,0x11e5}},  //  3687, depth=1
{0x000007ffffffff00,   -1,{0xffff,0x1e22}},  //  3688
{0xff8000000001ffff, 3265,{0xffff,0x1259}},  //  3689, depth=2
{0x003ffff000000000,   -1,{0xffff,0x1711}},  //  3690, depth=1
{0xfff80001ffffffff,   -1,{0xffff,0x136d}},  //  3691, depth=1
{0x00003ffc00003ffc,   -1,{0xffff,0x078b}},  //  3692, depth=1
{0x0000000083ffffff, 3267,{0x005a,0xffff}},  //  3693, depth=3
{0x000003fc00000000,   -1,{0xffff,0x1787}},  //  3694, depth=1
{0xc0000000000007ff,   -1,{0xffff,0x108c}},  //  3695
{0x000000fffffffc00, 3697,{0xffff,0x1d9d}},  //  3696
{0x00001ffffffc0000,   -1,{0xffff,0x1b9a}},  //  3697
{0xfe00000fffffffff,   -1,{0xffff,0x11ea}},  //  3698
{0xe0000007ffffffff,   -1,{0xffff,0x10e5}},  //  3699, depth=1
{0x0000000000003f00,   -1,{0x0605,0x1e05}},  //  3700, depth=1
{0xfffffffffff0ffff,   -1,{0xffff,0x1b3b}},  //  3701, depth=1
{0x0000007ffff00000,   -1,{0xffff,0x1b12}},  //  3702
{0xf0000000000003ff,   -1,{0xffff,0x110d}},  //  3703, depth=1
{0x07ffffffffffffe0,   -1,{0xffff,0x1ef5}},  //  3704
{0xffffffc00001ffff,   -1,{0xffff,0x16aa}},  //  3705
{0x03fffffffff00000, 3275,{0xffff,0x1b25}},  //  3706, depth=2
{0xfffffff8000007ff, 3276,{0xffff,0x1767}},  //  3707, depth=2
{0x00003e0000003e00,   -1,{0xffff,0x05c4}},  //  3708, depth=1
{0x000001fffffffffc,   -1,{0xffff,0x1fa6}},  //  3709
{0x000001fffffffffe,   -1,{0xffff,0x1fe7}},  //  3710
{0x0000020000000000,   -1,{0xffff,0x15c0}},  //  3711
{0x00ffffffff000000,   -1,{0xffff,0x1a1f}},  //  3712
{0xff83ff83ff83ff83, 3277,{0xffff,0x026a}},  //  3713, depth=2
{0x00007ffc00000000,   -1,{0xffff,0x178c}},  //  3714
{0x07fffffffffffff8, 3716,{0xffff,0x1f77}},  //  3715
{0x0003e00000000000, 3723,{0xffff,0x14c4}},  //  3716
{0xffffc007ffffffff,   -1,{0xffff,0x14b4}},  //  3717, depth=1
{0x000000007ff80000, 3281,{0x034b,0x1b4b}},  //  3718, depth=2
{0x00000000f8ffffff,   -1,{0x015c,0xffff}},  //  3719, depth=1
{0x07fffffffffffe00,   -1,{0xffff,0x1df1}},  //  3720, depth=1
{0xe00003ffe00003ff, 3282,{0xffff,0x00cc}},  //  3721, depth=2
{0x0ffffffc00000000,   -1,{0xffff,0x1799}},  //  3722, depth=1
{0x00000000007f8000,   -1,{0x0447,0x1c47}},  //  3723
{0x00003fffffffff00,   -1,{0xffff,0x1e25}},  //  3724
{0xfe000000003fffff,   -1,{0xffff,0x11dc}},  //  3725, depth=1
{0x0003ffff00000000,   -1,{0xffff,0x1811}},  //  3726, depth=1
{0xffff83ffffff83ff,   -1,{0xffff,0x045a}},  //  3727, depth=1
{0x00ffffffc0000000,   -1,{0xffff,0x1899}},  //  3728, depth=1
{0x07fffffffffffffc,   -1,{0xffff,0x1fb8}},  //  3729
{0x0000000000780078, 3291,{0x0363,0xffff}},  //  3730, depth=2
{0xf8000000001fffff, 3293,{0xffff,0x1159}},  //  3731, depth=2
{0x07fffffffffffffe,   -1,{0xffff,0x1ff9}},  //  3732
{0x00000000ffc03fff,   -1,{0x0297,0xffff}},  //  3733, depth=1
{0xffffffe01fffffff,   -1,{0xffff,0x16f7}},  //  3734
{0xfffc0fffffffffff,   -1,{0xffff,0x13b9}},  //  3735
{0x0ffffe0000000000,   -1,{0xffff,0x15d2}},  //  3736
{0x07ff07ff07ff07ff, 3296,{0xffff,0x002a}},  //  3737, depth=2
{0x0000003800000000,   -1,{0xffff,0x1742}},  //  3738
{0xfffffc0007ffffff,   -1,{0xffff,0x15b0}},  //  3739
{0x001fc00000000000,   -1,{0xffff,0x1486}},  //  3740, depth=1
{0xffc0ffffffc0ffff, 3297,{0xffff,0x0299}},  //  3741, depth=2
{0x0700070007000700, 3298,{0xffff,0x0222}},  //  3742, depth=2
{0xffffffffffff81ff,   -1,{0xffff,0x1c79}},  //  3743, depth=1
{0x7ff87ff87ff87ff8,   -1,{0xffff,0x036b}},  //  3744, depth=1
{0xff80000fff80000f, 3302,{0xffff,0x024c}},  //  3745, depth=2
{0x0001e0000001e000,   -1,{0xffff,0x04c3}},  //  3746
{0x00000ffffff00000,   -1,{0xffff,0x1b17}},  //  3747
{0x00007f0000000000,   -1,{0xffff,0x1606}},  //  3748, depth=1
{0x00000000e07fe07f, 3303,{0x00e9,0xffff}},  //  3749, depth=2
{0x0000000003ffff80,   -1,{0x0652,0x1e52}},  //  3750, depth=1
{0xffc3ffffffffffff,   -1,{0xffff,0x12bb}},  //  3751
{0xffffffdfffffffff,   -1,{0xffff,0x16be}},  //  3752
{0xe0000007e0000007, 3304,{0xffff,0x00c5}},  //  3753, depth=2
{0x0000000007800780, 3306,{0x0263,0xffff}},  //  3754, depth=2
{0xfffff807ffffffff,   -1,{0xffff,0x1577}},  //  3755, depth=1
{0x0000000000380000, 3309,{0x0342,0x1b42}},  //  3756, depth=2
{0x07ffffffe0000000,   -1,{0xffff,0x18dd}},  //  3757
{0xffffffe000000001,   -1,{0xffff,0x16db}},  //  3758
{0xffff8007ffff8007,   -1,{0xffff,0x0453}},  //  3759, depth=1
{0x003fffffffffff00,   -1,{0xffff,0x1e2d}},  //  3760, depth=1
{0xc07fffffc07fffff,   -1,{0xffff,0x0098}},  //  3761, depth=1
{0x0001fffffffffc00,   -1,{0xffff,0x1da6}},  //  3762, depth=1
{0xff80001fffffffff,   -1,{0xffff,0x126d}},  //  3763, depth=1
{0x00007fe000007fe0,   -1,{0xffff,0x06c9}},  //  3764, depth=1
{0xffffffe000000003,   -1,{0xffff,0x16dc}},  //  3765
{0xfffffffffffc01ff,   -1,{0xffff,0x1bb6}},  //  3766
{0xffffff8000001fff,   -1,{0xffff,0x1665}},  //  3767
{0x00000000e00fffff,   -1,{0x00d6,0xffff}},  //  3768
{0xfffff000000007ff,   -1,{0xffff,0x151e}},  //  3769
{0x03ffc00003ffc000, 3311,{0xffff,0x048b}},  //  3770, depth=2
{0x0000000080003fff,   -1,{0x004e,0xffff}},  //  3771, depth=1
{0x0000003800000038,   -1,{0xffff,0x0742}},  //  3772
{0x00000000e0000fff, 3317,{0x00ce,0xffff}},  //  3773, depth=2
{0xfffffffffffc0000,   -1,{0xffff,0x1bad}},  //  3774, depth=1
{0xfffffffffffc0001,   -1,{0xffff,0x1bae}},  //  3775, depth=1
{0x0000007ffe000000,   -1,{0xffff,0x19cd}},  //  3776, depth=1
{0x0007000700070007, 3321,{0xffff,0x0022}},  //  3777, depth=2
{0x001ff80000000000,   -1,{0xffff,0x1549}},  //  3778, depth=1
{0xfe0000003fffffff, 3324,{0xffff,0x11e4}},  //  3779, depth=2
{0x00000001ffff8000,   -1,{0xffff,0x1c51}},  //  3780, depth=1
{0xf80000000fffffff, 3327,{0xffff,0x1160}},  //  3781, depth=2
{0x0000000040404040, 3328,{0x00b0,0xffff}},  //  3782, depth=3
{0x000000000fff0fff,   -1,{0x002b,0xffff}},  //  3783, depth=1
{0x7800780078007800, 3337,{0xffff,0x0163}},  //  3784, depth=2
{0x00000000f00003ff,   -1,{0x010d,0xffff}},  //  3785, depth=1
{0x00fffffe00fffffe,   -1,{0xffff,0x07d6}},  //  3786, depth=1
{0xe003ffffffffffff,   -1,{0xffff,0x10f4}},  //  3787, depth=1
{0xffc003ffffc003ff, 3790,{0xffff,0x0293}},  //  3788
{0x00000000fff800ff, 3338,{0x0354,0xffff}},  //  3789, depth=2
{0xfe000001fe000001,   -1,{0xffff,0x01c7}},  //  3790
{0xe000000003ffffff,   -1,{0xffff,0x10dc}},  //  3791
{0x000007e000000000,   -1,{0xffff,0x16c5}},  //  3792, depth=1
{0xfffc03ffffffffff,   -1,{0xffff,0x13b7}},  //  3793, depth=1
{0x00ffff0000ffff00,   -1,{0xffff,0x060f}},  //  3794, depth=1
{0xffc0007fffc0007f, 3343,{0xffff,0x0290}},  //  3795, depth=2
{0xffffffe00000001f, 3803,{0xffff,0x16df}},  //  3796
{0x00000000fffff7ff, 3346,{0x051e,0xffff}},  //  3797, depth=2
{0x00000000fffff800,   -1,{0x0554,0x1d54}},  //  3798, depth=1
{0x00000000fffff801, 3349,{0x0555,0xffff}},  //  3799, depth=2
{0x4444444444444444, 3351,{0xffff,0x00b8}},  //  3800, depth=2
{0x00000000fffff803, 3352,{0x0556,0xffff}},  //  3801, depth=2
{0x0000ffffffff8000,   -1,{0xffff,0x1c60}},  //  3802, depth=1
{0xffc000000000ffff,   -1,{0xffff,0x1299}},  //  3803
{0xfffffff0007fffff, 3811,{0xffff,0x1732}},  //  3804
{0x00000000fffff807, 3360,{0x0557,0xffff}},  //  3805, depth=2
{0x00001ffff8000000,   -1,{0xffff,0x1951}},  //  3806, depth=1
{0xffffffffe1ffffff,   -1,{0xffff,0x18fb}},  //  3807, depth=1
{0x0f8000000f800000,   -1,{0xffff,0x0244}},  //  3808, depth=1
{0xffffff0000000fff,   -1,{0xffff,0x1623}},  //  3809, depth=1
{0x000000003ff80000,   -1,{0x034a,0x1b4a}},  //  3810, depth=1
{0xfffe0003fffe0003, 3818,{0xffff,0x03d0}},  //  3811
{0x0000000000380038, 3361,{0x0362,0xffff}},  //  3812, depth=2
{0x00000000fffff80f,   -1,{0x0558,0xffff}},  //  3813, depth=1
{0x0000000003ffffc0, 3365,{0x0693,0x1e93}},  //  3814, depth=2
{0xfff3fffffff3ffff,   -1,{0xffff,0x031d}},  //  3815, depth=1
{0x0000000000007000,   -1,{0x0502,0x1d02}},  //  3816, depth=1
{0xffffdfffffffdfff, 3369,{0xffff,0x049e}},  //  3817, depth=2
{0x00000000ff000fff,   -1,{0x0213,0xffff}},  //  3818
{0x00000000c0000fff,   -1,{0x008d,0xffff}},  //  3819, depth=1
{0x00ffffff80000000,   -1,{0xffff,0x1858}},  //  3820, depth=1
{0xfffc7fffffffffff,   -1,{0xffff,0x13bc}},  //  3821
{0x000fffffffff0000,   -1,{0xffff,0x1c23}},  //  3822, depth=1
{0xfffffffff0001fff,   -1,{0xffff,0x1930}},  //  3823, depth=1
{0x00000001fffc0000,   -1,{0xffff,0x1b8e}},  //  3824
{0xe00000000000ffff, 3372,{0xffff,0x10d2}},  //  3825, depth=2
{0x00001ffffffffc00,   -1,{0xffff,0x1da2}},  //  3826
{0xffffc00003ffffff,   -1,{0xffff,0x14ab}},  //  3827
{0x0000000000003f80,   -1,{0x0646,0x1e46}},  //  3828, depth=1
{0x00000000fffff81f,   -1,{0x0559,0xffff}},  //  3829, depth=1
{0x0003ffe00003ffe0, 3374,{0xffff,0x06cc}},  //  3830, depth=2
{0x00000000e03fe03f,   -1,{0x00e8,0xffff}},  //  3831, depth=1
{0x0000001fffe00000,   -1,{0xffff,0x1acf}},  //  3832, depth=1
{0x00000000fe1fffff, 3375,{0x01db,0xffff}},  //  3833, depth=3
{0xf8ffffffffffffff,   -1,{0xffff,0x117c}},  //  3834
{0x00000000ffe003ff, 3378,{0x02d4,0xffff}},  //  3835, depth=2
{0xfff800003fffffff, 3838,{0xffff,0x136a}},  //  3836
{0xfc000000000003ff, 3379,{0xffff,0x118f}},  //  3837, depth=2
{0xffe07fffffe07fff,   -1,{0xffff,0x02d9}},  //  3838
{0x80003fffffffffff, 3381,{0xffff,0x106e}},  //  3839, depth=2
{0x0000000000000f00, 3382,{0x0603,0x1e03}},  //  3840, depth=2
{0xc000000000007fff,   -1,{0xffff,0x1090}},  //  3841, depth=1
{0x00000ffffe000000,   -1,{0xffff,0x19d2}},  //  3842, depth=1
{0xfff8000000ffffff, 3386,{0xffff,0x1364}},  //  3843, depth=2
{0xffffff00000fffff,   -1,{0xffff,0x162b}},  //  3844
{0x000003fffffff000, 3847,{0xffff,0x1d1d}},  //  3845
{0x0000000003ffffe0,   -1,{0x06d4,0x1ed4}},  //  3846, depth=1
{0x00007ffffff00000,   -1,{0xffff,0x1b1a}},  //  3847
{0x03ffc00000000000, 3398,{0xffff,0x148b}},  //  3848, depth=2
{0xffffffe00000003f,   -1,{0xffff,0x16e0}},  //  3849
{0x0000fffffffc0000,   -1,{0xffff,0x1b9d}},  //  3850
{0xfffffffffffe0001, 3852,{0xffff,0x1bef}},  //  3851
{0x00000000ffcfffcf,   -1,{0x02ad,0xffff}},  //  3852
{0xfffffff1fffffff1, 3399,{0xffff,0x071c}},  //  3853, depth=2
{0x00000007ff000000,   -1,{0xffff,0x1a0a}},  //  3854, depth=1
{0xfffffffffffe0003,   -1,{0xffff,0x1bf0}},  //  3855
{0x000000001ff80000,   -1,{0x0349,0x1b49}},  //  3856, depth=1
{0x9fffffffffffffff,   -1,{0xffff,0x107d}},  //  3857, depth=1
{0x000000003e3e3e3e,   -1,{0x01f4,0xffff}},  //  3858, depth=1
{0xfffffffffffe000f,   -1,{0xffff,0x1bf2}},  //  3859
{0x000c000000000000,   -1,{0xffff,0x1381}},  //  3860, depth=1
{0x00000000fffff83f,   -1,{0x055a,0xffff}},  //  3861, depth=1
{0x7f8000007f800000, 3400,{0xffff,0x0247}},  //  3862, depth=2
{0xfffffc000003ffff, 3401,{0xffff,0x15a7}},  //  3863, depth=2
{0x0000000040004000,   -1,{0x00a0,0xffff}},  //  3864, depth=1
{0x0ffffffffffff000,   -1,{0xffff,0x1d2f}},  //  3865
{0x3fe000003fe00000, 3405,{0xffff,0x02c8}},  //  3866, depth=3
{0xfe1ffffffe1fffff, 3409,{0xffff,0x01db}},  //  3867, depth=3
{0x00000000ffffe000,   -1,{0x04d2,0x1cd2}},  //  3868, depth=1
{0x00000000ffffe001,   -1,{0x04d3,0xffff}},  //  3869, depth=1
{0x0040000000400000, 3417,{0xffff,0x0280}},  //  3870, depth=2
{0x00000000ffffe003, 3421,{0x04d4,0xffff}},  //  3871, depth=2
{0xfe00000000000000,   -1,{0xffff,0x11c6}},  //  3872, depth=1
{0x00000000ffc00fff, 3425,{0x0295,0xffff}},  //  3873, depth=5
{0x00000000007ff800, 3434,{0x054b,0x1d4b}},  //  3874, depth=3
{0x00000000ffffe007, 3440,{0x04d5,0xffff}},  //  3875, depth=2
{0x0000000003fffffe, 3443,{0x07d8,0x1fd8}},  //  3876, depth=2
{0x0000000003ffffff, 3445,{0x0019,0x1019}},  //  3877, depth=2
{0x0000000004000000, 3448,{0x0180,0x1980}},  //  3878, depth=2
{0x00000000ff803fff, 3455,{0x0256,0xffff}},  //  3879, depth=3
{0x01ffc00001ffc000,   -1,{0xffff,0x048a}},  //  3880, depth=1
{0xfffff8000000ffff, 3460,{0xffff,0x1564}},  //  3881, depth=2
{0x1fffffffffffff80,   -1,{0xffff,0x1e75}},  //  3882
{0x00000000ffffe00f, 3461,{0x04d6,0xffff}},  //  3883, depth=2
{0xfe07ffffffffffff,   -1,{0xffff,0x11f9}},  //  3884
{0xf80001fff80001ff,   -1,{0xffff,0x014d}},  //  3885, depth=1
{0x03fff80000000000,   -1,{0xffff,0x154e}},  //  3886, depth=1
{0x000000008f8f8f8f, 3462,{0x0074,0xffff}},  //  3887, depth=2
{0x003fffffffffff80,   -1,{0xffff,0x1e6e}},  //  3888, depth=1
{0xff0000000fffffff,   -1,{0xffff,0x1223}},  //  3889
{0x0000000001fc0000,   -1,{0x0386,0x1b86}},  //  3890, depth=1
{0xfffffffffffe001f,   -1,{0xffff,0x1bf3}},  //  3891
{0x0000700000007000, 3464,{0xffff,0x0502}},  //  3892, depth=3
{0xfffffffff00007ff,   -1,{0xffff,0x192e}},  //  3893, depth=1
{0x00fe00fe00fe00fe,   -1,{0xffff,0x03e6}},  //  3894, depth=1
{0x003fffff00000000,   -1,{0xffff,0x1815}},  //  3895
{0xfffffffffff00fff, 3897,{0xffff,0x1b37}},  //  3896
{0xffff83ffffffffff, 3900,{0xffff,0x147a}},  //  3897
{0x1ffff0001ffff000,   -1,{0xffff,0x0510}},  //  3898, depth=1
{0x00000000ffffe01f, 3469,{0x04d7,0xffff}},  //  3899, depth=2
{0xffe3ffffffe3ffff,   -1,{0xffff,0x02dc}},  //  3900
{0xf8000003f8000003, 3471,{0xffff,0x0146}},  //  3901, depth=3
{0x00fe000000fe0000,   -1,{0xffff,0x03c6}},  //  3902, depth=1
{0xffffffcfffffffcf, 3474,{0xffff,0x069d}},  //  3903, depth=2
{0x00003ffffffe0000,   -1,{0xffff,0x1bdc}},  //  3904, depth=1
{0xfffffffffffffbff,   -1,{0xffff,0x1d7e}},  //  3905, depth=1
{0x07fffff000000000, 3475,{0xffff,0x1716}},  //  3906, depth=2
{0xfffffffffffffc01,   -1,{0xffff,0x1db6}},  //  3907, depth=1
{0xf0f0f0f0f0f0f0f0,   -1,{0xffff,0x0133}},  //  3908, depth=1
{0xfffff8fffffff8ff, 3478,{0xffff,0x055c}},  //  3909, depth=3
{0xffffc00000003fff, 3915,{0xffff,0x149f}},  //  3910
{0x0000000080000fff, 3488,{0x004c,0xffff}},  //  3911, depth=2
{0x000000c000000000,   -1,{0xffff,0x1681}},  //  3912, depth=1
{0x00000000dfffdfff, 3489,{0x00ae,0xffff}},  //  3913, depth=2
{0x01fc000000000000,   -1,{0xffff,0x1386}},  //  3914, depth=1
{0x8000000001ffffff,   -1,{0xffff,0x1059}},  //  3915
{0x00000000e0e0e0e0,   -1,{0x00f2,0xffff}},  //  3916, depth=1
{0x00000000f003ffff, 3490,{0x0115,0xffff}},  //  3917, depth=2
{0x0000000007000700, 3491,{0x0222,0xffff}},  //  3918, depth=2
{0xffffffe00000007f,   -1,{0xffff,0x16e1}},  //  3919
{0xfffffffffffe003f, 3938,{0xffff,0x1bf4}},  //  3920
{0xfffffffffffffc0f,   -1,{0xffff,0x1db9}},  //  3921, depth=1
{0x1ffffffffffe0000,   -1,{0xffff,0x1beb}},  //  3922, depth=1
{0xffffffffff0fffff,   -1,{0xffff,0x1a3b}},  //  3923, depth=1
{0x0000000000003fe0,   -1,{0x06c8,0x1ec8}},  //  3924, depth=1
{0x00000000fffff87f, 3493,{0x055b,0xffff}},  //  3925, depth=2
{0xfff0000000000000,   -1,{0xffff,0x130b}},  //  3926, depth=1
{0xfff0000000000001,   -1,{0xffff,0x130c}},  //  3927, depth=1
{0x00ffffffff800000,   -1,{0xffff,0x1a60}},  //  3928, depth=1
{0xfff0000000000003,   -1,{0xffff,0x130d}},  //  3929, depth=1
{0x001fe000001fe000,   -1,{0xffff,0x04c7}},  //  3930, depth=1
{0x00000000ffffe03f,   -1,{0x04d8,0xffff}},  //  3931, depth=1
{0x0000000000ff8000, 3494,{0x0448,0x1c48}},  //  3932, depth=2
{0xfff0000000000007,   -1,{0xffff,0x130e}},  //  3933, depth=1
{0x000000003c3c3c3c,   -1,{0x01b3,0xffff}},  //  3934, depth=1
{0xfe0000000000003f,   -1,{0xffff,0x11cc}},  //  3935, depth=1
{0x00000001f8000000, 3495,{0xffff,0x1945}},  //  3936, depth=2
{0x00000000ffffc7ff, 3498,{0x049c,0xffff}},  //  3937, depth=2
{0xff80003fffffffff,   -1,{0xffff,0x126e}},  //  3938
{0x00000000ffe3ffe3,   -1,{0x02ec,0xffff}},  //  3939, depth=1
{0x0000000000003ff0,   -1,{0x0709,0x1f09}},  //  3940, depth=1
{0xfff000000000000f,   -1,{0xffff,0x130f}},  //  3941, depth=1
{0x4040404040404040, 3501,{0xffff,0x00b0}},  //  3942, depth=2
{0xffffffffffe007ff,   -1,{0xffff,0x1af5}},  //  3943, depth=1
{0x00000000007fe000,   -1,{0x04c9,0x1cc9}},  //  3944, depth=1
{0x00000000fc001fff, 3502,{0x0192,0xffff}},  //  3945, depth=3
{0x7000700070007000, 3509,{0xffff,0x0122}},  //  3946, depth=2
{0xffc00000001fffff,   -1,{0xffff,0x129e}},  //  3947, depth=1
{0x0000000000003ff8, 3510,{0x074a,0x1f4a}},  //  3948, depth=2
{0x7ffffffffffc0000,   -1,{0xffff,0x1bac}},  //  3949
{0x00003fffffffffc0,   -1,{0xffff,0x1ea7}},  //  3950
{0xffffffcfffffffff,   -1,{0xffff,0x16bd}},  //  3951, depth=1
{0x0000000000003ffc, 3512,{0x078b,0x1f8b}},  //  3952, depth=2
{0xfffff00000007fff,   -1,{0xffff,0x1522}},  //  3953, depth=1
{0x0000000000003ffe, 3516,{0x07cc,0x1fcc}},  //  3954, depth=2
{0x0000000000003fff,   -1,{0x000d,0x100d}},  //  3955, depth=1
{0x0000000000004000,   -1,{0x0480,0x1c80}},  //  3956, depth=1
{0xfff000000000001f,   -1,{0xffff,0x1310}},  //  3957, depth=1
{0x0003f0000003f000, 3517,{0xffff,0x0505}},  //  3958, depth=2
{0xff800fffff800fff,   -1,{0xffff,0x0254}},  //  3959, depth=1
{0xfffffffffffffe01, 3963,{0xffff,0x1df7}},  //  3960
{0xc07fffffffffffff,   -1,{0xffff,0x10b8}},  //  3961, depth=1
{0x000001ffffffff00,   -1,{0xffff,0x1e20}},  //  3962, depth=1
{0xfffffff07fffffff, 3964,{0xffff,0x173a}},  //  3963
{0xfffffc7ffffffc7f,   -1,{0xffff,0x059c}},  //  3964
{0xfffffffff801ffff,   -1,{0xffff,0x1975}},  //  3965, depth=1
{0x0fffffc00fffffc0, 3520,{0xffff,0x0695}},  //  3966, depth=2
{0x00000000ffe3ffff, 3526,{0x02dc,0xffff}},  //  3967, depth=3
{0x0000000000000f80,   -1,{0x0644,0x1e44}},  //  3968, depth=1
{0xfc0000000003ffff, 3530,{0xffff,0x1197}},  //  3969, depth=2
{0x00003ffffffffe00,   -1,{0xffff,0x1de4}},  //  3970, depth=1
{0xfffffff80007ffff,   -1,{0xffff,0x176f}},  //  3971, depth=1
{0xfff7ffffffffffff,   -1,{0xffff,0x133e}},  //  3972
{0x000007fffffffff0,   -1,{0xffff,0x1f26}},  //  3973
{0x0000000000030000,   -1,{0x0401,0x1c01}},  //  3974, depth=1
{0xffffffffffffe3ff,   -1,{0xffff,0x1cfc}},  //  3975, depth=1
{0x07ffffffffffff00,   -1,{0xffff,0x1e32}},  //  3976, depth=1
{0x0000000000030003, 3532,{0x0021,0xffff}},  //  3977, depth=2
{0xfff8000000000003,   -1,{0xffff,0x134e}},  //  3978
{0xfffffffffe000fff,   -1,{0xffff,0x19f2}},  //  3979, depth=1
{0x000003fe000003fe,   -1,{0xffff,0x07c8}},  //  3980, depth=1
{0x80000000000fffff, 3536,{0xffff,0x1054}},  //  3981, depth=2
{0x000000007ffff800, 3538,{0x0553,0x1d53}},  //  3982, depth=2
{0xe1ffe1ffe1ffe1ff, 3540,{0xffff,0x00eb}},  //  3983, depth=2
{0x003fffffffffffe0,   -1,{0xffff,0x1ef0}},  //  3984, depth=1
{0xffffffe0007fffff,   -1,{0xffff,0x16f1}},  //  3985, depth=1
{0xfff8000000000007,   -1,{0xffff,0x134f}},  //  3986
{0x000007fffffffff8,   -1,{0xffff,0x1f67}},  //  3987
{0x0000030000000300, 3545,{0xffff,0x0601}},  //  3988, depth=2
{0x80007fff80007fff, 3546,{0xffff,0x004f}},  //  3989, depth=2
{0xffffffffc0ffffff,   -1,{0xffff,0x18b9}},  //  3990
{0xfff800000000000f,   -1,{0xffff,0x1350}},  //  3991
{0x0fc0000000000000,   -1,{0xffff,0x1285}},  //  3992
{0xe001e001e001e001,   -1,{0xffff,0x00e3}},  //  3993, depth=1
{0xfffffffffffffe1f,   -1,{0xffff,0x1dfb}},  //  3994
{0x9fffffff9fffffff, 3551,{0xffff,0x005d}},  //  3995, depth=2
{0x0001ffc00001ffc0,   -1,{0xffff,0x068a}},  //  3996, depth=1
{0xffc000000fffffff,   -1,{0xffff,0x12a5}},  //  3997, depth=1
{0x1ff8000000000000,   -1,{0xffff,0x1349}},  //  3998, depth=1
{0xfe0000000000007f,   -1,{0xffff,0x11cd}},  //  3999, depth=1
{0x003ffffffffffff0,   -1,{0xffff,0x1f31}},  //  4000, depth=1
{0x00000001fffffc00,   -1,{0xffff,0x1d96}},  //  4001
{0x03c003c003c003c0, 3552,{0xffff,0x02a3}},  //  4002, depth=2
{0x0000000080ff80ff,   -1,{0x0068,0xffff}},  //  4003, depth=1
{0x001ffe0000000000, 3553,{0xffff,0x15cb}},  //  4004, depth=2
{0xffffff03ffffffff,   -1,{0xffff,0x1639}},  //  4005
{0xfffff000fffff000, 3554,{0xffff,0x0513}},  //  4006, depth=2
{0xfff800000000001f, 4010,{0xffff,0x1351}},  //  4007
{0x003ffffffffffff8,   -1,{0xffff,0x1f72}},  //  4008, depth=1
{0x00000000fc1ffc1f,   -1,{0x01aa,0xffff}},  //  4009, depth=1
{0x00000000f0007fff,   -1,{0x0112,0xffff}},  //  4010
{0x00000000f87fffff, 3558,{0x015b,0xffff}},  //  4011, depth=2
{0x0000000600000006, 3560,{0xffff,0x07c1}},  //  4012, depth=2
{0xfffe7ffffffe7fff,   -1,{0xffff,0x03dd}},  //  4013, depth=1
{0x003ffffffffffffe,   -1,{0xffff,0x1ff4}},  //  4014, depth=1
{0x00000000fc0007ff, 3563,{0x0190,0xffff}},  //  4015, depth=2
{0x0040000000000000,   -1,{0xffff,0x1280}},  //  4016, depth=1
{0x0001fff000000000,   -1,{0xffff,0x170c}},  //  4017
{0x007ffff8007ffff8,   -1,{0xffff,0x0753}},  //  4018
{0x00000000ff800fff, 3568,{0x0254,0xffff}},  //  4019, depth=3
{0x00000000003ff800, 3576,{0x054a,0x1d4a}},  //  4020, depth=3
{0xfffe000003ffffff,   -1,{0xffff,0x13e8}},  //  4021, depth=1
{0x0000000001fffc00,   -1,{0x058e,0x1d8e}},  //  4022, depth=1
{0xc1c1c1c1c1c1c1c1,   -1,{0xffff,0x00b2}},  //  4023, depth=1
{0x0000000003c00000,   -1,{0x0283,0x1a83}},  //  4024, depth=1
{0xffffe7ffffffffff,   -1,{0xffff,0x14fd}},  //  4025, depth=1
{0x1fffffffffffffe0, 4027,{0xffff,0x1ef7}},  //  4026
{0x000f800000000000, 4035,{0xffff,0x1444}},  //  4027
{0x000000003f803f80,   -1,{0x0266,0xffff}},  //  4028, depth=1
{0xfffffffffffc00ff,   -1,{0xffff,0x1bb5}},  //  4029, depth=1
{0x7ffffffff8000000,   -1,{0xffff,0x1963}},  //  4030, depth=1
{0xfff001fffff001ff,   -1,{0xffff,0x0314}},  //  4031, depth=1
{0x0000000000000fc0,   -1,{0x0685,0x1e85}},  //  4032, depth=1
{0xfffffffffffffc7f,   -1,{0xffff,0x1dbc}},  //  4033, depth=1
{0x00000f8000000f80,   -1,{0xffff,0x0644}},  //  4034, depth=1
{0x0000000001fe0000,   -1,{0x03c7,0x1bc7}},  //  4035
{0x00003ffffffffff8,   -1,{0xffff,0x1f6a}},  //  4036
{0x0000fffffffffc00,   -1,{0xffff,0x1da5}},  //  4037
{0x00003ffffffffffe,   -1,{0xffff,0x1fec}},  //  4038
{0xfffffffffffffe3f,   -1,{0xffff,0x1dfc}},  //  4039
{0x1ffffffffffffff0,   -1,{0xffff,0x1f38}},  //  4040
{0x00000003ffffff00,   -1,{0xffff,0x1e19}},  //  4041
{0x0007f00000000000,   -1,{0xffff,0x1506}},  //  4042, depth=1
{0xf0007ffff0007fff, 3578,{0xffff,0x0112}},  //  4043, depth=2
{0x00001fc000000000,   -1,{0xffff,0x1686}},  //  4044, depth=1
{0x00000000fff801ff,   -1,{0x0355,0xffff}},  //  4045, depth=1
{0x00000000000e0000, 3595,{0x03c2,0x1bc2}},  //  4046, depth=2
{0xfff00003fff00003,   -1,{0xffff,0x030d}},  //  4047, depth=1
{0x00001ff800001ff8,   -1,{0xffff,0x0749}},  //  4048, depth=1
{0xffff80000007ffff,   -1,{0xffff,0x1463}},  //  4049
{0x1ffffffffffffff8,   -1,{0xffff,0x1f79}},  //  4050
{0x0007fc0000000000,   -1,{0xffff,0x1588}},  //  4051
{0x000000007fffe000,   -1,{0x04d1,0x1cd1}},  //  4052, depth=1
{0x00000000fffff8ff, 3599,{0x055c,0xffff}},  //  4053, depth=3
{0x1ffffffffffffffc,   -1,{0xffff,0x1fba}},  //  4054
{0x8003ffff8003ffff, 3606,{0xffff,0x0052}},  //  4055, depth=2
{0xfffffffe01ffffff, 4058,{0xffff,0x17f7}},  //  4056
{0x1111111111111111,   -1,{0xffff,0x0038}},  //  4057, depth=1
{0xf07fffffffffffff,   -1,{0xffff,0x113a}},  //  4058
{0xfff80000ffffffff,   -1,{0xffff,0x136c}},  //  4059, depth=1
{0x00000000000e000e, 3607,{0x03e2,0xffff}},  //  4060, depth=2
{0x000001ffff800000,   -1,{0xffff,0x1a51}},  //  4061
{0x003fffffe0000000,   -1,{0xffff,0x18d8}},  //  4062, depth=1
{0xfffffff03fffffff,   -1,{0xffff,0x1739}},  //  4063
{0x0000000000000fe0,   -1,{0x06c6,0x1ec6}},  //  4064, depth=1
{0xe0ffffffffffffff, 3608,{0xffff,0x10fa}},  //  4065, depth=2
{0x7e00000000000000,   -1,{0xffff,0x11c5}},  //  4066
{0x0000780000000000, 4068,{0xffff,0x1543}},  //  4067
{0x00fffffc00fffffc,   -1,{0xffff,0x0795}},  //  4068
{0x000007ffc0000000,   -1,{0xffff,0x188c}},  //  4069
{0xfffffffff803ffff,   -1,{0xffff,0x1976}},  //  4070
{0x3ffff80000000000,   -1,{0xffff,0x1552}},  //  4071
{0x0003000000000000,   -1,{0xffff,0x1401}},  //  4072, depth=1
{0xffffc03fffffc03f,   -1,{0xffff,0x0497}},  //  4073, depth=1
{0x000000003ffff800,   -1,{0x0552,0x1d52}},  //  4074, depth=1
{0xfffffffe000fffff,   -1,{0xffff,0x17f2}},  //  4075, depth=1
{0x007fffffffffc000,   -1,{0xffff,0x1ca8}},  //  4076, depth=1
{0xffe000000003ffff,   -1,{0xffff,0x12dc}},  //  4077, depth=1
{0xfffffff01fffffff,   -1,{0xffff,0x1738}},  //  4078
{0xff00ffffff00ffff,   -1,{0xffff,0x0217}},  //  4079
{0x00001c0000001c00, 3609,{0xffff,0x0582}},  //  4080, depth=3
{0xff000007ffffffff,   -1,{0xffff,0x122a}},  //  4081
{0x000000e000000000,   -1,{0xffff,0x16c2}},  //  4082
{0xf800000003ffffff,   -1,{0xffff,0x115e}},  //  4083
{0x3c3c3c3c3c3c3c3c,   -1,{0xffff,0x01b3}},  //  4084, depth=1
{0xfffffffffc1fffff,   -1,{0xffff,0x19ba}},  //  4085, depth=1
{0x0000000038383838,   -1,{0x0172,0xffff}},  //  4086, depth=1
{0xff80001fff80001f,   -1,{0xffff,0x024d}},  //  4087, depth=1
{0x0000000000000ff8,   -1,{0x0748,0x1f48}},  //  4088, depth=1
{0x0007800000078000,   -1,{0xffff,0x0443}},  //  4089
{0x00000000003fe000, 3614,{0x04c8,0x1cc8}},  //  4090, depth=2
{0x0ffffffffc000000,   -1,{0xffff,0x19a1}},  //  4091
{0x0000000000000ffc,   -1,{0x0789,0x1f89}},  //  4092, depth=1
{0xf80000000000ffff, 3618,{0xffff,0x1154}},  //  4093, depth=2
{0x0000000000000ffe, 3621,{0x07ca,0x1fca}},  //  4094, depth=2
{0x0000000000000fff, 3642,{0x000b,0x100b}},  //  4095, depth=2
{0x0000000000001000,   -1,{0x0500,0x1d00}},  //  4096, depth=1
{0xfffe07ffffffffff,   -1,{0xffff,0x13f9}},  //  4097
{0x000007ffffc00000,   -1,{0xffff,0x1a94}},  //  4098
{0x80000003ffffffff, 3644,{0xffff,0x1062}},  //  4099, depth=2
{0x1fffffffc0000000,   -1,{0xffff,0x189e}},  //  4100
{0xff003fffff003fff,   -1,{0xffff,0x0215}},  //  4101, depth=1
{0x01fff00000000000,   -1,{0xffff,0x150c}},  //  4102
{0x003fffffff000000,   -1,{0xffff,0x1a1d}},  //  4103
{0x000000c0000000c0, 3646,{0xffff,0x0681}},  //  4104, depth=2
{0x00000000fff9fff9,   -1,{0x036d,0xffff}},  //  4105, depth=1
{0x00007ff000007ff0,   -1,{0xffff,0x070a}},  //  4106, depth=1
{0xfffffe000000ffff,   -1,{0xffff,0x15e6}},  //  4107, depth=1
{0x0007ff8000000000, 3648,{0xffff,0x164b}},  //  4108, depth=2
{0xf003fffff003ffff,   -1,{0xffff,0x0115}},  //  4109, depth=1
{0x7fe07fe07fe07fe0,   -1,{0xffff,0x02e9}},  //  4110, depth=1
{0x00000000fff9ffff,   -1,{0x035d,0xffff}},  //  4111, depth=1
{0x00000000000ffe00, 3649,{0x05ca,0x1dca}},  //  4112, depth=3
{0xffffffefffffffff,   -1,{0xffff,0x16fe}},  //  4113
{0x000000000fe00fe0,   -1,{0x02e6,0xffff}},  //  4114, depth=1
{0xf83ff83ff83ff83f, 3653,{0xffff,0x016a}},  //  4115, depth=2
{0xfffffff000000000,   -1,{0xffff,0x171b}},  //  4116
{0xfc00003fffffffff,   -1,{0xffff,0x11ab}},  //  4117, depth=1
{0x000007f000000000,   -1,{0xffff,0x1706}},  //  4118, depth=1
{0xfffffff000000001,   -1,{0xffff,0x171c}},  //  4119
{0x000000001ffff800,   -1,{0x0551,0x1d51}},  //  4120, depth=1
{0xc000000000001fff,   -1,{0xffff,0x108e}},  //  4121, depth=1
{0x0000003fffff0000,   -1,{0xffff,0x1c15}},  //  4122, depth=1
{0x00000000ffffe0ff,   -1,{0x04da,0xffff}},  //  4123, depth=1
{0xfffffffffffe00ff,   -1,{0xffff,0x1bf6}},  //  4124
{0x000fff8000000000,   -1,{0xffff,0x164c}},  //  4125
{0x001ffffffffff000,   -1,{0xffff,0x1d28}},  //  4126, depth=1
{0xfe000000000000ff,   -1,{0xffff,0x11ce}},  //  4127, depth=1
{0x00000000fff0ffff,   -1,{0x031b,0xffff}},  //  4128
{0xffffffff01ffffff,   -1,{0xffff,0x1838}},  //  4129, depth=1
{0xfffffff00000000f,   -1,{0xffff,0x171f}},  //  4130
{0xffffffe0003fffff,   -1,{0xffff,0x16f0}},  //  4131, depth=1
{0x00000000000ffffc,   -1,{0x0791,0x1f91}},  //  4132
{0xfffe3fffffffffff,   -1,{0xffff,0x13fc}},  //  4133
{0x0001ffe000000000, 3656,{0xffff,0x16cb}},  //  4134, depth=2
{0xff800000000003ff, 3658,{0xffff,0x1252}},  //  4135, depth=2
{0x03ffffc003ffffc0,   -1,{0xffff,0x0693}},  //  4136
{0xffffffc0000001ff,   -1,{0xffff,0x16a2}},  //  4137, depth=1
{0x0000000078007800,   -1,{0x0163,0xffff}},  //  4138, depth=1
{0xffffe00001ffffff,   -1,{0xffff,0x14eb}},  //  4139
{0x00003fffffc00000,   -1,{0xffff,0x1a97}},  //  4140
{0xfc7fffffffffffff,   -1,{0xffff,0x11bc}},  //  4141
{0xffffff800007ffff,   -1,{0xffff,0x166b}},  //  4142
{0x8001ffffffffffff, 3662,{0xffff,0x1071}},  //  4143, depth=2
{0x000000003fffe000,   -1,{0x04d0,0x1cd0}},  //  4144, depth=1
{0xfffffff00000001f,   -1,{0xffff,0x1720}},  //  4145
{0x00000ffc00000000,   -1,{0xffff,0x1789}},  //  4146, depth=1
{0x00000000ffff7fff,   -1,{0x041e,0xffff}},  //  4147, depth=1
{0x00000000ffff8000,   -1,{0x0450,0x1c50}},  //  4148, depth=1
{0x00000000ffff8001,   -1,{0x0451,0xffff}},  //  4149, depth=1
{0x0000fffc0000fffc,   -1,{0xffff,0x078d}},  //  4150, depth=1
{0x00000000ffff8003, 3666,{0x0452,0xffff}},  //  4151, depth=2
{0xff03ffffffffffff,   -1,{0xffff,0x1239}},  //  4152
{0xff80000007ffffff,   -1,{0xffff,0x1263}},  //  4153
{0x007ffffffc000000, 3667,{0xffff,0x199c}},  //  4154, depth=2
{0x00000000ffff8007,   -1,{0x0453,0xffff}},  //  4155, depth=1
{0x00003ff000000000,   -1,{0xffff,0x1709}},  //  4156, depth=1
{0x00000000f83fffff, 3670,{0x015a,0xffff}},  //  4157, depth=3
{0x07ffe00007ffe000,   -1,{0xffff,0x04cd}},  //  4158
{0xffffe00fffffe00f,   -1,{0xffff,0x04d6}},  //  4159, depth=1
{0xfffffffffff807ff, 4165,{0xffff,0x1b77}},  //  4160
{0xfffffffffffffcff,   -1,{0xffff,0x1dbd}},  //  4161, depth=1
{0x3ffffffc3ffffffc, 3673,{0xffff,0x079b}},  //  4162, depth=2
{0x00000000ffff800f,   -1,{0x0454,0xffff}},  //  4163, depth=1
{0x00000003fffe0000,   -1,{0xffff,0x1bd0}},  //  4164, depth=1
{0xffffc1ffffffffff,   -1,{0xffff,0x14ba}},  //  4165
{0x0000fe000000fe00,   -1,{0xffff,0x05c6}},  //  4166, depth=1
{0x00000000807f807f,   -1,{0x0067,0xffff}},  //  4167, depth=1
{0x07ffffffffffffc0,   -1,{0xffff,0x1eb4}},  //  4168, depth=1
{0xfffffff00000003f,   -1,{0xffff,0x1721}},  //  4169
{0x0000000003800000, 3676,{0x0242,0x1a42}},  //  4170, depth=2
{0x00000000ff003fff, 3680,{0x0215,0xffff}},  //  4171, depth=2
{0x0003fff00003fff0,   -1,{0xffff,0x070d}},  //  4172, depth=1
{0xffffffffffffff00, 4175,{0xffff,0x1e37}},  //  4173
{0x1ffffff000000000,   -1,{0xffff,0x1718}},  //  4174, depth=1
{0x007c000000000000, 4177,{0xffff,0x1384}},  //  4175
{0x1ffff8001ffff800,   -1,{0xffff,0x0551}},  //  4176, depth=1
{0x000000000ff00000,   -1,{0x0307,0x1b07}},  //  4177
{0x3ffffe003ffffe00,   -1,{0xffff,0x05d4}},  //  4178, depth=1
{0x00000000ffff801f, 3686,{0x0455,0xffff}},  //  4179, depth=2
{0xffffffffe07fffff,   -1,{0xffff,0x18f9}},  //  4180
{0xfff00000000000ff,   -1,{0xffff,0x1313}},  //  4181, depth=1
{0xfffffc0000001fff,   -1,{0xffff,0x15a2}},  //  4182
{0xffc00007ffffffff,   -1,{0xffff,0x12ac}},  //  4183, depth=1
{0x000fffffffe00000,   -1,{0xffff,0x1ade}},  //  4184, depth=1
{0xffffffe03fffffff,   -1,{0xffff,0x16f8}},  //  4185, depth=1
{0x000001ffffffffe0,   -1,{0xffff,0x1ee3}},  //  4186, depth=1
{0xffffffffffffff0f, 4193,{0xffff,0x1e3b}},  //  4187
{0x01fffffff0000000, 3688,{0xffff,0x191c}},  //  4188, depth=2
{0xfe01fffffe01ffff,   -1,{0xffff,0x01d7}},  //  4189, depth=1
{0x000000001fffe000,   -1,{0x04cf,0x1ccf}},  //  4190, depth=1
{0x0000000087878787, 3695,{0x0073,0xffff}},  //  4191, depth=2
{0x000000003f003f00, 3696,{0x0225,0xffff}},  //  4192, depth=3
{0xff0000000000ffff,   -1,{0xffff,0x1217}},  //  4193
{0x00000000fffc0001,   -1,{0x038e,0xffff}},  //  4194
{0x00000000fe00ffff,   -1,{0x01d6,0xffff}},  //  4195, depth=1
{0x0000ffc000000000,   -1,{0xffff,0x1689}},  //  4196, depth=1
{0xf81ffffff81fffff, 3698,{0xffff,0x0159}},  //  4197, depth=2
{0xfffcfffcfffcfffc, 3702,{0xffff,0x03ad}},  //  4198, depth=2
{0xfffffff8001fffff,   -1,{0xffff,0x1771}},  //  4199
{0x003fffff80000000, 3704,{0xffff,0x1856}},  //  4200, depth=2
{0x000000fffffffe00,   -1,{0xffff,0x1dde}},  //  4201
{0x000001fffffffff0,   -1,{0xffff,0x1f24}},  //  4202, depth=1
{0xffffff03ffffff03, 3705,{0xffff,0x0619}},  //  4203, depth=2
{0x0000007c0000007c,   -1,{0xffff,0x0784}},  //  4204, depth=1
{0xffffe000000fffff,   -1,{0xffff,0x14e6}},  //  4205, depth=1
{0xffffffffffffff1f, 4209,{0xffff,0x1e3c}},  //  4206
{0xffffffffffe7ffff,   -1,{0xffff,0x1afd}},  //  4207, depth=1
{0x0001fffffffe0000,   -1,{0xffff,0x1bdf}},  //  4208, depth=1
{0x80001fffffffffff,   -1,{0xffff,0x106d}},  //  4209
{0x000001fffffffff8,   -1,{0xffff,0x1f65}},  //  4210, depth=1
{0x00000000ffff803f,   -1,{0x0456,0xffff}},  //  4211, depth=1
{0xffffffff00ffffff,   -1,{0xffff,0x1837}},  //  4212
{0xfcfffcfffcfffcff,   -1,{0xffff,0x01ad}},  //  4213, depth=1
{0x3ffe000000000000, 3709,{0xffff,0x13cc}},  //  4214, depth=2
{0xfffff00ffffff00f,   -1,{0xffff,0x0517}},  //  4215
{0x07fffffffffffff0, 3710,{0xffff,0x1f36}},  //  4216, depth=2
{0x000001ffffffffff,   -1,{0xffff,0x1028}},  //  4217, depth=1
{0xfc00fc00fc00fc00, 3711,{0xffff,0x01a5}},  //  4218, depth=2
{0x0001ffffffffffc0,   -1,{0xffff,0x1eaa}},  //  4219
{0xfffffff0fffffff0, 3712,{0xffff,0x071b}},  //  4220, depth=2
{0xf000000007ffffff,   -1,{0xffff,0x111e}},  //  4221, depth=1
{0x0fffe0000fffe000, 3714,{0xffff,0x04ce}},  //  4222, depth=2
{0x00000000f1fff1ff,   -1,{0x012c,0xffff}},  //  4223, depth=1
{0x00e0000000e00000, 3715,{0xffff,0x02c2}},  //  4224, depth=4
{0xffe0003fffffffff,   -1,{0xffff,0x12f0}},  //  4225, depth=1
{0x3838383838383838, 3724,{0xffff,0x0172}},  //  4226, depth=2
{0xffff8000000007ff,   -1,{0xffff,0x145b}},  //  4227, depth=1
{0x0000000ffff80000, 3729,{0xffff,0x1b50}},  //  4228, depth=2
{0x00000003fffffff0, 4240,{0xffff,0x1f1d}},  //  4229
{0x00000003fffffe00, 3732,{0xffff,0x1dd8}},  //  4230, depth=2
{0x07ffffffffffffff, 3734,{0xffff,0x103a}},  //  4231, depth=2
{0x0800000000000000,   -1,{0xffff,0x1140}},  //  4232, depth=1
{0xfffff00000001fff,   -1,{0xffff,0x1520}},  //  4233, depth=1
{0x03fffffffffff000,   -1,{0xffff,0x1d2d}},  //  4234, depth=1
{0xfffffff0ffffffff,   -1,{0xffff,0x173b}},  //  4235, depth=1
{0x0003f8000003f800,   -1,{0xffff,0x0546}},  //  4236, depth=1
{0x003fffff003fffff,   -1,{0xffff,0x0015}},  //  4237, depth=1
{0x01ffffffffe00000,   -1,{0xffff,0x1ae3}},  //  4238, depth=1
{0xfffe000000000fff,   -1,{0xffff,0x13da}},  //  4239, depth=1
{0x0000007ffffff000,   -1,{0xffff,0x1d1a}},  //  4240
{0x003f000000000000,   -1,{0xffff,0x1405}},  //  4241
{0x000007ffff800000,   -1,{0xffff,0x1a53}},  //  4242
{0xfff800007fffffff,   -1,{0xffff,0x136b}},  //  4243, depth=1
{0x1fffffffffffff00,   -1,{0xffff,0x1e34}},  //  4244, depth=1
{0xffffffdfffffffdf, 3735,{0xffff,0x069e}},  //  4245, depth=2
{0x0000000006000600, 3736,{0x01e1,0xffff}},  //  4246, depth=2
{0x00000003fffffffc,   -1,{0xffff,0x1f9f}},  //  4247
{0x0780000007800000, 3738,{0xffff,0x0243}},  //  4248, depth=2
{0x00000000803f803f, 3739,{0x0066,0xffff}},  //  4249, depth=2
{0x0000000400000000,   -1,{0xffff,0x1780}},  //  4250
{0xfff0000fffffffff,   -1,{0xffff,0x132f}},  //  4251, depth=1
{0x000000000e000000, 3746,{0x01c2,0x19c2}},  //  4252, depth=2
{0x00007e0000000000,   -1,{0xffff,0x15c5}},  //  4253
{0x003ffffe00000000,   -1,{0xffff,0x17d4}},  //  4254
{0xffffffffffff83ff,   -1,{0xffff,0x1c7a}},  //  4255, depth=1
{0xfff8000fffffffff,   -1,{0xffff,0x1370}},  //  4256
{0xfffffc00000007ff,   -1,{0xffff,0x15a0}},  //  4257
{0x3ffffffc00000000,   -1,{0xffff,0x179b}},  //  4258, depth=1
{0xc3ffffffffffffff,   -1,{0xffff,0x10bb}},  //  4259, depth=1
{0x000fffc0000fffc0,   -1,{0xffff,0x068d}},  //  4260, depth=1
{0xfffffffffff03fff,   -1,{0xffff,0x1b39}},  //  4261, depth=1
{0x0080008000800080,   -1,{0xffff,0x0260}},  //  4262, depth=1
{0x0001fffffffffff0,   -1,{0xffff,0x1f2c}},  //  4263
{0x000001ffe0000000, 3747,{0xffff,0x18cb}},  //  4264, depth=2
{0x0001fffffffffff8,   -1,{0xffff,0x1f6d}},  //  4265
{0x001fffffffffc000,   -1,{0xffff,0x1ca6}},  //  4266, depth=1
{0xff8000000003ffff,   -1,{0xffff,0x125a}},  //  4267, depth=1
{0x7fffffc000000000,   -1,{0xffff,0x1698}},  //  4268, depth=1
{0xffffffc00003ffff,   -1,{0xffff,0x16ab}},  //  4269
{0x6000600060006000,   -1,{0xffff,0x00e1}},  //  4270, depth=1
{0x800001ff800001ff, 3751,{0xffff,0x0049}},  //  4271, depth=2
{0x0001fffffffffffc, 4280,{0xffff,0x1fae}},  //  4272
{0xc07fc07fc07fc07f,   -1,{0xffff,0x00a8}},  //  4273, depth=1
{0x0001fffffffffe00,   -1,{0xffff,0x1de7}},  //  4274, depth=1
{0x00000000ffff807f,   -1,{0x0457,0xffff}},  //  4275, depth=1
{0x7fffe0007fffe000,   -1,{0xffff,0x04d1}},  //  4276, depth=1
{0xf03ff03ff03ff03f, 3752,{0xffff,0x0129}},  //  4277, depth=2
{0xffffffe000000000, 3757,{0xffff,0x16da}},  //  4278, depth=2
{0x00000000fc07ffff, 3758,{0x0198,0xffff}},  //  4279, depth=2
{0x3ffffe0000000000,   -1,{0xffff,0x15d4}},  //  4280
{0xfff80000003fffff, 3765,{0xffff,0x1362}},  //  4281, depth=2
{0x0007fffc0007fffc,   -1,{0xffff,0x0790}},  //  4282, depth=1
{0xfffffffc00000fff,   -1,{0xffff,0x17a9}},  //  4283, depth=1
{0xfffff800fffff800,   -1,{0xffff,0x0554}},  //  4284, depth=1
{0xffffffe000000007, 3766,{0xffff,0x16dd}},  //  4285, depth=2
{0x00000003e0000000,   -1,{0xffff,0x18c4}},  //  4286
{0x8000000380000003,   -1,{0xffff,0x0042}},  //  4287, depth=1
{0xffffffe0000001ff,   -1,{0xffff,0x16e3}},  //  4288
{0xffc0000003ffffff,   -1,{0xffff,0x12a3}},  //  4289
{0x00000ffffffff000,   -1,{0xffff,0x1d1f}},  //  4290
{0xfffffffffffc03ff, 4292,{0xffff,0x1bb7}},  //  4291
{0xffffe0ffffffffff, 4294,{0xffff,0x14fa}},  //  4292
{0xffffffe00000000f,   -1,{0xffff,0x16de}},  //  4293, depth=1
{0xfff8fffffff8ffff,   -1,{0xffff,0x035c}},  //  4294
{0xfffe01fffffe01ff,   -1,{0xffff,0x03d7}},  //  4295, depth=1
{0xffffffffffffff80,   -1,{0xffff,0x1e78}},  //  4296
{0xfe0000fffe0000ff,   -1,{0xffff,0x01ce}},  //  4297, depth=1
{0x0007fe000007fe00,   -1,{0xffff,0x05c9}},  //  4298, depth=1
{0xfc01fffffc01ffff, 3767,{0xffff,0x0196}},  //  4299, depth=2
{0xffffffffffffff87, 4302,{0xffff,0x1e7b}},  //  4300
{0x00000000c007c007, 3768,{0x00a4,0xffff}},  //  4301, depth=2
{0xff80000000007fff,   -1,{0xffff,0x1257}},  //  4302
{0x00000000f7ffffff, 3769,{0x011e,0xffff}},  //  4303, depth=2
{0x00000000f8000000, 3772,{0x0144,0x1944}},  //  4304, depth=2
{0x00000000f8000001, 3788,{0x0145,0xffff}},  //  4305, depth=3
{0x00000000f80001ff,   -1,{0x014d,0xffff}},  //  4306
{0x00000000f8000003, 3791,{0x0146,0xffff}},  //  4307, depth=2
{0x003fffffff800000,   -1,{0xffff,0x1a5e}},  //  4308, depth=1
{0x00000000fffff9ff, 3796,{0x055d,0xffff}},  //  4309, depth=3
{0x000001ffc0000000,   -1,{0xffff,0x188a}},  //  4310, depth=1
{0x00000000f8000007, 3804,{0x0147,0xffff}},  //  4311, depth=4
{0xffffffffffffff8f, 4313,{0xffff,0x1e7c}},  //  4312
{0xc0000fffffffffff,   -1,{0xffff,0x10ad}},  //  4313
{0x03c0000000000000,   -1,{0xffff,0x1283}},  //  4314, depth=1
{0xfffffffffff87fff,   -1,{0xffff,0x1b7b}},  //  4315, depth=1
{0x000007ffffffff80,   -1,{0xffff,0x1e63}},  //  4316, depth=1
{0x0000001ffffff800,   -1,{0xffff,0x1d59}},  //  4317
{0xffffffff807fffff, 4320,{0xffff,0x1877}},  //  4318
{0x00000000f800000f,   -1,{0x0148,0xffff}},  //  4319, depth=1
{0xfc1fffffffffffff, 4326,{0xffff,0x11ba}},  //  4320
{0xffc1ffc1ffc1ffc1, 3821,{0xffff,0x02aa}},  //  4321, depth=2
{0x0000000003fe03fe, 3824,{0x03e8,0xffff}},  //  4322, depth=2
{0xffffe003ffffffff,   -1,{0xffff,0x14f4}},  //  4323, depth=1
{0x07ffffffc0000000, 3826,{0xffff,0x189c}},  //  4324, depth=2
{0xf00001fff00001ff, 3827,{0xffff,0x010c}},  //  4325, depth=2
{0x00000000fffffbff,   -1,{0x055e,0xffff}},  //  4326
{0xff000000001fffff,   -1,{0xffff,0x121c}},  //  4327, depth=1
{0xffffffffe03fffff,   -1,{0xffff,0x18f8}},  //  4328
{0x0001ffffe0000000,   -1,{0xffff,0x18d3}},  //  4329
{0x0000ffe000000000,   -1,{0xffff,0x16ca}},  //  4330
{0x000000007fff7fff,   -1,{0x002e,0xffff}},  //  4331, depth=1
{0x000000007fff8000,   -1,{0x044f,0x1c4f}},  //  4332, depth=1
{0x83ff83ff83ff83ff, 3834,{0xffff,0x006a}},  //  4333, depth=2
{0xffff8fffffffffff,   -1,{0xffff,0x147c}},  //  4334
{0x00000000f800001f, 3836,{0x0149,0xffff}},  //  4335, depth=3
{0xfffffff0000000ff,   -1,{0xffff,0x1723}},  //  4336
{0xffc00007ffc00007, 3844,{0xffff,0x028c}},  //  4337, depth=2
{0x00000007fff00000,   -1,{0xffff,0x1b0e}},  //  4338
{0xfff83ffffff83fff,   -1,{0xffff,0x035a}},  //  4339, depth=1
{0x00000000fc00fc00, 3845,{0x01a5,0xffff}},  //  4340, depth=3
{0xf0000003f0000003, 3849,{0xffff,0x0105}},  //  4341, depth=2
{0xfc000000000007ff,   -1,{0xffff,0x1190}},  //  4342
{0x0000000083838383,   -1,{0x0072,0xffff}},  //  4343, depth=1
{0x001ffffffc000000, 3850,{0xffff,0x199a}},  //  4344, depth=2
{0xe03fffffe03fffff,   -1,{0xffff,0x00d8}},  //  4345, depth=1
{0xfffffffffffe01ff, 4348,{0xffff,0x1bf7}},  //  4346
{0x00000000c00fffff,   -1,{0x0095,0xffff}},  //  4347, depth=1
{0xfffff07fffffffff,   -1,{0xffff,0x153a}},  //  4348
{0x001c000000000000,   -1,{0xffff,0x1382}},  //  4349
{0x00ff000000000000,   -1,{0xffff,0x1407}},  //  4350, depth=1
{0xfffffffffffdffff,   -1,{0xffff,0x1bbe}},  //  4351, depth=1
{0xfffffffffffe0000,   -1,{0xffff,0x1bee}},  //  4352, depth=1
{0x8003800380038003, 3851,{0xffff,0x0062}},  //  4353, depth=3
{0x00003fffffffff80,   -1,{0xffff,0x1e66}},  //  4354, depth=1
{0xfc00000007ffffff, 3855,{0xffff,0x11a0}},  //  4355, depth=2
{0x0003ff0000000000,   -1,{0xffff,0x1609}},  //  4356, depth=1
{0xffffffffffffffc0,   -1,{0xffff,0x1eb9}},  //  4357
{0x001fffe0001fffe0,   -1,{0xffff,0x06cf}},  //  4358, depth=1
{0xfffffffffffe0007,   -1,{0xffff,0x1bf1}},  //  4359, depth=1
{0x0000000007ff0000,   -1,{0x040a,0x1c0a}},  //  4360, depth=1
{0xfffe01ffffffffff,   -1,{0xffff,0x13f7}},  //  4361, depth=1
{0x00007ffffffff000,   -1,{0xffff,0x1d22}},  //  4362
{0xfffff80003ffffff,   -1,{0xffff,0x156e}},  //  4363, depth=1
{0x000001ffffc00000,   -1,{0xffff,0x1a92}},  //  4364, depth=1
{0xc00003ffffffffff,   -1,{0xffff,0x10ab}},  //  4365, depth=1
{0xffffffffffffffc3,   -1,{0xffff,0x1ebb}},  //  4366
{0x00000000f800003f, 3859,{0x014a,0xffff}},  //  4367, depth=2
{0x00000000000fff00, 3865,{0x060b,0x1e0b}},  //  4368, depth=2
{0xffffff80000007ff,   -1,{0xffff,0x1663}},  //  4369, depth=1
{0x00000000003f8000,   -1,{0x0446,0x1c46}},  //  4370, depth=1
{0xffffffffffffffc7,   -1,{0xffff,0x1ebc}},  //  4371
{0x00fffffe00000000, 3882,{0xffff,0x17d6}},  //  4372, depth=2
{0xffffefffffffefff, 3884,{0xffff,0x04de}},  //  4373, depth=2
{0x03ffffffffffc000,   -1,{0xffff,0x1cab}},  //  4374, depth=1
{0xffffffffc03fffff, 4382,{0xffff,0x18b7}},  //  4375
{0x3ffffffffff00000,   -1,{0xffff,0x1b29}},  //  4376, depth=1
{0xf000000000007fff, 3889,{0xffff,0x1112}},  //  4377, depth=2
{0x07ffffffffc00000,   -1,{0xffff,0x1aa4}},  //  4378, depth=1
{0x00000000ffffe1ff,   -1,{0x04db,0xffff}},  //  4379, depth=1
{0x000007ffffffffc0,   -1,{0xffff,0x1ea4}},  //  4380, depth=1
{0xfff800001fffffff,   -1,{0xffff,0x1369}},  //  4381, depth=1
{0xfe0fffffffffffff,   -1,{0xffff,0x11fa}},  //  4382
{0xfe000000000001ff, 3891,{0xffff,0x11cf}},  //  4383, depth=2
{0x0078007800780078, 3895,{0xffff,0x0363}},  //  4384, depth=2
{0xe000000000003fff,   -1,{0xffff,0x10d0}},  //  4385, depth=1
{0xfffffffff01fffff,   -1,{0xffff,0x1938}},  //  4386
{0xffffffffffffffcf, 4389,{0xffff,0x1ebd}},  //  4387
{0x000001f0000001f0,   -1,{0xffff,0x0704}},  //  4388, depth=1
{0x00000000f801ffff,   -1,{0x0155,0xffff}},  //  4389
{0x0000000030303030,   -1,{0x0131,0xffff}},  //  4390, depth=1
{0xffffffffe0ffffff,   -1,{0xffff,0x18fa}},  //  4391, depth=1
{0x003fe00000000000,   -1,{0xffff,0x14c8}},  //  4392
{0xcfffffffffffffff,   -1,{0xffff,0x10bd}},  //  4393, depth=1
{0x3ffc00003ffc0000,   -1,{0xffff,0x038b}},  //  4394, depth=1
{0xfffe0003ffffffff,   -1,{0xffff,0x13f0}},  //  4395
{0x0000001fffffe000,   -1,{0xffff,0x1cd7}},  //  4396
{0x000000f000000000,   -1,{0xffff,0x1703}},  //  4397
{0x0000000001fc01fc,   -1,{0x03a6,0xffff}},  //  4398, depth=1
{0xffffc7ffffffffff,   -1,{0xffff,0x14bc}},  //  4399
{0xfe000000000003ff,   -1,{0xffff,0x11d0}},  //  4400
{0x00000000ffcfffff, 3896,{0x029d,0xffff}},  //  4401, depth=4
{0x000001ff80000000,   -1,{0xffff,0x1849}},  //  4402, depth=1
{0x00000000ffff80ff, 3910,{0x0458,0xffff}},  //  4403, depth=3
{0x0007fffffff80000,   -1,{0xffff,0x1b5f}},  //  4404, depth=1
{0xfffffc0000007fff, 3919,{0xffff,0x15a4}},  //  4405, depth=2
{0x07ffffc007ffffc0,   -1,{0xffff,0x0694}},  //  4406, depth=1
{0xfc0000fffc0000ff,   -1,{0xffff,0x018d}},  //  4407, depth=1
{0xffffffffffffffe0,   -1,{0xffff,0x1efa}},  //  4408
{0xffffffffffffffe1,   -1,{0xffff,0x1efb}},  //  4409
{0xffffffffffffffe3,   -1,{0xffff,0x1efc}},  //  4410
{0xfffffffff80003ff,   -1,{0xffff,0x196e}},  //  4411, depth=1
{0x000007ffffffffe0,   -1,{0xffff,0x1ee5}},  //  4412, depth=1
{0xffffffffffffffe7,   -1,{0xffff,0x1efd}},  //  4413
{0xe0003fffffffffff,   -1,{0xffff,0x10f0}},  //  4414
{0xfc000001fc000001, 3920,{0xffff,0x0186}},  //  4415, depth=3
{0x07ffffff80000000, 3949,{0xffff,0x185b}},  //  4416, depth=2
{0xfffffffffffffdff,   -1,{0xffff,0x1dbe}},  //  4417, depth=1
{0xfffffffffffffe00, 3950,{0xffff,0x1df6}},  //  4418, depth=2
{0x0000000081818181, 3960,{0x0071,0xffff}},  //  4419, depth=4
{0xfffffffffffffff1,   -1,{0xffff,0x1f3c}},  //  4420
{0xfffffffffffffe03,   -1,{0xffff,0x1df8}},  //  4421, depth=1
{0xfffffffffffffff3,   -1,{0xffff,0x1f3d}},  //  4422
{0x0000000001ffff80,   -1,{0x0651,0x1e51}},  //  4423
{0x000000003fff8000,   -1,{0x044e,0x1c4e}},  //  4424, depth=1
{0xfffffffffffffe07,   -1,{0xffff,0x1df9}},  //  4425, depth=1
{0xfffffffffffffff8, 4432,{0xffff,0x1f7c}},  //  4426
{0x00000000ffff1fff, 3972,{0x041c,0xffff}},  //  4427, depth=2
{0xfff8000000000000, 3973,{0xffff,0x134c}},  //  4428, depth=2
{0xfff8000000000001,   -1,{0xffff,0x134d}},  //  4429, depth=1
{0x000000000007f000,   -1,{0x0506,0x1d06}},  //  4430, depth=1
{0x00000000f800007f, 3978,{0x014b,0xffff}},  //  4431, depth=2
{0x0000fffffffe0000,   -1,{0xffff,0x1bde}},  //  4432
{0xfffffffffffffe0f,   -1,{0xffff,0x1dfa}},  //  4433, depth=1
{0x0000003ff0000000,   -1,{0xffff,0x1909}},  //  4434, depth=1
{0x00000000ff8fff8f, 3986,{0x026c,0xffff}},  //  4435, depth=2
{0x1fffffffffffffc0, 3987,{0xffff,0x1eb6}},  //  4436, depth=2
{0xfff00000000001ff, 3990,{0xffff,0x1314}},  //  4437, depth=2
{0x00000007fe000000,   -1,{0xffff,0x19c9}},  //  4438, depth=1
{0x00000000800fffff,   -1,{0x0054,0xffff}},  //  4439, depth=1
{0x000007fffffffffc,   -1,{0xffff,0x1fa8}},  //  4440, depth=1
{0xfffff80000003fff,   -1,{0xffff,0x1562}},  //  4441, depth=1
{0x000007fffffffffe,   -1,{0xffff,0x1fe9}},  //  4442, depth=1
{0x000007ffffffffff, 3991,{0xffff,0x102a}},  //  4443, depth=2
{0x0000080000000000,   -1,{0xffff,0x1540}},  //  4444, depth=1
{0xe03fffffffffffff,   -1,{0xffff,0x10f8}},  //  4445, depth=1
{0xfffffffffffffff9,   -1,{0xffff,0x1f7d}},  //  4446
{0xfffffffffc00ffff,   -1,{0xffff,0x19b5}},  //  4447, depth=1
{0x0000800000008000, 3992,{0xffff,0x0440}},  //  4448, depth=2
{0xfe0000000001ffff, 3994,{0xffff,0x11d7}},  //  4449, depth=2
{0x00003fffffffffe0,   -1,{0xffff,0x1ee8}},  //  4450, depth=1
{0xfffffffffffffffc, 4458,{0xffff,0x1fbd}},  //  4451
{0x03fffffffc000000,   -1,{0xffff,0x199f}},  //  4452, depth=1
{0xfe0ffffffe0fffff,   -1,{0xffff,0x01da}},  //  4453, depth=1
{0x2000000020000000, 4001,{0xffff,0x00c0}},  //  4454, depth=2
{0xc00000000007ffff, 4005,{0xffff,0x1094}},  //  4455, depth=2
{0x0030000000300000,   -1,{0xffff,0x0301}},  //  4456, depth=1
{0xfffffff0003fffff,   -1,{0xffff,0x1731}},  //  4457, depth=1
{0x0000000380000000,   -1,{0xffff,0x1842}},  //  4458
{0xc0003fffc0003fff, 4007,{0xffff,0x008f}},  //  4459, depth=3
{0x3fff80003fff8000, 4017,{0xffff,0x044e}},  //  4460, depth=2
{0xfffffffffffffffd,   -1,{0xffff,0x1fbe}},  //  4461
{0x0000000003000000, 4018,{0x0201,0x1a01}},  //  4462, depth=2
{0xffe0000007ffffff,   -1,{0xffff,0x12e5}},  //  4463, depth=1
{0x00000fffffffc000,   -1,{0xffff,0x1c9d}},  //  4464
{0x03ffffffffff0000,   -1,{0xffff,0x1c29}},  //  4465
{0x00003ffffffffff0,   -1,{0xffff,0x1f29}},  //  4466, depth=1
{0xfff80000000001ff,   -1,{0xffff,0x1355}},  //  4467
{0x0380000003800000, 4026,{0xffff,0x0242}},  //  4468, depth=4
{0x0000007ffc000000,   -1,{0xffff,0x198c}},  //  4469
{0x000000001fff8000,   -1,{0x044d,0x1c4d}},  //  4470, depth=1
{0xffff3fffffff3fff,   -1,{0xffff,0x041d}},  //  4471, depth=1
{0xfffffe03fffffe03,   -1,{0xffff,0x05d8}},  //  4472
{0x0003fffffff00000,   -1,{0xffff,0x1b1d}},  //  4473
{0x0000000000fc0000, 4036,{0x0385,0x1b85}},  //  4474, depth=2
{0xffff000001ffffff,   -1,{0xffff,0x1428}},  //  4475, depth=1
{0xe0e0e0e0e0e0e0e0, 4037,{0xffff,0x00f2}},  //  4476, depth=2
{0xfffff3ffffffffff,   -1,{0xffff,0x153d}},  //  4477, depth=1
{0x00003ffffffffffc,   -1,{0xffff,0x1fab}},  //  4478, depth=1
{0xfffffffffffe007f,   -1,{0xffff,0x1bf5}},  //  4479, depth=1
{0x000003ffffffc000, 4038,{0xffff,0x1c9b}},  //  4480, depth=2
{0x00003fffffffffff, 4039,{0xffff,0x102d}},  //  4481, depth=2
{0x0000400000000000,   -1,{0xffff,0x1480}},  //  4482, depth=1
{0xfff00000000003ff, 4485,{0xffff,0x1315}},  //  4483
{0x0000003fffe00000, 4040,{0xffff,0x1ad0}},  //  4484, depth=2
{0x00000000fffffc7f,   -1,{0x059c,0xffff}},  //  4485
{0x001ff000001ff000, 4041,{0xffff,0x0508}},  //  4486, depth=2
{0x000000ffffffff00,   -1,{0xffff,0x1e1f}},  //  4487
{0x0000000078000000,   -1,{0x0143,0x1943}},  //  4488, depth=1
{0x00000001fffffe00,   -1,{0xffff,0x1dd7}},  //  4489
{0x000007ffe0000000,   -1,{0xffff,0x18cd}},  //  4490, depth=1
{0xfff800000000003f, 4049,{0xffff,0x1352}},  //  4491, depth=2
{0x0000000ffffff800, 4050,{0xffff,0x1d58}},  //  4492, depth=2
{0x00000000e001e001,   -1,{0x00e3,0xffff}},  //  4493, depth=1
{0x007ffc00007ffc00, 4051,{0xffff,0x058c}},  //  4494, depth=2
{0x000000007f7f7f7f,   -1,{0x0036,0xffff}},  //  4495, depth=1
{0x00000000000fff80, 4054,{0x064c,0x1e4c}},  //  4496, depth=2
{0x00000000ffff07ff, 4056,{0x041a,0xffff}},  //  4497, depth=3
{0x1ffffffffffffffe,   -1,{0xffff,0x1ffb}},  //  4498, depth=1
{0x1fffffffffffffff,   -1,{0xffff,0x103c}},  //  4499, depth=1
{0x2000000000000000,   -1,{0xffff,0x10c0}},  //  4500, depth=1
{0xffffe01fffffe01f,   -1,{0xffff,0x04d7}},  //  4501, depth=1
{0x00ffe00000000000,   -1,{0xffff,0x14ca}},  //  4502, depth=1
{0xfff000000001ffff,   -1,{0xffff,0x131c}},  //  4503, depth=1
{0xffffffff803fffff,   -1,{0xffff,0x1876}},  //  4504
{0xffe00000000007ff,   -1,{0xffff,0x12d5}},  //  4505
{0x0070007000700070,   -1,{0xffff,0x0322}},  //  4506, depth=1
{0xfffffffffe0fffff,   -1,{0xffff,0x19fa}},  //  4507, depth=1
{0x0fffffffffffc000,   -1,{0xffff,0x1cad}},  //  4508, depth=1
{0x00000000ffe07fff,   -1,{0x02d9,0xffff}},  //  4509, depth=1
{0x3030303030303030, 4061,{0xffff,0x0131}},  //  4510, depth=2
{0xfc00000000007fff, 4063,{0xffff,0x1194}},  //  4511, depth=2
{0x03fc000000000000,   -1,{0xffff,0x1387}},  //  4512
{0x0000000001fe01fe,   -1,{0x03e7,0xffff}},  //  4513
{0x0004000000040000, 4066,{0xffff,0x0380}},  //  4514, depth=2
{0xff801fffff801fff,   -1,{0xffff,0x0255}},  //  4515, depth=1
{0x000fe000000fe000,   -1,{0xffff,0x04c6}},  //  4516, depth=1
{0xffffffffe0000007,   -1,{0xffff,0x18e5}},  //  4517
{0x3fffc00000000000,   -1,{0xffff,0x148f}},  //  4518, depth=1
{0xf801fffff801ffff,   -1,{0xffff,0x0155}},  //  4519, depth=1
{0x000000003e003e00, 4067,{0x01e4,0xffff}},  //  4520, depth=3
{0x3fc0000000000000,   -1,{0xffff,0x1287}},  //  4521
{0x01ff800000000000,   -1,{0xffff,0x1449}},  //  4522, depth=1
{0xfe00001fffffffff,   -1,{0xffff,0x11eb}},  //  4523, depth=1
{0x07ffffffff800000,   -1,{0xffff,0x1a63}},  //  4524, depth=1
{0xe000000000000fff,   -1,{0xffff,0x10ce}},  //  4525, depth=1
{0x0003fffffe000000,   -1,{0xffff,0x19d8}},  //  4526, depth=1
{0xffffffffe000000f,   -1,{0xffff,0x18e6}},  //  4527
{0x00003fffe0000000,   -1,{0xffff,0x18d0}},  //  4528, depth=1
{0xffffffff80ffffff,   -1,{0xffff,0x1878}},  //  4529, depth=1
{0x0001ffffffffff00,   -1,{0xffff,0x1e28}},  //  4530, depth=1
{0x0ffffff00ffffff0,   -1,{0xffff,0x0717}},  //  4531
{0x007fffffffe00000, 4535,{0xffff,0x1ae1}},  //  4532
{0xffffffe0000000ff,   -1,{0xffff,0x16e2}},  //  4533, depth=1
{0x0000000001fffe00,   -1,{0x05cf,0x1dcf}},  //  4534, depth=1
{0x0000000ff8000000,   -1,{0xffff,0x1948}},  //  4535
{0x00fffe0000fffe00, 4069,{0xffff,0x05ce}},  //  4536, depth=2
{0xffffffff007fffff,   -1,{0xffff,0x1836}},  //  4537
{0xfffffffffc003fff,   -1,{0xffff,0x19b3}},  //  4538
{0xffc0000000000fff,   -1,{0xffff,0x1295}},  //  4539
{0x003fff0000000000,   -1,{0xffff,0x160d}},  //  4540
{0xffffffffbfffffff,   -1,{0xffff,0x187e}},  //  4541
{0xffffffffc0000000, 4549,{0xffff,0x18a1}},  //  4542
{0xffffc00000000fff, 4070,{0xffff,0x149d}},  //  4543, depth=2
{0x000ff00000000000,   -1,{0xffff,0x1507}},  //  4544, depth=1
{0xfffffffffffffe7f,   -1,{0xffff,0x1dfd}},  //  4545, depth=1
{0x1fffffffe0000000,   -1,{0xffff,0x18df}},  //  4546, depth=1
{0x00000000ff8fffff,   -1,{0x025c,0xffff}},  //  4547, depth=1
{0x7ffffffffffffc00,   -1,{0xffff,0x1db4}},  //  4548, depth=1
{0x00001ff000000000,   -1,{0xffff,0x1708}},  //  4549
{0xffffffffc000000f,   -1,{0xffff,0x18a5}},  //  4550
{0x00000000f0fff0ff,   -1,{0x012b,0xffff}},  //  4551, depth=1
{0x0000000001fffff8,   -1,{0x0755,0x1f55}},  //  4552
{0xffffffffe000003f,   -1,{0xffff,0x18e8}},  //  4553
{0x0000001ffff80000,   -1,{0xffff,0x1b51}},  //  4554, depth=1
{0xfff800000000007f,   -1,{0xffff,0x1353}},  //  4555, depth=1
{0x0000000018001800, 4071,{0x0161,0xffff}},  //  4556, depth=2
{0x00000000fff803ff, 4078,{0x0356,0xffff}},  //  4557, depth=2
{0x000003fffc000000,   -1,{0xffff,0x198f}},  //  4558, depth=1
{0x00000000f80000ff, 4079,{0x014c,0xffff}},  //  4559, depth=2
{0x00000000000fffc0,   -1,{0x068d,0x1e8d}},  //  4560, depth=1
{0x000003ffffff0000, 4565,{0xffff,0x1c19}},  //  4561
{0x0000000fffffe000,   -1,{0xffff,0x1cd6}},  //  4562, depth=1
{0xfc0ffffffc0fffff, 4081,{0xffff,0x0199}},  //  4563, depth=2
{0x1e0000001e000000, 4082,{0xffff,0x01c3}},  //  4564, depth=2
{0x3ff83ff83ff83ff8,   -1,{0xffff,0x036a}},  //  4565
{0x0000000001fffffc,   -1,{0x0796,0x1f96}},  //  4566
{0xfffff0000007ffff,   -1,{0xffff,0x1526}},  //  4567, depth=1
{0x7ff0000000000000,   -1,{0xffff,0x130a}},  //  4568, depth=1
{0x3fffffffffffc000, 4572,{0xffff,0x1caf}},  //  4569
{0x000000000007c000,   -1,{0x0484,0x1c84}},  //  4570, depth=1
{0xfe7ffe7ffe7ffe7f,   -1,{0xffff,0x01ed}},  //  4571, depth=1
{0x0000000001fffffe,   -1,{0x07d7,0x1fd7}},  //  4572
{0x800000ffffffffff,   -1,{0xffff,0x1068}},  //  4573, depth=1
{0x00003fffc0000000,   -1,{0xffff,0x188f}},  //  4574, depth=1
{0x00000000f807f807, 4083,{0x0167,0xffff}},  //  4575, depth=2
{0xfffe0001fffe0001,   -1,{0xffff,0x03cf}},  //  4576
{0xfff0001fffffffff,   -1,{0xffff,0x1330}},  //  4577, depth=1
{0x0180000001800000,   -1,{0xffff,0x0241}},  //  4578, depth=1
{0x7ffffff07ffffff0,   -1,{0xffff,0x071a}},  //  4579
{0x0000000038000000, 4089,{0x0142,0x1942}},  //  4580, depth=2
{0xfffff80000000fff,   -1,{0xffff,0x1560}},  //  4581, depth=1
{0xffffffffc000001f,   -1,{0xffff,0x18a6}},  //  4582
{0x801fffff801fffff,   -1,{0xffff,0x0055}},  //  4583, depth=1
{0xffffe001ffffffff,   -1,{0xffff,0x14f3}},  //  4584
{0x7ffffffffffffe00,   -1,{0xffff,0x1df5}},  //  4585
{0x000001ff00000000, 4091,{0xffff,0x1808}},  //  4586, depth=2
{0xffffffefffffffef, 4097,{0xffff,0x06de}},  //  4587, depth=2
{0x00003fffff000000,   -1,{0xffff,0x1a15}},  //  4588
{0xfffffe0003ffffff,   -1,{0xffff,0x15f0}},  //  4589, depth=1
{0x00001f8000001f80, 4098,{0xffff,0x0645}},  //  4590, depth=2
{0xffffffff9fffffff,   -1,{0xffff,0x187d}},  //  4591
{0x00000000000fffe0, 4100,{0x06ce,0x1ece}},  //  4592, depth=2
{0xffffc000000fffff,   -1,{0xffff,0x14a5}},  //  4593
{0x00000000f007ffff,   -1,{0x0116,0xffff}},  //  4594
{0xfffffffffff81fff,   -1,{0xffff,0x1b79}},  //  4595, depth=1
{0xffffffffffbfffff,   -1,{0xffff,0x1a7e}},  //  4596
{0xffffffffffc00001,   -1,{0xffff,0x1aaa}},  //  4597
{0x7fc07fc07fc07fc0, 4102,{0xffff,0x02a8}},  //  4598, depth=2
{0x00000000ff0fff0f,   -1,{0x022b,0xffff}},  //  4599, depth=1
{0x07ffffff00000000, 4103,{0xffff,0x181a}},  //  4600, depth=2
{0xe03fe03fe03fe03f,   -1,{0xffff,0x00e8}},  //  4601, depth=1
{0x00000000fff0fff0,   -1,{0x032b,0xffff}},  //  4602, depth=1
{0xf81ff81ff81ff81f, 4113,{0xffff,0x0169}},  //  4603, depth=2
{0x003ffffe003ffffe, 4116,{0xffff,0x07d4}},  //  4604, depth=2
{0xfffc0000001fffff, 4119,{0xffff,0x13a2}},  //  4605, depth=2
{0x0000000000fffc00,   -1,{0x058d,0x1d8d}},  //  4606, depth=1
{0xfffffff000000003, 4124,{0xffff,0x171d}},  //  4607, depth=2
{0x00000000000ffff0,   -1,{0x070f,0x1f0f}},  //  4608, depth=1
{0x0001ffffff800000,   -1,{0xffff,0x1a59}},  //  4609
{0x03fe03fe03fe03fe, 4125,{0xffff,0x03e8}},  //  4610, depth=2
{0xfffffff000000007,   -1,{0xffff,0x171e}},  //  4611, depth=1
{0x003fff00003fff00,   -1,{0xffff,0x060d}},  //  4612, depth=1
{0xff00007fff00007f,   -1,{0xffff,0x020e}},  //  4613, depth=1
{0x00000003ffffff80,   -1,{0xffff,0x1e5a}},  //  4614, depth=1
{0xffffffffffc00003,   -1,{0xffff,0x1aab}},  //  4615
{0x00000000000ffff8,   -1,{0x0750,0x1f50}},  //  4616, depth=1
{0xffe001ffffe001ff, 4128,{0xffff,0x02d3}},  //  4617, depth=2
{0x000000001fe01fe0,   -1,{0x02e7,0xffff}},  //  4618
{0xffe0000000007fff, 4130,{0xffff,0x12d9}},  //  4619, depth=2
{0x0200020002000200, 4132,{0xffff,0x01e0}},  //  4620, depth=2
{0xfffffffe00ffffff,   -1,{0xffff,0x17f6}},  //  4621
{0x00000000000ffffe,   -1,{0x07d2,0x1fd2}},  //  4622, depth=1
{0x00000000000fffff,   -1,{0x0013,0x1013}},  //  4623, depth=1
{0x0000000000100000,   -1,{0x0300,0x1b00}},  //  4624, depth=1
{0x0101010101010101, 4133,{0xffff,0x0030}},  //  4625, depth=2
{0x0000000018000000, 4136,{0x0141,0x1941}},  //  4626, depth=2
{0xf80000fff80000ff, 4139,{0xffff,0x014c}},  //  4627, depth=2
{0x000007ff80000000, 4140,{0xffff,0x184b}},  //  4628, depth=2
{0xfffffffff8007fff,   -1,{0xffff,0x1973}},  //  4629
{0x0002000200020002,   -1,{0xffff,0x03e0}},  //  4630, depth=1
{0xc1ffc1ffc1ffc1ff, 4141,{0xffff,0x00aa}},  //  4631, depth=2
{0x00007ffffc000000,   -1,{0xffff,0x1994}},  //  4632
{0xffe00003ffe00003, 4142,{0xffff,0x02cc}},  //  4633, depth=2
{0x00ffff0000000000,   -1,{0xffff,0x160f}},  //  4634, depth=1
{0xf8000001f8000001, 4145,{0xffff,0x0145}},  //  4635, depth=2
{0x007fffffffff0000,   -1,{0xffff,0x1c26}},  //  4636, depth=1
{0xf01ffffff01fffff,   -1,{0xffff,0x0118}},  //  4637, depth=1
{0x000000fffffe0000,   -1,{0xffff,0x1bd6}},  //  4638, depth=1
{0xff80000000001fff,   -1,{0xffff,0x1255}},  //  4639
{0x0000000000100010,   -1,{0x0320,0xffff}},  //  4640, depth=1
{0xc001c001c001c001,   -1,{0xffff,0x00a2}},  //  4641, depth=1
{0x0003c00000000000,   -1,{0xffff,0x1483}},  //  4642
{0xffffffffffc0000f,   -1,{0xffff,0x1aad}},  //  4643
{0x07ffe00000000000,   -1,{0xffff,0x14cd}},  //  4644
{0xffff00ffffffffff,   -1,{0xffff,0x1437}},  //  4645, depth=1
{0x1fffffffffc00000,   -1,{0xffff,0x1aa6}},  //  4646, depth=1
{0xe00001ffffffffff,   -1,{0xffff,0x10eb}},  //  4647, depth=1
{0x00fffffc00000000,   -1,{0xffff,0x1795}},  //  4648
{0xffffffc0000003ff,   -1,{0xffff,0x16a3}},  //  4649, depth=1
{0x0fffffe00fffffe0,   -1,{0xffff,0x06d6}},  //  4650, depth=1
{0xfffff7fffffff7ff, 4152,{0xffff,0x051e}},  //  4651, depth=2
{0x0000000300000000,   -1,{0xffff,0x1801}},  //  4652
{0xf800000000003fff, 4153,{0xffff,0x1152}},  //  4653, depth=2
{0xffffffff7fffffff,   -1,{0xffff,0x183e}},  //  4654
{0xfffc00000fffffff,   -1,{0xffff,0x13a9}},  //  4655, depth=1
{0xffffffff80000001,   -1,{0xffff,0x1861}},  //  4656
{0xf000000000001fff,   -1,{0xffff,0x1110}},  //  4657, depth=1
{0x0001ffffffffff80,   -1,{0xffff,0x1e69}},  //  4658, depth=1
{0x00000000ffff81ff,   -1,{0x0459,0xffff}},  //  4659, depth=1
{0x00000000ffe7ffff,   -1,{0x02dd,0xffff}},  //  4660
{0xe7ffffffffffffff,   -1,{0xffff,0x10fd}},  //  4661, depth=1
{0x000000007fe07fe0, 4158,{0x02e9,0xffff}},  //  4662, depth=2
{0x000000f0000000f0,   -1,{0xffff,0x0703}},  //  4663
{0x01ffff8000000000,   -1,{0xffff,0x1651}},  //  4664, depth=1
{0xfff1fffffff1ffff, 4160,{0xffff,0x031c}},  //  4665, depth=3
{0x00003fff80000000,   -1,{0xffff,0x184e}},  //  4666, depth=1
{0xfffffe0000003fff, 4169,{0xffff,0x15e4}},  //  4667, depth=2
{0x0007fffffffff800,   -1,{0xffff,0x1d67}},  //  4668, depth=1
{0xffc001ffffffffff,   -1,{0xffff,0x12b2}},  //  4669
{0x00000e0000000000,   -1,{0xffff,0x15c2}},  //  4670, depth=1
{0xffffffff8000001f,   -1,{0xffff,0x1865}},  //  4671
{0xfe000000fe000000,   -1,{0xffff,0x01c6}},  //  4672, depth=1
{0xfffffffffffffeff,   -1,{0xffff,0x1dfe}},  //  4673, depth=1
{0x1c0000001c000000, 4173,{0xffff,0x0182}},  //  4674, depth=4
{0xffffffffffffff01,   -1,{0xffff,0x1e38}},  //  4675, depth=1
{0x00000000cccccccc,   -1,{0x00b9,0xffff}},  //  4676, depth=1
{0xffffffffffffff03,   -1,{0xffff,0x1e39}},  //  4677, depth=1
{0x00000003ffffffc0,   -1,{0xffff,0x1e9b}},  //  4678, depth=1
{0xffffffffc000007f,   -1,{0xffff,0x18a8}},  //  4679
{0x0001ffe00001ffe0,   -1,{0xffff,0x06cb}},  //  4680, depth=1
{0xffffffffffffff07,   -1,{0xffff,0x1e3a}},  //  4681, depth=1
{0x000000ffffffffc0,   -1,{0xffff,0x1ea1}},  //  4682
{0xfff80000000000ff, 4180,{0xffff,0x1354}},  //  4683, depth=2
{0x1fffffff80000000,   -1,{0xffff,0x185d}},  //  4684, depth=1
{0x00000000fffbfffb, 4182,{0x036e,0xffff}},  //  4685, depth=2
{0xff00000003ffffff, 4688,{0xffff,0x1221}},  //  4686
{0xf01fffffffffffff,   -1,{0xffff,0x1138}},  //  4687, depth=1
{0xe0000003ffffffff,   -1,{0xffff,0x10e4}},  //  4688
{0x00000000fffbffff, 4187,{0x035e,0xffff}},  //  4689, depth=3
{0x00000000fffc0000,   -1,{0x038d,0x1b8d}},  //  4690, depth=1
{0xff07ffffff07ffff, 4194,{0xffff,0x021a}},  //  4691, depth=2
{0x0007fffff8000000,   -1,{0xffff,0x1957}},  //  4692
{0x00000000fffc0003, 4199,{0x038f,0xffff}},  //  4693, depth=2
{0x0300000000000000,   -1,{0xffff,0x1201}},  //  4694, depth=1
{0xffffffff8000003f, 4698,{0xffff,0x1866}},  //  4695
{0x00001fe000000000,   -1,{0xffff,0x16c7}},  //  4696, depth=1
{0x00000000fffc0007,   -1,{0x0390,0xffff}},  //  4697, depth=1
{0xc0000000001fffff,   -1,{0xffff,0x1096}},  //  4698
{0x000000ffffffffe0,   -1,{0xffff,0x1ee2}},  //  4699
{0x001ffff0001ffff0,   -1,{0xffff,0x0710}},  //  4700, depth=1
{0x001fffffffe00000,   -1,{0xffff,0x1adf}},  //  4701
{0xffffc003ffffffff,   -1,{0xffff,0x14b3}},  //  4702
{0xfffff9ffffffffff,   -1,{0xffff,0x157d}},  //  4703, depth=1
{0x7fffffe07fffffe0, 4201,{0xffff,0x06d9}},  //  4704, depth=2
{0x00000000fffc000f, 4206,{0x0391,0xffff}},  //  4705, depth=3
{0x00003c0000003c00,   -1,{0xffff,0x0583}},  //  4706
{0xffffffffffc0007f,   -1,{0xffff,0x1ab0}},  //  4707
{0xffffffff3fffffff,   -1,{0xffff,0x183d}},  //  4708
{0x0000003ffff80000,   -1,{0xffff,0x1b52}},  //  4709
{0x00000003ffffffe0,   -1,{0xffff,0x1edc}},  //  4710, depth=1
{0xffffe00000ffffff,   -1,{0xffff,0x14ea}},  //  4711
{0xffff8000001fffff,   -1,{0xffff,0x1465}},  //  4712
{0xf83fffffffffffff, 4212,{0xffff,0x117a}},  //  4713, depth=2
{0x000003fc000003fc,   -1,{0xffff,0x0787}},  //  4714, depth=1
{0x00000000f07ff07f, 4215,{0x012a,0xffff}},  //  4715, depth=2
{0x000000fffffffffe,   -1,{0xffff,0x1fe6}},  //  4716
{0xe001ffffffffffff,   -1,{0xffff,0x10f3}},  //  4717
{0x0000010000000000,   -1,{0xffff,0x1600}},  //  4718
{0x00003ffe00000000,   -1,{0xffff,0x17cc}},  //  4719
{0x003e0000003e0000,   -1,{0xffff,0x03c4}},  //  4720, depth=1
{0x00000000fffc001f,   -1,{0x0392,0xffff}},  //  4721, depth=1
{0x0000000007e00000, 4219,{0x02c5,0x1ac5}},  //  4722, depth=2
{0x00000000e1ffffff,   -1,{0x00db,0xffff}},  //  4723, depth=1
{0x00001fffffffff80,   -1,{0xffff,0x1e65}},  //  4724
{0xffffffffff7fffff,   -1,{0xffff,0x1a3e}},  //  4725
{0x0000000000fc00fc, 4229,{0x03a5,0xffff}},  //  4726, depth=3
{0xf0000000000007ff,   -1,{0xffff,0x110e}},  //  4727, depth=1
{0x00000001ffffff00,   -1,{0xffff,0x1e18}},  //  4728
{0xffffffffc07fffff,   -1,{0xffff,0x18b8}},  //  4729, depth=1
{0x0000020000000200, 4241,{0xffff,0x05c0}},  //  4730, depth=2
{0xfffffff00000007f,   -1,{0xffff,0x1722}},  //  4731, depth=1
{0xffffffffff800001,   -1,{0xffff,0x1a69}},  //  4732
{0xfffff803ffffffff, 4735,{0xffff,0x1576}},  //  4733
{0x00000003fffffff8,   -1,{0xffff,0x1f5e}},  //  4734, depth=1
{0xc00000000fffffff,   -1,{0xffff,0x109d}},  //  4735
{0x01ffffc001ffffc0, 4242,{0xffff,0x0692}},  //  4736, depth=2
{0xffffffffffffff3f,   -1,{0xffff,0x1e3d}},  //  4737, depth=1
{0x0007ffffffffe000, 4247,{0xffff,0x1ce5}},  //  4738, depth=2
{0x00000000f0001fff,   -1,{0x0110,0xffff}},  //  4739, depth=1
{0x00000003fffffffe,   -1,{0xffff,0x1fe0}},  //  4740, depth=1
{0x00000003ffffffff,   -1,{0xffff,0x1021}},  //  4741, depth=1
{0x01f801f801f801f8, 4250,{0xffff,0x0365}},  //  4742, depth=2
{0xfffffff80fffffff,   -1,{0xffff,0x1778}},  //  4743, depth=1
{0x000000000001f800,   -1,{0x0545,0x1d45}},  //  4744, depth=1
{0xffffffffff800007,   -1,{0xffff,0x1a6b}},  //  4745
{0x0000000400000004, 4253,{0xffff,0x0780}},  //  4746, depth=2
{0x07ffff0000000000,   -1,{0xffff,0x1612}},  //  4747
{0xffffffffe00000ff,   -1,{0xffff,0x18ea}},  //  4748
{0x000000001fff1fff,   -1,{0x002c,0xffff}},  //  4749, depth=1
{0x0060006000600060,   -1,{0xffff,0x02e1}},  //  4750, depth=1
{0xc000007fffffffff,   -1,{0xffff,0x10a8}},  //  4751, depth=1
{0x00fc000000fc0000, 4254,{0xffff,0x0385}},  //  4752, depth=2
{0x00000000fffc003f, 4256,{0x0393,0xffff}},  //  4753, depth=2
{0x0001ffffffffffe0,   -1,{0xffff,0x1eeb}},  //  4754, depth=1
{0x00000000fe003fff, 4257,{0x01d4,0xffff}},  //  4755, depth=2
{0xfffffffc01ffffff,   -1,{0xffff,0x17b6}},  //  4756
{0xfffffffff000ffff,   -1,{0xffff,0x1933}},  //  4757
{0x03fffffff0000000,   -1,{0xffff,0x191d}},  //  4758
{0xffffff0001ffffff,   -1,{0xffff,0x1630}},  //  4759, depth=1
{0xff00000000003fff,   -1,{0xffff,0x1215}},  //  4760
{0xffffffff1fffffff,   -1,{0xffff,0x183c}},  //  4761
{0x007ffff800000000,   -1,{0xffff,0x1753}},  //  4762, depth=1
{0xffffffffff80001f,   -1,{0xffff,0x1a6d}},  //  4763
{0x001ff800001ff800,   -1,{0xffff,0x0549}},  //  4764, depth=1
{0xf01ff01ff01ff01f,   -1,{0xffff,0x0128}},  //  4765, depth=1
{0x00000000007c0000,   -1,{0x0384,0x1b84}},  //  4766, depth=1
{0xfffe0000000fffff,   -1,{0xffff,0x13e2}},  //  4767, depth=1
{0x3800000000000000,   -1,{0xffff,0x1142}},  //  4768, depth=1
{0x00000000c1ffffff,   -1,{0x009a,0xffff}},  //  4769, depth=1
{0x00001ffffffe0000, 4263,{0xffff,0x1bdb}},  //  4770, depth=2
{0xff80003fff80003f,   -1,{0xffff,0x024e}},  //  4771, depth=1
{0xffff0000ffffffff,   -1,{0xffff,0x142f}},  //  4772
{0xfff000fffff000ff,   -1,{0xffff,0x0313}},  //  4773, depth=1
{0x00003fffff800000,   -1,{0xffff,0x1a56}},  //  4774, depth=1
{0xffffffffffff87ff,   -1,{0xffff,0x1c7b}},  //  4775
{0xf000007fffffffff,   -1,{0xffff,0x112a}},  //  4776
{0xffff1fffffffffff,   -1,{0xffff,0x143c}},  //  4777, depth=1
{0x0ffff0000ffff000, 4265,{0xffff,0x050f}},  //  4778, depth=2
{0x03ffffffffe00000,   -1,{0xffff,0x1ae4}},  //  4779
{0xfffffffeffffffff, 4794,{0xffff,0x17fe}},  //  4780
{0xfff00001fff00001, 4269,{0xffff,0x030c}},  //  4781, depth=2
{0xfc000000fc000000, 4272,{0xffff,0x0185}},  //  4782, depth=3
{0xf80ffffff80fffff,   -1,{0xffff,0x0158}},  //  4783, depth=1
{0x0001fffffffffffe,   -1,{0xffff,0x1fef}},  //  4784, depth=1
{0x0001ffffffffffff,   -1,{0xffff,0x1030}},  //  4785, depth=1
{0x0002000000000000,   -1,{0xffff,0x13c0}},  //  4786, depth=1
{0xffff807fffffffff,   -1,{0xffff,0x1477}},  //  4787, depth=1
{0x000000e0000000e0, 4286,{0xffff,0x06c2}},  //  4788, depth=2
{0x00000000ffe01fff, 4288,{0x02d7,0xffff}},  //  4789, depth=2
{0x0000000001ffff00,   -1,{0x0610,0x1e10}},  //  4790, depth=1
{0xfc00000000001fff, 4289,{0xffff,0x1192}},  //  4791, depth=2
{0x1fffffffff800000, 4290,{0xffff,0x1a65}},  //  4792, depth=2
{0xf800000000000fff,   -1,{0xffff,0x1150}},  //  4793, depth=1
{0xff81ff81ff81ff81,   -1,{0xffff,0x0269}},  //  4794
{0xf3ffffffffffffff,   -1,{0xffff,0x113d}},  //  4795, depth=1
{0xffffffff00000001,   -1,{0xffff,0x1820}},  //  4796
{0x00000000f03ff03f, 4291,{0x0129,0xffff}},  //  4797, depth=4
{0xffffffff00000003,   -1,{0xffff,0x1821}},  //  4798
{0x0000000077777777,   -1,{0x003a,0xffff}},  //  4799, depth=1
{0xffffff00ffffff00,   -1,{0xffff,0x0617}},  //  4800
{0xffffffffffffff7f,   -1,{0xffff,0x1e3e}},  //  4801, depth=1
{0x000001ffff000000, 4296,{0xffff,0x1a10}},  //  4802, depth=2
{0xffffffffffffff81,   -1,{0xffff,0x1e79}},  //  4803, depth=1
{0x00fffffffe000000,   -1,{0xffff,0x19de}},  //  4804, depth=1
{0xffffffffffffff83,   -1,{0xffff,0x1e7a}},  //  4805, depth=1
{0xfffffc7fffffffff,   -1,{0xffff,0x15bc}},  //  4806
{0xfffffe0000000fff,   -1,{0xffff,0x15e2}},  //  4807, depth=1
{0x0001f0000001f000,   -1,{0xffff,0x0504}},  //  4808, depth=1
{0x00000000f00007ff, 4300,{0x010e,0xffff}},  //  4809, depth=3
{0x00001fffffffffe0,   -1,{0xffff,0x1ee7}},  //  4810
{0xfffffffc000fffff,   -1,{0xffff,0x17b1}},  //  4811, depth=1
{0x000007ff00000000,   -1,{0xffff,0x180a}},  //  4812, depth=1
{0x00000000fff00fff,   -1,{0x0317,0xffff}},  //  4813
{0x000000000001e000,   -1,{0x04c3,0x1cc3}},  //  4814, depth=1
{0x00007fff00007fff, 4306,{0xffff,0x000e}},  //  4815, depth=2
{0x07ffffffff000000,   -1,{0xffff,0x1a22}},  //  4816, depth=1
{0x00000000fffc007f, 4312,{0x0394,0xffff}},  //  4817, depth=3
{0x00ff800000ff8000, 4317,{0xffff,0x0448}},  //  4818, depth=2
{0xf8000000000fffff,   -1,{0xffff,0x1158}},  //  4819
{0xffffffffffc000ff,   -1,{0xffff,0x1ab1}},  //  4820
{0x1fffffff1fffffff, 4318,{0xffff,0x001c}},  //  4821, depth=4
{0x00000000fffffc00,   -1,{0x0595,0x1d95}},  //  4822, depth=1
{0x00000000fffffc01,   -1,{0x0596,0xffff}},  //  4823, depth=1
{0xff8003ffffffffff, 4827,{0xffff,0x1272}},  //  4824
{0x00000000fffffc03,   -1,{0x0597,0xffff}},  //  4825, depth=1
{0x001fffffffff0000,   -1,{0xffff,0x1c24}},  //  4826, depth=1
{0x00000000fffffdff,   -1,{0x059e,0xffff}},  //  4827
{0x0f00000000000000,   -1,{0xffff,0x1203}},  //  4828, depth=1
{0x00000000fffffc07, 4328,{0x0598,0xffff}},  //  4829, depth=2
{0x0003fffc00000000,   -1,{0xffff,0x178f}},  //  4830, depth=1
{0x00000000fffffe00,   -1,{0x05d6,0x1dd6}},  //  4831
{0x7ffff0007ffff000, 4329,{0xffff,0x0512}},  //  4832, depth=2
{0xffffffffffffff9f,   -1,{0xffff,0x1e7d}},  //  4833, depth=1
{0x00000003c0000000,   -1,{0xffff,0x1883}},  //  4834, depth=1
{0x80000001ffffffff,   -1,{0xffff,0x1061}},  //  4835, depth=1
{0x00001ffffffffe00,   -1,{0xffff,0x1de3}},  //  4836, depth=1
{0x00000000fffffc0f,   -1,{0x0599,0xffff}},  //  4837, depth=1
{0xfffffe00000fffff, 4840,{0xffff,0x15ea}},  //  4838
{0x00000000ff0fffff,   -1,{0x021b,0xffff}},  //  4839, depth=1
{0x00000000fffffe01,   -1,{0x05d7,0xffff}},  //  4840
{0xfffc0007ffffffff,   -1,{0xffff,0x13b0}},  //  4841, depth=1
{0x0000000fffff8000,   -1,{0xffff,0x1c54}},  //  4842, depth=1
{0x003ffffc003ffffc,   -1,{0xffff,0x0793}},  //  4843
{0x00001ffffffffffc,   -1,{0xffff,0x1faa}},  //  4844
{0xffffffffff80007f,   -1,{0xffff,0x1a6f}},  //  4845
{0x00000001ffffff80,   -1,{0xffff,0x1e59}},  //  4846
{0xf80ff80ff80ff80f,   -1,{0xffff,0x0168}},  //  4847, depth=1
{0x7e7e7e7e7e7e7e7e, 4330,{0xffff,0x01f5}},  //  4848, depth=2
{0xf0000fffffffffff,   -1,{0xffff,0x112f}},  //  4849
{0x00003fff00000000,   -1,{0xffff,0x180d}},  //  4850, depth=1
{0xfff8007ffff8007f,   -1,{0xffff,0x0353}},  //  4851, depth=1
{0x0003fe0000000000,   -1,{0xffff,0x15c8}},  //  4852
{0x00000000fffffc1f, 4334,{0x059a,0xffff}},  //  4853, depth=2
{0xffffff80001fffff,   -1,{0xffff,0x166d}},  //  4854
{0xffffffe00001ffff,   -1,{0xffff,0x16eb}},  //  4855, depth=1
{0xfffe00ffffffffff,   -1,{0xffff,0x13f6}},  //  4856
{0x8000ffffffffffff,   -1,{0xffff,0x1070}},  //  4857, depth=1
{0xffffffff0000003f,   -1,{0xffff,0x1825}},  //  4858
{0x00000000ffe007ff, 4336,{0x02d5,0xffff}},  //  4859, depth=2
{0x000000000ff80ff8, 4338,{0x0368,0xffff}},  //  4860, depth=2
{0x0000000081ffffff, 4342,{0x0059,0xffff}},  //  4861, depth=2
{0x000000ffffc00000,   -1,{0xffff,0x1a91}},  //  4862
{0xfffc7ffffffc7fff, 4346,{0xffff,0x039c}},  //  4863, depth=3
{0x01f001f001f001f0, 4349,{0xffff,0x0324}},  //  4864, depth=2
{0xffffffffffffffbf,   -1,{0xffff,0x1e7e}},  //  4865, depth=1
{0x0000007fffffc000, 4357,{0xffff,0x1c98}},  //  4866, depth=2
{0xffffffffffffffc1,   -1,{0xffff,0x1eba}},  //  4867, depth=1
{0x1fffffff00000000, 4362,{0xffff,0x181c}},  //  4868, depth=2
{0xffc0000000003fff, 4366,{0xffff,0x1297}},  //  4869, depth=2
{0x3f00000000000000,   -1,{0xffff,0x1205}},  //  4870
{0xfffffffff8001fff,   -1,{0xffff,0x1971}},  //  4871
{0xffffffffff3fffff,   -1,{0xffff,0x1a3d}},  //  4872
{0xe00007ffffffffff, 4371,{0xffff,0x10ed}},  //  4873, depth=2
{0x000000007ffc0000,   -1,{0x038c,0x1b8c}},  //  4874, depth=1
{0x8fffffff8fffffff, 4375,{0xffff,0x005c}},  //  4875, depth=3
{0xffffffff800000ff,   -1,{0xffff,0x1868}},  //  4876
{0x00000007fffff000,   -1,{0xffff,0x1d16}},  //  4877
{0x0001ffffc0000000,   -1,{0xffff,0x1892}},  //  4878, depth=1
{0x00000000efffefff, 4386,{0x00ee,0xffff}},  //  4879, depth=2
{0xffc03fffffffffff,   -1,{0xffff,0x12b7}},  //  4880
{0x0003ffff0003ffff, 4387,{0xffff,0x0011}},  //  4881, depth=3
{0x03ffe00003ffe000, 4392,{0xffff,0x04cc}},  //  4882, depth=2
{0x07fffffffe000000,   -1,{0xffff,0x19e1}},  //  4883
{0x000003ffffe00000,   -1,{0xffff,0x1ad4}},  //  4884
{0x00000000fffffc3f, 4395,{0x059b,0xffff}},  //  4885, depth=2
{0x00fff80000000000,   -1,{0xffff,0x154c}},  //  4886
{0x0007ffc000000000,   -1,{0xffff,0x168c}},  //  4887
{0x00000003ffc00000, 4396,{0xffff,0x1a8b}},  //  4888, depth=2
{0xfe00000007ffffff,   -1,{0xffff,0x11e1}},  //  4889
{0x00000000007c007c, 4397,{0x03a4,0xffff}},  //  4890, depth=2
{0x00000000ffffe3ff, 4399,{0x04dc,0xffff}},  //  4891, depth=2
{0x01ffffe001ffffe0,   -1,{0xffff,0x06d3}},  //  4892
{0xc0007fffffffffff,   -1,{0xffff,0x10b0}},  //  4893, depth=1
{0x1800000018000000,   -1,{0xffff,0x0141}},  //  4894, depth=1
{0x00000000fe000fff, 4400,{0x01d2,0xffff}},  //  4895, depth=2
{0xff8000000007ffff,   -1,{0xffff,0x125b}},  //  4896
{0xffffffffffffffdf,   -1,{0xffff,0x1ebe}},  //  4897, depth=1
{0x00000000007ffc00, 4408,{0x058c,0x1d8c}},  //  4898, depth=2
{0xffe0000000001fff, 4409,{0xffff,0x12d7}},  //  4899, depth=2
{0x00000001fffe0000,   -1,{0xffff,0x1bcf}},  //  4900, depth=1
{0xf00003ffffffffff, 4410,{0xffff,0x112d}},  //  4901, depth=2
{0x0000000004000400,   -1,{0x01a0,0xffff}},  //  4902, depth=1
{0xffffffff0000007f,   -1,{0xffff,0x1826}},  //  4903
{0x3ffffffffffff000,   -1,{0xffff,0x1d31}},  //  4904, depth=1
{0x8001ffff8001ffff, 4413,{0xffff,0x0051}},  //  4905, depth=2
{0x000ffffffff80000,   -1,{0xffff,0x1b60}},  //  4906, depth=1
{0xffff0001ffffffff,   -1,{0xffff,0x1430}},  //  4907, depth=1
{0xffffe000ffffffff,   -1,{0xffff,0x14f2}},  //  4908
{0x03fff00003fff000,   -1,{0xffff,0x050d}},  //  4909
{0x0000070000000000,   -1,{0xffff,0x1602}},  //  4910
{0x0000000080018001, 4414,{0x0061,0xffff}},  //  4911, depth=2
{0x00000000003c0000,   -1,{0x0383,0x1b83}},  //  4912, depth=1
{0xffffffffffffffef,   -1,{0xffff,0x1efe}},  //  4913, depth=1
{0xfffffffffffffff0,   -1,{0xffff,0x1f3b}},  //  4914, depth=1
{0xf80001ffffffffff, 4420,{0xffff,0x116d}},  //  4915, depth=2
{0x7fffffffffffff80, 4919,{0xffff,0x1e77}},  //  4916
{0xc000ffffc000ffff, 4422,{0xffff,0x0091}},  //  4917, depth=2
{0x4000400040004000, 4423,{0xffff,0x00a0}},  //  4918, depth=2
{0x003e000000000000, 4924,{0xffff,0x13c4}},  //  4919
{0x7fffffe000000000,   -1,{0xffff,0x16d9}},  //  4920, depth=1
{0xfffffffffffffff7,   -1,{0xffff,0x1f3e}},  //  4921, depth=1
{0x0000000022222222, 4426,{0x00f8,0xffff}},  //  4922, depth=3
{0xe0007fffe0007fff, 4446,{0xffff,0x00d1}},  //  4923, depth=2
{0x0000000007f80000,   -1,{0x0347,0x1b47}},  //  4924
{0xfffffffffffffffb,   -1,{0xffff,0x1f7e}},  //  4925, depth=1
{0x0078000000780000, 4451,{0xffff,0x0343}},  //  4926, depth=3
{0x00000000fe0ffe0f, 4461,{0x01ea,0xffff}},  //  4927, depth=2
{0xfffffffffffffffe,   -1,{0xffff,0x1ffe}},  //  4928, depth=1
{0xffff8007ffffffff,   -1,{0xffff,0x1473}},  //  4929
{0xffff00007fffffff, 4931,{0xffff,0x142e}},  //  4930
{0x00000000f803ffff,   -1,{0x0156,0xffff}},  //  4931
{0x0001ffffffc00000, 4464,{0xffff,0x1a9a}},  //  4932, depth=2
{0x0000ffffffffffe0,   -1,{0xffff,0x1eea}},  //  4933
{0xff00ff00ff00ff00, 4465,{0xffff,0x0227}},  //  4934, depth=2
{0x00000001fffffff8, 4936,{0xffff,0x1f5d}},  //  4935
{0x0000003ffffff800,   -1,{0xffff,0x1d5a}},  //  4936
{0xffffff01ffffff01,   -1,{0xffff,0x0618}},  //  4937, depth=1
{0xfffffffffffc7fff,   -1,{0xffff,0x1bbc}},  //  4938
{0xfe01fe01fe01fe01, 4467,{0xffff,0x01e7}},  //  4939, depth=2
{0x001f800000000000,   -1,{0xffff,0x1445}},  //  4940
{0x00000001fffffffe,   -1,{0xffff,0x1fdf}},  //  4941
{0xffffffffff8000ff,   -1,{0xffff,0x1a70}},  //  4942
{0x0000000200000000,   -1,{0xffff,0x17c0}},  //  4943
{0x000fffe0000fffe0, 4469,{0xffff,0x06ce}},  //  4944, depth=2
{0x00000000fffc00ff, 4472,{0x0395,0xffff}},  //  4945, depth=2
{0x07ff000000000000,   -1,{0xffff,0x140a}},  //  4946, depth=1
{0x00003f0000000000,   -1,{0xffff,0x1605}},  //  4947
{0x007ffffff0000000, 4473,{0xffff,0x191a}},  //  4948, depth=2
{0xfc03fc03fc03fc03, 4483,{0xffff,0x01a7}},  //  4949, depth=3
{0xfffffffe7fffffff,   -1,{0xffff,0x17fd}},  //  4950
{0xffffffffe00001ff,   -1,{0xffff,0x18eb}},  //  4951
{0x001ffff800000000,   -1,{0xffff,0x1751}},  //  4952, depth=1
{0xfffe1fffffffffff, 4955,{0xffff,0x13fb}},  //  4953
{0x000001fe00000000,   -1,{0xffff,0x17c7}},  //  4954, depth=1
{0xfc00000ffc00000f,   -1,{0xffff,0x0189}},  //  4955
{0x0000fffffffffff8,   -1,{0xffff,0x1f6c}},  //  4956
{0xffffc00001ffffff,   -1,{0xffff,0x14aa}},  //  4957
{0x0000fffffffffffc,   -1,{0xffff,0x1fad}},  //  4958
{0xf001fffff001ffff,   -1,{0xffff,0x0114}},  //  4959, depth=1
{0x01fffffffff80000, 4487,{0xffff,0x1b65}},  //  4960, depth=2
{0xfffffc07fffffc07,   -1,{0xffff,0x0598}},  //  4961, depth=1
{0xffff0000003fffff,   -1,{0xffff,0x1425}},  //  4962
{0x0000fffffffffffe, 4964,{0xffff,0x1fee}},  //  4963
{0x1fffff0000000000,   -1,{0xffff,0x1614}},  //  4964
{0x7fffffffffffffc0,   -1,{0xffff,0x1eb8}},  //  4965
{0x000000003ffc0000, 4489,{0x038b,0x1b8b}},  //  4966, depth=2
{0xfffc00000000ffff, 4504,{0xffff,0x139d}},  //  4967, depth=2
{0x07fffffe00000000,   -1,{0xffff,0x17d9}},  //  4968, depth=1
{0xf807f807f807f807, 4505,{0xffff,0x0167}},  //  4969, depth=2
{0x0001ffff80000000,   -1,{0xffff,0x1851}},  //  4970, depth=1
{0xc000007fc000007f,   -1,{0xffff,0x0088}},  //  4971, depth=1
{0x00000000003c003c, 4512,{0x03a3,0xffff}},  //  4972, depth=2
{0xff0001ffffffffff,   -1,{0xffff,0x1230}},  //  4973
{0xfffffffffeffffff,   -1,{0xffff,0x19fe}},  //  4974
{0xffffffffdfffffff,   -1,{0xffff,0x18be}},  //  4975, depth=1
{0xffffffffe0000000,   -1,{0xffff,0x18e2}},  //  4976, depth=1
{0xffffffffe0000001,   -1,{0xffff,0x18e3}},  //  4977, depth=1
{0x0380038003800380, 4513,{0xffff,0x0262}},  //  4978, depth=2
{0xffffffffe0000003,   -1,{0xffff,0x18e4}},  //  4979, depth=1
{0xffffffffff000000,   -1,{0xffff,0x1a27}},  //  4980
{0xffffffffff000001,   -1,{0xffff,0x1a28}},  //  4981
{0x0000000001ffffc0,   -1,{0x0692,0x1e92}},  //  4982, depth=1
{0xffff007fffff007f, 4517,{0xffff,0x0416}},  //  4983, depth=2
{0x0000000003c003c0, 4521,{0x02a3,0xffff}},  //  4984, depth=2
{0x0000000ffffffc00,   -1,{0xffff,0x1d99}},  //  4985
{0xffffffffff000003,   -1,{0xffff,0x1a29}},  //  4986
{0x8000000000ffffff,   -1,{0xffff,0x1058}},  //  4987, depth=1
{0x0000fffffffffe00,   -1,{0xffff,0x1de6}},  //  4988, depth=1
{0xe003ffffe003ffff,   -1,{0xffff,0x00d4}},  //  4989, depth=1
{0xfffff007ffffffff, 4995,{0xffff,0x1536}},  //  4990
{0xf00000000007ffff, 4527,{0xffff,0x1116}},  //  4991, depth=2
{0x01ffe00001ffe000, 4531,{0xffff,0x04cb}},  //  4992, depth=2
{0xfffff80ffffff80f,   -1,{0xffff,0x0558}},  //  4993, depth=1
{0x7ffffffffffe0000,   -1,{0xffff,0x1bed}},  //  4994, depth=1
{0x800000001fffffff,   -1,{0xffff,0x105d}},  //  4995
{0x000ffc0000000000,   -1,{0xffff,0x1589}},  //  4996, depth=1
{0xffffffffff00000f,   -1,{0xffff,0x1a2b}},  //  4997
{0x0000000020202020, 4532,{0x00f0,0xffff}},  //  4998, depth=3
{0xffffffffffffe7ff,   -1,{0xffff,0x1cfd}},  //  4999, depth=1
{0xfffff80001ffffff,   -1,{0xffff,0x156d}},  //  5000
{0x00000000fc1fffff,   -1,{0x019a,0xffff}},  //  5001, depth=1
{0xf8000000f8000000,   -1,{0xffff,0x0144}},  //  5002, depth=1
{0xfffffff803ffffff,   -1,{0xffff,0x1776}},  //  5003
{0x007fff80007fff80,   -1,{0xffff,0x064f}},  //  5004, depth=1
{0xfff800000001ffff, 4537,{0xffff,0x135d}},  //  5005, depth=2
{0x000000007ffffc00,   -1,{0x0594,0x1d94}},  //  5006, depth=1
{0xffffffffe000001f, 4538,{0xffff,0x18e7}},  //  5007, depth=2
{0x00007fffffffc000,   -1,{0xffff,0x1ca0}},  //  5008, depth=1
{0xf00ff00ff00ff00f, 4539,{0xffff,0x0127}},  //  5009, depth=2
{0x00000ffffc000000,   -1,{0xffff,0x1991}},  //  5010, depth=1
{0x7fffffffffffffe0,   -1,{0xffff,0x1ef9}},  //  5011
{0x000000001ffc0000,   -1,{0x038a,0x1b8a}},  //  5012, depth=1
{0x800000ff800000ff,   -1,{0xffff,0x0048}},  //  5013, depth=1
{0x0000000001ffffe0, 4540,{0x06d3,0x1ed3}},  //  5014, depth=2
{0x001ff00000000000,   -1,{0xffff,0x1508}},  //  5015
{0x0000007800000000,   -1,{0xffff,0x1743}},  //  5016
{0xffffffffe001ffff,   -1,{0xffff,0x18f3}},  //  5017
{0x0007ffffffff8000,   -1,{0xffff,0x1c63}},  //  5018, depth=1
{0xfffffc000007ffff,   -1,{0xffff,0x15a8}},  //  5019, depth=1
{0x7ffffffffffffff0,   -1,{0xffff,0x1f3a}},  //  5020
{0x8000000180000001, 4541,{0xffff,0x0041}},  //  5021, depth=2
{0x0001fff00001fff0, 4542,{0xffff,0x070c}},  //  5022, depth=3
{0xffffffffc0000001,   -1,{0xffff,0x18a2}},  //  5023, depth=1
{0x0000000000ffffc0,   -1,{0x0691,0x1e91}},  //  5024
{0xffffffffc0000003,   -1,{0xffff,0x18a3}},  //  5025, depth=1
{0x7ffffffffffffffc,   -1,{0xffff,0x1fbc}},  //  5026
{0xfe0000000003ffff,   -1,{0xffff,0x11d8}},  //  5027, depth=1
{0x000007ffff000000,   -1,{0xffff,0x1a12}},  //  5028, depth=1
{0xffffffffc0000007,   -1,{0xffff,0x18a4}},  //  5029, depth=1
{0x0000000001fffff0,   -1,{0x0714,0x1f14}},  //  5030, depth=1
{0xfe00000000007fff,   -1,{0xffff,0x11d5}},  //  5031
{0x7ffffffffffffffe, 5033,{0xffff,0x1ffd}},  //  5032
{0x00000001c0000000,   -1,{0xffff,0x1882}},  //  5033
{0x00000003ff800000,   -1,{0xffff,0x1a4a}},  //  5034, depth=1
{0xfc0003ffffffffff,   -1,{0xffff,0x11af}},  //  5035, depth=1
{0x7fffffffffffffff,   -1,{0xffff,0x103e}},  //  5036
{0xfffe00fffffe00ff, 4550,{0xffff,0x03d6}},  //  5037, depth=2
{0x3fc000003fc00000, 4552,{0xffff,0x0287}},  //  5038, depth=2
{0xfdfffffffdffffff, 4553,{0xffff,0x019e}},  //  5039, depth=2
{0x1ff000001ff00000, 4561,{0xffff,0x0308}},  //  5040, depth=3
{0xfffffffe3fffffff, 5043,{0xffff,0x17fc}},  //  5041
{0x0020000000200000, 4566,{0xffff,0x02c0}},  //  5042, depth=2
{0x8000000000000001,   -1,{0xffff,0x1041}},  //  5043
{0x00000000003ffc00, 4569,{0x058b,0x1d8b}},  //  5044, depth=3
{0x0000000001ffffff, 4576,{0x0018,0x1018}},  //  5045, depth=2
{0x0000000002000000, 4579,{0x01c0,0x19c0}},  //  5046, depth=2
{0x00000000c01fc01f,   -1,{0x00a6,0xffff}},  //  5047, depth=1
{0x01ffffffffff8000,   -1,{0xffff,0x1c69}},  //  5048
{0xc007ffffc007ffff,   -1,{0xffff,0x0094}},  //  5049, depth=1
{0x01fffc0000000000,   -1,{0xffff,0x158e}},  //  5050, depth=1
{0xfff7fff7fff7fff7,   -1,{0xffff,0x032e}},  //  5051, depth=1
{0x0000000000fe0000,   -1,{0x03c6,0x1bc6}},  //  5052, depth=1
{0xe0000000000fffff, 4582,{0xffff,0x10d6}},  //  5053, depth=2
{0xffff80000000ffff,   -1,{0xffff,0x1460}},  //  5054
{0xfffffff80000ffff,   -1,{0xffff,0x176c}},  //  5055, depth=1
{0x0ffff8000ffff800,   -1,{0xffff,0x0550}},  //  5056, depth=1
{0xfffff01ffffff01f,   -1,{0xffff,0x0518}},  //  5057, depth=1
{0x007f0000007f0000,   -1,{0xffff,0x0406}},  //  5058, depth=1
{0xffff000000ffffff, 4584,{0xffff,0x1427}},  //  5059, depth=2
{0x03fffff800000000, 4585,{0xffff,0x1756}},  //  5060, depth=2
{0xffffffffff00003f,   -1,{0xffff,0x1a2d}},  //  5061
{0xfff800000003ffff,   -1,{0xffff,0x135e}},  //  5062
{0xffffffffffc001ff,   -1,{0xffff,0x1ab2}},  //  5063
{0x00fe000000000000,   -1,{0xffff,0x13c6}},  //  5064, depth=1
{0x00000000fffffeff,   -1,{0x05de,0xffff}},  //  5065
{0x0000000003800380, 4588,{0x0262,0xffff}},  //  5066, depth=2
{0xf00000fff00000ff, 4591,{0xffff,0x010b}},  //  5067, depth=2
{0x0fffffffffff0000,   -1,{0xffff,0x1c2b}},  //  5068, depth=1
{0xffffffffffffcfff,   -1,{0xffff,0x1cbd}},  //  5069, depth=1
{0x7ff8000000000000,   -1,{0xffff,0x134b}},  //  5070, depth=1
{0xfbfbfbfbfbfbfbfb, 4593,{0xffff,0x0176}},  //  5071, depth=2
{0x000ff000000ff000,   -1,{0xffff,0x0507}},  //  5072, depth=1
{0x00000000e003e003, 4594,{0x00e4,0xffff}},  //  5073, depth=2
{0x000000001e1e1e1e,   -1,{0x01f3,0xffff}},  //  5074, depth=1
{0xf0000001f0000001, 4596,{0xffff,0x0104}},  //  5075, depth=2
{0xffffffffffc00000,   -1,{0xffff,0x1aa9}},  //  5076, depth=1
{0x00000000fffffcff, 4597,{0x059d,0xffff}},  //  5077, depth=2
{0x2020202020202020, 4609,{0xffff,0x00f0}},  //  5078, depth=2
{0xfff8001fffffffff, 4615,{0xffff,0x1371}},  //  5079, depth=2
{0x3800380038003800, 4618,{0xffff,0x0162}},  //  5080, depth=2
{0xfff000000003ffff, 4621,{0xffff,0x131d}},  //  5081, depth=2
{0x00000000ffffff00,   -1,{0x0617,0x1e17}},  //  5082
{0xffffffffffc00007,   -1,{0xffff,0x1aac}},  //  5083, depth=1
{0x1fffffffff000000,   -1,{0xffff,0x1a24}},  //  5084, depth=1
{0xffffffffc000003f, 4629,{0xffff,0x18a7}},  //  5085, depth=2
{0x0001f8000001f800, 4632,{0xffff,0x0545}},  //  5086, depth=2
{0xffffff87ffffffff,   -1,{0xffff,0x167b}},  //  5087
{0x000000ffffffff80,   -1,{0xffff,0x1e60}},  //  5088, depth=1
{0xe01fe01fe01fe01f, 4639,{0xffff,0x00e7}},  //  5089, depth=2
{0x07ffffe007ffffe0, 4642,{0xffff,0x06d5}},  //  5090, depth=2
{0xf7fff7fff7fff7ff, 4643,{0xffff,0x012e}},  //  5091, depth=2
{0x00001fffffffff00,   -1,{0xffff,0x1e24}},  //  5092, depth=1
{0xffe00fffffffffff,   -1,{0xffff,0x12f6}},  //  5093
{0x0000000000018000,   -1,{0x0441,0x1c41}},  //  5094, depth=1
{0xffff00001fffffff,   -1,{0xffff,0x142c}},  //  5095
{0xffffffffc001ffff,   -1,{0xffff,0x18b2}},  //  5096
{0x000001ff000001ff,   -1,{0xffff,0x0008}},  //  5097, depth=1
{0x000000003ffffc00, 4644,{0x0593,0x1d93}},  //  5098, depth=2
{0x00000000ffe7ffe7,   -1,{0x02ed,0xffff}},  //  5099, depth=1
{0xffffffc000001fff,   -1,{0xffff,0x16a6}},  //  5100
{0x003ffffffff00000, 5102,{0xffff,0x1b21}},  //  5101
{0x00000007fc000000,   -1,{0xffff,0x1988}},  //  5102
{0xffffffffe000007f,   -1,{0xffff,0x18e9}},  //  5103, depth=1
{0x000007fffe000000,   -1,{0xffff,0x19d1}},  //  5104
{0x001fff8000000000,   -1,{0xffff,0x164d}},  //  5105
{0x0ffc000000000000,   -1,{0xffff,0x1389}},  //  5106, depth=1
{0xffffffffffc0001f,   -1,{0xffff,0x1aae}},  //  5107, depth=1
{0x01e001e001e001e0, 4648,{0xffff,0x02e3}},  //  5108, depth=2
{0xfffff800000fffff,   -1,{0xffff,0x1568}},  //  5109, depth=1
{0x7ffff8007ffff800, 4652,{0xffff,0x0553}},  //  5110, depth=2
{0x7fffffffe0000000, 5112,{0xffff,0x18e1}},  //  5111
{0x00000ff800000000,   -1,{0xffff,0x1748}},  //  5112
{0x0000000300000003, 4654,{0xffff,0x0001}},  //  5113, depth=2
{0xffffffff80000000,   -1,{0xffff,0x1860}},  //  5114, depth=1
{0xfffffff0000001ff, 4656,{0xffff,0x1724}},  //  5115, depth=2
{0xfffe0001ffffffff,   -1,{0xffff,0x13ef}},  //  5116
{0xffffffff80000003,   -1,{0xffff,0x1862}},  //  5117, depth=1
{0x0000000000fffe00,   -1,{0x05ce,0x1dce}},  //  5118, depth=1
{0xff9fffffffffffff,   -1,{0xffff,0x127d}},  //  5119, depth=1
{0x0000000000fffffc,   -1,{0x0795,0x1f95}},  //  5120
{0xffffffff80000007,   -1,{0xffff,0x1863}},  //  5121, depth=1
{0x3ffffffffc000000,   -1,{0xffff,0x19a3}},  //  5122, depth=1
{0xff0003ffff0003ff, 4660,{0xffff,0x0211}},  //  5123, depth=2
{0x000007c0000007c0,   -1,{0xffff,0x0684}},  //  5124, depth=1
{0xfc0000000007ffff,   -1,{0xffff,0x1198}},  //  5125, depth=1
{0xfffffff0000003ff,   -1,{0xffff,0x1725}},  //  5126
{0x0000000000fffffe,   -1,{0x07d6,0x1fd6}},  //  5127
{0x0003f80000000000,   -1,{0xffff,0x1546}},  //  5128, depth=1
{0xffffffff8000000f,   -1,{0xffff,0x1864}},  //  5129, depth=1
{0x0000000000070000, 4663,{0x0402,0x1c02}},  //  5130, depth=2
{0xf007f007f007f007, 4669,{0xffff,0x0126}},  //  5131, depth=2
{0x00000000c0c0c0c0,   -1,{0x00b1,0xffff}},  //  5132, depth=1
{0xffffffffffff0fff,   -1,{0xffff,0x1c3b}},  //  5133
{0x3ffffff83ffffff8,   -1,{0xffff,0x075a}},  //  5134
{0xffffff00007fffff,   -1,{0xffff,0x162e}},  //  5135
{0x00001fffff800000,   -1,{0xffff,0x1a55}},  //  5136
{0x0000000000070007,   -1,{0x0022,0xffff}},  //  5137, depth=1
{0x001ffffff0000000,   -1,{0xffff,0x1918}},  //  5138, depth=1
{0xffffffffffc0003f,   -1,{0xffff,0x1aaf}},  //  5139, depth=1
{0xffff000000000001, 5142,{0xffff,0x1410}},  //  5140
{0xf80007ffffffffff,   -1,{0xffff,0x116f}},  //  5141, depth=1
{0xe00000ffffffffff, 5146,{0xffff,0x10ea}},  //  5142
{0xffe003ffffffffff,   -1,{0xffff,0x12f4}},  //  5143, depth=1
{0x000000001ffffc00,   -1,{0x0592,0x1d92}},  //  5144, depth=1
{0xfffc01fffffc01ff, 4671,{0xffff,0x0396}},  //  5145, depth=2
{0x00000000c07fffff,   -1,{0x0098,0xffff}},  //  5146
{0xfffffffffc000fff,   -1,{0xffff,0x19b1}},  //  5147, depth=1
{0xfff00000fff00000,   -1,{0xffff,0x030b}},  //  5148, depth=1
{0xfbfffffffbffffff, 4679,{0xffff,0x015e}},  //  5149, depth=2
{0x000000001c1c1c1c,   -1,{0x01b2,0xffff}},  //  5150, depth=1
{0xffffffff003fffff,   -1,{0xffff,0x1835}},  //  5151, depth=1
{0x00000000001ff000, 4682,{0x0508,0x1d08}},  //  5152, depth=2
{0x0000ffffffc00000,   -1,{0xffff,0x1a99}},  //  5153
{0x0001ffff00000000,   -1,{0xffff,0x1810}},  //  5154, depth=1
{0xff000007ff000007,   -1,{0xffff,0x020a}},  //  5155, depth=1
{0xffff000000000003,   -1,{0xffff,0x1411}},  //  5156
{0x00003ffffe000000,   -1,{0xffff,0x19d4}},  //  5157
{0xfffff0000000ffff,   -1,{0xffff,0x1523}},  //  5158
{0xff800000000007ff,   -1,{0xffff,0x1253}},  //  5159, depth=1
{0x00003ff800003ff8,   -1,{0xffff,0x074a}},  //  5160, depth=1
{0xfffc0003fffc0003,   -1,{0xffff,0x038f}},  //  5161, depth=1
{0x3ff03ff03ff03ff0,   -1,{0xffff,0x0329}},  //  5162, depth=1
{0x0001e00000000000, 5165,{0xffff,0x14c3}},  //  5163
{0x0000000007f007f0,   -1,{0x0326,0xffff}},  //  5164, depth=1
{0x03fffff003fffff0,   -1,{0xffff,0x0715}},  //  5165
{0x000003f800000000,   -1,{0xffff,0x1746}},  //  5166, depth=1
{0xffffffc007ffffff,   -1,{0xffff,0x16b4}},  //  5167, depth=1
{0x0000001fffff8000,   -1,{0xffff,0x1c55}},  //  5168, depth=1
{0x800fffff800fffff,   -1,{0xffff,0x0054}},  //  5169, depth=1
{0x000ffffffffff800,   -1,{0xffff,0x1d68}},  //  5170, depth=1
{0x00000000ffff83ff, 4686,{0x045a,0xffff}},  //  5171, depth=3
{0xffffffffffc1ffff,   -1,{0xffff,0x1aba}},  //  5172
{0xffefffefffefffef,   -1,{0xffff,0x02ee}},  //  5173, depth=1
{0x0000fff000000000, 4692,{0xffff,0x170b}},  //  5174, depth=2
{0xfe00003fffffffff,   -1,{0xffff,0x11ec}},  //  5175, depth=1
{0x000000003c003c00,   -1,{0x01a3,0xffff}},  //  5176, depth=1
{0x00000000fff03fff, 4695,{0x0319,0xffff}},  //  5177, depth=3
{0x03fff00000000000,   -1,{0xffff,0x150d}},  //  5178
{0xf8003fffffffffff,   -1,{0xffff,0x1172}},  //  5179, depth=1
{0x000007fe00000000,   -1,{0xffff,0x17c9}},  //  5180, depth=1
{0xfffffff00001ffff,   -1,{0xffff,0x172c}},  //  5181, depth=1
{0x00007ffe00007ffe,   -1,{0xffff,0x07cd}},  //  5182, depth=1
{0x0000000180000000,   -1,{0xffff,0x1841}},  //  5183
{0x003ffffffe000000, 4699,{0xffff,0x19dc}},  //  5184, depth=2
{0xffffe03fffffe03f,   -1,{0xffff,0x04d8}},  //  5185, depth=1
{0xcccccccccccccccc,   -1,{0xffff,0x00b9}},  //  5186, depth=1
{0xffffc0ffffffffff,   -1,{0xffff,0x14b9}},  //  5187
{0x1ffffffe1ffffffe, 4701,{0xffff,0x07db}},  //  5188, depth=2
{0xfffe000001ffffff, 4702,{0xffff,0x13e7}},  //  5189, depth=2
{0x00007f0000007f00,   -1,{0xffff,0x0606}},  //  5190, depth=1
{0xffffff01ffffffff,   -1,{0xffff,0x1638}},  //  5191, depth=1
{0x0000000001c00000, 4706,{0x0282,0x1a82}},  //  5192, depth=2
{0xfffffffdffffffff,   -1,{0xffff,0x17be}},  //  5193
{0x0ffffff800000000,   -1,{0xffff,0x1758}},  //  5194, depth=1
{0xfffffffe00000000,   -1,{0xffff,0x17de}},  //  5195
{0x1fffff001fffff00,   -1,{0xffff,0x0614}},  //  5196, depth=1
{0xfffffffe00000001,   -1,{0xffff,0x17df}},  //  5197
{0xffe07fffffffffff,   -1,{0xffff,0x12f9}},  //  5198
{0x0000007800000078,   -1,{0xffff,0x0743}},  //  5199
{0x000000fffffffff0,   -1,{0xffff,0x1f23}},  //  5200, depth=1
{0x00000000fffc01ff,   -1,{0x0396,0xffff}},  //  5201, depth=1
{0x000000000ffff000,   -1,{0x050f,0x1d0f}},  //  5202, depth=1
{0x80000000ffffffff, 4707,{0xffff,0x1060}},  //  5203, depth=2
{0xfffffffe00000007,   -1,{0xffff,0x17e1}},  //  5204
{0xe00001ffe00001ff, 4708,{0xffff,0x00cb}},  //  5205, depth=2
{0x7ffe7ffe7ffe7ffe, 4709,{0xffff,0x03ed}},  //  5206, depth=2
{0xfffffffffc0fffff,   -1,{0xffff,0x19b9}},  //  5207
{0x000000fffffffff8,   -1,{0xffff,0x1f64}},  //  5208, depth=1
{0xffffffffffff9fff,   -1,{0xffff,0x1c7d}},  //  5209, depth=1
{0x0003fffffc000000,   -1,{0xffff,0x1997}},  //  5210
{0xffff81ffffff81ff, 4711,{0xffff,0x0459}},  //  5211, depth=2
{0x000000fffffffffc,   -1,{0xffff,0x1fa5}},  //  5212, depth=1
{0xf7f7f7f7f7f7f7f7, 4712,{0xffff,0x0136}},  //  5213, depth=2
{0x1fff000000000000, 4716,{0xffff,0x140c}},  //  5214, depth=2
{0x000000ffffffffff, 4717,{0xffff,0x1027}},  //  5215, depth=2
{0x7e007e007e007e00, 4718,{0xffff,0x01e5}},  //  5216, depth=2
{0xfffffffff807ffff, 5219,{0xffff,0x1977}},  //  5217
{0x07fff00007fff000, 4719,{0xffff,0x050e}},  //  5218, depth=2
{0xffc1ffffffffffff, 5228,{0xffff,0x12ba}},  //  5219
{0x1c1c1c1c1c1c1c1c, 4724,{0xffff,0x01b2}},  //  5220, depth=2
{0xe0000003e0000003, 4725,{0xffff,0x00c4}},  //  5221, depth=2
{0xffffffffff800000, 4728,{0xffff,0x1a68}},  //  5222, depth=2
{0xf00ffffff00fffff, 4732,{0xffff,0x0117}},  //  5223, depth=2
{0x01fffffffffff800,   -1,{0xffff,0x1d6d}},  //  5224, depth=1
{0xffffffffff800003,   -1,{0xffff,0x1a6a}},  //  5225, depth=1
{0x00fffffffff00000,   -1,{0xffff,0x1b23}},  //  5226, depth=1
{0xffff8003ffff8003, 4733,{0xffff,0x0452}},  //  5227, depth=3
{0x800000000000007f,   -1,{0xffff,0x1047}},  //  5228
{0xfff0003fffffffff, 4745,{0xffff,0x1331}},  //  5229, depth=2
{0x0000000003000300, 4747,{0x0221,0xffff}},  //  5230, depth=2
{0x00000000fff87fff, 4748,{0x035b,0xffff}},  //  5231, depth=2
{0xfffff8ffffffffff,   -1,{0xffff,0x157c}},  //  5232
{0xffe000000007ffff, 4756,{0xffff,0x12dd}},  //  5233, depth=2
{0xf0000000001fffff,   -1,{0xffff,0x1118}},  //  5234
{0x00001fff00000000,   -1,{0xffff,0x180c}},  //  5235
{0x1ffffffe00000000,   -1,{0xffff,0x17db}},  //  5236, depth=1
{0xffffffffff80000f,   -1,{0xffff,0x1a6c}},  //  5237, depth=1
{0x0040004000400040,   -1,{0xffff,0x02a0}},  //  5238, depth=1
{0xffffff800000ffff,   -1,{0xffff,0x1668}},  //  5239
{0x000fffffffffe000,   -1,{0xffff,0x1ce6}},  //  5240, depth=1
{0xffffffff8000007f, 4757,{0xffff,0x1867}},  //  5241, depth=2
{0x3000300030003000,   -1,{0xffff,0x0121}},  //  5242, depth=1
{0x7fffffffffc00000,   -1,{0xffff,0x1aa8}},  //  5243
{0x0000ffffffffff00,   -1,{0xffff,0x1e27}},  //  5244, depth=1
{0xff80007fffffffff,   -1,{0xffff,0x126f}},  //  5245
{0x7ffffff000000000, 4758,{0xffff,0x171a}},  //  5246, depth=2
{0x03ffff8000000000,   -1,{0xffff,0x1652}},  //  5247
{0x0003fffe0003fffe,   -1,{0xffff,0x07d0}},  //  5248, depth=1
{0xc03fc03fc03fc03f, 4760,{0xffff,0x00a7}},  //  5249, depth=2
{0xfffffffff3ffffff,   -1,{0xffff,0x193d}},  //  5250
{0xffe1ffffffe1ffff, 4761,{0xffff,0x02db}},  //  5251, depth=2
{0x01fffffff8000000,   -1,{0xffff,0x195d}},  //  5252
{0xefffefffefffefff, 4763,{0xffff,0x00ee}},  //  5253, depth=2
{0xfff8003fffffffff,   -1,{0xffff,0x1372}},  //  5254
{0xffffffffff8001ff,   -1,{0xffff,0x1a71}},  //  5255
{0x0003ff000003ff00,   -1,{0xffff,0x0609}},  //  5256, depth=1
{0xff0007ffffffffff, 5258,{0xffff,0x1232}},  //  5257
{0xc01fc01fc01fc01f,   -1,{0xffff,0x00a6}},  //  5258
{0xf9fffffff9ffffff,   -1,{0xffff,0x015d}},  //  5259, depth=1
{0xfffffc00001fffff,   -1,{0xffff,0x15aa}},  //  5260
{0xffffff007fffffff,   -1,{0xffff,0x1636}},  //  5261
{0x000000ffe0000000,   -1,{0xffff,0x18ca}},  //  5262, depth=1
{0xffff80000fffffff,   -1,{0xffff,0x146c}},  //  5263, depth=1
{0x01e0000000000000,   -1,{0xffff,0x12c3}},  //  5264, depth=1
{0x00000000fffdfffd,   -1,{0x03ae,0xffff}},  //  5265, depth=1
{0xffffe0001fffffff,   -1,{0xffff,0x14ef}},  //  5266
{0x00000000fffdffff,   -1,{0x039e,0xffff}},  //  5267, depth=1
{0x00000000fffe0000,   -1,{0x03ce,0x1bce}},  //  5268, depth=1
{0x00000000fffe0001,   -1,{0x03cf,0xffff}},  //  5269, depth=1
{0xfffffff800000fff,   -1,{0xffff,0x1768}},  //  5270
{0x00000000fffe0003,   -1,{0x03d0,0xffff}},  //  5271, depth=1
{0x000000007fffff00,   -1,{0x0616,0x1e16}},  //  5272
{0x001ffffe001ffffe,   -1,{0xffff,0x07d3}},  //  5273
{0xc0ffffffffffffff, 5276,{0xffff,0x10b9}},  //  5274
{0x00000000fffe0007, 4772,{0x03d1,0xffff}},  //  5275, depth=2
{0x00000000fffe01ff,   -1,{0x03d7,0xffff}},  //  5276
{0xffffffffc00000ff,   -1,{0xffff,0x18a9}},  //  5277, depth=1
{0xffffff00003fffff, 5280,{0xffff,0x162d}},  //  5278
{0x80000000007fffff, 4775,{0xffff,0x1057}},  //  5279, depth=2
{0x00000000ffffffbf,   -1,{0x065e,0xffff}},  //  5280
{0x00000000ffffffc0,   -1,{0x0699,0x1e99}},  //  5281
{0x007f800000000000,   -1,{0xffff,0x1447}},  //  5282, depth=1
{0x00000000fffe000f,   -1,{0x03d2,0xffff}},  //  5283, depth=1
{0x00001fffffffffc0,   -1,{0xffff,0x1ea6}},  //  5284, depth=1
{0xffffffffff80003f,   -1,{0xffff,0x1a6e}},  //  5285, depth=1
{0x000ffff0000ffff0,   -1,{0xffff,0x070f}},  //  5286, depth=1
{0xc0ffffffc0ffffff, 4776,{0xffff,0x0099}},  //  5287, depth=2
{0x00000000ffffffc1,   -1,{0x069a,0xffff}},  //  5288
{0xfffff000001fffff,   -1,{0xffff,0x1528}},  //  5289, depth=1
{0x0001ff0000000000,   -1,{0xffff,0x1608}},  //  5290
{0xffffffffffe0ffff,   -1,{0xffff,0x1afa}},  //  5291, depth=1
{0x00000000001fc000,   -1,{0x0486,0x1c86}},  //  5292, depth=1
{0xff800fffffffffff,   -1,{0xffff,0x1274}},  //  5293
{0x01ffffffffffe000,   -1,{0xffff,0x1ceb}},  //  5294, depth=1
{0xffe7ffe7ffe7ffe7,   -1,{0xffff,0x02ed}},  //  5295, depth=1
{0xfffffffefffffffe, 4779,{0xffff,0x07de}},  //  5296, depth=2
{0x000000008fff8fff, 4780,{0x006c,0xffff}},  //  5297, depth=3
{0xffffffff00000000,   -1,{0xffff,0x181f}},  //  5298, depth=1
{0x00000000fffe001f, 4796,{0x03d3,0xffff}},  //  5299, depth=2
{0x0000007fffe00000,   -1,{0xffff,0x1ad1}},  //  5300
{0xffffffe0000003ff, 4798,{0xffff,0x16e4}},  //  5301, depth=2
{0x0000000018181818,   -1,{0x0171,0xffff}},  //  5302, depth=1
{0xfffc01ffffffffff, 5311,{0xffff,0x13b6}},  //  5303
{0x1ffe00001ffe0000, 4800,{0xffff,0x03cb}},  //  5304, depth=2
{0xffffffff00000007,   -1,{0xffff,0x1822}},  //  5305, depth=1
{0x0000000000fe00fe,   -1,{0x03e6,0xffff}},  //  5306, depth=1
{0x00000000e0ffffff,   -1,{0x00da,0xffff}},  //  5307, depth=1
{0x000000ffc0000000,   -1,{0xffff,0x1889}},  //  5308, depth=1
{0xff3fffffffffffff,   -1,{0xffff,0x123d}},  //  5309, depth=1
{0x03ffffe003ffffe0,   -1,{0xffff,0x06d4}},  //  5310, depth=1
{0x00000000ffffffc7,   -1,{0x069c,0xffff}},  //  5311
{0xfffffc000fffffff,   -1,{0xffff,0x15b1}},  //  5312
{0xffffffff0000000f,   -1,{0xffff,0x1823}},  //  5313, depth=1
{0xfffffffe0000007f,   -1,{0xffff,0x17e5}},  //  5314
{0x87ffffff87ffffff, 4806,{0xffff,0x005b}},  //  5315, depth=2
{0x7fffffffffffff00, 4810,{0xffff,0x1e36}},  //  5316, depth=2
{0xfe0007fffe0007ff, 4813,{0xffff,0x01d1}},  //  5317, depth=2
{0xffffdfffffffffff,   -1,{0xffff,0x14be}},  //  5318
{0xffffe00000000000,   -1,{0xffff,0x14d2}},  //  5319
{0xffffe00000000001,   -1,{0xffff,0x14d3}},  //  5320
{0x00000000f81ff81f, 4819,{0x0169,0xffff}},  //  5321, depth=2
{0x000000000003f800,   -1,{0x0546,0x1d46}},  //  5322, depth=1
{0xffffffc000007fff,   -1,{0xffff,0x16a8}},  //  5323, depth=1
{0x0000001ff8000000,   -1,{0xffff,0x1949}},  //  5324, depth=1
{0xffffffffff81ffff,   -1,{0xffff,0x1a79}},  //  5325
{0x00000003ff000000,   -1,{0xffff,0x1a09}},  //  5326, depth=1
{0x00000000f80003ff,   -1,{0x014e,0xffff}},  //  5327, depth=1
{0xfffffffffe7fffff,   -1,{0xffff,0x19fd}},  //  5328
{0xffffffff0000001f,   -1,{0xffff,0x1824}},  //  5329, depth=1
{0x000001fffff00000,   -1,{0xffff,0x1b14}},  //  5330
{0x00000000fffe003f, 4820,{0x03d4,0xffff}},  //  5331, depth=2
{0x00001ffffffffff0,   -1,{0xffff,0x1f28}},  //  5332, depth=1
{0xe00fe00fe00fe00f, 4824,{0xffff,0x00e6}},  //  5333, depth=3
{0x1000000010000000, 4831,{0xffff,0x0100}},  //  5334, depth=2
{0xfffff81ffffff81f, 4838,{0xffff,0x0559}},  //  5335, depth=3
{0xffffffff000001ff,   -1,{0xffff,0x1828}},  //  5336
{0x00000000fffffe03,   -1,{0x05d8,0xffff}},  //  5337, depth=1
{0x0000000001800000, 4843,{0x0241,0x1a41}},  //  5338, depth=2
{0xfffff0000fffffff,   -1,{0xffff,0x152f}},  //  5339, depth=1
{0x00001ffffffffff8,   -1,{0xffff,0x1f69}},  //  5340, depth=1
{0x00000000fffffe07,   -1,{0x05d9,0xffff}},  //  5341, depth=1
{0x000000000fffc000,   -1,{0x048d,0x1c8d}},  //  5342, depth=1
{0xffffff0000ffffff,   -1,{0xffff,0x162f}},  //  5343, depth=1
{0x00000000007e0000, 4844,{0x03c5,0x1bc5}},  //  5344, depth=2
{0x007ffc0000000000,   -1,{0xffff,0x158c}},  //  5345
{0x00001ffffffffffe,   -1,{0xffff,0x1feb}},  //  5346, depth=1
{0x00001fffffffffff,   -1,{0xffff,0x102c}},  //  5347, depth=1
{0x0000200000000000,   -1,{0xffff,0x14c0}},  //  5348, depth=1
{0x00000000fffffe0f, 4845,{0x05da,0xffff}},  //  5349, depth=2
{0x000ff800000ff800, 4846,{0xffff,0x0548}},  //  5350, depth=2
{0xffff0000007fffff,   -1,{0xffff,0x1426}},  //  5351, depth=1
{0x000003fff0000000,   -1,{0xffff,0x190d}},  //  5352, depth=1
{0x00000000c0ffffff, 4849,{0x0099,0xffff}},  //  5353, depth=2
{0x003ffe00003ffe00, 4852,{0xffff,0x05cc}},  //  5354, depth=2
{0xf3f3f3f3f3f3f3f3, 4854,{0xffff,0x0135}},  //  5355, depth=2
{0xfffffffc00ffffff,   -1,{0xffff,0x17b5}},  //  5356
{0xffc007ffffffffff,   -1,{0xffff,0x12b4}},  //  5357, depth=1
{0x007ff00000000000,   -1,{0xffff,0x150a}},  //  5358, depth=1
{0xffe000ffffe000ff, 4856,{0xffff,0x02d2}},  //  5359, depth=2
{0x0038003800380038,   -1,{0xffff,0x0362}},  //  5360, depth=1
{0xfff803fffff803ff, 4858,{0xffff,0x0356}},  //  5361, depth=2
{0x1818181818181818, 4862,{0xffff,0x0171}},  //  5362, depth=2
{0x00000000fc00ffff,   -1,{0x0195,0xffff}},  //  5363, depth=1
{0x0002000000020000, 4870,{0xffff,0x03c0}},  //  5364, depth=2
{0x00000000fffffe1f, 4871,{0x05db,0xffff}},  //  5365, depth=2
{0x1fffe00000000000,   -1,{0xffff,0x14cf}},  //  5366, depth=1
{0xffe00001ffe00001, 4872,{0xffff,0x02cb}},  //  5367, depth=2
{0x00ffc00000000000,   -1,{0xffff,0x1489}},  //  5368, depth=1
{0xf7fffffff7ffffff, 4876,{0xffff,0x011e}},  //  5369, depth=2
{0x0001ffffff000000,   -1,{0xffff,0x1a18}},  //  5370, depth=1
{0xfffffffffffc07ff,   -1,{0xffff,0x1bb8}},  //  5371
{0x0000ffffffffff80,   -1,{0xffff,0x1e68}},  //  5372, depth=1
{0xfffffffe007fffff,   -1,{0xffff,0x17f5}},  //  5373, depth=1
{0x0000000000ffff00,   -1,{0x060f,0x1e0f}},  //  5374, depth=1
{0xfffffffffffe03ff,   -1,{0xffff,0x1bf8}},  //  5375, depth=1
{0x01fff80001fff800,   -1,{0xffff,0x054d}},  //  5376
{0xffffe0000000001f,   -1,{0xffff,0x14d7}},  //  5377
{0x00000000c000c000,   -1,{0x00a1,0xffff}},  //  5378, depth=1
{0x0000038000000000,   -1,{0xffff,0x1642}},  //  5379
{0x0ffffffff0000000,   -1,{0xffff,0x191f}},  //  5380, depth=1
{0xfe00000ffe00000f,   -1,{0xffff,0x01ca}},  //  5381, depth=1
{0x00000000fffffff3,   -1,{0x071d,0xffff}},  //  5382
{0x00000000fffffff9, 5385,{0x075d,0xffff}},  //  5383
{0x0000000ffffc0000,   -1,{0xffff,0x1b91}},  //  5384, depth=1
{0xe001ffffe001ffff,   -1,{0xffff,0x00d3}},  //  5385
{0x000001fffe000000,   -1,{0xffff,0x19cf}},  //  5386, depth=1
{0x800000007fffffff,   -1,{0xffff,0x105f}},  //  5387, depth=1
{0x8000000080000000, 4877,{0xffff,0x0040}},  //  5388, depth=2
{0xff00000000000fff,   -1,{0xffff,0x1213}},  //  5389, depth=1
{0x0000001ffffffc00,   -1,{0xffff,0x1d9a}},  //  5390
{0x0000000100000000,   -1,{0xffff,0x1800}},  //  5391
{0x000000000003e000,   -1,{0x04c4,0x1cc4}},  //  5392, depth=1
{0xfff80007fff80007,   -1,{0xffff,0x034f}},  //  5393, depth=1
{0x00001fffe0000000,   -1,{0xffff,0x18cf}},  //  5394, depth=1
{0x00000000fffe007f, 4880,{0x03d5,0xffff}},  //  5395, depth=2
{0x00c0000000c00000,   -1,{0xffff,0x0281}},  //  5396, depth=1
{0x00000000fffffe3f,   -1,{0x05dc,0xffff}},  //  5397, depth=1
{0xfc000000001fffff,   -1,{0xffff,0x119a}},  //  5398
{0x0fffff8000000000,   -1,{0xffff,0x1654}},  //  5399
{0x000000ff80000000, 4883,{0xffff,0x1848}},  //  5400, depth=2
{0x7fffffffff800000,   -1,{0xffff,0x1a67}},  //  5401
{0x00000fc000000fc0, 4884,{0xffff,0x0685}},  //  5402, depth=2
{0x00000000803fffff,   -1,{0x0056,0xffff}},  //  5403
{0x00000000007fffe0,   -1,{0x06d1,0x1ed1}},  //  5404
{0xffffff800fffffff,   -1,{0xffff,0x1674}},  //  5405, depth=1
{0x3fe03fe03fe03fe0, 4886,{0xffff,0x02e8}},  //  5406, depth=2
{0x001e0000001e0000,   -1,{0xffff,0x03c3}},  //  5407
{0x000000007ff87ff8,   -1,{0x036b,0xffff}},  //  5408, depth=1
{0x001fffff001fffff,   -1,{0xffff,0x0014}},  //  5409, depth=1
{0x00000000007ffe00,   -1,{0x05cd,0x1dcd}},  //  5410, depth=1
{0x00ffffffffffc000,   -1,{0xffff,0x1ca9}},  //  5411
{0xfffe0000fffe0000, 4887,{0xffff,0x03ce}},  //  5412, depth=2
{0xc0000007ffffffff, 4889,{0xffff,0x10a4}},  //  5413, depth=2
{0x00000001ffffffc0,   -1,{0xffff,0x1e9a}},  //  5414, depth=1
{0xe7ffe7ffe7ffe7ff,   -1,{0xffff,0x00ed}},  //  5415, depth=1
{0xfffc0001fffc0001,   -1,{0xffff,0x038e}},  //  5416
{0xffdfffdfffdfffdf,   -1,{0xffff,0x02ae}},  //  5417, depth=1
{0xfc0000000fffffff,   -1,{0xffff,0x11a1}},  //  5418
{0xffff0000000000ff,   -1,{0xffff,0x1417}},  //  5419
{0x000000000c000000, 4892,{0x0181,0x1981}},  //  5420, depth=2
{0xfc00007fffffffff,   -1,{0xffff,0x11ac}},  //  5421, depth=1
{0x000000007fffff80,   -1,{0x0657,0x1e57}},  //  5422
{0x00000000fe0fffff, 4896,{0x01da,0xffff}},  //  5423, depth=2
{0x007fff8000000000,   -1,{0xffff,0x164f}},  //  5424, depth=1
{0x80000000003fffff, 4903,{0xffff,0x1056}},  //  5425, depth=2
{0x0000007fffff0000,   -1,{0xffff,0x1c16}},  //  5426, depth=1
{0xfc0007fffc0007ff, 4908,{0xffff,0x0190}},  //  5427, depth=2
{0xe0000001ffffffff,   -1,{0xffff,0x10e3}},  //  5428
{0xf0007fffffffffff,   -1,{0xffff,0x1132}},  //  5429, depth=1
{0x0fffffffffe00000,   -1,{0xffff,0x1ae6}},  //  5430, depth=1
{0x001ffffffff80000, 5434,{0xffff,0x1b61}},  //  5431
{0x07fffff007fffff0,   -1,{0xffff,0x0716}},  //  5432, depth=1
{0xffffffe00003ffff,   -1,{0xffff,0x16ec}},  //  5433, depth=1
{0x00000003fe000000,   -1,{0xffff,0x19c8}},  //  5434
{0xfffffffff80007ff,   -1,{0xffff,0x196f}},  //  5435, depth=1
{0x0000ffffffffffc0,   -1,{0xffff,0x1ea9}},  //  5436, depth=1
{0x000fffc000000000,   -1,{0xffff,0x168d}},  //  5437
{0x000000003ff03ff0, 4909,{0x0329,0xffff}},  //  5438, depth=2
{0xf8000003ffffffff,   -1,{0xffff,0x1166}},  //  5439, depth=1
{0x00001fffc0000000,   -1,{0xffff,0x188e}},  //  5440, depth=1
{0xffffc07fffffc07f,   -1,{0xffff,0x0498}},  //  5441, depth=1
{0xf0000000f0000000, 4910,{0xffff,0x0103}},  //  5442, depth=2
{0x9999999999999999,   -1,{0xffff,0x0079}},  //  5443, depth=1
{0x0e0000000e000000, 4916,{0xffff,0x01c2}},  //  5444, depth=4
{0x0000000080ffffff,   -1,{0x0058,0xffff}},  //  5445, depth=1
{0x00000001ffffffe0,   -1,{0xffff,0x1edb}},  //  5446, depth=1
{0x3ffffffff0000000, 5448,{0xffff,0x1921}},  //  5447
{0x000007fc00000000,   -1,{0xffff,0x1788}},  //  5448
{0xfffc000003ffffff, 4929,{0xffff,0x13a7}},  //  5449, depth=2
{0x00000000007ffffe,   -1,{0x07d5,0x1fd5}},  //  5450
{0xfff80000000003ff,   -1,{0xffff,0x1356}},  //  5451, depth=1
{0x000000007ffe0000,   -1,{0x03cd,0x1bcd}},  //  5452, depth=1
{0xfffffe03ffffffff,   -1,{0xffff,0x15f8}},  //  5453, depth=1
{0x0180000000000000,   -1,{0xffff,0x1241}},  //  5454, depth=1
{0xe007e007e007e007,   -1,{0xffff,0x00e5}},  //  5455, depth=1
{0xfffffffe000000ff, 5457,{0xffff,0x17e6}},  //  5456
{0x00000000fdfdfdfd,   -1,{0x01b6,0xffff}},  //  5457
{0x1ffffffc1ffffffc,   -1,{0xffff,0x079a}},  //  5458
{0x00000000f001f001, 4930,{0x0124,0xffff}},  //  5459, depth=3
{0xffffe0000000007f, 5463,{0xffff,0x14d9}},  //  5460
{0x00000000fffffe7f,   -1,{0x05dd,0xffff}},  //  5461, depth=1
{0x00000001fffffff0,   -1,{0xffff,0x1f1c}},  //  5462, depth=1
{0x00000000fc003fff, 5465,{0x0193,0xffff}},  //  5463
{0x000001fe000001fe,   -1,{0xffff,0x07c7}},  //  5464, depth=1
{0xe3e3e3e3e3e3e3e3,   -1,{0xffff,0x00f4}},  //  5465
{0xffffc001ffffffff,   -1,{0xffff,0x14b2}},  //  5466
{0xfff0000007ffffff,   -1,{0xffff,0x1326}},  //  5467, depth=1
{0x0000000003f00000, 4933,{0x0305,0x1b05}},  //  5468, depth=2
{0xfffffeffffffffff, 5473,{0xffff,0x15fe}},  //  5469
{0x00000000007e007e, 4935,{0x03e5,0xffff}},  //  5470, depth=3
{0xffffff87ffffff87, 4938,{0xffff,0x065b}},  //  5471, depth=2
{0x0000010000000100, 4940,{0xffff,0x0600}},  //  5472, depth=2
{0x00000000ffbfffbf,   -1,{0x026e,0xffff}},  //  5473
{0x00000001fffffffc,   -1,{0xffff,0x1f9e}},  //  5474, depth=1
{0xe000000001ffffff,   -1,{0xffff,0x10db}},  //  5475, depth=1
{0x0003fffffffff000, 4941,{0xffff,0x1d25}},  //  5476, depth=2
{0x00000001ffffffff, 4942,{0xffff,0x1020}},  //  5477, depth=2
{0x00fc00fc00fc00fc, 4943,{0xffff,0x03a5}},  //  5478, depth=2
{0x800000003fffffff,   -1,{0xffff,0x105e}},  //  5479, depth=1
{0x0000000200000002, 4947,{0xffff,0x07c0}},  //  5480, depth=2
{0xc00003ffc00003ff, 4950,{0xffff,0x008b}},  //  5481, depth=2
{0x0030003000300030,   -1,{0xffff,0x0321}},  //  5482, depth=1
{0xf801ffffffffffff,   -1,{0xffff,0x1175}},  //  5483, depth=1
{0x0000fffffffffff0,   -1,{0xffff,0x1f2b}},  //  5484, depth=1
{0xffffff0000000000,   -1,{0xffff,0x1617}},  //  5485
{0xffffff0000000001,   -1,{0xffff,0x1618}},  //  5486
{0xfc000fffffffffff, 4951,{0xffff,0x11b1}},  //  5487, depth=2
{0x003ffffc00000000,   -1,{0xffff,0x1793}},  //  5488, depth=1
{0xffffffffffff3fff,   -1,{0xffff,0x1c3d}},  //  5489, depth=1
{0x00000000003e0000,   -1,{0x03c4,0x1bc4}},  //  5490, depth=1
{0x0000000080038003, 4953,{0x0062,0xffff}},  //  5491, depth=3
{0x00000fffffff0000, 4956,{0xffff,0x1c1b}},  //  5492, depth=2
{0xffff03ffffff03ff, 4957,{0xffff,0x0419}},  //  5493, depth=2
{0x00001fffffc00000,   -1,{0xffff,0x1a96}},  //  5494, depth=1
{0x00007fffffe00000,   -1,{0xffff,0x1ad9}},  //  5495
{0x07fff80007fff800, 4958,{0xffff,0x054f}},  //  5496, depth=2
{0xefefefefefefefef, 4962,{0xffff,0x00f6}},  //  5497, depth=2
{0x7e0000007e000000, 4963,{0xffff,0x01c5}},  //  5498, depth=3
{0x0000ffffffffffff,   -1,{0xffff,0x102f}},  //  5499, depth=1
{0x0001000000000000,   -1,{0xffff,0x1400}},  //  5500, depth=1
{0xc003ffffffffffff,   -1,{0xffff,0x10b3}},  //  5501, depth=1
{0x0000000000ffff80,   -1,{0x0650,0x1e50}},  //  5502, depth=1
{0xffffe000007fffff,   -1,{0xffff,0x14e9}},  //  5503, depth=1
{0x00001fffff000000,   -1,{0xffff,0x1a14}},  //  5504
{0xffffffc00007ffff, 5506,{0xffff,0x16ac}},  //  5505
{0x8001800180018001,   -1,{0xffff,0x0061}},  //  5506
{0xffc07fffffc07fff,   -1,{0xffff,0x0298}},  //  5507, depth=1
{0x000000ffff800000, 4965,{0xffff,0x1a50}},  //  5508, depth=2
{0x1ffffe001ffffe00,   -1,{0xffff,0x05d3}},  //  5509
{0x7fffffff00000000,   -1,{0xffff,0x181e}},  //  5510
{0x00000000fff81fff, 4973,{0x0359,0xffff}},  //  5511, depth=2
{0xfffffffff0000fff,   -1,{0xffff,0x192f}},  //  5512
{0xc0000007c0000007, 4974,{0xffff,0x0084}},  //  5513, depth=2
{0x000000000000f000, 4980,{0x0503,0x1d03}},  //  5514, depth=2
{0xffefffffffefffff, 4981,{0xffff,0x02de}},  //  5515, depth=2
{0x007fc000007fc000, 4985,{0xffff,0x0488}},  //  5516, depth=2
{0xe01fffffe01fffff, 4986,{0xffff,0x00d7}},  //  5517, depth=2
{0x000000007ffffe00,   -1,{0x05d5,0x1dd5}},  //  5518, depth=1
{0xfffffffe003fffff,   -1,{0xffff,0x17f4}},  //  5519, depth=1
{0x000fffffffff8000,   -1,{0xffff,0x1c64}},  //  5520, depth=1
{0xffffffffff000007,   -1,{0xffff,0x1a2a}},  //  5521, depth=1
{0x0001fffe00000000,   -1,{0xffff,0x17cf}},  //  5522, depth=1
{0x00000000fffe00ff,   -1,{0x03d6,0xffff}},  //  5523, depth=1
{0x00000001e0000000,   -1,{0xffff,0x18c3}},  //  5524, depth=1
{0xffff0007ffff0007, 4990,{0xffff,0x0412}},  //  5525, depth=3
{0xfffffffffe3fffff,   -1,{0xffff,0x19fc}},  //  5526
{0xfffc000000003fff,   -1,{0xffff,0x139b}},  //  5527, depth=1
{0x00000007ffffc000,   -1,{0xffff,0x1c94}},  //  5528, depth=1
{0xffe0007fffffffff, 4997,{0xffff,0x12f1}},  //  5529, depth=2
{0xffff80ffffff80ff,   -1,{0xffff,0x0458}},  //  5530
{0x3f3f3f3f3f3f3f3f, 5000,{0xffff,0x0035}},  //  5531, depth=2
{0x00001fff80000000,   -1,{0xffff,0x184d}},  //  5532, depth=1
{0xffffffffc00001ff,   -1,{0xffff,0x18aa}},  //  5533, depth=1
{0x0001fffffe000000,   -1,{0xffff,0x19d7}},  //  5534
{0x83ffffff83ffffff,   -1,{0xffff,0x005a}},  //  5535, depth=1
{0xffe00000001fffff, 5538,{0xffff,0x12df}},  //  5536
{0xffc00000000fffff, 5003,{0xffff,0x129d}},  //  5537, depth=2
{0x00000000e000003f,   -1,{0x00c8,0xffff}},  //  5538
{0xc00000000000ffff,   -1,{0xffff,0x1091}},  //  5539, depth=1
{0x0000003fffffe000, 5011,{0xffff,0x1cd8}},  //  5540, depth=2
{0xffffff000000001f,   -1,{0xffff,0x161c}},  //  5541
{0x3fffffffffe00000,   -1,{0xffff,0x1ae8}},  //  5542
{0x01ffffc000000000, 5549,{0xffff,0x1692}},  //  5543
{0x000000003ffe0000,   -1,{0x03cc,0x1bcc}},  //  5544, depth=1
{0xffffffffff00001f,   -1,{0xffff,0x1a2c}},  //  5545, depth=1
{0x0000ffffe0000000,   -1,{0xffff,0x18d2}},  //  5546, depth=1
{0xfe3ffe3ffe3ffe3f,   -1,{0xffff,0x01ec}},  //  5547, depth=1
{0x01fff00001fff000, 5015,{0xffff,0x050c}},  //  5548, depth=2
{0x0000000000c000c0,   -1,{0x02a1,0xffff}},  //  5549
{0xffff000fffffffff, 5555,{0xffff,0x1433}},  //  5550
{0xffff00003fffffff,   -1,{0xffff,0x142d}},  //  5551, depth=1
{0x00000000003e003e, 5016,{0x03e4,0xffff}},  //  5552, depth=2
{0xffffffff000000ff, 5017,{0xffff,0x1827}},  //  5553, depth=2
{0x0c0000000c000000,   -1,{0xffff,0x0181}},  //  5554, depth=1
{0xfff8000007ffffff,   -1,{0xffff,0x1367}},  //  5555
{0x00000000003ffe00, 5020,{0x05cc,0x1dcc}},  //  5556, depth=2
{0x00fffffffc000000,   -1,{0xffff,0x199d}},  //  5557
{0x0000000002000200,   -1,{0x01e0,0xffff}},  //  5558, depth=1
{0xffffff00ffffffff,   -1,{0xffff,0x1637}},  //  5559, depth=1
{0x0007fffffffc0000,   -1,{0xffff,0x1ba0}},  //  5560, depth=1
{0xffffc007ffffc007,   -1,{0xffff,0x0494}},  //  5561
{0xffffffffffe00fff, 5565,{0xffff,0x1af6}},  //  5562
{0xfc007fffffffffff,   -1,{0xffff,0x11b4}},  //  5563, depth=1
{0x7ffffffffffffff8,   -1,{0xffff,0x1f7b}},  //  5564, depth=1
{0xffffff000000003f,   -1,{0xffff,0x161d}},  //  5565
{0x2000200020002000, 5024,{0xffff,0x00e0}},  //  5566, depth=2
{0x000000003fffff80,   -1,{0x0656,0x1e56}},  //  5567
{0x00007fffffff0000, 5026,{0xffff,0x1c1e}},  //  5568, depth=2
{0x807f807f807f807f, 5031,{0xffff,0x0067}},  //  5569, depth=2
{0x003c0000003c0000, 5032,{0xffff,0x0383}},  //  5570, depth=3
{0x00000000c7ffc7ff, 5036,{0x00ac,0xffff}},  //  5571, depth=2
{0x8000000000000000,   -1,{0xffff,0x1040}},  //  5572, depth=1
{0xffc3ffffffc3ffff, 5041,{0xffff,0x029b}},  //  5573, depth=3
{0x7f807f807f807f80, 5048,{0xffff,0x0267}},  //  5574, depth=2
{0x8000000000000003, 5054,{0xffff,0x1042}},  //  5575, depth=2
{0x00000000ffc00003,   -1,{0x028b,0xffff}},  //  5576
{0xdfffdfffdfffdfff, 5061,{0xffff,0x00ae}},  //  5577, depth=2
{0x000000007fffffe0,   -1,{0x06d9,0x1ed9}},  //  5578
{0x8000000000000007,   -1,{0xffff,0x1043}},  //  5579, depth=1
{0x03ff800000000000,   -1,{0xffff,0x144a}},  //  5580, depth=1
{0x00000000fff807ff,   -1,{0x0357,0xffff}},  //  5581, depth=1
{0xffffffffe00003ff,   -1,{0xffff,0x18ec}},  //  5582
{0x00000000fc0ffc0f, 5062,{0x01a9,0xffff}},  //  5583, depth=2
{0x000000ff00000000,   -1,{0xffff,0x1807}},  //  5584, depth=1
{0x00000000f8007fff,   -1,{0x0153,0xffff}},  //  5585
{0x003ffe0000000000,   -1,{0xffff,0x15cc}},  //  5586
{0x800000000000000f, 5063,{0xffff,0x1044}},  //  5587, depth=2
{0xfffffffffff8ffff,   -1,{0xffff,0x1b7c}},  //  5588
{0xf3fffffff3ffffff, 5065,{0xffff,0x011d}},  //  5589, depth=2
{0x000000001ffe0000, 5082,{0x03cb,0x1bcb}},  //  5590, depth=2
{0x00000000ffffff01, 5087,{0x0618,0xffff}},  //  5591, depth=2
{0x0000ffffc0000000,   -1,{0xffff,0x1891}},  //  5592, depth=1
{0x00000000ffffff03,   -1,{0x0619,0xffff}},  //  5593, depth=1
{0xfffff83ffffff83f,   -1,{0xffff,0x055a}},  //  5594
{0xfe000ffffe000fff, 5093,{0xffff,0x01d2}},  //  5595, depth=2
{0x01c001c001c001c0,   -1,{0xffff,0x02a2}},  //  5596, depth=1
{0x00000000ffffff07, 5095,{0x061a,0xffff}},  //  5597, depth=2
{0x0000000000ffffe0,   -1,{0x06d2,0x1ed2}},  //  5598, depth=1
{0x00000000e07fffff, 5096,{0x00d9,0xffff}},  //  5599, depth=2
{0x00fffc0000fffc00,   -1,{0xffff,0x058d}},  //  5600
{0x0000000ffffffe00,   -1,{0xffff,0x1dda}},  //  5601
{0xfffffffcfffffffc,   -1,{0xffff,0x079d}},  //  5602
{0x800000000000001f, 5100,{0xffff,0x1045}},  //  5603, depth=2
{0x3fffffffffff0000,   -1,{0xffff,0x1c2d}},  //  5604, depth=1
{0x00000000ffffff0f,   -1,{0x061b,0xffff}},  //  5605, depth=1
{0x0000000010101010, 5101,{0x0130,0xffff}},  //  5606, depth=3
{0xfffffffcffffffff, 5609,{0xffff,0x17bd}},  //  5607
{0x7c0000007c000000,   -1,{0xffff,0x0184}},  //  5608, depth=1
{0x800007ff800007ff,   -1,{0xffff,0x004b}},  //  5609
{0x000000003ffffe00,   -1,{0x05d4,0x1dd4}},  //  5610, depth=1
{0xffffe0007fffffff,   -1,{0xffff,0x14f1}},  //  5611, depth=1
{0xc0c0c0c0c0c0c0c0, 5104,{0xffff,0x00b1}},  //  5612, depth=2
{0xff000003ffffffff,   -1,{0xffff,0x1229}},  //  5613
{0x0000000000fffff0, 5105,{0x0713,0x1f13}},  //  5614, depth=2
{0xfffffff800003fff,   -1,{0xffff,0x176a}},  //  5615, depth=1
{0x0003ffffffffc000,   -1,{0xffff,0x1ca3}},  //  5616, depth=1
{0xf003ffffffffffff,   -1,{0xffff,0x1135}},  //  5617
{0x0000fff80000fff8, 5111,{0xffff,0x074c}},  //  5618, depth=3
{0xfffffffe1fffffff,   -1,{0xffff,0x17fb}},  //  5619, depth=1
{0xffffffffc00003ff, 5631,{0xffff,0x18ab}},  //  5620
{0x00000000ffffff1f, 5116,{0x061c,0xffff}},  //  5621, depth=2
{0x0000000000fffff8,   -1,{0x0754,0x1f54}},  //  5622, depth=1
{0xfff0000000007fff,   -1,{0xffff,0x131a}},  //  5623, depth=1
{0x00000001ffc00000,   -1,{0xffff,0x1a8a}},  //  5624, depth=1
{0xffffffff800001ff,   -1,{0xffff,0x1869}},  //  5625, depth=1
{0x1fe000001fe00000, 5120,{0xffff,0x02c7}},  //  5626, depth=2
{0xff803fffff803fff, 5126,{0xffff,0x0256}},  //  5627, depth=2
{0x0010000000100000, 5127,{0xffff,0x0300}},  //  5628, depth=2
{0x0000000000ffffff, 5133,{0x0017,0x1017}},  //  5629, depth=2
{0x0000000001000000, 5134,{0x0200,0x1a00}},  //  5630, depth=2
{0xf8001fffffffffff,   -1,{0xffff,0x1171}},  //  5631
{0x00fffe0000000000,   -1,{0xffff,0x15ce}},  //  5632, depth=1
{0x00000000003ffff0,   -1,{0x0711,0x1f11}},  //  5633
{0xfffc3fffffffffff,   -1,{0xffff,0x13bb}},  //  5634
{0x800000000000003f, 5135,{0xffff,0x1046}},  //  5635, depth=2
{0x003f8000003f8000,   -1,{0xffff,0x0446}},  //  5636, depth=1
{0x007fffffffffe000,   -1,{0xffff,0x1ce9}},  //  5637
{0xffffff000000007f,   -1,{0xffff,0x161e}},  //  5638
{0xffff800003ffffff,   -1,{0xffff,0x146a}},  //  5639
{0x0000000001c001c0, 5136,{0x02a2,0xffff}},  //  5640, depth=2
{0xffffffffff00007f,   -1,{0xffff,0x1a2e}},  //  5641, depth=1
{0x3ffc000000000000,   -1,{0xffff,0x138b}},  //  5642, depth=1
{0xfffeffffffffffff,   -1,{0xffff,0x13fe}},  //  5643, depth=1
{0xffff000000000000,   -1,{0xffff,0x140f}},  //  5644, depth=1
{0x81ffffff81ffffff, 5140,{0xffff,0x0059}},  //  5645, depth=4
{0x1010101010101010, 5153,{0xffff,0x0130}},  //  5646, depth=2
{0xf80007fff80007ff, 5156,{0xffff,0x014f}},  //  5647, depth=2
{0x000000003fffffc0,   -1,{0x0697,0x1e97}},  //  5648
{0xffffe000003fffff,   -1,{0xffff,0x14e8}},  //  5649, depth=1
{0x0000fc000000fc00, 5157,{0xffff,0x0585}},  //  5650, depth=2
{0xffff000000000007, 5158,{0xffff,0x1412}},  //  5651, depth=2
{0x00000000f800f800, 5163,{0x0164,0xffff}},  //  5652, depth=3
{0x00000000ffffff3f, 5172,{0x061d,0xffff}},  //  5653, depth=2
{0x000000000000c000,   -1,{0x0481,0x1c81}},  //  5654, depth=1
{0xffc00003ffffffff,   -1,{0xffff,0x12ab}},  //  5655, depth=1
{0x000000001ffffe00, 5178,{0x05d3,0x1dd3}},  //  5656, depth=2
{0x000ffffffffc0000, 5658,{0xffff,0x1ba1}},  //  5657
{0x00000001ff000000,   -1,{0xffff,0x1a08}},  //  5658
{0xffff00000000000f,   -1,{0xffff,0x1413}},  //  5659, depth=1
{0x07fe000000000000,   -1,{0xffff,0x13c9}},  //  5660, depth=1
{0xffcfffcfffcfffcf,   -1,{0xffff,0x02ad}},  //  5661, depth=1
{0x3ffffc003ffffc00, 5183,{0xffff,0x0593}},  //  5662, depth=2
{0xfffffffdfffffffd, 5187,{0xffff,0x079e}},  //  5663, depth=2
{0x7fffffffc0000000,   -1,{0xffff,0x18a0}},  //  5664, depth=1
{0xff03ff03ff03ff03, 5193,{0xffff,0x0229}},  //  5665, depth=2
{0x00000000007fff00, 5195,{0x060e,0x1e0e}},  //  5666, depth=2
{0xfffc000000000fff, 5197,{0xffff,0x1399}},  //  5667, depth=2
{0x1ffffffffe000000,   -1,{0xffff,0x19e3}},  //  5668, depth=1
{0xfffffffe00000003,   -1,{0xffff,0x17e0}},  //  5669, depth=1
{0x00000000fc000fff,   -1,{0x0191,0xffff}},  //  5670
{0xfffffefffffffeff, 5198,{0xffff,0x05de}},  //  5671, depth=2
{0x0000000000038000, 5199,{0x0442,0x1c42}},  //  5672, depth=2
{0xffffffc0000007ff, 5204,{0xffff,0x16a4}},  //  5673, depth=2
{0x0007ffe000000000,   -1,{0xffff,0x16cd}},  //  5674
{0xffff00000000001f, 5207,{0xffff,0x1414}},  //  5675, depth=2
{0x000ffffff8000000,   -1,{0xffff,0x1958}},  //  5676, depth=1
{0xfffe0000007fffff, 5678,{0xffff,0x13e5}},  //  5677
{0xdfdfdfdfdfdfdfdf,   -1,{0xffff,0x00b6}},  //  5678
{0xfffffe01fffffe01,   -1,{0xffff,0x05d7}},  //  5679, depth=1
{0x0ffffffe0ffffffe,   -1,{0xffff,0x07da}},  //  5680
{0xfffffffe0000000f,   -1,{0xffff,0x17e2}},  //  5681, depth=1
{0x000000000e0e0e0e,   -1,{0x01f2,0xffff}},  //  5682, depth=1
{0x00003ffffff00000,   -1,{0xffff,0x1b19}},  //  5683
{0x0000ffff80000000,   -1,{0xffff,0x1850}},  //  5684, depth=1
{0x8007ffffffffffff,   -1,{0xffff,0x1073}},  //  5685
{0x0fffff000fffff00,   -1,{0xffff,0x0613}},  //  5686
{0xfc000007ffffffff,   -1,{0xffff,0x11a8}},  //  5687
{0x1ff81ff81ff81ff8,   -1,{0xffff,0x0369}},  //  5688, depth=1
{0xfe7fffffffffffff,   -1,{0xffff,0x11fd}},  //  5689, depth=1
{0x000001fc00000000,   -1,{0xffff,0x1786}},  //  5690, depth=1
{0xffffffff8001ffff,   -1,{0xffff,0x1871}},  //  5691, depth=1
{0x0007fffffffffc00,   -1,{0xffff,0x1da8}},  //  5692, depth=1
{0x0000ffffff000000,   -1,{0xffff,0x1a17}},  //  5693
{0x00007ff800000000, 5210,{0xffff,0x174b}},  //  5694, depth=2
{0x1ffffffffff00000,   -1,{0xffff,0x1b28}},  //  5695
{0xffffffffff01ffff, 5698,{0xffff,0x1a38}},  //  5696
{0xfffffffe0000001f,   -1,{0xffff,0x17e3}},  //  5697, depth=1
{0x00000000ff3fff3f,   -1,{0x022d,0xffff}},  //  5698
{0xf1fffffff1ffffff, 5217,{0xffff,0x011c}},  //  5699, depth=4
{0x6666666666666666,   -1,{0xffff,0x00f9}},  //  5700, depth=1
{0x0fffffff0fffffff, 5232,{0xffff,0x001b}},  //  5701, depth=2
{0x00003f8000003f80,   -1,{0xffff,0x0646}},  //  5702, depth=1
{0xffffe0003fffffff,   -1,{0xffff,0x14f0}},  //  5703, depth=1
{0x07fffffc00000000,   -1,{0xffff,0x1798}},  //  5704, depth=1
{0xfc000ffffc000fff,   -1,{0xffff,0x0191}},  //  5705, depth=1
{0xfe0003ffffffffff,   -1,{0xffff,0x11f0}},  //  5706
{0xffff00000000003f,   -1,{0xffff,0x1415}},  //  5707, depth=1
{0x0000000007fff800,   -1,{0x054f,0x1d4f}},  //  5708, depth=1
{0xfffffffffdffffff,   -1,{0xffff,0x19be}},  //  5709
{0x0000001ffffc0000,   -1,{0xffff,0x1b92}},  //  5710, depth=1
{0xf800000ff800000f,   -1,{0xffff,0x0148}},  //  5711, depth=1
{0xfffffffffe000000,   -1,{0xffff,0x19e6}},  //  5712
{0x00000000fffc03ff, 5234,{0x0397,0xffff}},  //  5713, depth=2
{0x0fff800000000000,   -1,{0xffff,0x144c}},  //  5714, depth=1
{0xffffffffffc003ff, 5724,{0xffff,0x1ab3}},  //  5715
{0x03fff80003fff800, 5235,{0xffff,0x054e}},  //  5716, depth=2
{0x00000000ffffff7f, 5239,{0x061e,0xffff}},  //  5717, depth=2
{0x00000000ffffff80, 5243,{0x0658,0x1e58}},  //  5718, depth=2
{0x00000000ffffff81, 5245,{0x0659,0xffff}},  //  5719, depth=2
{0x007ffffffff80000,   -1,{0xffff,0x1b63}},  //  5720, depth=1
{0x00000000ffffff83,   -1,{0x065a,0xffff}},  //  5721, depth=1
{0x0000000001800180, 5247,{0x0261,0xffff}},  //  5722, depth=2
{0xfe00001ffe00001f, 5250,{0xffff,0x01cb}},  //  5723, depth=2
{0xfffffffffe000001,   -1,{0xffff,0x19e7}},  //  5724
{0x00000000ffffff87,   -1,{0x065b,0xffff}},  //  5725, depth=1
{0x0020002000200020,   -1,{0xffff,0x02e0}},  //  5726, depth=1
{0xfffffffffe000003,   -1,{0xffff,0x19e8}},  //  5727
{0x1800180018001800,   -1,{0xffff,0x0161}},  //  5728, depth=1
{0xfffffffe0000003f,   -1,{0xffff,0x17e4}},  //  5729, depth=1
{0x3ffffff800000000, 5252,{0xffff,0x175a}},  //  5730, depth=2
{0x0001ffff0001ffff, 5254,{0xffff,0x0010}},  //  5731, depth=2
{0xfffffc000000ffff, 5735,{0xffff,0x15a5}},  //  5732
{0x00000000ffffff8f, 5255,{0x065c,0xffff}},  //  5733, depth=2
{0xffe00000ffe00000,   -1,{0xffff,0x02ca}},  //  5734, depth=1
{0xe00fffffe00fffff,   -1,{0xffff,0x00d6}},  //  5735
{0x000000001fffffc0,   -1,{0x0696,0x1e96}},  //  5736
{0x00000000807fffff, 5257,{0x0057,0xffff}},  //  5737, depth=3
{0x0000007ff0000000,   -1,{0xffff,0x190a}},  //  5738, depth=1
{0xfffffffffe000007,   -1,{0xffff,0x19e9}},  //  5739
{0x000000003ffffff0,   -1,{0x0719,0x1f19}},  //  5740
{0xfffff03ffffff03f, 5260,{0xffff,0x0519}},  //  5741, depth=2
{0xffffe00fffffffff, 5744,{0xffff,0x14f6}},  //  5742
{0xf800000001ffffff, 5261,{0xffff,0x115d}},  //  5743, depth=2
{0xfffe000ffffe000f,   -1,{0xffff,0x03d2}},  //  5744
{0x00000000e03fffff,   -1,{0x00d8,0xffff}},  //  5745, depth=1
{0x7ffffffe7ffffffe,   -1,{0xffff,0x07dd}},  //  5746
{0xff3fffffff3fffff,   -1,{0xffff,0x021d}},  //  5747, depth=1
{0x003fc00000000000,   -1,{0xffff,0x1487}},  //  5748, depth=1
{0x00000000ffffff9f, 5266,{0x065d,0xffff}},  //  5749, depth=2
{0x0007fff80007fff8,   -1,{0xffff,0x074f}},  //  5750, depth=1
{0xfffffffffffc1fff,   -1,{0xffff,0x1bba}},  //  5751, depth=1
{0xfff8000000007fff,   -1,{0xffff,0x135b}},  //  5752
{0xfffffffffe00001f,   -1,{0xffff,0x19eb}},  //  5753
{0x00fffffffffff000,   -1,{0xffff,0x1d2b}},  //  5754, depth=1
{0x7fffffff7fffffff, 5270,{0xffff,0x001e}},  //  5755, depth=2
{0x7fffffff80000000,   -1,{0xffff,0x185f}},  //  5756, depth=1
{0xfffffe0001ffffff,   -1,{0xffff,0x15ef}},  //  5757, depth=1
{0x000000000c0c0c0c,   -1,{0x01b1,0xffff}},  //  5758, depth=1
{0xfffffff00003ffff,   -1,{0xffff,0x172d}},  //  5759, depth=1
{0x003ffffffffff000,   -1,{0xffff,0x1d29}},  //  5760
{0x000000001fffffe0,   -1,{0x06d7,0x1ed7}},  //  5761
{0x01fffff001fffff0,   -1,{0xffff,0x0714}},  //  5762, depth=1
{0xf00000000fffffff,   -1,{0xffff,0x111f}},  //  5763, depth=1
{0xffffffff800003ff,   -1,{0xffff,0x186a}},  //  5764
{0x0007fffffffe0000, 5766,{0xffff,0x1be1}},  //  5765
{0x00000000ff800000,   -1,{0x0248,0x1a48}},  //  5766
{0xff80007fff80007f,   -1,{0xffff,0x024f}},  //  5767
{0x000000000001fc00,   -1,{0x0586,0x1d86}},  //  5768, depth=1
{0xffffffffff0000ff,   -1,{0xffff,0x1a2f}},  //  5769, depth=1
{0x00000001ff800000,   -1,{0xffff,0x1a49}},  //  5770, depth=1
{0xffff00000000007f,   -1,{0xffff,0x1416}},  //  5771, depth=1
{0x00000000ff800003,   -1,{0x024a,0xffff}},  //  5772
{0xfffe000000ffffff,   -1,{0xffff,0x13e6}},  //  5773, depth=1
{0x0800000008000000, 5272,{0xffff,0x0140}},  //  5774, depth=2
{0xff003fffffffffff,   -1,{0xffff,0x1235}},  //  5775, depth=1
{0x0000000000c00000, 5273,{0x0281,0x1a81}},  //  5776, depth=2
{0xe0001fffffffffff,   -1,{0xffff,0x10ef}},  //  5777, depth=1
{0x0000000007ffe000,   -1,{0x04cd,0x1ccd}},  //  5778, depth=1
{0xfffdfffffffdffff, 5274,{0xffff,0x039e}},  //  5779, depth=3
{0xfffffff007ffffff, 5787,{0xffff,0x1736}},  //  5780
{0xe7e7e7e7e7e7e7e7, 5278,{0xffff,0x00f5}},  //  5781, depth=3
{0x0007fc000007fc00, 5281,{0xffff,0x0588}},  //  5782, depth=2
{0xffc7ffc7ffc7ffc7, 5288,{0xffff,0x02ac}},  //  5783, depth=2
{0x001fff00001fff00, 5290,{0xffff,0x060c}},  //  5784, depth=2
{0x00000000ffffffc3, 5293,{0x069b,0xffff}},  //  5785, depth=2
{0x003ff80000000000,   -1,{0xffff,0x154a}},  //  5786, depth=1
{0xff800000001fffff,   -1,{0xffff,0x125d}},  //  5787
{0x0c0c0c0c0c0c0c0c, 5300,{0xffff,0x01b1}},  //  5788, depth=2
{0xffc001ffffc001ff, 5303,{0xffff,0x0292}},  //  5789, depth=3
{0x0ffff00000000000,   -1,{0xffff,0x150f}},  //  5790, depth=1
{0x00000000c03fffff, 5312,{0x0097,0xffff}},  //  5791, depth=2
{0x0000ffffff800000,   -1,{0xffff,0x1a58}},  //  5792, depth=1
{0xfff007fffff007ff, 5314,{0xffff,0x0316}},  //  5793, depth=2
{0x00000000007fff80,   -1,{0x064f,0x1e4f}},  //  5794, depth=1
{0x00000000f01fffff, 5318,{0x0118,0xffff}},  //  5795, depth=2
{0x0000000060006000, 5319,{0x00e1,0xffff}},  //  5796, depth=2
{0x00000000ffffffcf, 5320,{0x069d,0xffff}},  //  5797, depth=2
{0xfff003ffffffffff,   -1,{0xffff,0x1335}},  //  5798
{0xffffe00000000003, 5325,{0xffff,0x14d4}},  //  5799, depth=2
{0x000000ffff000000,   -1,{0xffff,0x1a0f}},  //  5800, depth=1
{0xfffffffff0003fff,   -1,{0xffff,0x1931}},  //  5801, depth=1
{0x800000000001ffff,   -1,{0xffff,0x1051}},  //  5802
{0xffffe00000000007,   -1,{0xffff,0x14d5}},  //  5803, depth=1
{0x00000ffff0000000,   -1,{0xffff,0x190f}},  //  5804, depth=1
{0xffc00003ffc00003, 5328,{0xffff,0x028b}},  //  5805, depth=2
{0xfe001ffffe001fff,   -1,{0xffff,0x01d3}},  //  5806
{0x0ffffffffff80000,   -1,{0xffff,0x1b68}},  //  5807
{0x000007e0000007e0, 5330,{0xffff,0x06c5}},  //  5808, depth=2
{0xefffffffefffffff, 5336,{0xffff,0x00de}},  //  5809, depth=2
{0x1ff01ff01ff01ff0, 5345,{0xffff,0x0328}},  //  5810, depth=2
{0xffffe0000000000f,   -1,{0xffff,0x14d6}},  //  5811, depth=1
{0x00000000003fff00,   -1,{0x060d,0x1e0d}},  //  5812, depth=1
{0x00000000ffffffdf,   -1,{0x069e,0xffff}},  //  5813, depth=1
{0x00000000ffffffe0,   -1,{0x06da,0x1eda}},  //  5814, depth=1
{0x00000000ffffffe1,   -1,{0x06db,0xffff}},  //  5815, depth=1
{0x00000000f000ffff,   -1,{0x0113,0xffff}},  //  5816
{0x00000000ffffffe3, 5356,{0x06dc,0xffff}},  //  5817, depth=2
{0x000000001ffffff8,   -1,{0x0759,0x1f59}},  //  5818
{0xffffffe007ffffff,   -1,{0xffff,0x16f5}},  //  5819, depth=1
{0x0000003fffff8000,   -1,{0xffff,0x1c56}},  //  5820, depth=1
{0x00000000ffffffe7, 5371,{0x06dd,0xffff}},  //  5821, depth=2
{0x07fffffffff00000,   -1,{0xffff,0x1b26}},  //  5822, depth=1
{0xfffe00007fffffff,   -1,{0xffff,0x13ed}},  //  5823
{0x001ffffffffff800,   -1,{0xffff,0x1d69}},  //  5824
{0x0003ffffffff0000, 5832,{0xffff,0x1c21}},  //  5825
{0x000000001ff81ff8, 5376,{0x0369,0xffff}},  //  5826, depth=2
{0x80000000000000ff, 5377,{0xffff,0x1048}},  //  5827, depth=2
{0x7800000078000000, 5379,{0xffff,0x0143}},  //  5828, depth=2
{0x00000000ffffffef,   -1,{0x06de,0xffff}},  //  5829, depth=1
{0x00000000fffffff0,   -1,{0x071b,0x1f1b}},  //  5830, depth=1
{0x00000000fffffff1,   -1,{0x071c,0xffff}},  //  5831, depth=1
{0x00000000ffe0ffe0,   -1,{0x02ea,0xffff}},  //  5832
{0xfc00001ffc00001f, 5382,{0xffff,0x018a}},  //  5833, depth=2
{0x00c0000000000000,   -1,{0xffff,0x1281}},  //  5834, depth=1
{0xffffff003fffffff,   -1,{0xffff,0x1635}},  //  5835, depth=1
{0xffffffffc003ffff,   -1,{0xffff,0x18b3}},  //  5836
{0x00000000fffffff7,   -1,{0x071e,0xffff}},  //  5837, depth=1
{0x00000000fffffff8,   -1,{0x075c,0x1f5c}},  //  5838, depth=1
{0x000000ff000000ff, 5383,{0xffff,0x0007}},  //  5839, depth=3
{0xffffff00000000ff,   -1,{0xffff,0x161f}},  //  5840
{0x00000000fffffffb,   -1,{0x075e,0xffff}},  //  5841, depth=1
{0x00000000fffffffc, 5390,{0x079d,0x1f9d}},  //  5842, depth=2
{0x00000000fffffffd,   -1,{0x079e,0xffff}},  //  5843, depth=1
{0x00000000fffffffe,   -1,{0x07de,0x1fde}},  //  5844, depth=1
{0x00000000ffffffff,   -1,{0xffff,0x101f}},  //  5845, depth=1
{0x007e007e007e007e, 5391,{0xffff,0x03e5}},  //  5846, depth=2
{0x0000000100000001, 5398,{0xffff,0x0000}},  //  5847, depth=2
{0x0018001800180018,   -1,{0xffff,0x0361}},  //  5848, depth=1
{0xfe00000000001fff,   -1,{0xffff,0x11d3}},  //  5849, depth=1
{0x000ffffffffffc00,   -1,{0xffff,0x1da9}},  //  5850
{0xffffffffffe03fff,   -1,{0xffff,0x1af8}},  //  5851, depth=1
{0x00000000001f0000,   -1,{0x0404,0x1c04}},  //  5852, depth=1
{0xfffc0000000fffff, 5855,{0xffff,0x13a1}},  //  5853
{0x00000fffffe00000,   -1,{0xffff,0x1ad6}},  //  5854, depth=1
{0xff8000000fffffff,   -1,{0xffff,0x1264}},  //  5855
{0x3f0000003f000000, 5399,{0xffff,0x0205}},  //  5856, depth=2
{0xfff0000ffff0000f,   -1,{0xffff,0x030f}},  //  5857, depth=1
{0x00000000007fffc0,   -1,{0x0690,0x1e90}},  //  5858, depth=1
{0xffffe0000000003f,   -1,{0xffff,0x14d8}},  //  5859, depth=1
{0xffffff8003ffffff,   -1,{0xffff,0x1672}},  //  5860
{0xff807fffffffffff,   -1,{0xffff,0x1277}},  //  5861, depth=1
{0x0007fffffffffe00,   -1,{0xffff,0x1de9}},  //  5862
{0xfc0000000000ffff,   -1,{0xffff,0x1195}},  //  5863
{0x0000000000007800, 5401,{0x0543,0x1d43}},  //  5864, depth=2
{0xfffff00000003fff,   -1,{0xffff,0x1521}},  //  5865
{0x000000003fffff00,   -1,{0x0615,0x1e15}},  //  5866, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5867
{0x0000ffff00000000,   -1,{0xffff,0x180f}},  //  5868, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5869
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5870
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5871
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5872
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5873
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5874
{0xffffffff0001ffff,   -1,{0xffff,0x1830}},  //  5875, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5876
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5877
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5878
{0xfe1fffffffffffff,   -1,{0xffff,0x11fb}},  //  5879, depth=1
{0x00007ffff0000000,   -1,{0xffff,0x1912}},  //  5880, depth=1
{0xffffff001fffffff,   -1,{0xffff,0x1634}},  //  5881, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5882
{0x00000000001f001f, 5403,{0x0024,0xffff}},  //  5883, depth=2
{0x0600000006000000,   -1,{0xffff,0x01c1}},  //  5884, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5885
{0x0000000001000100,   -1,{0x0220,0xffff}},  //  5886, depth=1
{0xfffc00fffffc00ff,   -1,{0xffff,0x0395}},  //  5887, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5888
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5889
{0x1000100010001000, 5404,{0xffff,0x0120}},  //  5890, depth=2
{0x00000000dfffffff,   -1,{0x009e,0xffff}},  //  5891, depth=1
{0x00000000e0000000, 5407,{0x00c2,0x18c2}},  //  5892, depth=2
{0x00000000e0000001,   -1,{0x00c3,0xffff}},  //  5893, depth=1
{0x3fc03fc03fc03fc0, 5411,{0xffff,0x02a7}},  //  5894, depth=2
{0x00000000e0000003, 5416,{0x00c4,0xffff}},  //  5895, depth=2
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5896
{0x8000000fffffffff, 5418,{0xffff,0x1064}},  //  5897, depth=2
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5898
{0x00000000e0000007, 5419,{0x00c5,0xffff}},  //  5899, depth=2
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5900
{0xcfffcfffcfffcfff,   -1,{0xffff,0x00ad}},  //  5901, depth=1
{0x000000000fff0000, 5422,{0x040b,0x1c0b}},  //  5902, depth=2
{0xfff0000000001fff,   -1,{0xffff,0x1318}},  //  5903, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5904
{0xffbfffbfffbfffbf,   -1,{0xffff,0x026e}},  //  5905, depth=1
{0x00000000007ffff0,   -1,{0x0712,0x1f12}},  //  5906, depth=1
{0x00000000e000000f, 5428,{0x00c6,0xffff}},  //  5907, depth=2
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5908
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5909
{0x0000000008080808, 5431,{0x0170,0xffff}},  //  5910, depth=3
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5911
{0x000000001fffff00,   -1,{0x0614,0x1e14}},  //  5912, depth=1
{0xf80000ffffffffff,   -1,{0xffff,0x116c}},  //  5913, depth=1
{0x00000000007ffff8, 5437,{0x0753,0x1f53}},  //  5914, depth=2
{0x00000000ffffe7ff,   -1,{0x04dd,0xffff}},  //  5915, depth=1
{0x00007ffc00007ffc, 5447,{0xffff,0x078c}},  //  5916, depth=3
{0xff000000000fffff,   -1,{0xffff,0x121b}},  //  5917, depth=1
{0x00000000007ffffc,   -1,{0x0794,0x1f94}},  //  5918, depth=1
{0xfe000000000007ff,   -1,{0xffff,0x11d1}},  //  5919, depth=1
{0x0ff000000ff00000, 5450,{0xffff,0x0307}},  //  5920, depth=2
{0x00000000007fffff, 5456,{0x0016,0x1016}},  //  5921, depth=3
{0x0000000000800000, 5458,{0x0240,0x1a40}},  //  5922, depth=2
{0x00000000e000001f, 5460,{0x00c7,0xffff}},  //  5923, depth=4
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5924
{0xf8000ffff8000fff, 5466,{0xffff,0x0150}},  //  5925, depth=2
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5926
{0x81ff81ff81ff81ff, 5469,{0xffff,0x0069}},  //  5927, depth=3
{0x1ffe000000000000, 5485,{0xffff,0x13cb}},  //  5928, depth=2
{0xe000ffffffffffff, 5486,{0xffff,0x10f2}},  //  5929, depth=2
{0x0808080808080808, 5495,{0xffff,0x0170}},  //  5930, depth=2
{0xffffff0000000003,   -1,{0xffff,0x1619}},  //  5931, depth=1
{0x00007e0000007e00, 5504,{0xffff,0x05c5}},  //  5932, depth=2
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5933
{0x0000000000006000,   -1,{0x04c1,0x1cc1}},  //  5934, depth=1
{0xffffff0000000007,   -1,{0xffff,0x161a}},  //  5935, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5936
{0x00000000bfffffff, 5505,{0x005e,0xffff}},  //  5937, depth=3
{0x00000000c0000000, 5509,{0x0081,0x1881}},  //  5938, depth=2
{0x00000000c0000001,   -1,{0x0082,0xffff}},  //  5939, depth=1
{0x00000000003fff80, 5510,{0x064e,0x1e4e}},  //  5940, depth=2
{0x00000000c0000003, 5512,{0x0083,0xffff}},  //  5941, depth=2
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5942
{0xffffff000000000f,   -1,{0xffff,0x161b}},  //  5943, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5944
{0x00000000c0000007,   -1,{0x0084,0xffff}},  //  5945, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5946
{0xe00000ffe00000ff,   -1,{0xffff,0x00ca}},  //  5947, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5948
{0xf0000007ffffffff,   -1,{0xffff,0x1126}},  //  5949, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5950
{0xffffc3ffffffc3ff, 5526,{0xffff,0x049b}},  //  5951, depth=2
{0x000000fe00000000,   -1,{0xffff,0x17c6}},  //  5952, depth=1
{0x00000000c000000f, 5530,{0x0085,0xffff}},  //  5953, depth=2
{0x00003ffc00000000, 5534,{0xffff,0x178b}},  //  5954, depth=2
{0xe0000001e0000001, 5536,{0xffff,0x00c3}},  //  5955, depth=3
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5956
{0x3333333333333333,   -1,{0xffff,0x0039}},  //  5957, depth=1
{0x00001fc000001fc0,   -1,{0xffff,0x0686}},  //  5958, depth=1
{0xffc0000001ffffff, 5541,{0xffff,0x12a2}},  //  5959, depth=2
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5960
{0xffff8001ffff8001,   -1,{0xffff,0x0451}},  //  5961, depth=1
{0x0000000ffffe0000,   -1,{0xffff,0x1bd2}},  //  5962, depth=1
{0xfffff801ffffffff,   -1,{0xffff,0x1575}},  //  5963, depth=1
{0x07ffc00000000000,   -1,{0xffff,0x148c}},  //  5964, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5965
{0x000000007fffffc0, 5542,{0x0698,0x1e98}},  //  5966, depth=2
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5967
{0x00000000aaaaaaaa, 5543,{0x007c,0xffff}},  //  5968, depth=3
{0x00000000c000001f, 5550,{0x0086,0xffff}},  //  5969, depth=3
{0x0010001000100010,   -1,{0xffff,0x0320}},  //  5970, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5971
{0x1ffffffc00000000, 5557,{0xffff,0x179a}},  //  5972, depth=2
{0xfff00000000007ff,   -1,{0xffff,0x1316}},  //  5973, depth=1
{0x7ff000007ff00000,   -1,{0xffff,0x030a}},  //  5974, depth=1
{0xffffffe000007fff,   -1,{0xffff,0x16e9}},  //  5975, depth=1
{0x0000003ff8000000,   -1,{0xffff,0x194a}},  //  5976, depth=1
{0xfffffc07ffffffff,   -1,{0xffff,0x15b8}},  //  5977, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5978
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5979
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5980
{0xc00fc00fc00fc00f,   -1,{0xffff,0x00a5}},  //  5981, depth=1
{0x0003fffc0003fffc,   -1,{0xffff,0x078f}},  //  5982, depth=1
{0x000000009fffffff, 5561,{0x005d,0xffff}},  //  5983, depth=2
{0x007ffffffffff800,   -1,{0xffff,0x1d6b}},  //  5984, depth=1
{0x00000000ffffcfff,   -1,{0x049d,0xffff}},  //  5985, depth=1
{0x0000000006060606,   -1,{0x01f1,0xffff}},  //  5986, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5987
{0x00fffff800fffff8,   -1,{0xffff,0x0754}},  //  5988, depth=1
{0xfffe0000ffffffff,   -1,{0xffff,0x13ee}},  //  5989, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  5990
{0x00000000ffbfffff, 5562,{0x025e,0xffff}},  //  5991, depth=3
{0x00000000ffc00000,   -1,{0x0289,0x1a89}},  //  5992, depth=1
{0x00000000ffc00001,   -1,{0x028a,0xffff}},  //  5993, depth=1
{0x0400000004000000, 5567,{0xffff,0x0180}},  //  5994, depth=2
{0x000fffff000fffff, 5576,{0xffff,0x0013}},  //  5995, depth=2
{0x0000000003fff000,   -1,{0x050d,0x1d0d}},  //  5996, depth=1
{0x00000000fbfbfbfb,   -1,{0x0176,0xffff}},  //  5997, depth=1
{0x0003fe000003fe00, 5578,{0xffff,0x05c8}},  //  5998, depth=2
{0x00000000ffc00007, 5582,{0x028c,0xffff}},  //  5999, depth=2
{0x001ffc0000000000,   -1,{0xffff,0x158a}},  //  6000, depth=1
{0x00000000c000003f, 5585,{0x0087,0xffff}},  //  6001, depth=2
{0x07fff80000000000,   -1,{0xffff,0x154f}},  //  6002, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6003
{0x00000000003fffc0,   -1,{0x068f,0x1e8f}},  //  6004, depth=1
{0xffe000000fffffff,   -1,{0xffff,0x12e6}},  //  6005, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6006
{0x00000000ffc0000f,   -1,{0x028d,0xffff}},  //  6007, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6008
{0x00000000ff7fff7f,   -1,{0x022e,0xffff}},  //  6009, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6010
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6011
{0x0ff80ff80ff80ff8, 5586,{0xffff,0x0368}},  //  6012, depth=2
{0xffffff0fffffff0f, 5588,{0xffff,0x061b}},  //  6013, depth=2
{0x000000007ffffff0,   -1,{0x071a,0x1f1a}},  //  6014, depth=1
{0xfffc07ffffffffff,   -1,{0xffff,0x13b8}},  //  6015, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6016
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6017
{0x03fffffffff80000,   -1,{0xffff,0x1b66}},  //  6018, depth=1
{0x00000000e000007f, 5594,{0x00c9,0xffff}},  //  6019, depth=2
{0x000000000ffc0ffc, 5600,{0x03a9,0xffff}},  //  6020, depth=2
{0xc000000003ffffff,   -1,{0xffff,0x109b}},  //  6021, depth=1
{0x000000007ffffff8,   -1,{0x075b,0x1f5b}},  //  6022, depth=1
{0x00000000ffc0001f,   -1,{0x028e,0xffff}},  //  6023, depth=1
{0x0060000000000000,   -1,{0xffff,0x12c1}},  //  6024, depth=1
{0xffffffffff0001ff,   -1,{0xffff,0x1a30}},  //  6025, depth=1
{0x000000007ffffffc,   -1,{0x079c,0x1f9c}},  //  6026, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6027
{0x000000007ffffffe, 5601,{0x07dd,0x1fdd}},  //  6028, depth=2
{0x000000007fffffff,   -1,{0x001e,0x101e}},  //  6029, depth=1
{0x0000000080000000, 5602,{0x0040,0x1840}},  //  6030, depth=2
{0x0000000080000001,   -1,{0x0041,0xffff}},  //  6031, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6032
{0x0000000080000003, 5607,{0x0042,0xffff}},  //  6033, depth=3
{0x000007fffff00000,   -1,{0xffff,0x1b16}},  //  6034, depth=1
{0xfe1ffe1ffe1ffe1f, 5613,{0xffff,0x01eb}},  //  6035, depth=2
{0x00000000003fffe0,   -1,{0x06d0,0x1ed0}},  //  6036, depth=1
{0x0000000080000007, 5617,{0x0043,0xffff}},  //  6037, depth=2
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6038
{0x00000000f003f003,   -1,{0x0125,0xffff}},  //  6039, depth=1
{0x000000001fffff80,   -1,{0x0655,0x1e55}},  //  6040, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6041
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6042
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6043
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6044
{0x000000008000000f, 5620,{0x0044,0xffff}},  //  6045, depth=3
{0xfffffe00fffffe00,   -1,{0xffff,0x05d6}},  //  6046, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6047
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6048
{0xfffffffffffe7fff,   -1,{0xffff,0x1bfd}},  //  6049, depth=1
{0x0000000000800080,   -1,{0x0260,0xffff}},  //  6050, depth=1
{0xffffe000000000ff,   -1,{0xffff,0x14da}},  //  6051, depth=1
{0x0800080008000800, 5633,{0xffff,0x0160}},  //  6052, depth=2
{0xf800001ff800001f, 5634,{0xffff,0x0149}},  //  6053, depth=2
{0x1fe01fe01fe01fe0, 5637,{0xffff,0x02e7}},  //  6054, depth=2
{0x00000000ffc0003f, 5638,{0x028f,0xffff}},  //  6055, depth=2
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6056
{0xfffe07fffffe07ff, 5639,{0xffff,0x03d9}},  //  6057, depth=2
{0x0000000007ff8000, 5648,{0x044b,0x1c4b}},  //  6058, depth=2
{0xe000000000ffffff,   -1,{0xffff,0x10da}},  //  6059, depth=1
{0x00000000003ffff8,   -1,{0x0752,0x1f52}},  //  6060, depth=1
{0x000000008000001f,   -1,{0x0045,0xffff}},  //  6061, depth=1
{0x0000000004040404, 5657,{0x01b0,0xffff}},  //  6062, depth=3
{0xc7ffc7ffc7ffc7ff, 5670,{0xffff,0x00ac}},  //  6063, depth=2
{0x00000000003ffffc, 5674,{0x0793,0x1f93}},  //  6064, depth=2
{0x00000000c000007f, 5677,{0x0088,0xffff}},  //  6065, depth=3
{0x00000000003ffffe,   -1,{0x07d4,0x1fd4}},  //  6066, depth=1
{0x00000000003fffff,   -1,{0x0015,0x1015}},  //  6067, depth=1
{0x0000000000400000, 5680,{0x0280,0x1a80}},  //  6068, depth=2
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6069
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6070
{0xfff00ffffff00fff,   -1,{0xffff,0x0317}},  //  6071, depth=1
{0x0404040404040404, 5683,{0xffff,0x01b0}},  //  6072, depth=2
{0x00000000f9f9f9f9, 5685,{0x0175,0xffff}},  //  6073, depth=2
{0x0000000000003000,   -1,{0x0501,0x1d01}},  //  6074, depth=1
{0xf00000000000ffff,   -1,{0xffff,0x1113}},  //  6075, depth=1
{0x0000000060000000, 5686,{0x00c1,0x18c1}},  //  6076, depth=2
{0xffffc00000ffffff,   -1,{0xffff,0x14a9}},  //  6077, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6078
{0xff01ffffffffffff,   -1,{0xffff,0x1238}},  //  6079, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6080
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6081
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6082
{0x80000000000001ff, 5687,{0xffff,0x1049}},  //  6083, depth=2
{0x00001ffe00000000, 5693,{0xffff,0x17cb}},  //  6084, depth=2
{0xff80ffffff80ffff,   -1,{0xffff,0x0258}},  //  6085, depth=1
{0x00000fe000000fe0,   -1,{0xffff,0x06c6}},  //  6086, depth=1
{0xf00007fff00007ff,   -1,{0xffff,0x010e}},  //  6087, depth=1
{0x00000007ffff0000,   -1,{0xffff,0x1c12}},  //  6088, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6089
{0x000000003fffffe0, 5695,{0x06d8,0x1ed8}},  //  6090, depth=2
{0x0000000055555555, 5696,{0x003c,0xffff}},  //  6091, depth=3
{0x0008000800080008,   -1,{0xffff,0x0360}},  //  6092, depth=1
{0x000000008000003f, 5706,{0x0046,0xffff}},  //  6093, depth=2
{0x3ff800003ff80000,   -1,{0xffff,0x034a}},  //  6094, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6095
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6096
{0x8000000f8000000f, 5709,{0xffff,0x0044}},  //  6097, depth=2
{0x0001fffe0001fffe, 5712,{0xffff,0x07cf}},  //  6098, depth=2
{0xc000000000003fff, 5715,{0xffff,0x108f}},  //  6099, depth=3
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6100
{0xffdfffffffdfffff, 5727,{0xffff,0x029e}},  //  6101, depth=2
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6102
{0xc007c007c007c007, 5732,{0xffff,0x00a4}},  //  6103, depth=3
{0x0200000002000000, 5736,{0xffff,0x01c0}},  //  6104, depth=2
{0xc03fffffc03fffff, 5739,{0xffff,0x0097}},  //  6105, depth=2
{0x0001ff000001ff00, 5740,{0xffff,0x0608}},  //  6106, depth=2
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6107
{0x03fffc0000000000,   -1,{0xffff,0x158f}},  //  6108, depth=1
{0xfffffffc007fffff,   -1,{0xffff,0x17b4}},  //  6109, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6110
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6111
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6112
{0xfffffffffe00000f,   -1,{0xffff,0x19ea}},  //  6113, depth=1
{0x000000003ffffff8,   -1,{0x075a,0x1f5a}},  //  6114, depth=1
{0xfffff80000ffffff,   -1,{0xffff,0x156c}},  //  6115, depth=1
{0x01fffffffffc0000,   -1,{0xffff,0x1ba6}},  //  6116, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6117
{0x000000003ffffffc,   -1,{0x079b,0x1f9b}},  //  6118, depth=1
{0x00000000ffc0007f,   -1,{0x0290,0xffff}},  //  6119, depth=1
{0x000000003ffffffe,   -1,{0x07dc,0x1fdc}},  //  6120, depth=1
{0x000000003fffffff, 5742,{0x001d,0x101d}},  //  6121, depth=3
{0x0000000040000000, 5746,{0x0080,0x1880}},  //  6122, depth=2
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6123
{0x000003fffff80000,   -1,{0xffff,0x1b56}},  //  6124, depth=1
{0x00000000ffff9fff, 5752,{0x045d,0xffff}},  //  6125, depth=2
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6126
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6127
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6128
{0xffc000ffffffffff, 5753,{0xffff,0x12b1}},  //  6129, depth=2
{0x7fffff007fffff00,   -1,{0xffff,0x0616}},  //  6130, depth=1
{0xffffffffe003ffff,   -1,{0xffff,0x18f4}},  //  6131, depth=1
{0x0000000000400040,   -1,{0x02a0,0xffff}},  //  6132, depth=1
{0xfffff00003ffffff,   -1,{0xffff,0x152d}},  //  6133, depth=1
{0x0ff00ff00ff00ff0, 5760,{0xffff,0x0327}},  //  6134, depth=2
{0xffff800000003fff,   -1,{0xffff,0x145e}},  //  6135, depth=1
{0x0000000003ffc000, 5761,{0x048b,0x1c8b}},  //  6136, depth=2
{0x00000000ff7fffff, 5764,{0x021e,0xffff}},  //  6137, depth=2
{0x0000000002020202, 5765,{0x01f0,0xffff}},  //  6138, depth=3
{0x00000000ff800001, 5767,{0x0249,0xffff}},  //  6139, depth=2
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6140
{0x07ffffff07ffffff, 5772,{0xffff,0x001a}},  //  6141, depth=2
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6142
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6143
{0x0000000000001800,   -1,{0x0541,0x1d41}},  //  6144, depth=1
{0x00000000ff800007, 5780,{0x024b,0xffff}},  //  6145, depth=3
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6146
{0x00000000e00000ff, 5798,{0x00ca,0xffff}},  //  6147, depth=2
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6148
{0x00000000f7f7f7f7, 5802,{0x0136,0xffff}},  //  6149, depth=2
{0x000007f0000007f0,   -1,{0xffff,0x0706}},  //  6150, depth=1
{0xf000000ff000000f, 5806,{0xffff,0x0107}},  //  6151, depth=2
{0x000000001ffffff0, 5807,{0x0718,0x1f18}},  //  6152, depth=2
{0x00000000ff80000f,   -1,{0x024c,0xffff}},  //  6153, depth=1
{0x1ffc00001ffc0000,   -1,{0xffff,0x038a}},  //  6154, depth=1
{0xffff0000000001ff,   -1,{0xffff,0x1418}},  //  6155, depth=1
{0x7fffffffff000000,   -1,{0xffff,0x1a26}},  //  6156, depth=1
{0x000000008000007f, 5816,{0x0047,0xffff}},  //  6157, depth=2
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6158
{0xc03fffffffffffff,   -1,{0xffff,0x10b7}},  //  6159, depth=1
{0x0000ff800000ff80, 5818,{0xffff,0x0648}},  //  6160, depth=2
{0xfffffffffe00003f,   -1,{0xffff,0x19ec}},  //  6161, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6162
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6163
{0x000000001ffffffc,   -1,{0x079a,0x1f9a}},  //  6164, depth=1
{0xfc7ffc7ffc7ffc7f,   -1,{0xffff,0x01ac}},  //  6165, depth=1
{0x000000001ffffffe,   -1,{0x07db,0x1fdb}},  //  6166, depth=1
{0x000000001fffffff,   -1,{0x001c,0x101c}},  //  6167, depth=1
{0x0000000020000000,   -1,{0x00c0,0x18c0}},  //  6168, depth=1
{0x00000000ff80001f,   -1,{0x024d,0xffff}},  //  6169, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6170
{0xfc03ffffffffffff,   -1,{0xffff,0x11b7}},  //  6171, depth=1
{0x3fffff803fffff80,   -1,{0xffff,0x0656}},  //  6172, depth=1
{0x00000000fefffeff, 5823,{0x01ee,0xffff}},  //  6173, depth=2
{0x07f807f807f807f8, 5824,{0xffff,0x0367}},  //  6174, depth=2
{0xf8000001ffffffff,   -1,{0xffff,0x1165}},  //  6175, depth=1
{0x000000007fc00000, 5825,{0x0288,0x1a88}},  //  6176, depth=3
{0xfffffffe000001ff, 5836,{0xffff,0x17e7}},  //  6177, depth=2
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6178
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6179
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6180
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6181
{0x000003f8000003f8,   -1,{0xffff,0x0746}},  //  6182, depth=1
{0xfe0000000007ffff, 5840,{0xffff,0x11d9}},  //  6183, depth=2
{0x0ffe00000ffe0000,   -1,{0xffff,0x03ca}},  //  6184, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6185
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6186
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6187
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6188
{0xfffffe01ffffffff,   -1,{0xffff,0x15f7}},  //  6189, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6190
{0xffe00007ffffffff,   -1,{0xffff,0x12ed}},  //  6191, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6192
{0x00000000c00000ff,   -1,{0x0089,0xffff}},  //  6193, depth=1
{0x03fc03fc03fc03fc, 5850,{0xffff,0x03a7}},  //  6194, depth=2
{0x00000000ffff87ff, 5853,{0x045b,0xffff}},  //  6195, depth=3
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6196
{0xf800ffffffffffff,   -1,{0xffff,0x1174}},  //  6197, depth=1
{0x000001fc000001fc,   -1,{0xffff,0x0786}},  //  6198, depth=1
{0xfff0001ffff0001f, 5860,{0xffff,0x0310}},  //  6199, depth=2
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6200
{0x00000000ff80003f,   -1,{0x024e,0xffff}},  //  6201, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6202
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6203
{0x01fe01fe01fe01fe, 5862,{0xffff,0x03e7}},  //  6204, depth=2
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6205
{0x000000fe000000fe,   -1,{0xffff,0x07c6}},  //  6206, depth=1
{0x00000000ffe0ffff,   -1,{0x02da,0xffff}},  //  6207, depth=1
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6208
{0x00ff00ff00ff00ff, 5863,{0xffff,0x0027}},  //  6209, depth=2
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6210
{0xfffe0000003fffff, 5865,{0xffff,0x13e4}},  //  6211, depth=2
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6212
{0xbaadf00dbaadf00d,   -1,{0xffff,0xffff}},  //  6213
};
#endif // ARM64_INCLUDE_LOGICAL_IMMEDIATE_TABLE
// Average depth (hit): 1.73
// Average depth (fail): 1.35
// Maximum depth: 5
