const pacote = require('pacote')
const { URL } = require('url')
const log = require('../utils/log-shim')
const hostedFromMani = require('../utils/hosted-git-info-from-manifest.js')
const openUrl = require('../utils/open-url.js')

const BaseCommand = require('../base-command.js')
class Repo extends BaseCommand {
  static description = 'Open package repository page in the browser'
  static name = 'repo'
  static params = ['browser', 'workspace', 'workspaces', 'include-workspace-root']
  static usage = ['[<pkgname> [<pkgname> ...]]']
  static ignoreImplicitWorkspace = false

  async exec (args) {
    if (!args || !args.length) {
      args = ['.']
    }

    await Promise.all(args.map(pkg => this.get(pkg)))
  }

  async execWorkspaces (args, filters) {
    await this.setWorkspaces(filters)
    return this.exec(this.workspacePaths)
  }

  async get (pkg) {
    // XXX It is very odd that `where` is how pacote knows to look anywhere
    // other than the cwd.
    const opts = {
      ...this.npm.flatOptions,
      where: this.npm.localPrefix,
      fullMetadata: true,
    }
    const mani = await pacote.manifest(pkg, opts)

    const r = mani.repository
    const rurl = !r ? null
      : typeof r === 'string' ? r
      : typeof r === 'object' && typeof r.url === 'string' ? r.url
      : null

    if (!rurl) {
      throw Object.assign(new Error('no repository'), {
        pkgid: pkg,
      })
    }

    const info = hostedFromMani(mani)
    const url = info ?
      info.browse(mani.repository.directory) : unknownHostedUrl(rurl)

    if (!url) {
      throw Object.assign(new Error('no repository: could not get url'), {
        pkgid: pkg,
      })
    }

    log.silly('docs', 'url', url)
    await openUrl(this.npm, url, `${mani.name} repo available at the following URL`)
  }
}
module.exports = Repo

const unknownHostedUrl = url => {
  try {
    const {
      protocol,
      hostname,
      pathname,
    } = new URL(url)

    /* istanbul ignore next - URL ctor should prevent this */
    if (!protocol || !hostname) {
      return null
    }

    const proto = /(git\+)http:$/.test(protocol) ? 'http:' : 'https:'
    const path = pathname.replace(/\.git$/, '')
    return `${proto}//${hostname}${path}`
  } catch (e) {
    return null
  }
}
