"use strict";

var _interopRequire = function (obj) { return obj && obj.__esModule ? obj["default"] : obj; };

var _createClass = (function () { function defineProperties(target, props) { for (var key in props) { var prop = props[key]; prop.configurable = true; if (prop.value) prop.writable = true; } Object.defineProperties(target, props); } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; })();

var _get = function get(object, property, receiver) { var desc = Object.getOwnPropertyDescriptor(object, property); if (desc === undefined) { var parent = Object.getPrototypeOf(object); if (parent === null) { return undefined; } else { return get(parent, property, receiver); } } else if ("value" in desc && desc.writable) { return desc.value; } else { var getter = desc.get; if (getter === undefined) { return undefined; } return getter.call(receiver); } };

var _inherits = function (subClass, superClass) { if (typeof superClass !== "function" && superClass !== null) { throw new TypeError("Super expression must either be null or a function, not " + typeof superClass); } subClass.prototype = Object.create(superClass && superClass.prototype, { constructor: { value: subClass, enumerable: false, writable: true, configurable: true } }); if (superClass) subClass.__proto__ = superClass; };

var _classCallCheck = function (instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } };

/*
  Copyright (C) 2015 Yusuke Suzuki <utatane.tea@gmail.com>

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

var Syntax = require("estraverse").Syntax;

var esrecurse = _interopRequire(require("esrecurse"));

var Reference = _interopRequire(require("./reference"));

var Variable = _interopRequire(require("./variable"));

var _definition = require("./definition");

var ParameterDefinition = _definition.ParameterDefinition;
var Definition = _definition.Definition;

var assert = _interopRequire(require("assert"));

var PatternVisitor = (function (_esrecurse$Visitor) {
    function PatternVisitor(rootPattern, callback) {
        _classCallCheck(this, PatternVisitor);

        _get(Object.getPrototypeOf(PatternVisitor.prototype), "constructor", this).call(this);
        this.rootPattern = rootPattern;
        this.callback = callback;
        this.assignments = [];
        this.rightHandNodes = [];
        this.restElements = [];
    }

    _inherits(PatternVisitor, _esrecurse$Visitor);

    _createClass(PatternVisitor, {
        Identifier: {
            value: function Identifier(pattern) {
                var lastRestElement = getLast(this.restElements);
                this.callback(pattern, {
                    topLevel: pattern === this.rootPattern,
                    rest: lastRestElement != null && lastRestElement.argument === pattern,
                    assignments: this.assignments
                });
            }
        },
        ObjectPattern: {
            value: function ObjectPattern(pattern) {
                var i, iz, property;
                for (i = 0, iz = pattern.properties.length; i < iz; ++i) {
                    property = pattern.properties[i];

                    // Computed property's key is a right hand node.
                    if (property.computed) {
                        this.rightHandNodes.push(property.key);
                    }

                    // If it's shorthand, its key is same as its value.
                    // If it's shorthand and has its default value, its key is same as its value.left (the value is AssignmentPattern).
                    // If it's not shorthand, the name of new variable is its value's.
                    this.visit(property.value);
                }
            }
        },
        ArrayPattern: {
            value: function ArrayPattern(pattern) {
                var i, iz, element;
                for (i = 0, iz = pattern.elements.length; i < iz; ++i) {
                    element = pattern.elements[i];
                    this.visit(element);
                }
            }
        },
        AssignmentPattern: {
            value: function AssignmentPattern(pattern) {
                this.assignments.push(pattern);
                this.visit(pattern.left);
                this.rightHandNodes.push(pattern.right);
                this.assignments.pop();
            }
        },
        RestElement: {
            value: function RestElement(pattern) {
                this.restElements.push(pattern);
                this.visit(pattern.argument);
                this.restElements.pop();
            }
        },
        MemberExpression: {
            value: function MemberExpression(node) {
                // Computed property's key is a right hand node.
                if (node.computed) {
                    this.rightHandNodes.push(node.property);
                }
                // the object is only read, write to its property.
                this.rightHandNodes.push(node.object);
            }
        },
        SpreadElement: {

            //
            // ForInStatement.left and AssignmentExpression.left are LeftHandSideExpression.
            // By spec, LeftHandSideExpression is Pattern or MemberExpression.
            //   (see also: https://github.com/estree/estree/pull/20#issuecomment-74584758)
            // But espree 2.0 and esprima 2.0 parse to ArrayExpression, ObjectExpression, etc...
            //

            value: function SpreadElement(node) {
                this.visit(node.argument);
            }
        },
        ArrayExpression: {
            value: function ArrayExpression(node) {
                node.elements.forEach(this.visit, this);
            }
        },
        ObjectExpression: {
            value: function ObjectExpression(node) {
                var _this = this;

                node.properties.forEach(function (property) {
                    // Computed property's key is a right hand node.
                    if (property.computed) {
                        _this.rightHandNodes.push(property.key);
                    }
                    _this.visit(property.value);
                });
            }
        },
        AssignmentExpression: {
            value: function AssignmentExpression(node) {
                this.assignments.push(node);
                this.visit(node.left);
                this.rightHandNodes.push(node.right);
                this.assignments.pop();
            }
        },
        CallExpression: {
            value: function CallExpression(node) {
                var _this = this;

                // arguments are right hand nodes.
                node.arguments.forEach(function (a) {
                    _this.rightHandNodes.push(a);
                });
                this.visit(node.callee);
            }
        }
    });

    return PatternVisitor;
})(esrecurse.Visitor);

function getLast(xs) {
    return xs[xs.length - 1] || null;
}

function traverseIdentifierInPattern(rootPattern, referencer, callback) {
    // Call the callback at left hand identifier nodes, and Collect right hand nodes.
    var visitor = new PatternVisitor(rootPattern, callback);
    visitor.visit(rootPattern);

    // Process the right hand nodes recursively.
    if (referencer != null) {
        visitor.rightHandNodes.forEach(referencer.visit, referencer);
    }
}

function isPattern(node) {
    var nodeType = node.type;
    return nodeType === Syntax.Identifier || nodeType === Syntax.ObjectPattern || nodeType === Syntax.ArrayPattern || nodeType === Syntax.SpreadElement || nodeType === Syntax.RestElement || nodeType === Syntax.AssignmentPattern;
}

// Importing ImportDeclaration.
// http://people.mozilla.org/~jorendorff/es6-draft.html#sec-moduledeclarationinstantiation
// https://github.com/estree/estree/blob/master/es6.md#importdeclaration
// FIXME: Now, we don't create module environment, because the context is
// implementation dependent.

var Importer = (function (_esrecurse$Visitor2) {
    function Importer(declaration, referencer) {
        _classCallCheck(this, Importer);

        _get(Object.getPrototypeOf(Importer.prototype), "constructor", this).call(this);
        this.declaration = declaration;
        this.referencer = referencer;
    }

    _inherits(Importer, _esrecurse$Visitor2);

    _createClass(Importer, {
        visitImport: {
            value: function visitImport(id, specifier) {
                var _this = this;

                this.referencer.visitPattern(id, function (pattern) {
                    _this.referencer.currentScope().__define(pattern, new Definition(Variable.ImportBinding, pattern, specifier, _this.declaration, null, null));
                });
            }
        },
        ImportNamespaceSpecifier: {
            value: function ImportNamespaceSpecifier(node) {
                var local = node.local || node.id;
                if (local) {
                    this.visitImport(local, node);
                }
            }
        },
        ImportDefaultSpecifier: {
            value: function ImportDefaultSpecifier(node) {
                var local = node.local || node.id;
                this.visitImport(local, node);
            }
        },
        ImportSpecifier: {
            value: function ImportSpecifier(node) {
                var local = node.local || node.id;
                if (node.name) {
                    this.visitImport(node.name, node);
                } else {
                    this.visitImport(local, node);
                }
            }
        }
    });

    return Importer;
})(esrecurse.Visitor);

// Referencing variables and creating bindings.

var Referencer = (function (_esrecurse$Visitor3) {
    function Referencer(scopeManager) {
        _classCallCheck(this, Referencer);

        _get(Object.getPrototypeOf(Referencer.prototype), "constructor", this).call(this);
        this.scopeManager = scopeManager;
        this.parent = null;
        this.isInnerMethodDefinition = false;
    }

    _inherits(Referencer, _esrecurse$Visitor3);

    _createClass(Referencer, {
        currentScope: {
            value: function currentScope() {
                return this.scopeManager.__currentScope;
            }
        },
        close: {
            value: function close(node) {
                while (this.currentScope() && node === this.currentScope().block) {
                    this.scopeManager.__currentScope = this.currentScope().__close(this.scopeManager);
                }
            }
        },
        pushInnerMethodDefinition: {
            value: function pushInnerMethodDefinition(isInnerMethodDefinition) {
                var previous = this.isInnerMethodDefinition;
                this.isInnerMethodDefinition = isInnerMethodDefinition;
                return previous;
            }
        },
        popInnerMethodDefinition: {
            value: function popInnerMethodDefinition(isInnerMethodDefinition) {
                this.isInnerMethodDefinition = isInnerMethodDefinition;
            }
        },
        materializeTDZScope: {
            value: function materializeTDZScope(node, iterationNode) {
                // http://people.mozilla.org/~jorendorff/es6-draft.html#sec-runtime-semantics-forin-div-ofexpressionevaluation-abstract-operation
                // TDZ scope hides the declaration's names.
                this.scopeManager.__nestTDZScope(node, iterationNode);
                this.visitVariableDeclaration(this.currentScope(), Variable.TDZ, iterationNode.left, 0, true);
            }
        },
        materializeIterationScope: {
            value: function materializeIterationScope(node) {
                var _this = this;

                // Generate iteration scope for upper ForIn/ForOf Statements.
                var letOrConstDecl;
                this.scopeManager.__nestForScope(node);
                letOrConstDecl = node.left;
                this.visitVariableDeclaration(this.currentScope(), Variable.Variable, letOrConstDecl, 0);
                this.visitPattern(letOrConstDecl.declarations[0].id, function (pattern) {
                    _this.currentScope().__referencing(pattern, Reference.WRITE, node.right, null, true, true);
                });
            }
        },
        referencingDefaultValue: {
            value: function referencingDefaultValue(pattern, assignments, maybeImplicitGlobal, init) {
                var scope = this.currentScope();
                assignments.forEach(function (assignment) {
                    scope.__referencing(pattern, Reference.WRITE, assignment.right, maybeImplicitGlobal, pattern !== assignment.left, init);
                });
            }
        },
        visitPattern: {
            value: function visitPattern(node, options, callback) {
                if (typeof options === "function") {
                    callback = options;
                    options = { processRightHandNodes: false };
                }
                traverseIdentifierInPattern(node, options.processRightHandNodes ? this : null, callback);
            }
        },
        visitFunction: {
            value: function visitFunction(node) {
                var _this = this;

                var i, iz;
                // FunctionDeclaration name is defined in upper scope
                // NOTE: Not referring variableScope. It is intended.
                // Since
                //  in ES5, FunctionDeclaration should be in FunctionBody.
                //  in ES6, FunctionDeclaration should be block scoped.
                if (node.type === Syntax.FunctionDeclaration) {
                    // id is defined in upper scope
                    this.currentScope().__define(node.id, new Definition(Variable.FunctionName, node.id, node, null, null, null));
                }

                // FunctionExpression with name creates its special scope;
                // FunctionExpressionNameScope.
                if (node.type === Syntax.FunctionExpression && node.id) {
                    this.scopeManager.__nestFunctionExpressionNameScope(node);
                }

                // Consider this function is in the MethodDefinition.
                this.scopeManager.__nestFunctionScope(node, this.isInnerMethodDefinition);

                // Process parameter declarations.
                for (i = 0, iz = node.params.length; i < iz; ++i) {
                    this.visitPattern(node.params[i], { processRightHandNodes: true }, function (pattern, info) {
                        _this.currentScope().__define(pattern, new ParameterDefinition(pattern, node, i, info.rest));

                        _this.referencingDefaultValue(pattern, info.assignments, null, true);
                    });
                }

                // if there's a rest argument, add that
                if (node.rest) {
                    this.visitPattern({
                        type: "RestElement",
                        argument: node.rest
                    }, function (pattern) {
                        _this.currentScope().__define(pattern, new ParameterDefinition(pattern, node, node.params.length, true));
                    });
                }

                // Skip BlockStatement to prevent creating BlockStatement scope.
                if (node.body.type === Syntax.BlockStatement) {
                    this.visitChildren(node.body);
                } else {
                    this.visit(node.body);
                }

                this.close(node);
            }
        },
        visitClass: {
            value: function visitClass(node) {
                if (node.type === Syntax.ClassDeclaration) {
                    this.currentScope().__define(node.id, new Definition(Variable.ClassName, node.id, node, null, null, null));
                }

                // FIXME: Maybe consider TDZ.
                this.visit(node.superClass);

                this.scopeManager.__nestClassScope(node);

                if (node.id) {
                    this.currentScope().__define(node.id, new Definition(Variable.ClassName, node.id, node));
                }
                this.visit(node.body);

                this.close(node);
            }
        },
        visitProperty: {
            value: function visitProperty(node) {
                var previous, isMethodDefinition;
                if (node.computed) {
                    this.visit(node.key);
                }

                isMethodDefinition = node.type === Syntax.MethodDefinition || node.method;
                if (isMethodDefinition) {
                    previous = this.pushInnerMethodDefinition(true);
                }
                this.visit(node.value);
                if (isMethodDefinition) {
                    this.popInnerMethodDefinition(previous);
                }
            }
        },
        visitForIn: {
            value: function visitForIn(node) {
                var _this = this;

                if (node.left.type === Syntax.VariableDeclaration && node.left.kind !== "var") {
                    this.materializeTDZScope(node.right, node);
                    this.visit(node.right);
                    this.close(node.right);

                    this.materializeIterationScope(node);
                    this.visit(node.body);
                    this.close(node);
                } else {
                    if (node.left.type === Syntax.VariableDeclaration) {
                        this.visit(node.left);
                        this.visitPattern(node.left.declarations[0].id, function (pattern) {
                            _this.currentScope().__referencing(pattern, Reference.WRITE, node.right, null, true, true);
                        });
                    } else {
                        this.visitPattern(node.left, { processRightHandNodes: true }, function (pattern, info) {
                            var maybeImplicitGlobal = null;
                            if (!_this.currentScope().isStrict) {
                                maybeImplicitGlobal = {
                                    pattern: pattern,
                                    node: node
                                };
                            }
                            _this.referencingDefaultValue(pattern, info.assignments, maybeImplicitGlobal, false);
                            _this.currentScope().__referencing(pattern, Reference.WRITE, node.right, maybeImplicitGlobal, true, false);
                        });
                    }
                    this.visit(node.right);
                    this.visit(node.body);
                }
            }
        },
        visitVariableDeclaration: {
            value: function visitVariableDeclaration(variableTargetScope, type, node, index, fromTDZ) {
                var _this = this;

                // If this was called to initialize a TDZ scope, this needs to make definitions, but doesn't make references.
                var decl, init;

                decl = node.declarations[index];
                init = decl.init;
                this.visitPattern(decl.id, { processRightHandNodes: !fromTDZ }, function (pattern, info) {
                    variableTargetScope.__define(pattern, new Definition(type, pattern, decl, node, index, node.kind));

                    if (!fromTDZ) {
                        _this.referencingDefaultValue(pattern, info.assignments, null, true);
                    }
                    if (init) {
                        _this.currentScope().__referencing(pattern, Reference.WRITE, init, null, !info.topLevel, true);
                    }
                });
            }
        },
        AssignmentExpression: {
            value: function AssignmentExpression(node) {
                var _this = this;

                if (isPattern(node.left)) {
                    if (node.operator === "=") {
                        this.visitPattern(node.left, { processRightHandNodes: true }, function (pattern, info) {
                            var maybeImplicitGlobal = null;
                            if (!_this.currentScope().isStrict) {
                                maybeImplicitGlobal = {
                                    pattern: pattern,
                                    node: node
                                };
                            }
                            _this.referencingDefaultValue(pattern, info.assignments, maybeImplicitGlobal, false);
                            _this.currentScope().__referencing(pattern, Reference.WRITE, node.right, maybeImplicitGlobal, !info.topLevel, false);
                        });
                    } else {
                        this.currentScope().__referencing(node.left, Reference.RW, node.right);
                    }
                } else {
                    this.visit(node.left);
                }
                this.visit(node.right);
            }
        },
        CatchClause: {
            value: function CatchClause(node) {
                var _this = this;

                this.scopeManager.__nestCatchScope(node);

                this.visitPattern(node.param, { processRightHandNodes: true }, function (pattern, info) {
                    _this.currentScope().__define(pattern, new Definition(Variable.CatchClause, node.param, node, null, null, null));
                    _this.referencingDefaultValue(pattern, info.assignments, null, true);
                });
                this.visit(node.body);

                this.close(node);
            }
        },
        Program: {
            value: function Program(node) {
                this.scopeManager.__nestGlobalScope(node);

                if (this.scopeManager.__isNodejsScope()) {
                    // Force strictness of GlobalScope to false when using node.js scope.
                    this.currentScope().isStrict = false;
                    this.scopeManager.__nestFunctionScope(node, false);
                }

                if (this.scopeManager.__isES6() && this.scopeManager.isModule()) {
                    this.scopeManager.__nestModuleScope(node);
                }

                this.visitChildren(node);
                this.close(node);
            }
        },
        Identifier: {
            value: function Identifier(node) {
                this.currentScope().__referencing(node);
            }
        },
        UpdateExpression: {
            value: function UpdateExpression(node) {
                if (isPattern(node.argument)) {
                    this.currentScope().__referencing(node.argument, Reference.RW, null);
                } else {
                    this.visitChildren(node);
                }
            }
        },
        MemberExpression: {
            value: function MemberExpression(node) {
                this.visit(node.object);
                if (node.computed) {
                    this.visit(node.property);
                }
            }
        },
        Property: {
            value: function Property(node) {
                this.visitProperty(node);
            }
        },
        MethodDefinition: {
            value: function MethodDefinition(node) {
                this.visitProperty(node);
            }
        },
        BreakStatement: {
            value: function BreakStatement() {}
        },
        ContinueStatement: {
            value: function ContinueStatement() {}
        },
        LabeledStatement: {
            value: function LabeledStatement(node) {
                this.visit(node.body);
            }
        },
        ForStatement: {
            value: function ForStatement(node) {
                // Create ForStatement declaration.
                // NOTE: In ES6, ForStatement dynamically generates
                // per iteration environment. However, escope is
                // a static analyzer, we only generate one scope for ForStatement.
                if (node.init && node.init.type === Syntax.VariableDeclaration && node.init.kind !== "var") {
                    this.scopeManager.__nestForScope(node);
                }

                this.visitChildren(node);

                this.close(node);
            }
        },
        ClassExpression: {
            value: function ClassExpression(node) {
                this.visitClass(node);
            }
        },
        ClassDeclaration: {
            value: function ClassDeclaration(node) {
                this.visitClass(node);
            }
        },
        CallExpression: {
            value: function CallExpression(node) {
                // Check this is direct call to eval
                if (!this.scopeManager.__ignoreEval() && node.callee.type === Syntax.Identifier && node.callee.name === "eval") {
                    // NOTE: This should be `variableScope`. Since direct eval call always creates Lexical environment and
                    // let / const should be enclosed into it. Only VariableDeclaration affects on the caller's environment.
                    this.currentScope().variableScope.__detectEval();
                }
                this.visitChildren(node);
            }
        },
        BlockStatement: {
            value: function BlockStatement(node) {
                if (this.scopeManager.__isES6()) {
                    this.scopeManager.__nestBlockScope(node);
                }

                this.visitChildren(node);

                this.close(node);
            }
        },
        ThisExpression: {
            value: function ThisExpression() {
                this.currentScope().variableScope.__detectThis();
            }
        },
        WithStatement: {
            value: function WithStatement(node) {
                this.visit(node.object);
                // Then nest scope for WithStatement.
                this.scopeManager.__nestWithScope(node);

                this.visit(node.body);

                this.close(node);
            }
        },
        VariableDeclaration: {
            value: function VariableDeclaration(node) {
                var variableTargetScope, i, iz, decl;
                variableTargetScope = node.kind === "var" ? this.currentScope().variableScope : this.currentScope();
                for (i = 0, iz = node.declarations.length; i < iz; ++i) {
                    decl = node.declarations[i];
                    this.visitVariableDeclaration(variableTargetScope, Variable.Variable, node, i);
                    if (decl.init) {
                        this.visit(decl.init);
                    }
                }
            }
        },
        SwitchStatement: {

            // sec 13.11.8

            value: function SwitchStatement(node) {
                var i, iz;

                this.visit(node.discriminant);

                if (this.scopeManager.__isES6()) {
                    this.scopeManager.__nestSwitchScope(node);
                }

                for (i = 0, iz = node.cases.length; i < iz; ++i) {
                    this.visit(node.cases[i]);
                }

                this.close(node);
            }
        },
        FunctionDeclaration: {
            value: function FunctionDeclaration(node) {
                this.visitFunction(node);
            }
        },
        FunctionExpression: {
            value: function FunctionExpression(node) {
                this.visitFunction(node);
            }
        },
        ForOfStatement: {
            value: function ForOfStatement(node) {
                this.visitForIn(node);
            }
        },
        ForInStatement: {
            value: function ForInStatement(node) {
                this.visitForIn(node);
            }
        },
        ArrowFunctionExpression: {
            value: function ArrowFunctionExpression(node) {
                this.visitFunction(node);
            }
        },
        ImportDeclaration: {
            value: function ImportDeclaration(node) {
                var importer;

                assert(this.scopeManager.__isES6() && this.scopeManager.isModule(), "ImportDeclaration should appear when the mode is ES6 and in the module context.");

                importer = new Importer(node, this);
                importer.visit(node);
            }
        },
        visitExportDeclaration: {
            value: function visitExportDeclaration(node) {
                if (node.source) {
                    return;
                }
                if (node.declaration) {
                    this.visit(node.declaration);
                    return;
                }

                this.visitChildren(node);
            }
        },
        ExportDeclaration: {
            value: function ExportDeclaration(node) {
                this.visitExportDeclaration(node);
            }
        },
        ExportNamedDeclaration: {
            value: function ExportNamedDeclaration(node) {
                this.visitExportDeclaration(node);
            }
        },
        ExportSpecifier: {
            value: function ExportSpecifier(node) {
                var local = node.id || node.local;
                this.visit(local);
            }
        }
    });

    return Referencer;
})(esrecurse.Visitor);

module.exports = Referencer;

/* vim: set sw=4 ts=4 et tw=80 : */
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbInJlZmVyZW5jZXIuanMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0lBdUJTLE1BQU0sV0FBUSxZQUFZLEVBQTFCLE1BQU07O0lBQ1IsU0FBUywyQkFBTSxXQUFXOztJQUMxQixTQUFTLDJCQUFNLGFBQWE7O0lBQzVCLFFBQVEsMkJBQU0sWUFBWTs7MEJBQ2UsY0FBYzs7SUFBckQsbUJBQW1CLGVBQW5CLG1CQUFtQjtJQUFFLFVBQVUsZUFBVixVQUFVOztJQUNqQyxNQUFNLDJCQUFNLFFBQVE7O0lBRXJCLGNBQWM7QUFDTCxhQURULGNBQWMsQ0FDSixXQUFXLEVBQUUsUUFBUSxFQUFFOzhCQURqQyxjQUFjOztBQUVaLG1DQUZGLGNBQWMsNkNBRUo7QUFDUixZQUFJLENBQUMsV0FBVyxHQUFHLFdBQVcsQ0FBQztBQUMvQixZQUFJLENBQUMsUUFBUSxHQUFHLFFBQVEsQ0FBQztBQUN6QixZQUFJLENBQUMsV0FBVyxHQUFHLEVBQUUsQ0FBQztBQUN0QixZQUFJLENBQUMsY0FBYyxHQUFHLEVBQUUsQ0FBQztBQUN6QixZQUFJLENBQUMsWUFBWSxHQUFHLEVBQUUsQ0FBQztLQUMxQjs7Y0FSQyxjQUFjOztpQkFBZCxjQUFjO0FBVWhCLGtCQUFVO21CQUFBLG9CQUFDLE9BQU8sRUFBRTtBQUNoQixvQkFBTSxlQUFlLEdBQUcsT0FBTyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQztBQUNuRCxvQkFBSSxDQUFDLFFBQVEsQ0FBQyxPQUFPLEVBQUU7QUFDbkIsNEJBQVEsRUFBRSxPQUFPLEtBQUssSUFBSSxDQUFDLFdBQVc7QUFDdEMsd0JBQUksRUFBRSxlQUFlLElBQUksSUFBSSxJQUFJLGVBQWUsQ0FBQyxRQUFRLEtBQUssT0FBTztBQUNyRSwrQkFBVyxFQUFFLElBQUksQ0FBQyxXQUFXO2lCQUNoQyxDQUFDLENBQUM7YUFDTjs7QUFFRCxxQkFBYTttQkFBQSx1QkFBQyxPQUFPLEVBQUU7QUFDbkIsb0JBQUksQ0FBQyxFQUFFLEVBQUUsRUFBRSxRQUFRLENBQUM7QUFDcEIscUJBQUssQ0FBQyxHQUFHLENBQUMsRUFBRSxFQUFFLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxHQUFHLEVBQUUsRUFBRSxFQUFFLENBQUMsRUFBRTtBQUNyRCw0QkFBUSxHQUFHLE9BQU8sQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUM7OztBQUdqQyx3QkFBSSxRQUFRLENBQUMsUUFBUSxFQUFFO0FBQ25CLDRCQUFJLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLENBQUM7cUJBQzFDOzs7OztBQUtELHdCQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQztpQkFDOUI7YUFDSjs7QUFFRCxvQkFBWTttQkFBQSxzQkFBQyxPQUFPLEVBQUU7QUFDbEIsb0JBQUksQ0FBQyxFQUFFLEVBQUUsRUFBRSxPQUFPLENBQUM7QUFDbkIscUJBQUssQ0FBQyxHQUFHLENBQUMsRUFBRSxFQUFFLEdBQUcsT0FBTyxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxHQUFHLEVBQUUsRUFBRSxFQUFFLENBQUMsRUFBRTtBQUNuRCwyQkFBTyxHQUFHLE9BQU8sQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUM7QUFDOUIsd0JBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUM7aUJBQ3ZCO2FBQ0o7O0FBRUQseUJBQWlCO21CQUFBLDJCQUFDLE9BQU8sRUFBRTtBQUN2QixvQkFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUM7QUFDL0Isb0JBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDO0FBQ3pCLG9CQUFJLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUM7QUFDeEMsb0JBQUksQ0FBQyxXQUFXLENBQUMsR0FBRyxFQUFFLENBQUM7YUFDMUI7O0FBRUQsbUJBQVc7bUJBQUEscUJBQUMsT0FBTyxFQUFFO0FBQ2pCLG9CQUFJLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQztBQUNoQyxvQkFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLENBQUM7QUFDN0Isb0JBQUksQ0FBQyxZQUFZLENBQUMsR0FBRyxFQUFFLENBQUM7YUFDM0I7O0FBRUQsd0JBQWdCO21CQUFBLDBCQUFDLElBQUksRUFBRTs7QUFFbkIsb0JBQUksSUFBSSxDQUFDLFFBQVEsRUFBRTtBQUNmLHdCQUFJLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7aUJBQzNDOztBQUVELG9CQUFJLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7YUFDekM7O0FBU0QscUJBQWE7Ozs7Ozs7OzttQkFBQSx1QkFBQyxJQUFJLEVBQUU7QUFDaEIsb0JBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDO2FBQzdCOztBQUVELHVCQUFlO21CQUFBLHlCQUFDLElBQUksRUFBRTtBQUNsQixvQkFBSSxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxJQUFJLENBQUMsQ0FBQzthQUMzQzs7QUFFRCx3QkFBZ0I7bUJBQUEsMEJBQUMsSUFBSSxFQUFFOzs7QUFDbkIsb0JBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDLFVBQUEsUUFBUSxFQUFJOztBQUVoQyx3QkFBSSxRQUFRLENBQUMsUUFBUSxFQUFFO0FBQ25CLDhCQUFLLGNBQWMsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxDQUFDO3FCQUMxQztBQUNELDBCQUFLLEtBQUssQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUM7aUJBQzlCLENBQUMsQ0FBQzthQUNOOztBQUVELDRCQUFvQjttQkFBQSw4QkFBQyxJQUFJLEVBQUU7QUFDdkIsb0JBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO0FBQzVCLG9CQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztBQUN0QixvQkFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO0FBQ3JDLG9CQUFJLENBQUMsV0FBVyxDQUFDLEdBQUcsRUFBRSxDQUFDO2FBQzFCOztBQUVELHNCQUFjO21CQUFBLHdCQUFDLElBQUksRUFBRTs7OztBQUVqQixvQkFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsVUFBQSxDQUFDLEVBQUk7QUFBRSwwQkFBSyxjQUFjLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO2lCQUFFLENBQUMsQ0FBQztBQUM5RCxvQkFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7YUFDM0I7Ozs7V0F0R0MsY0FBYztHQUFTLFNBQVMsQ0FBQyxPQUFPOztBQXlHOUMsU0FBUyxPQUFPLENBQUMsRUFBRSxFQUFFO0FBQ2pCLFdBQU8sRUFBRSxDQUFDLEVBQUUsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLElBQUksSUFBSSxDQUFDO0NBQ3BDOztBQUVELFNBQVMsMkJBQTJCLENBQUMsV0FBVyxFQUFFLFVBQVUsRUFBRSxRQUFRLEVBQUU7O0FBRXBFLFFBQUksT0FBTyxHQUFHLElBQUksY0FBYyxDQUFDLFdBQVcsRUFBRSxRQUFRLENBQUMsQ0FBQztBQUN4RCxXQUFPLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDOzs7QUFHM0IsUUFBSSxVQUFVLElBQUksSUFBSSxFQUFFO0FBQ3BCLGVBQU8sQ0FBQyxjQUFjLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDLENBQUM7S0FDaEU7Q0FDSjs7QUFFRCxTQUFTLFNBQVMsQ0FBQyxJQUFJLEVBQUU7QUFDckIsUUFBSSxRQUFRLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQztBQUN6QixXQUNJLFFBQVEsS0FBSyxNQUFNLENBQUMsVUFBVSxJQUM5QixRQUFRLEtBQUssTUFBTSxDQUFDLGFBQWEsSUFDakMsUUFBUSxLQUFLLE1BQU0sQ0FBQyxZQUFZLElBQ2hDLFFBQVEsS0FBSyxNQUFNLENBQUMsYUFBYSxJQUNqQyxRQUFRLEtBQUssTUFBTSxDQUFDLFdBQVcsSUFDL0IsUUFBUSxLQUFLLE1BQU0sQ0FBQyxpQkFBaUIsQ0FDdkM7Q0FDTDs7Ozs7Ozs7SUFRSyxRQUFRO0FBQ0MsYUFEVCxRQUFRLENBQ0UsV0FBVyxFQUFFLFVBQVUsRUFBRTs4QkFEbkMsUUFBUTs7QUFFTixtQ0FGRixRQUFRLDZDQUVFO0FBQ1IsWUFBSSxDQUFDLFdBQVcsR0FBRyxXQUFXLENBQUM7QUFDL0IsWUFBSSxDQUFDLFVBQVUsR0FBRyxVQUFVLENBQUM7S0FDaEM7O2NBTEMsUUFBUTs7aUJBQVIsUUFBUTtBQU9WLG1CQUFXO21CQUFBLHFCQUFDLEVBQUUsRUFBRSxTQUFTLEVBQUU7OztBQUN2QixvQkFBSSxDQUFDLFVBQVUsQ0FBQyxZQUFZLENBQUMsRUFBRSxFQUFFLFVBQUMsT0FBTyxFQUFLO0FBQzFDLDBCQUFLLFVBQVUsQ0FBQyxZQUFZLEVBQUUsQ0FBQyxRQUFRLENBQUMsT0FBTyxFQUMzQyxJQUFJLFVBQVUsQ0FDVixRQUFRLENBQUMsYUFBYSxFQUN0QixPQUFPLEVBQ1AsU0FBUyxFQUNULE1BQUssV0FBVyxFQUNoQixJQUFJLEVBQ0osSUFBSSxDQUNILENBQUMsQ0FBQztpQkFDZCxDQUFDLENBQUM7YUFDTjs7QUFFRCxnQ0FBd0I7bUJBQUEsa0NBQUMsSUFBSSxFQUFFO0FBQzNCLG9CQUFJLEtBQUssR0FBSSxJQUFJLENBQUMsS0FBSyxJQUFJLElBQUksQ0FBQyxFQUFFLEFBQUMsQ0FBQztBQUNwQyxvQkFBSSxLQUFLLEVBQUU7QUFDUCx3QkFBSSxDQUFDLFdBQVcsQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLENBQUM7aUJBQ2pDO2FBQ0o7O0FBRUQsOEJBQXNCO21CQUFBLGdDQUFDLElBQUksRUFBRTtBQUN6QixvQkFBSSxLQUFLLEdBQUksSUFBSSxDQUFDLEtBQUssSUFBSSxJQUFJLENBQUMsRUFBRSxBQUFDLENBQUM7QUFDcEMsb0JBQUksQ0FBQyxXQUFXLENBQUMsS0FBSyxFQUFFLElBQUksQ0FBQyxDQUFDO2FBQ2pDOztBQUVELHVCQUFlO21CQUFBLHlCQUFDLElBQUksRUFBRTtBQUNsQixvQkFBSSxLQUFLLEdBQUksSUFBSSxDQUFDLEtBQUssSUFBSSxJQUFJLENBQUMsRUFBRSxBQUFDLENBQUM7QUFDcEMsb0JBQUksSUFBSSxDQUFDLElBQUksRUFBRTtBQUNYLHdCQUFJLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUM7aUJBQ3JDLE1BQU07QUFDSCx3QkFBSSxDQUFDLFdBQVcsQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLENBQUM7aUJBQ2pDO2FBQ0o7Ozs7V0F4Q0MsUUFBUTtHQUFTLFNBQVMsQ0FBQyxPQUFPOzs7O0lBNENuQixVQUFVO0FBQ2hCLGFBRE0sVUFBVSxDQUNmLFlBQVksRUFBRTs4QkFEVCxVQUFVOztBQUV2QixtQ0FGYSxVQUFVLDZDQUVmO0FBQ1IsWUFBSSxDQUFDLFlBQVksR0FBRyxZQUFZLENBQUM7QUFDakMsWUFBSSxDQUFDLE1BQU0sR0FBRyxJQUFJLENBQUM7QUFDbkIsWUFBSSxDQUFDLHVCQUF1QixHQUFHLEtBQUssQ0FBQztLQUN4Qzs7Y0FOZ0IsVUFBVTs7aUJBQVYsVUFBVTtBQVEzQixvQkFBWTttQkFBQSx3QkFBRztBQUNYLHVCQUFPLElBQUksQ0FBQyxZQUFZLENBQUMsY0FBYyxDQUFDO2FBQzNDOztBQUVELGFBQUs7bUJBQUEsZUFBQyxJQUFJLEVBQUU7QUFDUix1QkFBTyxJQUFJLENBQUMsWUFBWSxFQUFFLElBQUksSUFBSSxLQUFLLElBQUksQ0FBQyxZQUFZLEVBQUUsQ0FBQyxLQUFLLEVBQUU7QUFDOUQsd0JBQUksQ0FBQyxZQUFZLENBQUMsY0FBYyxHQUFHLElBQUksQ0FBQyxZQUFZLEVBQUUsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDO2lCQUNyRjthQUNKOztBQUVELGlDQUF5QjttQkFBQSxtQ0FBQyx1QkFBdUIsRUFBRTtBQUMvQyxvQkFBSSxRQUFRLEdBQUcsSUFBSSxDQUFDLHVCQUF1QixDQUFDO0FBQzVDLG9CQUFJLENBQUMsdUJBQXVCLEdBQUcsdUJBQXVCLENBQUM7QUFDdkQsdUJBQU8sUUFBUSxDQUFDO2FBQ25COztBQUVELGdDQUF3QjttQkFBQSxrQ0FBQyx1QkFBdUIsRUFBRTtBQUM5QyxvQkFBSSxDQUFDLHVCQUF1QixHQUFHLHVCQUF1QixDQUFDO2FBQzFEOztBQUVELDJCQUFtQjttQkFBQSw2QkFBQyxJQUFJLEVBQUUsYUFBYSxFQUFFOzs7QUFHckMsb0JBQUksQ0FBQyxZQUFZLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxhQUFhLENBQUMsQ0FBQztBQUN0RCxvQkFBSSxDQUFDLHdCQUF3QixDQUFDLElBQUksQ0FBQyxZQUFZLEVBQUUsRUFBRSxRQUFRLENBQUMsR0FBRyxFQUFFLGFBQWEsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO2FBQ2pHOztBQUVELGlDQUF5QjttQkFBQSxtQ0FBQyxJQUFJLEVBQUU7Ozs7QUFFNUIsb0JBQUksY0FBYyxDQUFDO0FBQ25CLG9CQUFJLENBQUMsWUFBWSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsQ0FBQztBQUN2Qyw4QkFBYyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUM7QUFDM0Isb0JBQUksQ0FBQyx3QkFBd0IsQ0FBQyxJQUFJLENBQUMsWUFBWSxFQUFFLEVBQUUsUUFBUSxDQUFDLFFBQVEsRUFBRSxjQUFjLEVBQUUsQ0FBQyxDQUFDLENBQUM7QUFDekYsb0JBQUksQ0FBQyxZQUFZLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsVUFBQyxPQUFPLEVBQUs7QUFDOUQsMEJBQUssWUFBWSxFQUFFLENBQUMsYUFBYSxDQUFDLE9BQU8sRUFBRSxTQUFTLENBQUMsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQztpQkFDN0YsQ0FBQyxDQUFDO2FBQ047O0FBRUQsK0JBQXVCO21CQUFBLGlDQUFDLE9BQU8sRUFBRSxXQUFXLEVBQUUsbUJBQW1CLEVBQUUsSUFBSSxFQUFFO0FBQ3JFLG9CQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsWUFBWSxFQUFFLENBQUM7QUFDbEMsMkJBQVcsQ0FBQyxPQUFPLENBQUMsVUFBQSxVQUFVLEVBQUk7QUFDOUIseUJBQUssQ0FBQyxhQUFhLENBQ2YsT0FBTyxFQUNQLFNBQVMsQ0FBQyxLQUFLLEVBQ2YsVUFBVSxDQUFDLEtBQUssRUFDaEIsbUJBQW1CLEVBQ25CLE9BQU8sS0FBSyxVQUFVLENBQUMsSUFBSSxFQUMzQixJQUFJLENBQUMsQ0FBQztpQkFDYixDQUFDLENBQUM7YUFDTjs7QUFFRCxvQkFBWTttQkFBQSxzQkFBQyxJQUFJLEVBQUUsT0FBTyxFQUFFLFFBQVEsRUFBRTtBQUNsQyxvQkFBSSxPQUFPLE9BQU8sS0FBSyxVQUFVLEVBQUU7QUFDL0IsNEJBQVEsR0FBRyxPQUFPLENBQUM7QUFDbkIsMkJBQU8sR0FBRyxFQUFDLHFCQUFxQixFQUFFLEtBQUssRUFBQyxDQUFBO2lCQUMzQztBQUNELDJDQUEyQixDQUN2QixJQUFJLEVBQ0osT0FBTyxDQUFDLHFCQUFxQixHQUFHLElBQUksR0FBRyxJQUFJLEVBQzNDLFFBQVEsQ0FBQyxDQUFDO2FBQ2pCOztBQUVELHFCQUFhO21CQUFBLHVCQUFDLElBQUksRUFBRTs7O0FBQ2hCLG9CQUFJLENBQUMsRUFBRSxFQUFFLENBQUM7Ozs7OztBQU1WLG9CQUFJLElBQUksQ0FBQyxJQUFJLEtBQUssTUFBTSxDQUFDLG1CQUFtQixFQUFFOztBQUUxQyx3QkFBSSxDQUFDLFlBQVksRUFBRSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRSxFQUM1QixJQUFJLFVBQVUsQ0FDVixRQUFRLENBQUMsWUFBWSxFQUNyQixJQUFJLENBQUMsRUFBRSxFQUNQLElBQUksRUFDSixJQUFJLEVBQ0osSUFBSSxFQUNKLElBQUksQ0FDUCxDQUFDLENBQUM7aUJBQ2Q7Ozs7QUFJRCxvQkFBSSxJQUFJLENBQUMsSUFBSSxLQUFLLE1BQU0sQ0FBQyxrQkFBa0IsSUFBSSxJQUFJLENBQUMsRUFBRSxFQUFFO0FBQ3BELHdCQUFJLENBQUMsWUFBWSxDQUFDLGlDQUFpQyxDQUFDLElBQUksQ0FBQyxDQUFDO2lCQUM3RDs7O0FBR0Qsb0JBQUksQ0FBQyxZQUFZLENBQUMsbUJBQW1CLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDOzs7QUFHMUUscUJBQUssQ0FBQyxHQUFHLENBQUMsRUFBRSxFQUFFLEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsQ0FBQyxHQUFHLEVBQUUsRUFBRSxFQUFFLENBQUMsRUFBRTtBQUM5Qyx3QkFBSSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUMscUJBQXFCLEVBQUUsSUFBSSxFQUFDLEVBQUUsVUFBQyxPQUFPLEVBQUUsSUFBSSxFQUFLO0FBQ2hGLDhCQUFLLFlBQVksRUFBRSxDQUFDLFFBQVEsQ0FBQyxPQUFPLEVBQ2hDLElBQUksbUJBQW1CLENBQ25CLE9BQU8sRUFDUCxJQUFJLEVBQ0osQ0FBQyxFQUNELElBQUksQ0FBQyxJQUFJLENBQ1osQ0FBQyxDQUFDOztBQUVQLDhCQUFLLHVCQUF1QixDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsV0FBVyxFQUFFLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQztxQkFDdkUsQ0FBQyxDQUFDO2lCQUNOOzs7QUFHRCxvQkFBSSxJQUFJLENBQUMsSUFBSSxFQUFFO0FBQ1gsd0JBQUksQ0FBQyxZQUFZLENBQUM7QUFDZCw0QkFBSSxFQUFFLGFBQWE7QUFDbkIsZ0NBQVEsRUFBRSxJQUFJLENBQUMsSUFBSTtxQkFDdEIsRUFBRSxVQUFDLE9BQU8sRUFBSztBQUNaLDhCQUFLLFlBQVksRUFBRSxDQUFDLFFBQVEsQ0FBQyxPQUFPLEVBQ2hDLElBQUksbUJBQW1CLENBQ25CLE9BQU8sRUFDUCxJQUFJLEVBQ0osSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQ2xCLElBQUksQ0FDUCxDQUFDLENBQUM7cUJBQ1YsQ0FBQyxDQUFDO2lCQUNOOzs7QUFHRCxvQkFBSSxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxNQUFNLENBQUMsY0FBYyxFQUFFO0FBQzFDLHdCQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztpQkFDakMsTUFBTTtBQUNILHdCQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztpQkFDekI7O0FBRUQsb0JBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7YUFDcEI7O0FBRUQsa0JBQVU7bUJBQUEsb0JBQUMsSUFBSSxFQUFFO0FBQ2Isb0JBQUksSUFBSSxDQUFDLElBQUksS0FBSyxNQUFNLENBQUMsZ0JBQWdCLEVBQUU7QUFDdkMsd0JBQUksQ0FBQyxZQUFZLEVBQUUsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsRUFDNUIsSUFBSSxVQUFVLENBQ1YsUUFBUSxDQUFDLFNBQVMsRUFDbEIsSUFBSSxDQUFDLEVBQUUsRUFDUCxJQUFJLEVBQ0osSUFBSSxFQUNKLElBQUksRUFDSixJQUFJLENBQ1AsQ0FBQyxDQUFDO2lCQUNkOzs7QUFHRCxvQkFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUM7O0FBRTVCLG9CQUFJLENBQUMsWUFBWSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxDQUFDOztBQUV6QyxvQkFBSSxJQUFJLENBQUMsRUFBRSxFQUFFO0FBQ1Qsd0JBQUksQ0FBQyxZQUFZLEVBQUUsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsRUFDNUIsSUFBSSxVQUFVLENBQ1YsUUFBUSxDQUFDLFNBQVMsRUFDbEIsSUFBSSxDQUFDLEVBQUUsRUFDUCxJQUFJLENBQ1AsQ0FBQyxDQUFDO2lCQUNkO0FBQ0Qsb0JBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDOztBQUV0QixvQkFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQzthQUNwQjs7QUFFRCxxQkFBYTttQkFBQSx1QkFBQyxJQUFJLEVBQUU7QUFDaEIsb0JBQUksUUFBUSxFQUFFLGtCQUFrQixDQUFDO0FBQ2pDLG9CQUFJLElBQUksQ0FBQyxRQUFRLEVBQUU7QUFDZix3QkFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7aUJBQ3hCOztBQUVELGtDQUFrQixHQUFHLElBQUksQ0FBQyxJQUFJLEtBQUssTUFBTSxDQUFDLGdCQUFnQixJQUFJLElBQUksQ0FBQyxNQUFNLENBQUM7QUFDMUUsb0JBQUksa0JBQWtCLEVBQUU7QUFDcEIsNEJBQVEsR0FBRyxJQUFJLENBQUMseUJBQXlCLENBQUMsSUFBSSxDQUFDLENBQUM7aUJBQ25EO0FBQ0Qsb0JBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO0FBQ3ZCLG9CQUFJLGtCQUFrQixFQUFFO0FBQ3BCLHdCQUFJLENBQUMsd0JBQXdCLENBQUMsUUFBUSxDQUFDLENBQUM7aUJBQzNDO2FBQ0o7O0FBRUQsa0JBQVU7bUJBQUEsb0JBQUMsSUFBSSxFQUFFOzs7QUFDYixvQkFBSSxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxNQUFNLENBQUMsbUJBQW1CLElBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUssS0FBSyxFQUFFO0FBQzNFLHdCQUFJLENBQUMsbUJBQW1CLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxJQUFJLENBQUMsQ0FBQztBQUMzQyx3QkFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7QUFDdkIsd0JBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDOztBQUV2Qix3QkFBSSxDQUFDLHlCQUF5QixDQUFDLElBQUksQ0FBQyxDQUFDO0FBQ3JDLHdCQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztBQUN0Qix3QkFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQztpQkFDcEIsTUFBTTtBQUNILHdCQUFJLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxLQUFLLE1BQU0sQ0FBQyxtQkFBbUIsRUFBRTtBQUMvQyw0QkFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7QUFDdEIsNEJBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLFVBQUMsT0FBTyxFQUFLO0FBQ3pELGtDQUFLLFlBQVksRUFBRSxDQUFDLGFBQWEsQ0FBQyxPQUFPLEVBQUUsU0FBUyxDQUFDLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSyxFQUFFLElBQUksRUFBRSxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUM7eUJBQzdGLENBQUMsQ0FBQztxQkFDTixNQUFNO0FBQ0gsNEJBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxFQUFDLHFCQUFxQixFQUFFLElBQUksRUFBQyxFQUFFLFVBQUMsT0FBTyxFQUFFLElBQUksRUFBSztBQUMzRSxnQ0FBSSxtQkFBbUIsR0FBRyxJQUFJLENBQUM7QUFDL0IsZ0NBQUksQ0FBQyxNQUFLLFlBQVksRUFBRSxDQUFDLFFBQVEsRUFBRTtBQUMvQixtREFBbUIsR0FBRztBQUNsQiwyQ0FBTyxFQUFFLE9BQU87QUFDaEIsd0NBQUksRUFBRSxJQUFJO2lDQUNiLENBQUM7NkJBQ0w7QUFDRCxrQ0FBSyx1QkFBdUIsQ0FBQyxPQUFPLEVBQUUsSUFBSSxDQUFDLFdBQVcsRUFBRSxtQkFBbUIsRUFBRSxLQUFLLENBQUMsQ0FBQztBQUNwRixrQ0FBSyxZQUFZLEVBQUUsQ0FBQyxhQUFhLENBQUMsT0FBTyxFQUFFLFNBQVMsQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssRUFBRSxtQkFBbUIsRUFBRSxJQUFJLEVBQUUsS0FBSyxDQUFDLENBQUM7eUJBQzdHLENBQUMsQ0FBQztxQkFDTjtBQUNELHdCQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztBQUN2Qix3QkFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7aUJBQ3pCO2FBQ0o7O0FBRUQsZ0NBQXdCO21CQUFBLGtDQUFDLG1CQUFtQixFQUFFLElBQUksRUFBRSxJQUFJLEVBQUUsS0FBSyxFQUFFLE9BQU8sRUFBRTs7OztBQUV0RSxvQkFBSSxJQUFJLEVBQUUsSUFBSSxDQUFDOztBQUVmLG9CQUFJLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsQ0FBQztBQUNoQyxvQkFBSSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUM7QUFDakIsb0JBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLEVBQUUsRUFBRSxFQUFDLHFCQUFxQixFQUFFLENBQUMsT0FBTyxFQUFDLEVBQUUsVUFBQyxPQUFPLEVBQUUsSUFBSSxFQUFLO0FBQzdFLHVDQUFtQixDQUFDLFFBQVEsQ0FBQyxPQUFPLEVBQ2hDLElBQUksVUFBVSxDQUNWLElBQUksRUFDSixPQUFPLEVBQ1AsSUFBSSxFQUNKLElBQUksRUFDSixLQUFLLEVBQ0wsSUFBSSxDQUFDLElBQUksQ0FDWixDQUFDLENBQUM7O0FBRVAsd0JBQUksQ0FBQyxPQUFPLEVBQUU7QUFDViw4QkFBSyx1QkFBdUIsQ0FBQyxPQUFPLEVBQUUsSUFBSSxDQUFDLFdBQVcsRUFBRSxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUM7cUJBQ3ZFO0FBQ0Qsd0JBQUksSUFBSSxFQUFFO0FBQ04sOEJBQUssWUFBWSxFQUFFLENBQUMsYUFBYSxDQUFDLE9BQU8sRUFBRSxTQUFTLENBQUMsS0FBSyxFQUFFLElBQUksRUFBRSxJQUFJLEVBQUUsQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxDQUFDO3FCQUNqRztpQkFDSixDQUFDLENBQUM7YUFDTjs7QUFFRCw0QkFBb0I7bUJBQUEsOEJBQUMsSUFBSSxFQUFFOzs7QUFDdkIsb0JBQUksU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRTtBQUN0Qix3QkFBSSxJQUFJLENBQUMsUUFBUSxLQUFLLEdBQUcsRUFBRTtBQUN2Qiw0QkFBSSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLEVBQUMscUJBQXFCLEVBQUUsSUFBSSxFQUFDLEVBQUUsVUFBQyxPQUFPLEVBQUUsSUFBSSxFQUFLO0FBQzNFLGdDQUFJLG1CQUFtQixHQUFHLElBQUksQ0FBQztBQUMvQixnQ0FBSSxDQUFDLE1BQUssWUFBWSxFQUFFLENBQUMsUUFBUSxFQUFFO0FBQy9CLG1EQUFtQixHQUFHO0FBQ2xCLDJDQUFPLEVBQUUsT0FBTztBQUNoQix3Q0FBSSxFQUFFLElBQUk7aUNBQ2IsQ0FBQzs2QkFDTDtBQUNELGtDQUFLLHVCQUF1QixDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsV0FBVyxFQUFFLG1CQUFtQixFQUFFLEtBQUssQ0FBQyxDQUFDO0FBQ3BGLGtDQUFLLFlBQVksRUFBRSxDQUFDLGFBQWEsQ0FBQyxPQUFPLEVBQUUsU0FBUyxDQUFDLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSyxFQUFFLG1CQUFtQixFQUFFLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxLQUFLLENBQUMsQ0FBQzt5QkFDdkgsQ0FBQyxDQUFDO3FCQUNOLE1BQU07QUFDSCw0QkFBSSxDQUFDLFlBQVksRUFBRSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLFNBQVMsQ0FBQyxFQUFFLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO3FCQUMxRTtpQkFDSixNQUFNO0FBQ0gsd0JBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO2lCQUN6QjtBQUNELG9CQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQzthQUMxQjs7QUFFRCxtQkFBVzttQkFBQSxxQkFBQyxJQUFJLEVBQUU7OztBQUNkLG9CQUFJLENBQUMsWUFBWSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxDQUFDOztBQUV6QyxvQkFBSSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLEVBQUMscUJBQXFCLEVBQUUsSUFBSSxFQUFDLEVBQUUsVUFBQyxPQUFPLEVBQUUsSUFBSSxFQUFLO0FBQzVFLDBCQUFLLFlBQVksRUFBRSxDQUFDLFFBQVEsQ0FBQyxPQUFPLEVBQ2hDLElBQUksVUFBVSxDQUNWLFFBQVEsQ0FBQyxXQUFXLEVBQ3BCLElBQUksQ0FBQyxLQUFLLEVBQ1YsSUFBSSxFQUNKLElBQUksRUFDSixJQUFJLEVBQ0osSUFBSSxDQUNQLENBQUMsQ0FBQztBQUNQLDBCQUFLLHVCQUF1QixDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsV0FBVyxFQUFFLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQztpQkFDdkUsQ0FBQyxDQUFDO0FBQ0gsb0JBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDOztBQUV0QixvQkFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQzthQUNwQjs7QUFFRCxlQUFPO21CQUFBLGlCQUFDLElBQUksRUFBRTtBQUNWLG9CQUFJLENBQUMsWUFBWSxDQUFDLGlCQUFpQixDQUFDLElBQUksQ0FBQyxDQUFDOztBQUUxQyxvQkFBSSxJQUFJLENBQUMsWUFBWSxDQUFDLGVBQWUsRUFBRSxFQUFFOztBQUVyQyx3QkFBSSxDQUFDLFlBQVksRUFBRSxDQUFDLFFBQVEsR0FBRyxLQUFLLENBQUM7QUFDckMsd0JBQUksQ0FBQyxZQUFZLENBQUMsbUJBQW1CLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxDQUFDO2lCQUN0RDs7QUFFRCxvQkFBSSxJQUFJLENBQUMsWUFBWSxDQUFDLE9BQU8sRUFBRSxJQUFJLElBQUksQ0FBQyxZQUFZLENBQUMsUUFBUSxFQUFFLEVBQUU7QUFDN0Qsd0JBQUksQ0FBQyxZQUFZLENBQUMsaUJBQWlCLENBQUMsSUFBSSxDQUFDLENBQUM7aUJBQzdDOztBQUVELG9CQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxDQUFDO0FBQ3pCLG9CQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO2FBQ3BCOztBQUVELGtCQUFVO21CQUFBLG9CQUFDLElBQUksRUFBRTtBQUNiLG9CQUFJLENBQUMsWUFBWSxFQUFFLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxDQUFDO2FBQzNDOztBQUVELHdCQUFnQjttQkFBQSwwQkFBQyxJQUFJLEVBQUU7QUFDbkIsb0JBQUksU0FBUyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsRUFBRTtBQUMxQix3QkFBSSxDQUFDLFlBQVksRUFBRSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLFNBQVMsQ0FBQyxFQUFFLEVBQUUsSUFBSSxDQUFDLENBQUM7aUJBQ3hFLE1BQU07QUFDSCx3QkFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsQ0FBQztpQkFDNUI7YUFDSjs7QUFFRCx3QkFBZ0I7bUJBQUEsMEJBQUMsSUFBSSxFQUFFO0FBQ25CLG9CQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztBQUN4QixvQkFBSSxJQUFJLENBQUMsUUFBUSxFQUFFO0FBQ2Ysd0JBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDO2lCQUM3QjthQUNKOztBQUVELGdCQUFRO21CQUFBLGtCQUFDLElBQUksRUFBRTtBQUNYLG9CQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxDQUFDO2FBQzVCOztBQUVELHdCQUFnQjttQkFBQSwwQkFBQyxJQUFJLEVBQUU7QUFDbkIsb0JBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLENBQUM7YUFDNUI7O0FBRUQsc0JBQWM7bUJBQUEsMEJBQUcsRUFBRTs7QUFFbkIseUJBQWlCO21CQUFBLDZCQUFHLEVBQUU7O0FBRXRCLHdCQUFnQjttQkFBQSwwQkFBQyxJQUFJLEVBQUU7QUFDbkIsb0JBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO2FBQ3pCOztBQUVELG9CQUFZO21CQUFBLHNCQUFDLElBQUksRUFBRTs7Ozs7QUFLZixvQkFBSSxJQUFJLENBQUMsSUFBSSxJQUFJLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxLQUFLLE1BQU0sQ0FBQyxtQkFBbUIsSUFBSSxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxLQUFLLEVBQUU7QUFDeEYsd0JBQUksQ0FBQyxZQUFZLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxDQUFDO2lCQUMxQzs7QUFFRCxvQkFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsQ0FBQzs7QUFFekIsb0JBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7YUFDcEI7O0FBRUQsdUJBQWU7bUJBQUEseUJBQUMsSUFBSSxFQUFFO0FBQ2xCLG9CQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxDQUFDO2FBQ3pCOztBQUVELHdCQUFnQjttQkFBQSwwQkFBQyxJQUFJLEVBQUU7QUFDbkIsb0JBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLENBQUM7YUFDekI7O0FBRUQsc0JBQWM7bUJBQUEsd0JBQUMsSUFBSSxFQUFFOztBQUVqQixvQkFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsWUFBWSxFQUFFLElBQUksSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLEtBQUssTUFBTSxDQUFDLFVBQVUsSUFBSSxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksS0FBSyxNQUFNLEVBQUU7OztBQUc1Ryx3QkFBSSxDQUFDLFlBQVksRUFBRSxDQUFDLGFBQWEsQ0FBQyxZQUFZLEVBQUUsQ0FBQztpQkFDcEQ7QUFDRCxvQkFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsQ0FBQzthQUM1Qjs7QUFFRCxzQkFBYzttQkFBQSx3QkFBQyxJQUFJLEVBQUU7QUFDakIsb0JBQUksSUFBSSxDQUFDLFlBQVksQ0FBQyxPQUFPLEVBQUUsRUFBRTtBQUM3Qix3QkFBSSxDQUFDLFlBQVksQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsQ0FBQztpQkFDNUM7O0FBRUQsb0JBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLENBQUM7O0FBRXpCLG9CQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO2FBQ3BCOztBQUVELHNCQUFjO21CQUFBLDBCQUFHO0FBQ2Isb0JBQUksQ0FBQyxZQUFZLEVBQUUsQ0FBQyxhQUFhLENBQUMsWUFBWSxFQUFFLENBQUM7YUFDcEQ7O0FBRUQscUJBQWE7bUJBQUEsdUJBQUMsSUFBSSxFQUFFO0FBQ2hCLG9CQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQzs7QUFFeEIsb0JBQUksQ0FBQyxZQUFZLENBQUMsZUFBZSxDQUFDLElBQUksQ0FBQyxDQUFDOztBQUV4QyxvQkFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7O0FBRXRCLG9CQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO2FBQ3BCOztBQUVELDJCQUFtQjttQkFBQSw2QkFBQyxJQUFJLEVBQUU7QUFDdEIsb0JBQUksbUJBQW1CLEVBQUUsQ0FBQyxFQUFFLEVBQUUsRUFBRSxJQUFJLENBQUM7QUFDckMsbUNBQW1CLEdBQUcsQUFBQyxJQUFJLENBQUMsSUFBSSxLQUFLLEtBQUssR0FBSSxJQUFJLENBQUMsWUFBWSxFQUFFLENBQUMsYUFBYSxHQUFHLElBQUksQ0FBQyxZQUFZLEVBQUUsQ0FBQztBQUN0RyxxQkFBSyxDQUFDLEdBQUcsQ0FBQyxFQUFFLEVBQUUsR0FBRyxJQUFJLENBQUMsWUFBWSxDQUFDLE1BQU0sRUFBRSxDQUFDLEdBQUcsRUFBRSxFQUFFLEVBQUUsQ0FBQyxFQUFFO0FBQ3BELHdCQUFJLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsQ0FBQztBQUM1Qix3QkFBSSxDQUFDLHdCQUF3QixDQUFDLG1CQUFtQixFQUFFLFFBQVEsQ0FBQyxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDO0FBQy9FLHdCQUFJLElBQUksQ0FBQyxJQUFJLEVBQUU7QUFDWCw0QkFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7cUJBQ3pCO2lCQUNKO2FBQ0o7O0FBR0QsdUJBQWU7Ozs7bUJBQUEseUJBQUMsSUFBSSxFQUFFO0FBQ2xCLG9CQUFJLENBQUMsRUFBRSxFQUFFLENBQUM7O0FBRVYsb0JBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDOztBQUU5QixvQkFBSSxJQUFJLENBQUMsWUFBWSxDQUFDLE9BQU8sRUFBRSxFQUFFO0FBQzdCLHdCQUFJLENBQUMsWUFBWSxDQUFDLGlCQUFpQixDQUFDLElBQUksQ0FBQyxDQUFDO2lCQUM3Qzs7QUFFRCxxQkFBSyxDQUFDLEdBQUcsQ0FBQyxFQUFFLEVBQUUsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sRUFBRSxDQUFDLEdBQUcsRUFBRSxFQUFFLEVBQUUsQ0FBQyxFQUFFO0FBQzdDLHdCQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztpQkFDN0I7O0FBRUQsb0JBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7YUFDcEI7O0FBRUQsMkJBQW1CO21CQUFBLDZCQUFDLElBQUksRUFBRTtBQUN0QixvQkFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsQ0FBQzthQUM1Qjs7QUFFRCwwQkFBa0I7bUJBQUEsNEJBQUMsSUFBSSxFQUFFO0FBQ3JCLG9CQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxDQUFDO2FBQzVCOztBQUVELHNCQUFjO21CQUFBLHdCQUFDLElBQUksRUFBRTtBQUNqQixvQkFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsQ0FBQzthQUN6Qjs7QUFFRCxzQkFBYzttQkFBQSx3QkFBQyxJQUFJLEVBQUU7QUFDakIsb0JBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLENBQUM7YUFDekI7O0FBRUQsK0JBQXVCO21CQUFBLGlDQUFDLElBQUksRUFBRTtBQUMxQixvQkFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsQ0FBQzthQUM1Qjs7QUFFRCx5QkFBaUI7bUJBQUEsMkJBQUMsSUFBSSxFQUFFO0FBQ3BCLG9CQUFJLFFBQVEsQ0FBQzs7QUFFYixzQkFBTSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsT0FBTyxFQUFFLElBQUksSUFBSSxDQUFDLFlBQVksQ0FBQyxRQUFRLEVBQUUsRUFBRSxpRkFBaUYsQ0FBQyxDQUFDOztBQUV2Six3QkFBUSxHQUFHLElBQUksUUFBUSxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQztBQUNwQyx3QkFBUSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQzthQUN4Qjs7QUFFRCw4QkFBc0I7bUJBQUEsZ0NBQUMsSUFBSSxFQUFFO0FBQ3pCLG9CQUFJLElBQUksQ0FBQyxNQUFNLEVBQUU7QUFDYiwyQkFBTztpQkFDVjtBQUNELG9CQUFJLElBQUksQ0FBQyxXQUFXLEVBQUU7QUFDbEIsd0JBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxDQUFDO0FBQzdCLDJCQUFPO2lCQUNWOztBQUVELG9CQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxDQUFDO2FBQzVCOztBQUVELHlCQUFpQjttQkFBQSwyQkFBQyxJQUFJLEVBQUU7QUFDcEIsb0JBQUksQ0FBQyxzQkFBc0IsQ0FBQyxJQUFJLENBQUMsQ0FBQzthQUNyQzs7QUFFRCw4QkFBc0I7bUJBQUEsZ0NBQUMsSUFBSSxFQUFFO0FBQ3pCLG9CQUFJLENBQUMsc0JBQXNCLENBQUMsSUFBSSxDQUFDLENBQUM7YUFDckM7O0FBRUQsdUJBQWU7bUJBQUEseUJBQUMsSUFBSSxFQUFFO0FBQ2xCLG9CQUFJLEtBQUssR0FBSSxJQUFJLENBQUMsRUFBRSxJQUFJLElBQUksQ0FBQyxLQUFLLEFBQUMsQ0FBQztBQUNwQyxvQkFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQzthQUNyQjs7OztXQTlkZ0IsVUFBVTtHQUFTLFNBQVMsQ0FBQyxPQUFPOztpQkFBcEMsVUFBVSIsImZpbGUiOiJyZWZlcmVuY2VyLmpzIiwic291cmNlc0NvbnRlbnQiOlsiLypcbiAgQ29weXJpZ2h0IChDKSAyMDE1IFl1c3VrZSBTdXp1a2kgPHV0YXRhbmUudGVhQGdtYWlsLmNvbT5cblxuICBSZWRpc3RyaWJ1dGlvbiBhbmQgdXNlIGluIHNvdXJjZSBhbmQgYmluYXJ5IGZvcm1zLCB3aXRoIG9yIHdpdGhvdXRcbiAgbW9kaWZpY2F0aW9uLCBhcmUgcGVybWl0dGVkIHByb3ZpZGVkIHRoYXQgdGhlIGZvbGxvd2luZyBjb25kaXRpb25zIGFyZSBtZXQ6XG5cbiAgICAqIFJlZGlzdHJpYnV0aW9ucyBvZiBzb3VyY2UgY29kZSBtdXN0IHJldGFpbiB0aGUgYWJvdmUgY29weXJpZ2h0XG4gICAgICBub3RpY2UsIHRoaXMgbGlzdCBvZiBjb25kaXRpb25zIGFuZCB0aGUgZm9sbG93aW5nIGRpc2NsYWltZXIuXG4gICAgKiBSZWRpc3RyaWJ1dGlvbnMgaW4gYmluYXJ5IGZvcm0gbXVzdCByZXByb2R1Y2UgdGhlIGFib3ZlIGNvcHlyaWdodFxuICAgICAgbm90aWNlLCB0aGlzIGxpc3Qgb2YgY29uZGl0aW9ucyBhbmQgdGhlIGZvbGxvd2luZyBkaXNjbGFpbWVyIGluIHRoZVxuICAgICAgZG9jdW1lbnRhdGlvbiBhbmQvb3Igb3RoZXIgbWF0ZXJpYWxzIHByb3ZpZGVkIHdpdGggdGhlIGRpc3RyaWJ1dGlvbi5cblxuICBUSElTIFNPRlRXQVJFIElTIFBST1ZJREVEIEJZIFRIRSBDT1BZUklHSFQgSE9MREVSUyBBTkQgQ09OVFJJQlVUT1JTIFwiQVMgSVNcIlxuICBBTkQgQU5ZIEVYUFJFU1MgT1IgSU1QTElFRCBXQVJSQU5USUVTLCBJTkNMVURJTkcsIEJVVCBOT1QgTElNSVRFRCBUTywgVEhFXG4gIElNUExJRUQgV0FSUkFOVElFUyBPRiBNRVJDSEFOVEFCSUxJVFkgQU5EIEZJVE5FU1MgRk9SIEEgUEFSVElDVUxBUiBQVVJQT1NFXG4gIEFSRSBESVNDTEFJTUVELiBJTiBOTyBFVkVOVCBTSEFMTCA8Q09QWVJJR0hUIEhPTERFUj4gQkUgTElBQkxFIEZPUiBBTllcbiAgRElSRUNULCBJTkRJUkVDVCwgSU5DSURFTlRBTCwgU1BFQ0lBTCwgRVhFTVBMQVJZLCBPUiBDT05TRVFVRU5USUFMIERBTUFHRVNcbiAgKElOQ0xVRElORywgQlVUIE5PVCBMSU1JVEVEIFRPLCBQUk9DVVJFTUVOVCBPRiBTVUJTVElUVVRFIEdPT0RTIE9SIFNFUlZJQ0VTO1xuICBMT1NTIE9GIFVTRSwgREFUQSwgT1IgUFJPRklUUzsgT1IgQlVTSU5FU1MgSU5URVJSVVBUSU9OKSBIT1dFVkVSIENBVVNFRCBBTkRcbiAgT04gQU5ZIFRIRU9SWSBPRiBMSUFCSUxJVFksIFdIRVRIRVIgSU4gQ09OVFJBQ1QsIFNUUklDVCBMSUFCSUxJVFksIE9SIFRPUlRcbiAgKElOQ0xVRElORyBORUdMSUdFTkNFIE9SIE9USEVSV0lTRSkgQVJJU0lORyBJTiBBTlkgV0FZIE9VVCBPRiBUSEUgVVNFIE9GXG4gIFRISVMgU09GVFdBUkUsIEVWRU4gSUYgQURWSVNFRCBPRiBUSEUgUE9TU0lCSUxJVFkgT0YgU1VDSCBEQU1BR0UuXG4qL1xuaW1wb3J0IHsgU3ludGF4IH0gZnJvbSAnZXN0cmF2ZXJzZSc7XG5pbXBvcnQgZXNyZWN1cnNlIGZyb20gJ2VzcmVjdXJzZSc7XG5pbXBvcnQgUmVmZXJlbmNlIGZyb20gJy4vcmVmZXJlbmNlJztcbmltcG9ydCBWYXJpYWJsZSBmcm9tICcuL3ZhcmlhYmxlJztcbmltcG9ydCB7IFBhcmFtZXRlckRlZmluaXRpb24sIERlZmluaXRpb24gfSBmcm9tICcuL2RlZmluaXRpb24nO1xuaW1wb3J0IGFzc2VydCBmcm9tICdhc3NlcnQnO1xuXG5jbGFzcyBQYXR0ZXJuVmlzaXRvciBleHRlbmRzIGVzcmVjdXJzZS5WaXNpdG9yIHtcbiAgICBjb25zdHJ1Y3Rvcihyb290UGF0dGVybiwgY2FsbGJhY2spIHtcbiAgICAgICAgc3VwZXIoKTtcbiAgICAgICAgdGhpcy5yb290UGF0dGVybiA9IHJvb3RQYXR0ZXJuO1xuICAgICAgICB0aGlzLmNhbGxiYWNrID0gY2FsbGJhY2s7XG4gICAgICAgIHRoaXMuYXNzaWdubWVudHMgPSBbXTtcbiAgICAgICAgdGhpcy5yaWdodEhhbmROb2RlcyA9IFtdO1xuICAgICAgICB0aGlzLnJlc3RFbGVtZW50cyA9IFtdO1xuICAgIH1cblxuICAgIElkZW50aWZpZXIocGF0dGVybikge1xuICAgICAgICBjb25zdCBsYXN0UmVzdEVsZW1lbnQgPSBnZXRMYXN0KHRoaXMucmVzdEVsZW1lbnRzKTtcbiAgICAgICAgdGhpcy5jYWxsYmFjayhwYXR0ZXJuLCB7XG4gICAgICAgICAgICB0b3BMZXZlbDogcGF0dGVybiA9PT0gdGhpcy5yb290UGF0dGVybixcbiAgICAgICAgICAgIHJlc3Q6IGxhc3RSZXN0RWxlbWVudCAhPSBudWxsICYmIGxhc3RSZXN0RWxlbWVudC5hcmd1bWVudCA9PT0gcGF0dGVybixcbiAgICAgICAgICAgIGFzc2lnbm1lbnRzOiB0aGlzLmFzc2lnbm1lbnRzXG4gICAgICAgIH0pO1xuICAgIH1cblxuICAgIE9iamVjdFBhdHRlcm4ocGF0dGVybikge1xuICAgICAgICB2YXIgaSwgaXosIHByb3BlcnR5O1xuICAgICAgICBmb3IgKGkgPSAwLCBpeiA9IHBhdHRlcm4ucHJvcGVydGllcy5sZW5ndGg7IGkgPCBpejsgKytpKSB7XG4gICAgICAgICAgICBwcm9wZXJ0eSA9IHBhdHRlcm4ucHJvcGVydGllc1tpXTtcblxuICAgICAgICAgICAgLy8gQ29tcHV0ZWQgcHJvcGVydHkncyBrZXkgaXMgYSByaWdodCBoYW5kIG5vZGUuXG4gICAgICAgICAgICBpZiAocHJvcGVydHkuY29tcHV0ZWQpIHtcbiAgICAgICAgICAgICAgICB0aGlzLnJpZ2h0SGFuZE5vZGVzLnB1c2gocHJvcGVydHkua2V5KTtcbiAgICAgICAgICAgIH1cblxuICAgICAgICAgICAgLy8gSWYgaXQncyBzaG9ydGhhbmQsIGl0cyBrZXkgaXMgc2FtZSBhcyBpdHMgdmFsdWUuXG4gICAgICAgICAgICAvLyBJZiBpdCdzIHNob3J0aGFuZCBhbmQgaGFzIGl0cyBkZWZhdWx0IHZhbHVlLCBpdHMga2V5IGlzIHNhbWUgYXMgaXRzIHZhbHVlLmxlZnQgKHRoZSB2YWx1ZSBpcyBBc3NpZ25tZW50UGF0dGVybikuXG4gICAgICAgICAgICAvLyBJZiBpdCdzIG5vdCBzaG9ydGhhbmQsIHRoZSBuYW1lIG9mIG5ldyB2YXJpYWJsZSBpcyBpdHMgdmFsdWUncy5cbiAgICAgICAgICAgIHRoaXMudmlzaXQocHJvcGVydHkudmFsdWUpO1xuICAgICAgICB9XG4gICAgfVxuXG4gICAgQXJyYXlQYXR0ZXJuKHBhdHRlcm4pIHtcbiAgICAgICAgdmFyIGksIGl6LCBlbGVtZW50O1xuICAgICAgICBmb3IgKGkgPSAwLCBpeiA9IHBhdHRlcm4uZWxlbWVudHMubGVuZ3RoOyBpIDwgaXo7ICsraSkge1xuICAgICAgICAgICAgZWxlbWVudCA9IHBhdHRlcm4uZWxlbWVudHNbaV07XG4gICAgICAgICAgICB0aGlzLnZpc2l0KGVsZW1lbnQpO1xuICAgICAgICB9XG4gICAgfVxuXG4gICAgQXNzaWdubWVudFBhdHRlcm4ocGF0dGVybikge1xuICAgICAgICB0aGlzLmFzc2lnbm1lbnRzLnB1c2gocGF0dGVybik7XG4gICAgICAgIHRoaXMudmlzaXQocGF0dGVybi5sZWZ0KTtcbiAgICAgICAgdGhpcy5yaWdodEhhbmROb2Rlcy5wdXNoKHBhdHRlcm4ucmlnaHQpO1xuICAgICAgICB0aGlzLmFzc2lnbm1lbnRzLnBvcCgpO1xuICAgIH1cblxuICAgIFJlc3RFbGVtZW50KHBhdHRlcm4pIHtcbiAgICAgICAgdGhpcy5yZXN0RWxlbWVudHMucHVzaChwYXR0ZXJuKTtcbiAgICAgICAgdGhpcy52aXNpdChwYXR0ZXJuLmFyZ3VtZW50KTtcbiAgICAgICAgdGhpcy5yZXN0RWxlbWVudHMucG9wKCk7XG4gICAgfVxuXG4gICAgTWVtYmVyRXhwcmVzc2lvbihub2RlKSB7XG4gICAgICAgIC8vIENvbXB1dGVkIHByb3BlcnR5J3Mga2V5IGlzIGEgcmlnaHQgaGFuZCBub2RlLlxuICAgICAgICBpZiAobm9kZS5jb21wdXRlZCkge1xuICAgICAgICAgICAgdGhpcy5yaWdodEhhbmROb2Rlcy5wdXNoKG5vZGUucHJvcGVydHkpO1xuICAgICAgICB9XG4gICAgICAgIC8vIHRoZSBvYmplY3QgaXMgb25seSByZWFkLCB3cml0ZSB0byBpdHMgcHJvcGVydHkuXG4gICAgICAgIHRoaXMucmlnaHRIYW5kTm9kZXMucHVzaChub2RlLm9iamVjdCk7XG4gICAgfVxuXG4gICAgLy9cbiAgICAvLyBGb3JJblN0YXRlbWVudC5sZWZ0IGFuZCBBc3NpZ25tZW50RXhwcmVzc2lvbi5sZWZ0IGFyZSBMZWZ0SGFuZFNpZGVFeHByZXNzaW9uLlxuICAgIC8vIEJ5IHNwZWMsIExlZnRIYW5kU2lkZUV4cHJlc3Npb24gaXMgUGF0dGVybiBvciBNZW1iZXJFeHByZXNzaW9uLlxuICAgIC8vICAgKHNlZSBhbHNvOiBodHRwczovL2dpdGh1Yi5jb20vZXN0cmVlL2VzdHJlZS9wdWxsLzIwI2lzc3VlY29tbWVudC03NDU4NDc1OClcbiAgICAvLyBCdXQgZXNwcmVlIDIuMCBhbmQgZXNwcmltYSAyLjAgcGFyc2UgdG8gQXJyYXlFeHByZXNzaW9uLCBPYmplY3RFeHByZXNzaW9uLCBldGMuLi5cbiAgICAvL1xuXG4gICAgU3ByZWFkRWxlbWVudChub2RlKSB7XG4gICAgICAgIHRoaXMudmlzaXQobm9kZS5hcmd1bWVudCk7XG4gICAgfVxuXG4gICAgQXJyYXlFeHByZXNzaW9uKG5vZGUpIHtcbiAgICAgICAgbm9kZS5lbGVtZW50cy5mb3JFYWNoKHRoaXMudmlzaXQsIHRoaXMpO1xuICAgIH1cblxuICAgIE9iamVjdEV4cHJlc3Npb24obm9kZSkge1xuICAgICAgICBub2RlLnByb3BlcnRpZXMuZm9yRWFjaChwcm9wZXJ0eSA9PiB7XG4gICAgICAgICAgICAvLyBDb21wdXRlZCBwcm9wZXJ0eSdzIGtleSBpcyBhIHJpZ2h0IGhhbmQgbm9kZS5cbiAgICAgICAgICAgIGlmIChwcm9wZXJ0eS5jb21wdXRlZCkge1xuICAgICAgICAgICAgICAgIHRoaXMucmlnaHRIYW5kTm9kZXMucHVzaChwcm9wZXJ0eS5rZXkpO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgdGhpcy52aXNpdChwcm9wZXJ0eS52YWx1ZSk7XG4gICAgICAgIH0pO1xuICAgIH1cblxuICAgIEFzc2lnbm1lbnRFeHByZXNzaW9uKG5vZGUpIHtcbiAgICAgICAgdGhpcy5hc3NpZ25tZW50cy5wdXNoKG5vZGUpO1xuICAgICAgICB0aGlzLnZpc2l0KG5vZGUubGVmdCk7XG4gICAgICAgIHRoaXMucmlnaHRIYW5kTm9kZXMucHVzaChub2RlLnJpZ2h0KTtcbiAgICAgICAgdGhpcy5hc3NpZ25tZW50cy5wb3AoKTtcbiAgICB9XG5cbiAgICBDYWxsRXhwcmVzc2lvbihub2RlKSB7XG4gICAgICAgIC8vIGFyZ3VtZW50cyBhcmUgcmlnaHQgaGFuZCBub2Rlcy5cbiAgICAgICAgbm9kZS5hcmd1bWVudHMuZm9yRWFjaChhID0+IHsgdGhpcy5yaWdodEhhbmROb2Rlcy5wdXNoKGEpOyB9KTtcbiAgICAgICAgdGhpcy52aXNpdChub2RlLmNhbGxlZSk7XG4gICAgfVxufVxuXG5mdW5jdGlvbiBnZXRMYXN0KHhzKSB7XG4gICAgcmV0dXJuIHhzW3hzLmxlbmd0aCAtIDFdIHx8IG51bGw7XG59XG5cbmZ1bmN0aW9uIHRyYXZlcnNlSWRlbnRpZmllckluUGF0dGVybihyb290UGF0dGVybiwgcmVmZXJlbmNlciwgY2FsbGJhY2spIHtcbiAgICAvLyBDYWxsIHRoZSBjYWxsYmFjayBhdCBsZWZ0IGhhbmQgaWRlbnRpZmllciBub2RlcywgYW5kIENvbGxlY3QgcmlnaHQgaGFuZCBub2Rlcy5cbiAgICB2YXIgdmlzaXRvciA9IG5ldyBQYXR0ZXJuVmlzaXRvcihyb290UGF0dGVybiwgY2FsbGJhY2spO1xuICAgIHZpc2l0b3IudmlzaXQocm9vdFBhdHRlcm4pO1xuXG4gICAgLy8gUHJvY2VzcyB0aGUgcmlnaHQgaGFuZCBub2RlcyByZWN1cnNpdmVseS5cbiAgICBpZiAocmVmZXJlbmNlciAhPSBudWxsKSB7XG4gICAgICAgIHZpc2l0b3IucmlnaHRIYW5kTm9kZXMuZm9yRWFjaChyZWZlcmVuY2VyLnZpc2l0LCByZWZlcmVuY2VyKTtcbiAgICB9XG59XG5cbmZ1bmN0aW9uIGlzUGF0dGVybihub2RlKSB7XG4gICAgdmFyIG5vZGVUeXBlID0gbm9kZS50eXBlO1xuICAgIHJldHVybiAoXG4gICAgICAgIG5vZGVUeXBlID09PSBTeW50YXguSWRlbnRpZmllciB8fFxuICAgICAgICBub2RlVHlwZSA9PT0gU3ludGF4Lk9iamVjdFBhdHRlcm4gfHxcbiAgICAgICAgbm9kZVR5cGUgPT09IFN5bnRheC5BcnJheVBhdHRlcm4gfHxcbiAgICAgICAgbm9kZVR5cGUgPT09IFN5bnRheC5TcHJlYWRFbGVtZW50IHx8XG4gICAgICAgIG5vZGVUeXBlID09PSBTeW50YXguUmVzdEVsZW1lbnQgfHxcbiAgICAgICAgbm9kZVR5cGUgPT09IFN5bnRheC5Bc3NpZ25tZW50UGF0dGVyblxuICAgICk7XG59XG5cbi8vIEltcG9ydGluZyBJbXBvcnREZWNsYXJhdGlvbi5cbi8vIGh0dHA6Ly9wZW9wbGUubW96aWxsYS5vcmcvfmpvcmVuZG9yZmYvZXM2LWRyYWZ0Lmh0bWwjc2VjLW1vZHVsZWRlY2xhcmF0aW9uaW5zdGFudGlhdGlvblxuLy8gaHR0cHM6Ly9naXRodWIuY29tL2VzdHJlZS9lc3RyZWUvYmxvYi9tYXN0ZXIvZXM2Lm1kI2ltcG9ydGRlY2xhcmF0aW9uXG4vLyBGSVhNRTogTm93LCB3ZSBkb24ndCBjcmVhdGUgbW9kdWxlIGVudmlyb25tZW50LCBiZWNhdXNlIHRoZSBjb250ZXh0IGlzXG4vLyBpbXBsZW1lbnRhdGlvbiBkZXBlbmRlbnQuXG5cbmNsYXNzIEltcG9ydGVyIGV4dGVuZHMgZXNyZWN1cnNlLlZpc2l0b3Ige1xuICAgIGNvbnN0cnVjdG9yKGRlY2xhcmF0aW9uLCByZWZlcmVuY2VyKSB7XG4gICAgICAgIHN1cGVyKCk7XG4gICAgICAgIHRoaXMuZGVjbGFyYXRpb24gPSBkZWNsYXJhdGlvbjtcbiAgICAgICAgdGhpcy5yZWZlcmVuY2VyID0gcmVmZXJlbmNlcjtcbiAgICB9XG5cbiAgICB2aXNpdEltcG9ydChpZCwgc3BlY2lmaWVyKSB7XG4gICAgICAgIHRoaXMucmVmZXJlbmNlci52aXNpdFBhdHRlcm4oaWQsIChwYXR0ZXJuKSA9PiB7XG4gICAgICAgICAgICB0aGlzLnJlZmVyZW5jZXIuY3VycmVudFNjb3BlKCkuX19kZWZpbmUocGF0dGVybixcbiAgICAgICAgICAgICAgICBuZXcgRGVmaW5pdGlvbihcbiAgICAgICAgICAgICAgICAgICAgVmFyaWFibGUuSW1wb3J0QmluZGluZyxcbiAgICAgICAgICAgICAgICAgICAgcGF0dGVybixcbiAgICAgICAgICAgICAgICAgICAgc3BlY2lmaWVyLFxuICAgICAgICAgICAgICAgICAgICB0aGlzLmRlY2xhcmF0aW9uLFxuICAgICAgICAgICAgICAgICAgICBudWxsLFxuICAgICAgICAgICAgICAgICAgICBudWxsXG4gICAgICAgICAgICAgICAgICAgICkpO1xuICAgICAgICB9KTtcbiAgICB9XG5cbiAgICBJbXBvcnROYW1lc3BhY2VTcGVjaWZpZXIobm9kZSkge1xuICAgICAgICBsZXQgbG9jYWwgPSAobm9kZS5sb2NhbCB8fCBub2RlLmlkKTtcbiAgICAgICAgaWYgKGxvY2FsKSB7XG4gICAgICAgICAgICB0aGlzLnZpc2l0SW1wb3J0KGxvY2FsLCBub2RlKTtcbiAgICAgICAgfVxuICAgIH1cblxuICAgIEltcG9ydERlZmF1bHRTcGVjaWZpZXIobm9kZSkge1xuICAgICAgICBsZXQgbG9jYWwgPSAobm9kZS5sb2NhbCB8fCBub2RlLmlkKTtcbiAgICAgICAgdGhpcy52aXNpdEltcG9ydChsb2NhbCwgbm9kZSk7XG4gICAgfVxuXG4gICAgSW1wb3J0U3BlY2lmaWVyKG5vZGUpIHtcbiAgICAgICAgbGV0IGxvY2FsID0gKG5vZGUubG9jYWwgfHwgbm9kZS5pZCk7XG4gICAgICAgIGlmIChub2RlLm5hbWUpIHtcbiAgICAgICAgICAgIHRoaXMudmlzaXRJbXBvcnQobm9kZS5uYW1lLCBub2RlKTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIHRoaXMudmlzaXRJbXBvcnQobG9jYWwsIG5vZGUpO1xuICAgICAgICB9XG4gICAgfVxufVxuXG4vLyBSZWZlcmVuY2luZyB2YXJpYWJsZXMgYW5kIGNyZWF0aW5nIGJpbmRpbmdzLlxuZXhwb3J0IGRlZmF1bHQgY2xhc3MgUmVmZXJlbmNlciBleHRlbmRzIGVzcmVjdXJzZS5WaXNpdG9yIHtcbiAgICBjb25zdHJ1Y3RvcihzY29wZU1hbmFnZXIpIHtcbiAgICAgICAgc3VwZXIoKTtcbiAgICAgICAgdGhpcy5zY29wZU1hbmFnZXIgPSBzY29wZU1hbmFnZXI7XG4gICAgICAgIHRoaXMucGFyZW50ID0gbnVsbDtcbiAgICAgICAgdGhpcy5pc0lubmVyTWV0aG9kRGVmaW5pdGlvbiA9IGZhbHNlO1xuICAgIH1cblxuICAgIGN1cnJlbnRTY29wZSgpIHtcbiAgICAgICAgcmV0dXJuIHRoaXMuc2NvcGVNYW5hZ2VyLl9fY3VycmVudFNjb3BlO1xuICAgIH1cblxuICAgIGNsb3NlKG5vZGUpIHtcbiAgICAgICAgd2hpbGUgKHRoaXMuY3VycmVudFNjb3BlKCkgJiYgbm9kZSA9PT0gdGhpcy5jdXJyZW50U2NvcGUoKS5ibG9jaykge1xuICAgICAgICAgICAgdGhpcy5zY29wZU1hbmFnZXIuX19jdXJyZW50U2NvcGUgPSB0aGlzLmN1cnJlbnRTY29wZSgpLl9fY2xvc2UodGhpcy5zY29wZU1hbmFnZXIpO1xuICAgICAgICB9XG4gICAgfVxuXG4gICAgcHVzaElubmVyTWV0aG9kRGVmaW5pdGlvbihpc0lubmVyTWV0aG9kRGVmaW5pdGlvbikge1xuICAgICAgICB2YXIgcHJldmlvdXMgPSB0aGlzLmlzSW5uZXJNZXRob2REZWZpbml0aW9uO1xuICAgICAgICB0aGlzLmlzSW5uZXJNZXRob2REZWZpbml0aW9uID0gaXNJbm5lck1ldGhvZERlZmluaXRpb247XG4gICAgICAgIHJldHVybiBwcmV2aW91cztcbiAgICB9XG5cbiAgICBwb3BJbm5lck1ldGhvZERlZmluaXRpb24oaXNJbm5lck1ldGhvZERlZmluaXRpb24pIHtcbiAgICAgICAgdGhpcy5pc0lubmVyTWV0aG9kRGVmaW5pdGlvbiA9IGlzSW5uZXJNZXRob2REZWZpbml0aW9uO1xuICAgIH1cblxuICAgIG1hdGVyaWFsaXplVERaU2NvcGUobm9kZSwgaXRlcmF0aW9uTm9kZSkge1xuICAgICAgICAvLyBodHRwOi8vcGVvcGxlLm1vemlsbGEub3JnL35qb3JlbmRvcmZmL2VzNi1kcmFmdC5odG1sI3NlYy1ydW50aW1lLXNlbWFudGljcy1mb3Jpbi1kaXYtb2ZleHByZXNzaW9uZXZhbHVhdGlvbi1hYnN0cmFjdC1vcGVyYXRpb25cbiAgICAgICAgLy8gVERaIHNjb3BlIGhpZGVzIHRoZSBkZWNsYXJhdGlvbidzIG5hbWVzLlxuICAgICAgICB0aGlzLnNjb3BlTWFuYWdlci5fX25lc3RURFpTY29wZShub2RlLCBpdGVyYXRpb25Ob2RlKTtcbiAgICAgICAgdGhpcy52aXNpdFZhcmlhYmxlRGVjbGFyYXRpb24odGhpcy5jdXJyZW50U2NvcGUoKSwgVmFyaWFibGUuVERaLCBpdGVyYXRpb25Ob2RlLmxlZnQsIDAsIHRydWUpO1xuICAgIH1cblxuICAgIG1hdGVyaWFsaXplSXRlcmF0aW9uU2NvcGUobm9kZSkge1xuICAgICAgICAvLyBHZW5lcmF0ZSBpdGVyYXRpb24gc2NvcGUgZm9yIHVwcGVyIEZvckluL0Zvck9mIFN0YXRlbWVudHMuXG4gICAgICAgIHZhciBsZXRPckNvbnN0RGVjbDtcbiAgICAgICAgdGhpcy5zY29wZU1hbmFnZXIuX19uZXN0Rm9yU2NvcGUobm9kZSk7XG4gICAgICAgIGxldE9yQ29uc3REZWNsID0gbm9kZS5sZWZ0O1xuICAgICAgICB0aGlzLnZpc2l0VmFyaWFibGVEZWNsYXJhdGlvbih0aGlzLmN1cnJlbnRTY29wZSgpLCBWYXJpYWJsZS5WYXJpYWJsZSwgbGV0T3JDb25zdERlY2wsIDApO1xuICAgICAgICB0aGlzLnZpc2l0UGF0dGVybihsZXRPckNvbnN0RGVjbC5kZWNsYXJhdGlvbnNbMF0uaWQsIChwYXR0ZXJuKSA9PiB7XG4gICAgICAgICAgICB0aGlzLmN1cnJlbnRTY29wZSgpLl9fcmVmZXJlbmNpbmcocGF0dGVybiwgUmVmZXJlbmNlLldSSVRFLCBub2RlLnJpZ2h0LCBudWxsLCB0cnVlLCB0cnVlKTtcbiAgICAgICAgfSk7XG4gICAgfVxuXG4gICAgcmVmZXJlbmNpbmdEZWZhdWx0VmFsdWUocGF0dGVybiwgYXNzaWdubWVudHMsIG1heWJlSW1wbGljaXRHbG9iYWwsIGluaXQpIHtcbiAgICAgICAgY29uc3Qgc2NvcGUgPSB0aGlzLmN1cnJlbnRTY29wZSgpO1xuICAgICAgICBhc3NpZ25tZW50cy5mb3JFYWNoKGFzc2lnbm1lbnQgPT4ge1xuICAgICAgICAgICAgc2NvcGUuX19yZWZlcmVuY2luZyhcbiAgICAgICAgICAgICAgICBwYXR0ZXJuLFxuICAgICAgICAgICAgICAgIFJlZmVyZW5jZS5XUklURSxcbiAgICAgICAgICAgICAgICBhc3NpZ25tZW50LnJpZ2h0LFxuICAgICAgICAgICAgICAgIG1heWJlSW1wbGljaXRHbG9iYWwsXG4gICAgICAgICAgICAgICAgcGF0dGVybiAhPT0gYXNzaWdubWVudC5sZWZ0LFxuICAgICAgICAgICAgICAgIGluaXQpO1xuICAgICAgICB9KTtcbiAgICB9XG5cbiAgICB2aXNpdFBhdHRlcm4obm9kZSwgb3B0aW9ucywgY2FsbGJhY2spIHtcbiAgICAgICAgaWYgKHR5cGVvZiBvcHRpb25zID09PSAnZnVuY3Rpb24nKSB7XG4gICAgICAgICAgICBjYWxsYmFjayA9IG9wdGlvbnM7XG4gICAgICAgICAgICBvcHRpb25zID0ge3Byb2Nlc3NSaWdodEhhbmROb2RlczogZmFsc2V9XG4gICAgICAgIH1cbiAgICAgICAgdHJhdmVyc2VJZGVudGlmaWVySW5QYXR0ZXJuKFxuICAgICAgICAgICAgbm9kZSxcbiAgICAgICAgICAgIG9wdGlvbnMucHJvY2Vzc1JpZ2h0SGFuZE5vZGVzID8gdGhpcyA6IG51bGwsXG4gICAgICAgICAgICBjYWxsYmFjayk7XG4gICAgfVxuXG4gICAgdmlzaXRGdW5jdGlvbihub2RlKSB7XG4gICAgICAgIHZhciBpLCBpejtcbiAgICAgICAgLy8gRnVuY3Rpb25EZWNsYXJhdGlvbiBuYW1lIGlzIGRlZmluZWQgaW4gdXBwZXIgc2NvcGVcbiAgICAgICAgLy8gTk9URTogTm90IHJlZmVycmluZyB2YXJpYWJsZVNjb3BlLiBJdCBpcyBpbnRlbmRlZC5cbiAgICAgICAgLy8gU2luY2VcbiAgICAgICAgLy8gIGluIEVTNSwgRnVuY3Rpb25EZWNsYXJhdGlvbiBzaG91bGQgYmUgaW4gRnVuY3Rpb25Cb2R5LlxuICAgICAgICAvLyAgaW4gRVM2LCBGdW5jdGlvbkRlY2xhcmF0aW9uIHNob3VsZCBiZSBibG9jayBzY29wZWQuXG4gICAgICAgIGlmIChub2RlLnR5cGUgPT09IFN5bnRheC5GdW5jdGlvbkRlY2xhcmF0aW9uKSB7XG4gICAgICAgICAgICAvLyBpZCBpcyBkZWZpbmVkIGluIHVwcGVyIHNjb3BlXG4gICAgICAgICAgICB0aGlzLmN1cnJlbnRTY29wZSgpLl9fZGVmaW5lKG5vZGUuaWQsXG4gICAgICAgICAgICAgICAgICAgIG5ldyBEZWZpbml0aW9uKFxuICAgICAgICAgICAgICAgICAgICAgICAgVmFyaWFibGUuRnVuY3Rpb25OYW1lLFxuICAgICAgICAgICAgICAgICAgICAgICAgbm9kZS5pZCxcbiAgICAgICAgICAgICAgICAgICAgICAgIG5vZGUsXG4gICAgICAgICAgICAgICAgICAgICAgICBudWxsLFxuICAgICAgICAgICAgICAgICAgICAgICAgbnVsbCxcbiAgICAgICAgICAgICAgICAgICAgICAgIG51bGxcbiAgICAgICAgICAgICAgICAgICAgKSk7XG4gICAgICAgIH1cblxuICAgICAgICAvLyBGdW5jdGlvbkV4cHJlc3Npb24gd2l0aCBuYW1lIGNyZWF0ZXMgaXRzIHNwZWNpYWwgc2NvcGU7XG4gICAgICAgIC8vIEZ1bmN0aW9uRXhwcmVzc2lvbk5hbWVTY29wZS5cbiAgICAgICAgaWYgKG5vZGUudHlwZSA9PT0gU3ludGF4LkZ1bmN0aW9uRXhwcmVzc2lvbiAmJiBub2RlLmlkKSB7XG4gICAgICAgICAgICB0aGlzLnNjb3BlTWFuYWdlci5fX25lc3RGdW5jdGlvbkV4cHJlc3Npb25OYW1lU2NvcGUobm9kZSk7XG4gICAgICAgIH1cblxuICAgICAgICAvLyBDb25zaWRlciB0aGlzIGZ1bmN0aW9uIGlzIGluIHRoZSBNZXRob2REZWZpbml0aW9uLlxuICAgICAgICB0aGlzLnNjb3BlTWFuYWdlci5fX25lc3RGdW5jdGlvblNjb3BlKG5vZGUsIHRoaXMuaXNJbm5lck1ldGhvZERlZmluaXRpb24pO1xuXG4gICAgICAgIC8vIFByb2Nlc3MgcGFyYW1ldGVyIGRlY2xhcmF0aW9ucy5cbiAgICAgICAgZm9yIChpID0gMCwgaXogPSBub2RlLnBhcmFtcy5sZW5ndGg7IGkgPCBpejsgKytpKSB7XG4gICAgICAgICAgICB0aGlzLnZpc2l0UGF0dGVybihub2RlLnBhcmFtc1tpXSwge3Byb2Nlc3NSaWdodEhhbmROb2RlczogdHJ1ZX0sIChwYXR0ZXJuLCBpbmZvKSA9PiB7XG4gICAgICAgICAgICAgICAgdGhpcy5jdXJyZW50U2NvcGUoKS5fX2RlZmluZShwYXR0ZXJuLFxuICAgICAgICAgICAgICAgICAgICBuZXcgUGFyYW1ldGVyRGVmaW5pdGlvbihcbiAgICAgICAgICAgICAgICAgICAgICAgIHBhdHRlcm4sXG4gICAgICAgICAgICAgICAgICAgICAgICBub2RlLFxuICAgICAgICAgICAgICAgICAgICAgICAgaSxcbiAgICAgICAgICAgICAgICAgICAgICAgIGluZm8ucmVzdFxuICAgICAgICAgICAgICAgICAgICApKTtcblxuICAgICAgICAgICAgICAgIHRoaXMucmVmZXJlbmNpbmdEZWZhdWx0VmFsdWUocGF0dGVybiwgaW5mby5hc3NpZ25tZW50cywgbnVsbCwgdHJ1ZSk7XG4gICAgICAgICAgICB9KTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8vIGlmIHRoZXJlJ3MgYSByZXN0IGFyZ3VtZW50LCBhZGQgdGhhdFxuICAgICAgICBpZiAobm9kZS5yZXN0KSB7XG4gICAgICAgICAgICB0aGlzLnZpc2l0UGF0dGVybih7XG4gICAgICAgICAgICAgICAgdHlwZTogJ1Jlc3RFbGVtZW50JyxcbiAgICAgICAgICAgICAgICBhcmd1bWVudDogbm9kZS5yZXN0XG4gICAgICAgICAgICB9LCAocGF0dGVybikgPT4ge1xuICAgICAgICAgICAgICAgIHRoaXMuY3VycmVudFNjb3BlKCkuX19kZWZpbmUocGF0dGVybixcbiAgICAgICAgICAgICAgICAgICAgbmV3IFBhcmFtZXRlckRlZmluaXRpb24oXG4gICAgICAgICAgICAgICAgICAgICAgICBwYXR0ZXJuLFxuICAgICAgICAgICAgICAgICAgICAgICAgbm9kZSxcbiAgICAgICAgICAgICAgICAgICAgICAgIG5vZGUucGFyYW1zLmxlbmd0aCxcbiAgICAgICAgICAgICAgICAgICAgICAgIHRydWVcbiAgICAgICAgICAgICAgICAgICAgKSk7XG4gICAgICAgICAgICB9KTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8vIFNraXAgQmxvY2tTdGF0ZW1lbnQgdG8gcHJldmVudCBjcmVhdGluZyBCbG9ja1N0YXRlbWVudCBzY29wZS5cbiAgICAgICAgaWYgKG5vZGUuYm9keS50eXBlID09PSBTeW50YXguQmxvY2tTdGF0ZW1lbnQpIHtcbiAgICAgICAgICAgIHRoaXMudmlzaXRDaGlsZHJlbihub2RlLmJvZHkpO1xuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgdGhpcy52aXNpdChub2RlLmJvZHkpO1xuICAgICAgICB9XG5cbiAgICAgICAgdGhpcy5jbG9zZShub2RlKTtcbiAgICB9XG5cbiAgICB2aXNpdENsYXNzKG5vZGUpIHtcbiAgICAgICAgaWYgKG5vZGUudHlwZSA9PT0gU3ludGF4LkNsYXNzRGVjbGFyYXRpb24pIHtcbiAgICAgICAgICAgIHRoaXMuY3VycmVudFNjb3BlKCkuX19kZWZpbmUobm9kZS5pZCxcbiAgICAgICAgICAgICAgICAgICAgbmV3IERlZmluaXRpb24oXG4gICAgICAgICAgICAgICAgICAgICAgICBWYXJpYWJsZS5DbGFzc05hbWUsXG4gICAgICAgICAgICAgICAgICAgICAgICBub2RlLmlkLFxuICAgICAgICAgICAgICAgICAgICAgICAgbm9kZSxcbiAgICAgICAgICAgICAgICAgICAgICAgIG51bGwsXG4gICAgICAgICAgICAgICAgICAgICAgICBudWxsLFxuICAgICAgICAgICAgICAgICAgICAgICAgbnVsbFxuICAgICAgICAgICAgICAgICAgICApKTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8vIEZJWE1FOiBNYXliZSBjb25zaWRlciBURFouXG4gICAgICAgIHRoaXMudmlzaXQobm9kZS5zdXBlckNsYXNzKTtcblxuICAgICAgICB0aGlzLnNjb3BlTWFuYWdlci5fX25lc3RDbGFzc1Njb3BlKG5vZGUpO1xuXG4gICAgICAgIGlmIChub2RlLmlkKSB7XG4gICAgICAgICAgICB0aGlzLmN1cnJlbnRTY29wZSgpLl9fZGVmaW5lKG5vZGUuaWQsXG4gICAgICAgICAgICAgICAgICAgIG5ldyBEZWZpbml0aW9uKFxuICAgICAgICAgICAgICAgICAgICAgICAgVmFyaWFibGUuQ2xhc3NOYW1lLFxuICAgICAgICAgICAgICAgICAgICAgICAgbm9kZS5pZCxcbiAgICAgICAgICAgICAgICAgICAgICAgIG5vZGVcbiAgICAgICAgICAgICAgICAgICAgKSk7XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy52aXNpdChub2RlLmJvZHkpO1xuXG4gICAgICAgIHRoaXMuY2xvc2Uobm9kZSk7XG4gICAgfVxuXG4gICAgdmlzaXRQcm9wZXJ0eShub2RlKSB7XG4gICAgICAgIHZhciBwcmV2aW91cywgaXNNZXRob2REZWZpbml0aW9uO1xuICAgICAgICBpZiAobm9kZS5jb21wdXRlZCkge1xuICAgICAgICAgICAgdGhpcy52aXNpdChub2RlLmtleSk7XG4gICAgICAgIH1cblxuICAgICAgICBpc01ldGhvZERlZmluaXRpb24gPSBub2RlLnR5cGUgPT09IFN5bnRheC5NZXRob2REZWZpbml0aW9uIHx8IG5vZGUubWV0aG9kO1xuICAgICAgICBpZiAoaXNNZXRob2REZWZpbml0aW9uKSB7XG4gICAgICAgICAgICBwcmV2aW91cyA9IHRoaXMucHVzaElubmVyTWV0aG9kRGVmaW5pdGlvbih0cnVlKTtcbiAgICAgICAgfVxuICAgICAgICB0aGlzLnZpc2l0KG5vZGUudmFsdWUpO1xuICAgICAgICBpZiAoaXNNZXRob2REZWZpbml0aW9uKSB7XG4gICAgICAgICAgICB0aGlzLnBvcElubmVyTWV0aG9kRGVmaW5pdGlvbihwcmV2aW91cyk7XG4gICAgICAgIH1cbiAgICB9XG5cbiAgICB2aXNpdEZvckluKG5vZGUpIHtcbiAgICAgICAgaWYgKG5vZGUubGVmdC50eXBlID09PSBTeW50YXguVmFyaWFibGVEZWNsYXJhdGlvbiAmJiBub2RlLmxlZnQua2luZCAhPT0gJ3ZhcicpIHtcbiAgICAgICAgICAgIHRoaXMubWF0ZXJpYWxpemVURFpTY29wZShub2RlLnJpZ2h0LCBub2RlKTtcbiAgICAgICAgICAgIHRoaXMudmlzaXQobm9kZS5yaWdodCk7XG4gICAgICAgICAgICB0aGlzLmNsb3NlKG5vZGUucmlnaHQpO1xuXG4gICAgICAgICAgICB0aGlzLm1hdGVyaWFsaXplSXRlcmF0aW9uU2NvcGUobm9kZSk7XG4gICAgICAgICAgICB0aGlzLnZpc2l0KG5vZGUuYm9keSk7XG4gICAgICAgICAgICB0aGlzLmNsb3NlKG5vZGUpO1xuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgaWYgKG5vZGUubGVmdC50eXBlID09PSBTeW50YXguVmFyaWFibGVEZWNsYXJhdGlvbikge1xuICAgICAgICAgICAgICAgIHRoaXMudmlzaXQobm9kZS5sZWZ0KTtcbiAgICAgICAgICAgICAgICB0aGlzLnZpc2l0UGF0dGVybihub2RlLmxlZnQuZGVjbGFyYXRpb25zWzBdLmlkLCAocGF0dGVybikgPT4ge1xuICAgICAgICAgICAgICAgICAgICB0aGlzLmN1cnJlbnRTY29wZSgpLl9fcmVmZXJlbmNpbmcocGF0dGVybiwgUmVmZXJlbmNlLldSSVRFLCBub2RlLnJpZ2h0LCBudWxsLCB0cnVlLCB0cnVlKTtcbiAgICAgICAgICAgICAgICB9KTtcbiAgICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICAgICAgdGhpcy52aXNpdFBhdHRlcm4obm9kZS5sZWZ0LCB7cHJvY2Vzc1JpZ2h0SGFuZE5vZGVzOiB0cnVlfSwgKHBhdHRlcm4sIGluZm8pID0+IHtcbiAgICAgICAgICAgICAgICAgICAgdmFyIG1heWJlSW1wbGljaXRHbG9iYWwgPSBudWxsO1xuICAgICAgICAgICAgICAgICAgICBpZiAoIXRoaXMuY3VycmVudFNjb3BlKCkuaXNTdHJpY3QpIHtcbiAgICAgICAgICAgICAgICAgICAgICAgIG1heWJlSW1wbGljaXRHbG9iYWwgPSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgcGF0dGVybjogcGF0dGVybixcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBub2RlOiBub2RlXG4gICAgICAgICAgICAgICAgICAgICAgICB9O1xuICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgICAgIHRoaXMucmVmZXJlbmNpbmdEZWZhdWx0VmFsdWUocGF0dGVybiwgaW5mby5hc3NpZ25tZW50cywgbWF5YmVJbXBsaWNpdEdsb2JhbCwgZmFsc2UpO1xuICAgICAgICAgICAgICAgICAgICB0aGlzLmN1cnJlbnRTY29wZSgpLl9fcmVmZXJlbmNpbmcocGF0dGVybiwgUmVmZXJlbmNlLldSSVRFLCBub2RlLnJpZ2h0LCBtYXliZUltcGxpY2l0R2xvYmFsLCB0cnVlLCBmYWxzZSk7XG4gICAgICAgICAgICAgICAgfSk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICB0aGlzLnZpc2l0KG5vZGUucmlnaHQpO1xuICAgICAgICAgICAgdGhpcy52aXNpdChub2RlLmJvZHkpO1xuICAgICAgICB9XG4gICAgfVxuXG4gICAgdmlzaXRWYXJpYWJsZURlY2xhcmF0aW9uKHZhcmlhYmxlVGFyZ2V0U2NvcGUsIHR5cGUsIG5vZGUsIGluZGV4LCBmcm9tVERaKSB7XG4gICAgICAgIC8vIElmIHRoaXMgd2FzIGNhbGxlZCB0byBpbml0aWFsaXplIGEgVERaIHNjb3BlLCB0aGlzIG5lZWRzIHRvIG1ha2UgZGVmaW5pdGlvbnMsIGJ1dCBkb2Vzbid0IG1ha2UgcmVmZXJlbmNlcy5cbiAgICAgICAgdmFyIGRlY2wsIGluaXQ7XG5cbiAgICAgICAgZGVjbCA9IG5vZGUuZGVjbGFyYXRpb25zW2luZGV4XTtcbiAgICAgICAgaW5pdCA9IGRlY2wuaW5pdDtcbiAgICAgICAgdGhpcy52aXNpdFBhdHRlcm4oZGVjbC5pZCwge3Byb2Nlc3NSaWdodEhhbmROb2RlczogIWZyb21URFp9LCAocGF0dGVybiwgaW5mbykgPT4ge1xuICAgICAgICAgICAgdmFyaWFibGVUYXJnZXRTY29wZS5fX2RlZmluZShwYXR0ZXJuLFxuICAgICAgICAgICAgICAgIG5ldyBEZWZpbml0aW9uKFxuICAgICAgICAgICAgICAgICAgICB0eXBlLFxuICAgICAgICAgICAgICAgICAgICBwYXR0ZXJuLFxuICAgICAgICAgICAgICAgICAgICBkZWNsLFxuICAgICAgICAgICAgICAgICAgICBub2RlLFxuICAgICAgICAgICAgICAgICAgICBpbmRleCxcbiAgICAgICAgICAgICAgICAgICAgbm9kZS5raW5kXG4gICAgICAgICAgICAgICAgKSk7XG5cbiAgICAgICAgICAgIGlmICghZnJvbVREWikge1xuICAgICAgICAgICAgICAgIHRoaXMucmVmZXJlbmNpbmdEZWZhdWx0VmFsdWUocGF0dGVybiwgaW5mby5hc3NpZ25tZW50cywgbnVsbCwgdHJ1ZSk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBpZiAoaW5pdCkge1xuICAgICAgICAgICAgICAgIHRoaXMuY3VycmVudFNjb3BlKCkuX19yZWZlcmVuY2luZyhwYXR0ZXJuLCBSZWZlcmVuY2UuV1JJVEUsIGluaXQsIG51bGwsICFpbmZvLnRvcExldmVsLCB0cnVlKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfSk7XG4gICAgfVxuXG4gICAgQXNzaWdubWVudEV4cHJlc3Npb24obm9kZSkge1xuICAgICAgICBpZiAoaXNQYXR0ZXJuKG5vZGUubGVmdCkpIHtcbiAgICAgICAgICAgIGlmIChub2RlLm9wZXJhdG9yID09PSAnPScpIHtcbiAgICAgICAgICAgICAgICB0aGlzLnZpc2l0UGF0dGVybihub2RlLmxlZnQsIHtwcm9jZXNzUmlnaHRIYW5kTm9kZXM6IHRydWV9LCAocGF0dGVybiwgaW5mbykgPT4ge1xuICAgICAgICAgICAgICAgICAgICB2YXIgbWF5YmVJbXBsaWNpdEdsb2JhbCA9IG51bGw7XG4gICAgICAgICAgICAgICAgICAgIGlmICghdGhpcy5jdXJyZW50U2NvcGUoKS5pc1N0cmljdCkge1xuICAgICAgICAgICAgICAgICAgICAgICAgbWF5YmVJbXBsaWNpdEdsb2JhbCA9IHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBwYXR0ZXJuOiBwYXR0ZXJuLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgIG5vZGU6IG5vZGVcbiAgICAgICAgICAgICAgICAgICAgICAgIH07XG4gICAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICAgICAgdGhpcy5yZWZlcmVuY2luZ0RlZmF1bHRWYWx1ZShwYXR0ZXJuLCBpbmZvLmFzc2lnbm1lbnRzLCBtYXliZUltcGxpY2l0R2xvYmFsLCBmYWxzZSk7XG4gICAgICAgICAgICAgICAgICAgIHRoaXMuY3VycmVudFNjb3BlKCkuX19yZWZlcmVuY2luZyhwYXR0ZXJuLCBSZWZlcmVuY2UuV1JJVEUsIG5vZGUucmlnaHQsIG1heWJlSW1wbGljaXRHbG9iYWwsICFpbmZvLnRvcExldmVsLCBmYWxzZSk7XG4gICAgICAgICAgICAgICAgfSk7XG4gICAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgICAgIHRoaXMuY3VycmVudFNjb3BlKCkuX19yZWZlcmVuY2luZyhub2RlLmxlZnQsIFJlZmVyZW5jZS5SVywgbm9kZS5yaWdodCk7XG4gICAgICAgICAgICB9XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICB0aGlzLnZpc2l0KG5vZGUubGVmdCk7XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy52aXNpdChub2RlLnJpZ2h0KTtcbiAgICB9XG5cbiAgICBDYXRjaENsYXVzZShub2RlKSB7XG4gICAgICAgIHRoaXMuc2NvcGVNYW5hZ2VyLl9fbmVzdENhdGNoU2NvcGUobm9kZSk7XG5cbiAgICAgICAgdGhpcy52aXNpdFBhdHRlcm4obm9kZS5wYXJhbSwge3Byb2Nlc3NSaWdodEhhbmROb2RlczogdHJ1ZX0sIChwYXR0ZXJuLCBpbmZvKSA9PiB7XG4gICAgICAgICAgICB0aGlzLmN1cnJlbnRTY29wZSgpLl9fZGVmaW5lKHBhdHRlcm4sXG4gICAgICAgICAgICAgICAgbmV3IERlZmluaXRpb24oXG4gICAgICAgICAgICAgICAgICAgIFZhcmlhYmxlLkNhdGNoQ2xhdXNlLFxuICAgICAgICAgICAgICAgICAgICBub2RlLnBhcmFtLFxuICAgICAgICAgICAgICAgICAgICBub2RlLFxuICAgICAgICAgICAgICAgICAgICBudWxsLFxuICAgICAgICAgICAgICAgICAgICBudWxsLFxuICAgICAgICAgICAgICAgICAgICBudWxsXG4gICAgICAgICAgICAgICAgKSk7XG4gICAgICAgICAgICB0aGlzLnJlZmVyZW5jaW5nRGVmYXVsdFZhbHVlKHBhdHRlcm4sIGluZm8uYXNzaWdubWVudHMsIG51bGwsIHRydWUpO1xuICAgICAgICB9KTtcbiAgICAgICAgdGhpcy52aXNpdChub2RlLmJvZHkpO1xuXG4gICAgICAgIHRoaXMuY2xvc2Uobm9kZSk7XG4gICAgfVxuXG4gICAgUHJvZ3JhbShub2RlKSB7XG4gICAgICAgIHRoaXMuc2NvcGVNYW5hZ2VyLl9fbmVzdEdsb2JhbFNjb3BlKG5vZGUpO1xuXG4gICAgICAgIGlmICh0aGlzLnNjb3BlTWFuYWdlci5fX2lzTm9kZWpzU2NvcGUoKSkge1xuICAgICAgICAgICAgLy8gRm9yY2Ugc3RyaWN0bmVzcyBvZiBHbG9iYWxTY29wZSB0byBmYWxzZSB3aGVuIHVzaW5nIG5vZGUuanMgc2NvcGUuXG4gICAgICAgICAgICB0aGlzLmN1cnJlbnRTY29wZSgpLmlzU3RyaWN0ID0gZmFsc2U7XG4gICAgICAgICAgICB0aGlzLnNjb3BlTWFuYWdlci5fX25lc3RGdW5jdGlvblNjb3BlKG5vZGUsIGZhbHNlKTtcbiAgICAgICAgfVxuXG4gICAgICAgIGlmICh0aGlzLnNjb3BlTWFuYWdlci5fX2lzRVM2KCkgJiYgdGhpcy5zY29wZU1hbmFnZXIuaXNNb2R1bGUoKSkge1xuICAgICAgICAgICAgdGhpcy5zY29wZU1hbmFnZXIuX19uZXN0TW9kdWxlU2NvcGUobm9kZSk7XG4gICAgICAgIH1cblxuICAgICAgICB0aGlzLnZpc2l0Q2hpbGRyZW4obm9kZSk7XG4gICAgICAgIHRoaXMuY2xvc2Uobm9kZSk7XG4gICAgfVxuXG4gICAgSWRlbnRpZmllcihub2RlKSB7XG4gICAgICAgIHRoaXMuY3VycmVudFNjb3BlKCkuX19yZWZlcmVuY2luZyhub2RlKTtcbiAgICB9XG5cbiAgICBVcGRhdGVFeHByZXNzaW9uKG5vZGUpIHtcbiAgICAgICAgaWYgKGlzUGF0dGVybihub2RlLmFyZ3VtZW50KSkge1xuICAgICAgICAgICAgdGhpcy5jdXJyZW50U2NvcGUoKS5fX3JlZmVyZW5jaW5nKG5vZGUuYXJndW1lbnQsIFJlZmVyZW5jZS5SVywgbnVsbCk7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICB0aGlzLnZpc2l0Q2hpbGRyZW4obm9kZSk7XG4gICAgICAgIH1cbiAgICB9XG5cbiAgICBNZW1iZXJFeHByZXNzaW9uKG5vZGUpIHtcbiAgICAgICAgdGhpcy52aXNpdChub2RlLm9iamVjdCk7XG4gICAgICAgIGlmIChub2RlLmNvbXB1dGVkKSB7XG4gICAgICAgICAgICB0aGlzLnZpc2l0KG5vZGUucHJvcGVydHkpO1xuICAgICAgICB9XG4gICAgfVxuXG4gICAgUHJvcGVydHkobm9kZSkge1xuICAgICAgICB0aGlzLnZpc2l0UHJvcGVydHkobm9kZSk7XG4gICAgfVxuXG4gICAgTWV0aG9kRGVmaW5pdGlvbihub2RlKSB7XG4gICAgICAgIHRoaXMudmlzaXRQcm9wZXJ0eShub2RlKTtcbiAgICB9XG5cbiAgICBCcmVha1N0YXRlbWVudCgpIHt9XG5cbiAgICBDb250aW51ZVN0YXRlbWVudCgpIHt9XG5cbiAgICBMYWJlbGVkU3RhdGVtZW50KG5vZGUpIHtcbiAgICAgICAgdGhpcy52aXNpdChub2RlLmJvZHkpO1xuICAgIH1cblxuICAgIEZvclN0YXRlbWVudChub2RlKSB7XG4gICAgICAgIC8vIENyZWF0ZSBGb3JTdGF0ZW1lbnQgZGVjbGFyYXRpb24uXG4gICAgICAgIC8vIE5PVEU6IEluIEVTNiwgRm9yU3RhdGVtZW50IGR5bmFtaWNhbGx5IGdlbmVyYXRlc1xuICAgICAgICAvLyBwZXIgaXRlcmF0aW9uIGVudmlyb25tZW50LiBIb3dldmVyLCBlc2NvcGUgaXNcbiAgICAgICAgLy8gYSBzdGF0aWMgYW5hbHl6ZXIsIHdlIG9ubHkgZ2VuZXJhdGUgb25lIHNjb3BlIGZvciBGb3JTdGF0ZW1lbnQuXG4gICAgICAgIGlmIChub2RlLmluaXQgJiYgbm9kZS5pbml0LnR5cGUgPT09IFN5bnRheC5WYXJpYWJsZURlY2xhcmF0aW9uICYmIG5vZGUuaW5pdC5raW5kICE9PSAndmFyJykge1xuICAgICAgICAgICAgdGhpcy5zY29wZU1hbmFnZXIuX19uZXN0Rm9yU2NvcGUobm9kZSk7XG4gICAgICAgIH1cblxuICAgICAgICB0aGlzLnZpc2l0Q2hpbGRyZW4obm9kZSk7XG5cbiAgICAgICAgdGhpcy5jbG9zZShub2RlKTtcbiAgICB9XG5cbiAgICBDbGFzc0V4cHJlc3Npb24obm9kZSkge1xuICAgICAgICB0aGlzLnZpc2l0Q2xhc3Mobm9kZSk7XG4gICAgfVxuXG4gICAgQ2xhc3NEZWNsYXJhdGlvbihub2RlKSB7XG4gICAgICAgIHRoaXMudmlzaXRDbGFzcyhub2RlKTtcbiAgICB9XG5cbiAgICBDYWxsRXhwcmVzc2lvbihub2RlKSB7XG4gICAgICAgIC8vIENoZWNrIHRoaXMgaXMgZGlyZWN0IGNhbGwgdG8gZXZhbFxuICAgICAgICBpZiAoIXRoaXMuc2NvcGVNYW5hZ2VyLl9faWdub3JlRXZhbCgpICYmIG5vZGUuY2FsbGVlLnR5cGUgPT09IFN5bnRheC5JZGVudGlmaWVyICYmIG5vZGUuY2FsbGVlLm5hbWUgPT09ICdldmFsJykge1xuICAgICAgICAgICAgLy8gTk9URTogVGhpcyBzaG91bGQgYmUgYHZhcmlhYmxlU2NvcGVgLiBTaW5jZSBkaXJlY3QgZXZhbCBjYWxsIGFsd2F5cyBjcmVhdGVzIExleGljYWwgZW52aXJvbm1lbnQgYW5kXG4gICAgICAgICAgICAvLyBsZXQgLyBjb25zdCBzaG91bGQgYmUgZW5jbG9zZWQgaW50byBpdC4gT25seSBWYXJpYWJsZURlY2xhcmF0aW9uIGFmZmVjdHMgb24gdGhlIGNhbGxlcidzIGVudmlyb25tZW50LlxuICAgICAgICAgICAgdGhpcy5jdXJyZW50U2NvcGUoKS52YXJpYWJsZVNjb3BlLl9fZGV0ZWN0RXZhbCgpO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMudmlzaXRDaGlsZHJlbihub2RlKTtcbiAgICB9XG5cbiAgICBCbG9ja1N0YXRlbWVudChub2RlKSB7XG4gICAgICAgIGlmICh0aGlzLnNjb3BlTWFuYWdlci5fX2lzRVM2KCkpIHtcbiAgICAgICAgICAgIHRoaXMuc2NvcGVNYW5hZ2VyLl9fbmVzdEJsb2NrU2NvcGUobm9kZSk7XG4gICAgICAgIH1cblxuICAgICAgICB0aGlzLnZpc2l0Q2hpbGRyZW4obm9kZSk7XG5cbiAgICAgICAgdGhpcy5jbG9zZShub2RlKTtcbiAgICB9XG5cbiAgICBUaGlzRXhwcmVzc2lvbigpIHtcbiAgICAgICAgdGhpcy5jdXJyZW50U2NvcGUoKS52YXJpYWJsZVNjb3BlLl9fZGV0ZWN0VGhpcygpO1xuICAgIH1cblxuICAgIFdpdGhTdGF0ZW1lbnQobm9kZSkge1xuICAgICAgICB0aGlzLnZpc2l0KG5vZGUub2JqZWN0KTtcbiAgICAgICAgLy8gVGhlbiBuZXN0IHNjb3BlIGZvciBXaXRoU3RhdGVtZW50LlxuICAgICAgICB0aGlzLnNjb3BlTWFuYWdlci5fX25lc3RXaXRoU2NvcGUobm9kZSk7XG5cbiAgICAgICAgdGhpcy52aXNpdChub2RlLmJvZHkpO1xuXG4gICAgICAgIHRoaXMuY2xvc2Uobm9kZSk7XG4gICAgfVxuXG4gICAgVmFyaWFibGVEZWNsYXJhdGlvbihub2RlKSB7XG4gICAgICAgIHZhciB2YXJpYWJsZVRhcmdldFNjb3BlLCBpLCBpeiwgZGVjbDtcbiAgICAgICAgdmFyaWFibGVUYXJnZXRTY29wZSA9IChub2RlLmtpbmQgPT09ICd2YXInKSA/IHRoaXMuY3VycmVudFNjb3BlKCkudmFyaWFibGVTY29wZSA6IHRoaXMuY3VycmVudFNjb3BlKCk7XG4gICAgICAgIGZvciAoaSA9IDAsIGl6ID0gbm9kZS5kZWNsYXJhdGlvbnMubGVuZ3RoOyBpIDwgaXo7ICsraSkge1xuICAgICAgICAgICAgZGVjbCA9IG5vZGUuZGVjbGFyYXRpb25zW2ldO1xuICAgICAgICAgICAgdGhpcy52aXNpdFZhcmlhYmxlRGVjbGFyYXRpb24odmFyaWFibGVUYXJnZXRTY29wZSwgVmFyaWFibGUuVmFyaWFibGUsIG5vZGUsIGkpO1xuICAgICAgICAgICAgaWYgKGRlY2wuaW5pdCkge1xuICAgICAgICAgICAgICAgIHRoaXMudmlzaXQoZGVjbC5pbml0KTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgIH1cblxuICAgIC8vIHNlYyAxMy4xMS44XG4gICAgU3dpdGNoU3RhdGVtZW50KG5vZGUpIHtcbiAgICAgICAgdmFyIGksIGl6O1xuXG4gICAgICAgIHRoaXMudmlzaXQobm9kZS5kaXNjcmltaW5hbnQpO1xuXG4gICAgICAgIGlmICh0aGlzLnNjb3BlTWFuYWdlci5fX2lzRVM2KCkpIHtcbiAgICAgICAgICAgIHRoaXMuc2NvcGVNYW5hZ2VyLl9fbmVzdFN3aXRjaFNjb3BlKG5vZGUpO1xuICAgICAgICB9XG5cbiAgICAgICAgZm9yIChpID0gMCwgaXogPSBub2RlLmNhc2VzLmxlbmd0aDsgaSA8IGl6OyArK2kpIHtcbiAgICAgICAgICAgIHRoaXMudmlzaXQobm9kZS5jYXNlc1tpXSk7XG4gICAgICAgIH1cblxuICAgICAgICB0aGlzLmNsb3NlKG5vZGUpO1xuICAgIH1cblxuICAgIEZ1bmN0aW9uRGVjbGFyYXRpb24obm9kZSkge1xuICAgICAgICB0aGlzLnZpc2l0RnVuY3Rpb24obm9kZSk7XG4gICAgfVxuXG4gICAgRnVuY3Rpb25FeHByZXNzaW9uKG5vZGUpIHtcbiAgICAgICAgdGhpcy52aXNpdEZ1bmN0aW9uKG5vZGUpO1xuICAgIH1cblxuICAgIEZvck9mU3RhdGVtZW50KG5vZGUpIHtcbiAgICAgICAgdGhpcy52aXNpdEZvckluKG5vZGUpO1xuICAgIH1cblxuICAgIEZvckluU3RhdGVtZW50KG5vZGUpIHtcbiAgICAgICAgdGhpcy52aXNpdEZvckluKG5vZGUpO1xuICAgIH1cblxuICAgIEFycm93RnVuY3Rpb25FeHByZXNzaW9uKG5vZGUpIHtcbiAgICAgICAgdGhpcy52aXNpdEZ1bmN0aW9uKG5vZGUpO1xuICAgIH1cblxuICAgIEltcG9ydERlY2xhcmF0aW9uKG5vZGUpIHtcbiAgICAgICAgdmFyIGltcG9ydGVyO1xuXG4gICAgICAgIGFzc2VydCh0aGlzLnNjb3BlTWFuYWdlci5fX2lzRVM2KCkgJiYgdGhpcy5zY29wZU1hbmFnZXIuaXNNb2R1bGUoKSwgJ0ltcG9ydERlY2xhcmF0aW9uIHNob3VsZCBhcHBlYXIgd2hlbiB0aGUgbW9kZSBpcyBFUzYgYW5kIGluIHRoZSBtb2R1bGUgY29udGV4dC4nKTtcblxuICAgICAgICBpbXBvcnRlciA9IG5ldyBJbXBvcnRlcihub2RlLCB0aGlzKTtcbiAgICAgICAgaW1wb3J0ZXIudmlzaXQobm9kZSk7XG4gICAgfVxuXG4gICAgdmlzaXRFeHBvcnREZWNsYXJhdGlvbihub2RlKSB7XG4gICAgICAgIGlmIChub2RlLnNvdXJjZSkge1xuICAgICAgICAgICAgcmV0dXJuO1xuICAgICAgICB9XG4gICAgICAgIGlmIChub2RlLmRlY2xhcmF0aW9uKSB7XG4gICAgICAgICAgICB0aGlzLnZpc2l0KG5vZGUuZGVjbGFyYXRpb24pO1xuICAgICAgICAgICAgcmV0dXJuO1xuICAgICAgICB9XG5cbiAgICAgICAgdGhpcy52aXNpdENoaWxkcmVuKG5vZGUpO1xuICAgIH1cblxuICAgIEV4cG9ydERlY2xhcmF0aW9uKG5vZGUpIHtcbiAgICAgICAgdGhpcy52aXNpdEV4cG9ydERlY2xhcmF0aW9uKG5vZGUpO1xuICAgIH1cblxuICAgIEV4cG9ydE5hbWVkRGVjbGFyYXRpb24obm9kZSkge1xuICAgICAgICB0aGlzLnZpc2l0RXhwb3J0RGVjbGFyYXRpb24obm9kZSk7XG4gICAgfVxuXG4gICAgRXhwb3J0U3BlY2lmaWVyKG5vZGUpIHtcbiAgICAgICAgbGV0IGxvY2FsID0gKG5vZGUuaWQgfHwgbm9kZS5sb2NhbCk7XG4gICAgICAgIHRoaXMudmlzaXQobG9jYWwpO1xuICAgIH1cbn1cblxuLyogdmltOiBzZXQgc3c9NCB0cz00IGV0IHR3PTgwIDogKi9cbiJdLCJzb3VyY2VSb290IjoiL3NvdXJjZS8ifQ==