/*
Copyright (c) 2016 Niklas Knaack

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

( function() {

    function ThresholdingImageCrossfader( element, params ) {

        var settings = {

            images: [],
            width: 480,
            height: 480,
            delay: 5,
            step: 3,
            dir: 1,
            effectDir: 1,
            bgColor: '#000000',
            auto: true

        };
            
        //---

        if ( params !== undefined )
            for ( var prop in params )
                if ( params.hasOwnProperty( prop ) && settings.hasOwnProperty( prop ) )
                    settings[ prop ] = params[ prop ];

        //---

        if ( !settings.images.length )
            return false;

        //---

        var divHolder = document.createElement( 'div' );
            divHolder.style.width = settings.width + 'px';
            divHolder.style.height = settings.height + 'px';

        element.appendChild( divHolder );

        //---

        var outputCanvas = document.createElement( 'canvas' );
            outputCanvas.width = settings.width;
            outputCanvas.height = settings.height;

        divHolder.appendChild( outputCanvas );

        var outputCtx = outputCanvas.getContext( '2d' );
            outputCtx.fillStyle = settings.bgColor;
            outputCtx.fillRect( 0, 0, settings.width, settings.height );

        var outputImageData = outputCtx.getImageData( 0, 0, settings.width, settings.height );
        var outputData = outputImageData.data;

        //---

        var inputImages = [];
        var inputData;
        var inputImagesLoaded = 0;
        var inputImagesIndex = -1;

        //---

        var images = [];

        var tresholdValue;

        var delay = settings.delay * 1000;
        var delayPaused = false;
        var delayTimer;

        //---

        function loadImages() {

            for ( var i = 0, l = settings.images.length; i < l; i++ ) {

                var image = new Image();
                    image.onload = function() {

                        inputImagesLoaded++;

                        init();

                    };
                    //img.crossOrigin = 'anonymous';
                    image.src = settings.images[ i ];

                images.push( image );

            }

        };

        function getImagesData() {

            for ( var i = 0, l = images.length; i < l; i++ ) {

                var inputCanvas = document.createElement( 'canvas' );
                    inputCanvas.width = settings.width; 
                    inputCanvas.height = settings.height;

                var inputCtx = inputCanvas.getContext( '2d' );
                    inputCtx.drawImage( images[ i ], 0, 0 );

                inputImages.push( inputCtx.getImageData( 0, 0, settings.width, settings.height ).data );

            }

        };

        function init() {

            if ( inputImagesLoaded === settings.images.length ) {

                getImagesData();
                showImage( settings.dir );
                animloop();

            }

        };

        //---

        function crossfade( value ) {

            for ( var i = 0, l = outputData.length; i < l; i += 4 ) {

                var r = inputData[ i ];
                var g = inputData[ i + 1 ];
                var b = inputData[ i + 2 ]; 

                if ( settings.effectDir > 0 && ( r + g + b ) / 3 >= value || 
                     settings.effectDir < 0 && ( r + g + b ) / 3 <= value ) {

                    outputData[ i ]     = r;
                    outputData[ i + 1 ] = g;
                    outputData[ i + 2 ] = b;
                    outputData[ i + 3 ] = 255;

                }

            }

        };

        //---

        function switchImage() {

            delayTimer = setTimeout( function() { 

                showImage( settings.dir );

            }, delay );

        };

        function showImage( dir ) {

            settings.effectDir > 0 ? tresholdValue = 255 : tresholdValue = 0;

            inputImagesIndex += dir;

            if ( inputImagesIndex > inputImages.length - 1 )
                inputImagesIndex = 0;
            if ( inputImagesIndex < 0 )
                inputImagesIndex = inputImages.length - 1;

            inputData = inputImages[ inputImagesIndex ];

            delayPaused = false;

        };  

        //---

        window.requestAnimFrame = ( function() {

            return  window.requestAnimationFrame       ||
                    window.webkitRequestAnimationFrame ||
                    window.mozRequestAnimationFrame    ||
                    function( callback ) {
                        window.setTimeout( callback, 1000 / 60 );
                    };

        } )();

        function animloop() {

            requestAnimFrame( animloop );

            if ( !delayPaused )
                render();

        };

        function render() {

            crossfade( tresholdValue );

            outputCtx.putImageData( outputImageData, 0, 0 );

            //---

            if ( settings.effectDir > 0 )
                tresholdValue -= settings.step;
            else
                tresholdValue += settings.step;

            if ( tresholdValue < 0 || tresholdValue > 255 ) {
 
                delayPaused = true;

                if ( settings.auto )
                    switchImage();

            }

        };
        
        //---

        this.nextImage = function() {

            clearTimeout( delayTimer );

            showImage( 1 );

        };

        this.previousImage = function() {

            clearTimeout( delayTimer );

            showImage( -1 );

        };

        //---

        loadImages();
        
    };

    window.ThresholdingImageCrossfader = ThresholdingImageCrossfader;

} () );

if ( typeof jQuery !== 'undefined' ) {

    ( function( $ ) {
        
        $.fn.thresholdingImageCrossfader = function( params ) {

            return this.each( function() {

                if ( !$.data( this, 'plugin_ThresholdingImageCrossfader' ) ) {

                    $.data( this, 'plugin_ThresholdingImageCrossfader', new ThresholdingImageCrossfader( this, params ) );

                } else {

                    var plugin = $.data( this, 'plugin_ThresholdingImageCrossfader' );

                    if ( params[ 'nextImage' ] !== undefined ) {

                        plugin.nextImage();

                    } else if ( params[ 'previousImage' ] !== undefined ) {

                        plugin.previousImage();

                    }

                }

            } );

        };
        
    } ( jQuery ) );

}