/*!
 * bs-datepicker (jQuery Plugin)
 * -----------------------------------------------------------------------------
 * A lightweight Bootstrap 5–styled datepicker for jQuery with support for
 * single-date and date-range selection, inline or dropdown rendering,
 * locale-aware formatting via Intl, multi-month views, and configurable
 * disabled date rules.
 *
 * GitHub: https://github.com/ThomasDev-de/bs-datepicker
 *
 * Author: Thomas Kirsch <t.kirsch@webcito.de>
 * Version: 1.0.0
 *
 * Dependencies:
 * - jQuery >= 3.x
 * - Bootstrap >= 5.x
 * - (optional) Bootstrap Icons >= 1.x
 *
 * Events (namespace: bs.datepicker):
 * - init.bs.datepicker
 * - render.bs.datepicker
 * - show.bs.datepicker
 * - hide.bs.datepicker
 * - navigate.bs.datepicker
 * - changeDate.bs.datepicker   (detail.value: Date | [Date|null, Date|null] | null)
 * - clear.bs.datepicker
 * - destroy.bs.datepicker
 *
 * Public API:
 * - $(el).bsDatepicker('getDate')
 * - $(el).bsDatepicker('val')
 * - $(el).bsDatepicker('val', date | start, end | [start, end])
 * - $(el).bsDatepicker('setDate', dateOrRange)
 * - $(el).bsDatepicker('setLocale', locale)
 * - $(el).bsDatepicker('setDisableDates', config)
 * - $(el).bsDatepicker('getDisableDates')
 * - $(el).bsDatepicker('setMin', date)
 * - $(el).bsDatepicker('setMax', date)
 * - $(el).bsDatepicker('clearDisableDates')
 * - $(el).bsDatepicker('destroy')
 *
 * License: MIT
 * -----------------------------------------------------------------------------
 * Copyright (c) 2025 Thomas Kirsch
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND.
 */


(function ($) {

    // Defaults
    $.bsDatepicker = {
        version: '1.0.0',
        default: {
            locale: 'de-DE',           // Intl locale, e.g. 'de-DE', 'en-US'
            range: false,               // select a date range
            inline: false,              // render inline instead of dropdown
            startOnSunday: false,       // week starts on Sunday (else Monday)
            autoClose: true,            // close dropdown after selection (single) or after end (range)
            format: 'locale',           // 'locale' | 'iso' | custom function(date|[start,end]) -> string
            separator: ' – ',           // range separator for text output
            zIndex: 1080,               // for dropdown panel
            months: 1,                  // number of months to render side by side
            // Visual Theme: previously configurable; now always uses a subtle Bootstrap-based highlighting
            icons: {                    // Bootstrap Icons class names
                prevYear: 'bi bi-chevron-double-left',
                prev: 'bi bi-chevron-left',
                today: 'bi bi-record-circle',
                next: 'bi bi-chevron-right',
                nextYear: 'bi bi-chevron-double-right',
                clear: 'bi bi-x-lg'
            },
            // CSS classes for the visible display (no actual input field)
            classes: {
                display: 'form-control d-flex align-items-center justify-content-between',
                displayText: '',
                displayIcon: 'bi bi-calendar-event'
            },
            placeholder: 'Select period',
            // Disabled dates configuration
            // Forms:
            // - disabled: { before?: Date|string, after?: Date|string, min?: Date|string, max?: Date|string, dates?: (Date|string)[] }
            // Note: before => disables <= before, after => disables >= after
            disabled: null
        }
    };

    const NS = 'bs.datepicker';

    // Event emitter helper (all events share the same namespace: bs.datepicker)
    // Usage: emit(state, 'changeDate', { selected: Date|[Date,Date]|null })
    function emit(state, name, detail) {
        const $target = state.containerMode ? state.$root : (state.$input || state.$anchor || state.$root);
        if (!$target || !$target.length) return;
        try {
            const ev = $.Event(name + '.' + NS, { detail: detail || {} });
            $target.trigger(ev, [detail || {}]);
        } catch (e) {
            // no-op fail safe
        }
    }

    // Utilities
    function startOfDay(d) {
        const x = new Date(d.getTime());
        x.setHours(0, 0, 0, 0);
        return x;
    }
    function isSameDay(a, b) {
        if (!a || !b) return false;
        return a.getFullYear() === b.getFullYear() && a.getMonth() === b.getMonth() && a.getDate() === b.getDate();
    }
    function addMonths(date, n) {
        const d = new Date(date.getFullYear(), date.getMonth() + n, 1);
        return d;
    }
    function daysInMonth(year, month) {
        return new Date(year, month + 1, 0).getDate();
    }
    function clampRange(a, b) {
        if (!a || !b) return [a, b];
        return (a <= b) ? [a, b] : [b, a];
    }
    function inRange(d, a, b) {
        if (!a || !b) return false;
        const [s, e] = clampRange(a, b);
        return d >= startOfDay(s) && d <= startOfDay(e);
    }

    function toDateOrNull(v) {
        if (!v) return null;
        if (v instanceof Date) return startOfDay(v);
        if (typeof v === 'number') return startOfDay(new Date(v));
        if (typeof v === 'string') {
            // Expect ISO YYYY-MM-DD; fallback to Date parse
            const parts = v.match(/^(\d{4})-(\d{2})-(\d{2})$/);
            let d = parts ? new Date(Number(parts[1]), Number(parts[2]) - 1, Number(parts[3])) : new Date(v);
            if (isNaN(d.getTime())) return null;
            return startOfDay(d);
        }
        return null;
    }

    function normalizeDisabled(cfg) {
        if (!cfg) return { before: null, after: null, min: null, max: null, datesSet: new Set() };
        const out = { before: null, after: null, min: null, max: null, datesSet: new Set() };
        if (cfg.before) out.before = toDateOrNull(cfg.before);
        if (cfg.after) out.after = toDateOrNull(cfg.after);
        if (cfg.min) out.min = toDateOrNull(cfg.min);
        if (cfg.max) out.max = toDateOrNull(cfg.max);
        if (Array.isArray(cfg.dates)) {
            cfg.dates.forEach(function (x) {
                const d = toDateOrNull(x);
                if (d) out.datesSet.add(d.getTime());
            });
        }
        return out;
    }

    function isDisabledDate(d, state) {
        if (!state || !state.disabled) return false;
        const dd = startOfDay(d);
        const t = dd.getTime();
        const dis = state.disabled;
        if (dis.min && dd < dis.min) return true;
        if (dis.max && dd > dis.max) return true;
        if (dis.before && dd <= dis.before) return true;
        if (dis.after && dd >= dis.after) return true;
        if (dis.datesSet && dis.datesSet.has(t)) return true;
        return false;
    }
    function getWeekdayNames(locale, startOnSunday) {
        const fmt = new Intl.DateTimeFormat(locale, { weekday: 'short' });
        const base = new Date(2021, 7, 1); // arbitrary Sunday
        const days = [];
        for (let i = 0; i < 7; i++) {
            const d = new Date(base.getFullYear(), base.getMonth(), base.getDate() + i);
            days.push(fmt.format(d));
        }
        // base was Sunday → order is Sun..Sat
        if (startOnSunday) return days;
        // move Sunday to end for Monday as first day
        return days.slice(1).concat(days.slice(0, 1));
    }
    function getMonthYearTitle(date, locale) {
        // Full month name for per-month headers above each calendar
        const fmt = new Intl.DateTimeFormat(locale, { month: 'long', year: 'numeric' });
        return fmt.format(date);
    }
    function formatDateValue(value, opts) {
        const { format, locale, separator } = opts;
        if (typeof format === 'function') return format(value);
        const fmt = new Intl.DateTimeFormat(locale, { year: 'numeric', month: '2-digit', day: '2-digit' });
        const toLocalISO = (d) => {
            if (!d) return '';
            const y = d.getFullYear();
            const m = String(d.getMonth() + 1).padStart(2, '0');
            const day = String(d.getDate()).padStart(2, '0');
            return y + '-' + m + '-' + day;
        };
        if (Array.isArray(value)) {
            const [a, b] = value;
            if (!a && !b) return '';
            if (format === 'iso') {
                return [toLocalISO(a), toLocalISO(b)].filter(Boolean).join(separator);
            }
            return [a ? fmt.format(a) : '', b ? fmt.format(b) : ''].filter(Boolean).join(separator);
        } else if (value instanceof Date) {
            if (format === 'iso') return toLocalISO(value);
            return fmt.format(value);
        }
        return '';
    }

    function buildCalendarGrid(current, opts) {
        const year = current.getFullYear();
        const month = current.getMonth();
        const firstOfMonth = new Date(year, month, 1);
        const firstWeekday = firstOfMonth.getDay(); // 0=Sun..6=Sat
        const shift = opts.startOnSunday ? 0 : 1; // Monday start shifts
        // index of the first cell (0..6) that belongs to current month
        const startIdx = (firstWeekday - shift + 7) % 7;
        const dim = daysInMonth(year, month);
        const prevDim = daysInMonth(year, month - 1 < 0 ? 11 : month - 1);
        const cells = [];
        // 6 rows x 7 columns = 42 cells
        for (let i = 0; i < 42; i++) {
            const cell = { inMonth: false, date: null };
            if (i < startIdx) {
                const day = prevDim - (startIdx - 1 - i);
                const d = new Date(year, month, 1);
                d.setDate(d.getDate() - (startIdx - i));
                cell.inMonth = false;
                cell.date = startOfDay(d);
            } else if (i >= startIdx && i < startIdx + dim) {
                const day = i - startIdx + 1;
                cell.inMonth = true;
                cell.date = startOfDay(new Date(year, month, day));
            } else {
                const d = new Date(year, month, dim);
                d.setDate(d.getDate() + (i - (startIdx + dim)) + 1);
                cell.inMonth = false;
                cell.date = startOfDay(d);
            }
            cells.push(cell);
        }
        return cells;
    }

    function renderOneMonthBlock(currMonthDate, state) {
        const { opts, selected, rangeStart } = state;
        const weekdays = getWeekdayNames(opts.locale, opts.startOnSunday);
        const title = getMonthYearTitle(currMonthDate, opts.locale);
        const cells = buildCalendarGrid(currMonthDate, opts);
        const today = startOfDay(new Date());
        const isRange = !!opts.range;
        // Theme is fixed to 'subtle' (option removed to keep options compact)
        const theme = 'subtle';

        let html = '';
        html += '<div class="mb-2">';
        html += '  <div class="fw-semibold text-capitalize text-center mb-0">' + title + '</div>';
        html += '  <div class="table-responsive">';
        // Compact, content-based table; cells remain even (buttons fill the cell)
        // Make the table span the full tile width so the visual center aligns with the title
        // Use table-layout: fixed to keep 7 equal columns and prevent width jumps when locale changes
        html += '    <table class="table table-sm table-borderless mb-0 text-center align-middle user-select-none w-100" style="table-layout:fixed">';
        html += '      <thead><tr>';
        weekdays.forEach(w => { html += '<th class="text-muted small">' + w + '</th>'; });
        html += '      </tr></thead>';
        html += '      <tbody>';
        // Dynamic number of rows: only as many rows as the month needs
        const year = currMonthDate.getFullYear();
        const month = currMonthDate.getMonth();
        const firstOfMonth = new Date(year, month, 1);
        const firstWeekday = firstOfMonth.getDay(); // 0..6
        const shift = opts.startOnSunday ? 0 : 1;
        const startIdx = (firstWeekday - shift + 7) % 7;
        const dim = daysInMonth(year, month);
        const neededCells = startIdx + dim;
        const rows = Math.ceil(neededCells / 7);
        for (let r = 0; r < rows; r++) {
            html += '<tr>';
            for (let c = 0; c < 7; c++) {
                const idx = r * 7 + c;
                const dayIndex = idx - startIdx; // 0-based relative to month (can be <0 or >= dim)
                // Also compute dates for previous/next month
                const d = startOfDay(new Date(year, month, dayIndex + 1));
                const inMonth = dayIndex >= 0 && dayIndex < dim;
                const isToday = isSameDay(d, today);
                const isSelected = !isRange && selected && isSameDay(d, selected);
                const isStart = isRange && rangeStart && isSameDay(d, rangeStart);
                const isEnd = isRange && selected && !isStart && isSameDay(d, selected);
                const isBetween = isRange && rangeStart && selected && inRange(d, rangeStart, selected) && !isStart && !isEnd;
                const muted = !inMonth; // Dim days outside current month

                // TD classes: only remove padding, no colors/borders on TD
                let tdCls = 'p-0'; // no spacing between days

                // Button classes: base without rounded corners; full-width buttons fill cells
                // Subtle highlighting (Bootstrap 5.3 utilities)
                let btnCls = 'btn btn-sm w-100 border-0 rounded-0 ';
                const inRangeAny = isStart || isEnd || isBetween;

                // Subtle theme (always on)
                if (isRange) {
                    if (inRangeAny) {
                        // Middle and edges of the range get subtle fill
                        btnCls += ' bg-primary-subtle text-primary-emphasis ';
                    }
                    if (isStart || isEnd) {
                        // Emphasize the range edges
                        btnCls += ' border border-primary fw-semibold ';
                    }
                    // Rounded corners at the visual range boundaries (precise corners only)
                } else {
                    if (isSelected) {
                        // Single selection: use subtle filled background like range edges for better visibility
                        btnCls += ' bg-primary-subtle text-primary-emphasis fw-semibold border border-primary ';
                    }
                }
                if (muted) btnCls += ' text-muted ';
                // Today: subtle emphasis if not part of selection
                if (isToday && !(isSelected || inRangeAny)) btnCls += ' text-primary fw-semibold ';

                const disabled = isDisabledDate(d, state);
                html += '<td class="' + tdCls.trim() + '">';
                // Store data-date as millisecond timestamp to avoid TZ parsing pitfalls
                const actionAttr = disabled ? '' : ' data-action="pick"';
                const disAttr = disabled ? ' disabled aria-disabled="true"' : '';
                // Disabled: give a subtle gray background, analogous to the subtle blue of selection
                // Uses Bootstrap 5.3 utilities: bg-secondary-subtle + text-secondary-emphasis
                const disabledEnhance = disabled ? ' bg-secondary-subtle text-secondary-emphasis ' : '';
                // Add structural classes for styling without extra CSS files
                // dp-start / dp-end allow inline styles below to target edges precisely per button
                const structCls = (isStart ? ' dp-start ' : '') + (isEnd ? ' dp-end ' : '');
                const clsFinal = (btnCls + (disabled ? ' disabled ' : '') + disabledEnhance + structCls).trim();
                // Inline styles (no external CSS): draw thin primary caps at left/right for start/end
                // Also round desired single corners only
                let cornerStyle = '';
                if (isRange) {
                    if (isStart) cornerStyle += 'border-top-left-radius: var(--bs-border-radius); box-shadow: inset 2px 0 0 0 var(--bs-primary);';
                    if (isEnd) cornerStyle += 'border-bottom-right-radius: var(--bs-border-radius); box-shadow: inset -2px 0 0 0 var(--bs-primary);';
                }
                const styleAttr = cornerStyle ? ' style="' + cornerStyle + '"' : '';
                html += '<button type="button" class="' + clsFinal + '"' + styleAttr + actionAttr + disAttr + ' data-date="' + d.getTime() + '">';
                html += d.getDate();
                html += '</button>';
                html += '</td>';
            }
            html += '</tr>';
        }
        html += '      </tbody>';
        html += '    </table>';
        html += '  </div>';
        html += '</div>';
        return html;
    }

    function renderTemplate(state) {
        const { opts, current } = state;
        const months = Math.max(1, parseInt(opts.months || 1, 10));

        let html = '';
        // Modern, neutral panel look (no card)
        // Inline: minimal look; also shrink-to-content via d-inline-block (no unnecessary right whitespace)
        // Dropdown: inline-block so width follows content (month tiles) and avoids unused right whitespace
        var panelCls = opts.inline ? 'bg-transparent p-2 d-inline-block' : 'bg-body border rounded-3 shadow p-2 d-inline-block';
        html += '<div class="' + panelCls + '">';
        // Compact header: single row with three zones
        html += '  <div class="d-flex align-items-center justify-content-between gap-2 pb-2' + (opts.inline ? '' : ' border-bottom') + '">';
        // Left: PrevYear / Prev
        html += '    <div class="d-flex align-items-center gap-1">';
        html += '      <button type="button" class="btn btn-sm border-0 p-1" data-action="prevYear" title="Previous year" aria-label="Previous year"><i class="' + (opts.icons && (opts.icons.prevYear || opts.icons.prev) || $.bsDatepicker.default.icons.prevYear) + '"></i></button>';
        html += '      <button type="button" class="btn btn-sm border-0 p-1" data-action="prev" title="Previous month" aria-label="Previous month"><i class="' + (opts.icons && opts.icons.prev || $.bsDatepicker.default.icons.prev) + '"></i></button>';
        html += '    </div>';
        // Center: title (navigation – keep short format to stay compact)
        html += '    <div class="text-center flex-grow-1">';
        (function(){
            const fmtShort = new Intl.DateTimeFormat(opts.locale, { month: 'short', year: 'numeric' });
            const titleLeft = fmtShort.format(current);
            const titleRight = months > 1 ? fmtShort.format(addMonths(current, months - 1)) : '';
            html += '      <div class="small fw-semibold text-capitalize">' + titleLeft + (months > 1 ? ' … ' + titleRight : '') + '</div>';
        })();
        html += '    </div>';
        // Right: Next / NextYear / Today / Clear as icons
        html += '    <div class="d-flex align-items-center gap-1">';
        html += '      <button type="button" class="btn btn-sm border-0 p-1" data-action="next" title="Next month" aria-label="Next month"><i class="' + (opts.icons && opts.icons.next || $.bsDatepicker.default.icons.next) + '"></i></button>';
        html += '      <button type="button" class="btn btn-sm border-0 p-1" data-action="nextYear" title="Next year" aria-label="Next year"><i class="' + (opts.icons && (opts.icons.nextYear || opts.icons.next) || $.bsDatepicker.default.icons.nextYear) + '"></i></button>';
        html += '      <button type="button" class="btn btn-sm border-0 p-1" data-action="today" title="Today" aria-label="Today"><i class="' + (opts.icons && opts.icons.today || $.bsDatepicker.default.icons.today) + '"></i></button>';
        html += '      <button type="button" class="btn btn-sm border-0 p-1" data-action="clear" title="Clear" aria-label="Clear"><i class="' + (opts.icons && opts.icons.clear || $.bsDatepicker.default.icons.clear) + '"></i></button>';
        html += '    </div>';
        html += '  </div>';
        html += '  <div class="pt-2">';
        // Inline: small textual output of the current selection inside the panel
        if (opts.inline) {
            const isRange = !!opts.range;
            const dispOpts = $.extend({}, opts, { format: 'locale' });
            let text = '';
            if (isRange) {
                const pair = clampRange(state.rangeStart, state.selected);
                text = formatDateValue(pair, dispOpts);
            } else {
                text = formatDateValue(state.selected, dispOpts);
            }
            const placeholder = opts.placeholder;
            const clsMuted = text ? '' : ' text-muted';
            // Inline: subtle one-line selection summary
            html += '    <div class="mb-2 small text-center dp-inline-output' + clsMuted + '">' + (text || placeholder) + '</div>';
        }
        // Month tiles: fixed width, flex-wrap and capped total width for two columns
        // Goal: show two months side-by-side when space allows; on narrow screens auto-wrap to one column
        // gap-2 equals .5rem in Bootstrap → total ≈ 300px + .5rem + 300px
        const monthsWrapStyle = (months > 1)
            ? 'max-width: calc(600px + .5rem)'
            : 'max-width: 100%';
        // Left-align tiles per row (no centering) so with 3 months the third starts left on the next line
        html += '    <div class="dp-months d-inline-flex flex-wrap align-items-start gap-2" style="' + monthsWrapStyle + '">';
        for (let i = 0; i < months; i++) {
            // Fixed tile width and flex-basis so items do not stretch
            html += '      <div class="dp-month" style="width:300px; flex:0 0 300px">';
            html += renderOneMonthBlock(addMonths(current, i), state);
            html += '      </div>';
        }
        html += '    </div>';
        html += '  </div>';
        html += '</div>';
        return html;
    }

    function attachEvents(state) {
        const { $panel, opts } = state;
        $panel.off('.' + NS);
        $panel.on('click.' + NS, '[data-action="prev"]', function (e) {
            e.preventDefault();
            state.current = addMonths(state.current, -1);
            updatePanel(state);
            emit(state, 'navigate', { action: 'prev', current: new Date(state.current) });
        });
        $panel.on('click.' + NS, '[data-action="prevYear"]', function (e) {
            e.preventDefault();
            state.current = addMonths(state.current, -12);
            updatePanel(state);
            emit(state, 'navigate', { action: 'prevYear', current: new Date(state.current) });
        });
        $panel.on('click.' + NS, '[data-action="next"]', function (e) {
            e.preventDefault();
            state.current = addMonths(state.current, +1);
            updatePanel(state);
            emit(state, 'navigate', { action: 'next', current: new Date(state.current) });
        });
        $panel.on('click.' + NS, '[data-action="nextYear"]', function (e) {
            e.preventDefault();
            state.current = addMonths(state.current, +12);
            updatePanel(state);
            emit(state, 'navigate', { action: 'nextYear', current: new Date(state.current) });
        });
        $panel.on('click.' + NS, '[data-action="today"]', function (e) {
            e.preventDefault();
            const t = new Date();
            state.current = new Date(t.getFullYear(), t.getMonth(), 1);
            updatePanel(state);
            emit(state, 'navigate', { action: 'today', current: new Date(state.current) });
        });
        $panel.on('click.' + NS, '[data-action="clear"]', function (e) {
            e.preventDefault();
            // Clear selection
            state.rangeStart = null;
            state.selected = null;
            // Clear visible display / hidden inputs
            if (state.$display) updateDisplay(state);
            if (state.$inStart) state.$inStart.val('').trigger('change');
            if (state.$inEnd) state.$inEnd.val('').trigger('change');
            if (state.$input && !state.$display) state.$input.val('').trigger('change');
            updatePanel(state);
            emit(state, 'clear', {});
            emit(state, 'changeDate', { value: opts.range ? [null, null] : null });
        });
        $panel.on('click.' + NS, '[data-action="pick"]', function (e) {
            e.preventDefault();
            const stamp = $(this).data('date');
            const d = startOfDay(new Date(typeof stamp === 'number' ? stamp : Number(stamp)));
            if (isDisabledDate(d, state)) return; // ignore disabled days
            if (opts.range) {
                const S = state.rangeStart;
                const E = state.selected;

                // Case 3 (toggle off): clicking exactly on current start or end clears that edge
                if (S && isSameDay(d, S)) {
                    // If only start existed (no end), allow immediate re-set of start on same click
                    if (!E) {
                        // Interpret as re-pick of start
                        state.rangeStart = d;
                        updatePanel(state);
                        return;
                    }
                    // With both edges present: clear start only; user will pick a new start next
                    state.rangeStart = null;
                    updatePanel(state);
                    return;
                }
                if (E && isSameDay(d, E)) {
                    state.selected = null; // remove end
                    // If next click is before the remaining start, we will swap accordingly below
                    updatePanel(state);
                    return;
                }

                if (!S && !E) {
                    // No selection yet → set start
                    state.rangeStart = d;
                    state.selected = null;
                    updatePanel(state);
                } else if (S && !E) {
                    // Start picked, end open
                    if (d < S) {
                        // Click before current start → interpret as new start (swap behavior)
                        state.selected = S; // previous start becomes end
                        state.rangeStart = d; // new start
                    } else {
                        // After (or same day) → set as end
                        state.selected = d;
                    }
                    // Keep dropdown open for fine-tuning
                    updatePanel(state);
                } else if (!S && E) {
                    // Only end exists, start open (after deselecting start)
                    if (d <= E) {
                        // Click left/equal end -> set start to D
                        state.rangeStart = d;
                    } else {
                        // Click right of current end -> swap so that previous end becomes start
                        state.rangeStart = E;
                        state.selected = d;
                    }
                    updatePanel(state);
                } else if (S && E) {
                    // Complete range present → move the nearer edge
                    if (d <= S) {
                        // Click left/equal start → start becomes D
                        state.rangeStart = d;
                    } else if (d >= E) {
                        // Click right/equal end → end becomes D
                        state.selected = d;
                    } else {
                        // Click inside (S < D < E): move nearer edge
                        const distToStart = d - S;
                        const distToEnd = E - d;
                        if (distToStart <= distToEnd) {
                            state.rangeStart = d;
                        } else {
                            state.selected = d;
                        }
                    }
                    // Update values (display + hidden)
                    // Note: Do NOT auto-close when adjusting an existing range to allow fine tuning.
                    updatePanel(state);
                }
                // Emit changeDate for any pick in range mode (edge moved or set)
                const [a, b] = clampRange(state.rangeStart, state.selected);
                emit(state, 'changeDate', { value: [a || null, b || null] });
            } else {
                state.selected = d;
                // Update values (display + hidden)
                if (state.$display || state.$inStart) {
                    const toLocalISO = (x) => x ? (function(d){ const y=d.getFullYear(); const m=String(d.getMonth()+1).padStart(2,'0'); const day=String(d.getDate()).padStart(2,'0'); return y+'-'+m+'-'+day; })(x) : '';
                    if (state.$inStart) state.$inStart.val(toLocalISO(state.selected)).trigger('change');
                    if (state.$display) updateDisplay(state);
                } else if (state.$input) {
                    state.$input.val(formatDateValue(state.selected, state.opts)).trigger('change');
                }
                if (!opts.inline && opts.autoClose) hideDropdown(state);
                updatePanel(state);
                emit(state, 'changeDate', { value: state.selected });
            }
        });
    }

    function updatePanel(state) {
        const html = renderTemplate(state);
        state.$panel.html(html);
        emit(state, 'render', { current: new Date(state.current), range: !!state.opts.range });
        // After each render mirror the current values into display/hidden inputs
        // (e.g. after clear or external setDate usage)
        (function syncOutputs() {
            const isRange = !!state.opts.range;
            const dispOpts = $.extend({}, state.opts, { format: 'locale' });
            const toLocalISO = (d) => d ? (function(x){ const y=x.getFullYear(); const m=String(x.getMonth()+1).padStart(2,'0'); const day=String(x.getDate()).padStart(2,'0'); return y+'-'+m+'-'+day; })(d) : '';
            if (isRange) {
                // Mirror single-edge ranges into both inputs:
                // - Only start set  -> end mirrors start
                // - Only end set    -> start mirrors end
                // - Both set        -> clamp to [start, end]
                const S = state.rangeStart;
                const E = state.selected;
                let a = S || null, b = E || null;
                if (S && !E) { b = S; }
                else if (!S && E) { a = E; }
                else if (S && E) { const pair = clampRange(S, E); a = pair[0]; b = pair[1]; }

                if (state.$display) updateDisplay(state);
                if (state.$inStart) state.$inStart.val(toLocalISO(a));
                if (state.$inEnd) state.$inEnd.val(toLocalISO(b));
                if (state.$input && !state.$display) state.$input.val(formatDateValue([a, b], state.opts));
            } else {
                const d = state.selected;
                if (state.$display) updateDisplay(state);
                if (state.$inStart) state.$inStart.val(toLocalISO(d));
                if (state.$input && !state.$display) state.$input.val(formatDateValue(d, state.opts));
            }
        })();
        attachEvents(state);
        // If dropdown is visible: set width exactly to content or desired max width
        if (!state.opts.inline && state.$container && state.$container.is(':visible')) {
            applyCalculatedWidth(state);
        }
    }

    function showDropdown(state) {
        if (state.opts.inline) return; // no-op
        const $anchor = state.$anchor || state.$input;
        const off = $anchor.offset();
        const h = $anchor.outerHeight();
        state.$container
            .css({ position: 'absolute', top: off.top + h + 4, left: off.left, zIndex: state.opts.zIndex, width: 'auto' })
            .addClass('show')
            .show();
        // After showing, calculate and set width
        applyCalculatedWidth(state);
        emit(state, 'show', {});
        $(document).on('mousedown.' + NS, function (ev) {
            const $t = $(ev.target);
            if ($t.closest(state.$container).length === 0 && $t.closest($anchor).length === 0) {
                hideDropdown(state);
            }
        });
    }

    // Calculate a sensible max width based on month blocks:
    // - For 1 month: width = width of that month + horizontal padding
    // - For >= 2 months: width = sum of first two months + gap + padding
    function applyCalculatedWidth(state) {
        const $card = state.$panel.children('.card');
        if ($card.length === 0) return;
        const $body = $card.children('.card-body');
        // Padding of the card body (contains the months wrapper)
        let bodyPaddingX = 0;
        if ($body.length) {
            const bs = getComputedStyle($body[0]);
            bodyPaddingX = (parseFloat(bs.paddingLeft) || 0) + (parseFloat(bs.paddingRight) || 0);
        }
        // Months wrapper (first child in body)
        const $wrap = $body.children().first();
        if ($wrap.length === 0) return;
        // Find month elements depending on layout (grid: .col-auto, flex: .d-inline-block)
        let $months = $wrap.children('.col-auto');
        if ($months.length === 0) {
            $months = $wrap.children('.d-inline-block');
        }
        if ($months.length === 0) return;
        const m0 = $months.eq(0).outerWidth(true);
        let monthsWidth = m0;
        if ($months.length >= 2) {
            const m1 = $months.eq(1).outerWidth(true);
            // In Bootstrap grid the spacing is already in the column widths (padding),
            // so no extra gap addition necessary.
            monthsWidth = m0 + m1;
        }
        // Target width: strictly month sum + body padding (header may wrap)
        const targetWidth = monthsWidth + bodyPaddingX;
        // Set card itself to the target width so no inner 100% stretches it
        $card.css('width', Math.ceil(targetWidth + 1) + 'px');
        // Match container exactly to card width
        state.$container.css({ width: Math.ceil(targetWidth + 1) + 'px' });
    }
    function hideDropdown(state) {
        if (state.opts.inline) return; // no-op
        state.$container.removeClass('show').hide();
        // Blur prevents a focus event from immediately reopening the dropdown
        const $anchor = state.$anchor || state.$input;
        if ($anchor && $anchor.length) {
            $anchor.trigger('blur');
        }
        // short suppression to avoid immediate reopen (e.g. click/focus sequences)
        state.suppressOpenUntil = Date.now() + 150;
        $(document).off('mousedown.' + NS);
        emit(state, 'hide', {});
    }

    function create(state) {
        const { opts } = state;

        state.current = startOfDay(new Date());
        state.selected = null;      // single date or range end
        state.rangeStart = null;    // range start

        if (opts.inline) {
            // Render inline directly inside the wrapper
            // Use form-control styling but avoid taking full width: make it inline-block and width auto
            state.$container = $('<div class="bs-datepicker inline form-control p-2 d-inline-block w-auto"></div>').appendTo(state.$root);
        } else {
            // --bs-dropdown-min-width defaults to 10rem → set to auto for content-width dropdowns
            state.$container = $('<div class="bs-datepicker dropdown-menu p-0" style="display:none; --bs-dropdown-min-width:auto;"></div>').appendTo('body');
        }
        state.$panel = $('<div></div>').appendTo(state.$container);
        updatePanel(state);
        // Defensive: ensure dropdown is closed initially
        if (!opts.inline) {
            state.$container.removeClass('show').hide();
        }

        if (!opts.inline) {
            // Open only on click, not on focus (prevents reopen directly after blur)
            const $anchor = state.$anchor || state.$input;
            if ($anchor && $anchor.length) {
                $anchor.on('click.' + NS, function () {
                    if (state.$container.is(':visible')) return;
                    if (state.suppressOpenUntil && Date.now() < state.suppressOpenUntil) return;
                    showDropdown(state);
                });
                // Keyboard support: Enter/Space opens dropdown
                $anchor.on('keydown.' + NS, function (ev) {
                    const key = ev.key || ev.code;
                    if (key === 'Enter' || key === ' ' || key === 'Spacebar') {
                        ev.preventDefault();
                        if (state.$container.is(':visible')) return;
                        if (state.suppressOpenUntil && Date.now() < state.suppressOpenUntil) return;
                        showDropdown(state);
                    }
                });
            }
        }
        emit(state, 'init', { range: !!state.opts.range, inline: !!opts.inline, months: opts.months });
    }

    function destroy(state) {
        hideDropdown(state);
        if (state.$container) state.$container.remove();
        if (state.$anchor) state.$anchor.off('.' + NS);
        if (state.$input) state.$input.off('.' + NS);
        const $dataEl = state.containerMode ? state.$root : state.$input;
        if ($dataEl) $dataEl.removeData(NS);
        $(document).off('.' + NS);
        emit(state, 'destroy', {});
    }

    const methods = {
        getDate() {
            const state = this.data(NS);
            if (!state) return null;
            if (state.opts.range) {
                return [state.rangeStart, state.selected].filter(Boolean);
            }
            return state.selected;
        },
        // jQuery-like val():
        // - Getter (no args):
        //     single -> returns ISO string 'YYYY-MM-DD' or ''
        //     range  -> returns [startISO, endISO] ('' when empty)
        // - Setter:
        //     single -> val(date)
        //     range  -> val(start, end) or val([start, end])
        // Accepts Date|string|null; strings should be ISO 'YYYY-MM-DD'.
        val(a, b) {
            const state = this.data(NS);
            if (!state) return (arguments.length === 0 ? '' : this);

            function toIso(d) {
                if (!d) return '';
                const y = d.getFullYear();
                const m = ('0' + (d.getMonth() + 1)).slice(-2);
                const dd = ('0' + d.getDate()).slice(-2);
                return y + '-' + m + '-' + dd;
            }

            // Getter
            if (arguments.length === 0) {
                if (state.opts.range) {
                    // Prefer hidden inputs if vorhanden (Container-Modus)
                    if (state.$inStart || state.$inEnd) {
                        const s = state.$inStart ? state.$inStart.val() : '';
                        const e = state.$inEnd ? state.$inEnd.val() : '';
                        return [s, e];
                    }
                    // Legacy/Direct: aus internem State als ISO ableiten
                    return [toIso(state.rangeStart), toIso(state.selected)];
                }
                // Single
                if (state.$inStart) {
                    return state.$inStart.val();
                }
                return toIso(state.selected);
            }

            // Setter
            if (state.opts.range) {
                let a1 = a, b1 = b;
                if (Array.isArray(a)) {
                    a1 = a[0];
                    b1 = a[1];
                }
                state.rangeStart = a1 ? toDateOrNull(a1) : null;
                state.selected = b1 ? toDateOrNull(b1) : null;
                // Prevent disabled dates from being set via API
                if (state.rangeStart && isDisabledDate(state.rangeStart, state)) state.rangeStart = null;
                if (state.selected && isDisabledDate(state.selected, state)) state.selected = null;
            } else {
                state.selected = a ? toDateOrNull(a) : null;
                if (state.selected && isDisabledDate(state.selected, state)) state.selected = null;
            }
            updatePanel(state);
            return this;
        },
        setDate(dateOrRange) {
            const state = this.data(NS);
            if (!state) return this;
            if (Array.isArray(dateOrRange)) {
                state.rangeStart = dateOrRange[0] ? startOfDay(new Date(dateOrRange[0])) : null;
                state.selected = dateOrRange[1] ? startOfDay(new Date(dateOrRange[1])) : null;
                // Prevent disabled dates from being set via API
                if (state.rangeStart && isDisabledDate(state.rangeStart, state)) state.rangeStart = null;
                if (state.selected && isDisabledDate(state.selected, state)) state.selected = null;
            } else if (dateOrRange) {
                state.selected = startOfDay(new Date(dateOrRange));
                if (state.selected && isDisabledDate(state.selected, state)) state.selected = null;
            } else {
                state.selected = null; state.rangeStart = null;
            }
            updatePanel(state);
            return this;
        },
        destroy() {
            const state = this.data(NS);
            if (state) destroy(state);
            return this;
        }
    };

    $.fn.bsDatepicker = function (optionsOrMethod) {
        if (this.length === 0) return this;
        if (this.length > 1) {
            return this.each(function () {
                $(this).bsDatepicker(optionsOrMethod);
            });
        }
        const $element = this;
        if (typeof optionsOrMethod === 'string') {
            const method = optionsOrMethod;
            if (methods[method]) {
                return methods[method].apply($element, Array.prototype.slice.call(arguments, 1));
            }
            return $element;
        }
        const opts = $.extend({}, $.bsDatepicker.default, optionsOrMethod || {});

        // Support two modes:
        // 1) Legacy: initialized directly on a visible <input>
        // 2) Container: initialized on a wrapper (e.g. <div class="datepicker">) with 1–2 hidden inputs inside
        const isDirectInput = $element.is('input, textarea');

        // Base state structure
        const state = {
            $input: null,             // Legacy visible input element (direct input)
            $root: $element,          // Original initialization element
            $anchor: null,            // Anchor element for dropdown (input or display wrapper)
            $display: null,           // Visible display wrapper (container mode)
            $displayText: null,       // Text span inside display
            $inStart: null,           // Hidden start (range) or single
            $inEnd: null,             // Hidden end (range)
            containerMode: false,     // true when initialized on wrapper with hidden inputs
            opts,
            $container: null,
            $panel: null,
            current: null,
            selected: null,
            rangeStart: null,
            suppressOpenUntil: 0
        };

        function initBindings() {
            if (isDirectInput) {
                // Legacy usage: keep behavior
                state.$input = $element;
                state.$anchor = state.$input;
                state.containerMode = false;
            } else {
                // Container mode: find inputs inside wrapper
                const $inputs = state.$root.find('input');
                // Prefer hidden, otherwise any inputs
                const $hidden = $inputs.filter('[type="hidden"]');
                const list = $hidden.length ? $hidden : $inputs;
                if (list.length >= 1) state.$inStart = $(list[0]);
                if (list.length >= 2) state.$inEnd = $(list[1]);

                // Derive range from number of inputs
                if (list.length >= 2) state.opts.range = true; else state.opts.range = false;

                // Inline has NO visible display wrapper; calendars are rendered inline
                if (state.opts.inline) {
                    state.$display = null;
                    state.$displayText = null;
                    state.$anchor = null; // no dropdown anchor needed
                    state.containerMode = true;
                } else {
                    // Create visible display wrapper (no input) with text + icon (dropdown only)
                    const cls = state.opts.classes || {};
                    const dispCls = (cls.display || $.bsDatepicker.default.classes.display || '').trim();
                    const textCls = (cls.displayText || $.bsDatepicker.default.classes.displayText || '').trim();
                    const iconCls = (cls.displayIcon || $.bsDatepicker.default.classes.displayIcon || '').trim();
                    state.$display = $('<div role="button" tabindex="0" class="dp-display ' + dispCls + '"></div>');
                    state.$displayText = $('<span class="dp-display-text ' + textCls + '"></span>');
                    const $icon = $('<i class="dp-display-icon ' + iconCls + '"></i>');
                    state.$display.append(state.$displayText).append($icon);
                    state.$root.append(state.$display);
                    state.$anchor = state.$display;
                    state.containerMode = true;
                }
            }
        }

        // No separate setOutputs required, synchronization happens in updatePanel()

        initBindings();

        // Store state on the element (on anchor so method calls keep working)
        (state.containerMode ? state.$root : state.$input).data(NS, state);

        // create() expects state.$input (legacy) – for container only $anchor matters
        if (!state.containerMode) {
            state.$input = state.$anchor;
        }

        // Prepare disabled configuration and create
        state.disabled = normalizeDisabled(state.opts.disabled);
        create(state);

        return (state.containerMode ? state.$root : state.$input);
    };

    // Helper: update the text within the visible display wrapper
    function updateDisplay(state) {
        if (!state.$display) return;
        const isRange = !!state.opts.range;
        const dispOpts = $.extend({}, state.opts, { format: 'locale' });
        let text = '';
        if (isRange) {
            const [a, b] = clampRange(state.rangeStart, state.selected);
            text = formatDateValue([a, b], dispOpts);
        } else {
            text = formatDateValue(state.selected, dispOpts);
        }
        const placeholder = state.opts.placeholder;
        // If no dedicated text element exists (legacy), set text on wrapper as fallback
        if (state.$displayText && state.$displayText.length) {
            state.$displayText.text(text || placeholder);
            state.$display.toggleClass('text-muted', !text);
        } else {
            state.$display.text(text || placeholder);
        }
    }

    // Extend methods: setDisableDates / getDisableDates / setMin / setMax / clearDisableDates / setLocale
    const _old = $.fn.bsDatepicker;
    $.fn.bsDatepicker = function (optionsOrMethod) {
        if (typeof optionsOrMethod === 'string') {
            const args = Array.prototype.slice.call(arguments, 1);
            if (optionsOrMethod === 'setLocale') {
                return this.each(function () {
                    const $el = $(this);
                    const state = $el.data(NS) || $el.find('.dp-display').data(NS) || $el.data(NS);
                    if (!state) return;
                    const newLocale = args[0];
                    if (typeof newLocale === 'string' && newLocale.trim()) {
                        state.opts.locale = newLocale.trim();
                        updatePanel(state);
                        emit(state, 'setLocale', { locale: state.opts.locale });
                    }
                });
            }
            if (optionsOrMethod === 'setDisableDates') {
                return this.each(function () {
                    const $el = $(this);
                    const state = $el.data(NS) || $el.find('.dp-display').data(NS) || $el.data(NS);
                    if (!state) return;
                    state.disabled = normalizeDisabled(args[0] || null);
                    // Purge selection if it became invalid
                    const purge = function (d) { return d && isDisabledDate(d, state) ? null : d; };
                    state.selected = purge(state.selected);
                    state.rangeStart = purge(state.rangeStart);
                    updatePanel(state);
                    emit(state, 'setDisableDates', { disabled: state.disabled });
                });
            }
            if (optionsOrMethod === 'getDisableDates') {
                const $el = this.eq(0);
                const state = $el.data(NS) || $el.find('.dp-display').data(NS) || $el.data(NS);
                return state ? state.disabled : null;
            }
            if (optionsOrMethod === 'setMin') {
                return this.each(function () {
                    const state = $(this).data(NS);
                    if (!state) return;
                    const newMin = toDateOrNull(args[0]);
                    // Preserve all existing disabled rules, including individual dates
                    const cfg = {
                        before: state.disabled ? state.disabled.before : null,
                        after: state.disabled ? state.disabled.after : null,
                        min: newMin,
                        max: state.disabled ? state.disabled.max : null,
                        dates: state.disabled && state.disabled.datesSet ? Array.from(state.disabled.datesSet).map(function (t) { return new Date(t); }) : []
                    };
                    state.disabled = normalizeDisabled(cfg);
                    // Purge now-invalid selections
                    state.selected = (state.selected && isDisabledDate(state.selected, state)) ? null : state.selected;
                    state.rangeStart = (state.rangeStart && isDisabledDate(state.rangeStart, state)) ? null : state.rangeStart;
                    updatePanel(state);
                    emit(state, 'setMin', { min: state.disabled.min || null });
                });
            }
            if (optionsOrMethod === 'setMax') {
                return this.each(function () {
                    const state = $(this).data(NS);
                    if (!state) return;
                    const newMax = toDateOrNull(args[0]);
                    // Preserve all existing disabled rules, including individual dates
                    const cfg = {
                        before: state.disabled ? state.disabled.before : null,
                        after: state.disabled ? state.disabled.after : null,
                        min: state.disabled ? state.disabled.min : null,
                        max: newMax,
                        dates: state.disabled && state.disabled.datesSet ? Array.from(state.disabled.datesSet).map(function (t) { return new Date(t); }) : []
                    };
                    state.disabled = normalizeDisabled(cfg);
                    // Purge now-invalid selections
                    state.selected = (state.selected && isDisabledDate(state.selected, state)) ? null : state.selected;
                    state.rangeStart = (state.rangeStart && isDisabledDate(state.rangeStart, state)) ? null : state.rangeStart;
                    updatePanel(state);
                    emit(state, 'setMax', { max: state.disabled.max || null });
                });
            }
            if (optionsOrMethod === 'clearDisableDates') {
                return this.each(function () {
                    const state = $(this).data(NS);
                    if (!state) return;
                    state.disabled = normalizeDisabled(null);
                    updatePanel(state);
                    emit(state, 'clearDisableDates', {});
                });
            }
        }
        // Fallback: normal initialization
        return _old.apply(this, arguments);
    };

}(jQuery));